﻿// --------------------------------------------------------------------------------
// <copyright>
// Copyright (C)Nintendo. All rights reserved.
//
// These coded instructions, statements, and computer programs contain proprietary
// information of Nintendo and/or its licensed developers and are protected by
// national and international copyright laws. They may not be disclosed to third
// parties or copied or duplicated in any form, in whole or in part, without the
// prior written consent of Nintendo.
//
// The content herein is highly confidential and should be handled accordingly.
// </copyright>
// --------------------------------------------------------------------------------

using System;
using System.Collections.Generic;
using System.Collections.ObjectModel;
using System.ComponentModel;
using System.Linq;
using System.Text;
using System.Threading.Tasks;
using System.Xml;
using nw.g3d.nw4f_3dif;
using Opal.Security.Cryptography;
using Nintendo.ToolFoundation.Collections;
using Nintendo.ToolFoundation.Contracts;
using Nintendo.G3dTool.Entities.Internal;

namespace Nintendo.G3dTool.Entities
{
    public class StreamByte : Stream, IDeepCopyable<StreamByte>, IDeepCopyFrom<StreamByte>
    {
        private readonly ObservableList<Byte> @values = new ObservableList<Byte>();

        /// <summary>
        /// コンストラクタです。
        /// </summary>
        public StreamByte()
        {
            this.@values.CollectionChanged += OnValueCollectionChanged;
            this.@values.CollectionChanged += OnValuesCollectionChanged;
        }

        /// <summary>
        /// コピーコンストラクタです。
        /// </summary>
        /// <param name="source">設定するデータです。</param>
        public StreamByte(StreamByte source)
            : this()
        {
            this.DeepCopyFrom(source);
        }

        /// <summary>
        /// コンストラクタです。
        /// </summary>
        /// <param name="source">設定するデータです。</param>
        public StreamByte(streamType source)
            : base(source)
        {
            this.@values.CollectionChanged += OnValueCollectionChanged;
            this.@values.CollectionChanged += OnValuesCollectionChanged;
            Ensure.Argument.AreEqual(stream_typeType.@byte, source.type);
            {
                var valueArray = G3dDataParser.ParseByteArray(source.Value);
                if (valueArray != null)
                {
                    this.@values.Add(valueArray);
                }
            }
        }

        public StreamByte(IEnumerable<Byte> source)
            : this(source, 1)
        {
        }

        public StreamByte(IEnumerable<Byte> source, int column)
            : this()
        {
            this.Values.Add(source);
            this.Column = column;
        }

        public override stream_typeType Type
        {
            get
            {
                return stream_typeType.@byte;
            }
        }

        public override int Count
        {
            get
            {
                return this.values.Count;
            }
        }

        public ObservableList<Byte> Values
        {
            get
            {
                return this.@values;
            }
        }

        /// <summary>
        /// 出力データを作成します。
        /// </summary>
        /// <returns>出力データのインスタンスを返します。</returns>
        public override streamType CreateSerializableData()
        {
            var writeData = base.CreateSerializableData();
            writeData.@type = this.Type;
            writeData.@count = this.Count;
            G3dDataFormatter.FormatStreamData(writeData, this.Values.ToArray());
            return writeData;
        }

        /// <summary>
        /// 現在のインスタンスをディープコピーで複製した新規インスタンスを返します。
        /// </summary>
        StreamByte IDeepCopyable<StreamByte>.DeepCopy()
        {
            return new StreamByte(this);
        }

        /// <summary>
        /// 現在のインスタンスをディープコピーで複製した新規インスタンスを返します。
        /// </summary>
        private protected override Stream DeepCopy()
        {
            return new StreamByte(this);
        }

        /// <summary>
        /// 入力ファイルからディープコピーします。
        /// </summary>
        /// <param name="source">コピー元となる入力ファイルです。</param>
        public void DeepCopyFrom(StreamByte source)
        {
            base.DeepCopyFrom(source);
            CopyValues(source.@values, this.@values);
        }

        /// <summary>
        /// エンティティの CRC を作成します。（内部処理用）
        /// </summary>
        /// <returns>CRC の値を返します。</returns>
        protected override uint CreateCRCInternal()
        {
            CRC32 crc = new CRC32();
            List<byte> buffers = new List<byte>();
            buffers.AddRange(BitConverter.GetBytes(base.CreateCRCInternal()));
            foreach (var elem in this.Values)
            {
                buffers.AddRange(BitConverter.GetBytes(crc.ComputeHashUInt32(elem)));
            }
            return crc.ComputeHashUInt32(buffers.ToArray());
        }

        private void OnValuesCollectionChanged(object sender, System.Collections.Specialized.NotifyCollectionChangedEventArgs e)
        {
            this.RaisePropertyChanged("Count");
        }
    }
}
