﻿// --------------------------------------------------------------------------------
// <copyright>
// Copyright (C)Nintendo. All rights reserved.
//
// These coded instructions, statements, and computer programs contain proprietary
// information of Nintendo and/or its licensed developers and are protected by
// national and international copyright laws. They may not be disclosed to third
// parties or copied or duplicated in any form, in whole or in part, without the
// prior written consent of Nintendo.
//
// The content herein is highly confidential and should be handled accordingly.
// </copyright>
// --------------------------------------------------------------------------------

using System;
using System.Collections.Generic;
using System.Collections.ObjectModel;
using System.ComponentModel;
using System.Linq;
using System.Text;
using System.Threading.Tasks;
using System.Xml;
using nw.g3d.nw4f_3dif;
using Opal.Security.Cryptography;
using Nintendo.ToolFoundation.Collections;
using Nintendo.ToolFoundation.Contracts;
using Nintendo.G3dTool.Entities.Internal;
using System.Text.RegularExpressions;

namespace Nintendo.G3dTool.Entities
{
    public enum ShaderOptionChoiceType
    {
        Bool,
        IntRange,
        Enum,
    }

    public class ShaderOptionChoice : ObservableEntity<string>, IEquatable<ShaderOptionChoice>, IDeepCopyFrom<ShaderOptionChoice>
    {
        private readonly ObservableList<ChoiceValueString> values = new ObservableList<ChoiceValueString>();
        private ShaderOptionChoiceType type = ShaderOptionChoiceType.Enum;
        private int minValue = 0;
        private int maxValue = 0;

        public ShaderOptionChoice()
        {
            this.values.CollectionChanged += Values_CollectionChanged;
        }

        public ShaderOptionChoice(string source)
            : this()
        {
            this.SetFromShaderAnnotationFormat(source);
        }

        public ShaderOptionChoice(int minValue, int maxValue)
            : this()
        {
            this.SetIntRange(minValue, maxValue);
        }

        public ObservableList<ChoiceValueString> Values
        {
            get
            {
                return this.values;
            }
        }

        public ShaderOptionChoiceType Type
        {
            get
            {
                return this.type;
            }

            set
            {
                this.SetProperty<ShaderOptionChoiceType>(ref this.type, value, () => CalcCRC());
            }
        }

        public int MinValue
        {
            get
            {
                return this.minValue;
            }

            set
            {
                this.SetProperty<int>(ref this.minValue, value, () => CalcCRC());
            }
        }

        public int MaxValue
        {
            get
            {
                return this.maxValue;
            }

            set
            {
                this.SetProperty<int>(ref this.maxValue, value, () => CalcCRC());
            }
        }

        public void SetValue(string value)
        {
            this.values.Clear();
            this.values.Add(new ChoiceValueString(value, null));
            this.type = ShaderOptionChoiceType.Enum;
        }

        private void SetFromShaderAnnotationFormat(string source)
        {
            if (source == "bool")
            {
                this.SetBoolean();
            }
            else if (source.StartsWith("["))
            {
                Ensure.Argument.True(source.EndsWith("]"));

                string[] splited = source.Substring(1, source.Length - 2).Split(new char[] { ',' }, StringSplitOptions.RemoveEmptyEntries);
                Ensure.Argument.True(splited.Length == 2);

                int min, max;
                int.TryParse(splited[0], out min);
                int.TryParse(splited[1], out max);
                this.SetIntRange(min, max);
            }
            else
            {
                string[] splited = source.Split(new char[] { ',' }, StringSplitOptions.RemoveEmptyEntries);
                this.SetEnum(splited);
            }
        }

        public void SetEnum(IEnumerable<string> values)
        {
            this.Values.Clear();
            this.Values.Add(values.Select(x => new ChoiceValueString(x)));
            this.type = ShaderOptionChoiceType.Enum;
        }

        public void SetIntRange(int minValue, int maxValue)
        {
            this.minValue = minValue;
            this.maxValue = maxValue;
            this.type = ShaderOptionChoiceType.IntRange;
        }

        public void SetBoolean()
        {
            this.type = ShaderOptionChoiceType.Bool;
        }

        public bool Equals(ShaderOptionChoice obj)
        {
            if (obj == null)
            {
                return false;
            }

            if (this.Values.Count != obj.Values.Count)
            {
                return false;
            }

            for(int elemIndex = 0; elemIndex < this.Values.Count; ++elemIndex)
            {
                if (this.Values[elemIndex] != obj.Values[elemIndex])
                {
                    return false;
                }
            }

            return true;
        }

        public override string CreateSerializableData()
        {
            switch (this.Type)
            {
                case ShaderOptionChoiceType.Bool:
                    return "bool";
                case ShaderOptionChoiceType.IntRange:
                    Ensure.Operation.True(this.minValue <= this.maxValue, "MaxValue must be larger than or equal to MinValue");
                    return $"[{this.minValue}, {this.maxValue}]";
                case ShaderOptionChoiceType.Enum:
                    if (this.Values.Count == 0)
                    {
                        // アノテーションの仕様と合わせて何も設定されていない場合は bool 扱い
                        return "bool";
                    }

                    return string.Join(",", this.values.Select(x => x.CreateSerializableData()));
                default:
                    throw new Exception("Unexpected default");
            }
        }

        protected override uint CreateCRCInternal()
        {
            CRC32 crc = new CRC32();
            List<byte> buffers = new List<byte>();
            foreach (var elem in this.Values)
            {
                buffers.AddRange(BitConverter.GetBytes(crc.ComputeHashUInt32(elem)));
            }
            buffers.AddRange(BitConverter.GetBytes(this.minValue));
            buffers.AddRange(BitConverter.GetBytes(this.maxValue));
            buffers.AddRange(BitConverter.GetBytes(crc.ComputeHashUInt32(this.type)));
            return crc.ComputeHashUInt32(buffers.ToArray());
        }

        public void DeepCopyFrom(ShaderOptionChoice source)
        {
            CopyElements(source.values, this.values);
            this.minValue = source.minValue;
            this.maxValue = source.maxValue;
            this.type = source.type;
        }

        private void Values_CollectionChanged(object sender, System.Collections.Specialized.NotifyCollectionChangedEventArgs e)
        {
            // コレクション変更があった場合は列挙型とする
            this.type = ShaderOptionChoiceType.Enum;
        }
    }
}
