﻿// --------------------------------------------------------------------------------
// <copyright>
// Copyright (C)Nintendo. All rights reserved.
//
// These coded instructions, statements, and computer programs contain proprietary
// information of Nintendo and/or its licensed developers and are protected by
// national and international copyright laws. They may not be disclosed to third
// parties or copied or duplicated in any form, in whole or in part, without the
// prior written consent of Nintendo.
//
// The content herein is highly confidential and should be handled accordingly.
// </copyright>
// --------------------------------------------------------------------------------

using System;
using System.Collections.Generic;
using System.Collections.ObjectModel;
using System.ComponentModel;
using System.Linq;
using System.Text;
using System.Threading.Tasks;
using System.Xml;
using nw.g3d.nw4f_3dif;
using Opal.Security.Cryptography;
using Nintendo.ToolFoundation.Collections;
using Nintendo.ToolFoundation.Contracts;
using Nintendo.G3dTool.Entities.Internal;

namespace Nintendo.G3dTool.Entities
{
    public class ShaderAssign : ObservableEntity<shader_assignType>, IChildEntity, IDeepCopyable<ShaderAssign>, IDeepCopyFrom<ShaderAssign>
    {
        private readonly ObservableList<RenderInfo> @renderInfos = new ObservableList<RenderInfo>();
        private readonly ObservableList<ShaderOption> @shaderOptions = new ObservableList<ShaderOption>();
        private readonly ObservableList<SamplerAssign> @samplerAssigns = new ObservableList<SamplerAssign>();
        private readonly ObservableList<ShaderParam> @shaderParams = new ObservableList<ShaderParam>();
        private readonly ObservableList<AttribAssign> @attribAssigns = new ObservableList<AttribAssign>();
        private string @shaderArchive = string.Empty;
        private string @shadingModel = string.Empty;
        private int @revision;
        private Material parent = null;

        /// <summary>
        /// コンストラクタです。
        /// </summary>
        public ShaderAssign()
        {
            this.@renderInfos.CollectionChanged += OnCollectionChanged<RenderInfo>;
            this.@shaderOptions.CollectionChanged += OnCollectionChanged<ShaderOption>;
            this.@samplerAssigns.CollectionChanged += OnCollectionChanged<SamplerAssign>;
            this.@shaderParams.CollectionChanged += OnCollectionChanged<ShaderParam>;
            this.@attribAssigns.CollectionChanged += OnCollectionChanged<AttribAssign>;
        }

        /// <summary>
        /// コピーコンストラクタです。
        /// </summary>
        /// <param name="source">設定するデータです。</param>
        public ShaderAssign(ShaderAssign source)
            : this()
        {
            this.DeepCopyFrom(source);
        }

        /// <summary>
        /// コンストラクタです。
        /// </summary>
        /// <param name="source">設定するデータです。</param>
        public ShaderAssign(shader_assignType source)
        {
            this.@renderInfos.CollectionChanged += OnCollectionChanged<RenderInfo>;
            this.@shaderOptions.CollectionChanged += OnCollectionChanged<ShaderOption>;
            this.@samplerAssigns.CollectionChanged += OnCollectionChanged<SamplerAssign>;
            this.@shaderParams.CollectionChanged += OnCollectionChanged<ShaderParam>;
            this.@attribAssigns.CollectionChanged += OnCollectionChanged<AttribAssign>;
            if (source.@render_info_array != null)
            {
                this.@renderInfos.Clear();
                foreach (var elem in source.@render_info_array.Items)
                {
                    this.@renderInfos.Add(DataModelConverter.Convert(elem));
                }
            }
            if (source.@shader_option_array != null)
            {
                this.@shaderOptions.Clear();
                foreach (var elem in source.@shader_option_array.Items)
                {
                    this.@shaderOptions.Add(new ShaderOption(elem));
                }
            }
            if (source.@sampler_assign_array != null)
            {
                this.@samplerAssigns.Clear();
                foreach (var elem in source.@sampler_assign_array.Items)
                {
                    this.@samplerAssigns.Add(new SamplerAssign(elem));
                }
            }
            if (source.@shader_param_array != null)
            {
                this.@shaderParams.Clear();
                foreach (var elem in source.@shader_param_array.Items)
                {
                    this.@shaderParams.Add(DataModelConverter.Convert(elem));
                }
            }
            if (source.@attrib_assign_array != null)
            {
                this.@attribAssigns.Clear();
                foreach (var elem in source.@attrib_assign_array.Items)
                {
                    this.@attribAssigns.Add(new AttribAssign(elem));
                }
            }
            this.@shaderArchive = source.@shader_archive;
            this.@shadingModel = source.@shading_model;
            this.@revision = source.@revision;
        }

        public ObservableList<RenderInfo> RenderInfos
        {
            get
            {
                return this.@renderInfos;
            }
        }

        public ObservableList<ShaderOption> ShaderOptions
        {
            get
            {
                return this.@shaderOptions;
            }
        }

        public ObservableList<SamplerAssign> SamplerAssigns
        {
            get
            {
                return this.@samplerAssigns;
            }
        }

        public ObservableList<ShaderParam> ShaderParams
        {
            get
            {
                return this.@shaderParams;
            }
        }

        public ObservableList<AttribAssign> AttribAssigns
        {
            get
            {
                return this.@attribAssigns;
            }
        }

        public string ShaderArchive
        {
            get
            {
                return this.@shaderArchive;
            }

            set
            {
                if (this.@shaderArchive == value)
                {
                    return;
                }

                Ensure.Argument.NotNull(value);
                this.SetProperty(ref this.@shaderArchive, value, () => this.CalcCRC());
            }
        }

        public string ShadingModel
        {
            get
            {
                return this.@shadingModel;
            }

            set
            {
                if (this.@shadingModel == value)
                {
                    return;
                }

                Ensure.Argument.NotNull(value);
                this.SetProperty(ref this.@shadingModel, value, () => this.CalcCRC());
            }
        }

        public int Revision
        {
            get
            {
                return this.@revision;
            }

            set
            {
                if (this.@revision == value)
                {
                    return;
                }

                this.SetProperty(ref this.@revision, value, () => this.CalcCRC());
            }
        }

        IEntity IChildEntity.Parent
        {
            get
            {
                return this.parent as IEntity;
            }

            set
            {
                if (this.parent == value)
                {
                    return;
                }

                Ensure.Argument.True(value != null ? value is Material : true);
                this.SetProperty(ref this.parent, value as Material, () => this.CalcCRC());
            }
        }

        public Material Parent
        {
            get
            {
                return this.parent;
            }
        }

        Type IChildEntity.ParentType
        {
            get
            {
                return typeof(Material);
            }
        }

        /// <summary>
        /// 出力データを作成します。
        /// </summary>
        /// <returns>出力データのインスタンスを返します。</returns>
        public override shader_assignType CreateSerializableData()
        {
            if (string.IsNullOrEmpty(this.ShaderArchive) || string.IsNullOrEmpty(this.ShadingModel))
            {
                // シェーダー割り当て忘れに書き出し時に気が付けるようにシェーダーパラメーターが存在したら例外を投げる
                Ensure.Operation.AreEqual(0, this.RenderInfos.Count, "RenderInfos is not empty, but ShaderArchive or ShadingModel is empty.");
                Ensure.Operation.AreEqual(0, this.ShaderOptions.Count, "ShaderOptions is not empty, but ShaderArchive or ShadingModel is empty.");
                Ensure.Operation.AreEqual(0, this.SamplerAssigns.Count, "SamplerAssigns is not empty, but ShaderArchive or ShadingModel is empty.");
                Ensure.Operation.AreEqual(0, this.ShaderParams.Count, "ShaderParams is not empty, but ShaderArchive or ShadingModel is empty.");
                Ensure.Operation.AreEqual(0, this.AttribAssigns.Count, "AttribAssigns is not empty, but ShaderArchive or ShadingModel is empty.");
                return null;
            }
            var writeData = new shader_assignType();
            if (this.@RenderInfos.Count > 0)
            {
                writeData.@render_info_array = new render_info_arrayType();
                writeData.@render_info_array.Items = this.@renderInfos.Select(x => x.CreateSerializableData()).Where(x => x != null).ToArray();
                if (writeData.@render_info_array.Items.Length == 0)
                {
                    writeData.@render_info_array = null;
                }
            }
            if (this.@ShaderOptions.Count > 0)
            {
                writeData.@shader_option_array = new shader_option_arrayType();
                writeData.@shader_option_array.Items = this.@shaderOptions.Select(x => x.CreateSerializableData()).Where(x => x != null).ToArray();
                if (writeData.@shader_option_array.Items.Length == 0)
                {
                    writeData.@shader_option_array = null;
                }
            }
            if (this.@SamplerAssigns.Count > 0)
            {
                writeData.@sampler_assign_array = new sampler_assign_arrayType();
                writeData.@sampler_assign_array.Items = this.@samplerAssigns.Select(x => x.CreateSerializableData()).Where(x => x != null).ToArray();
                if (writeData.@sampler_assign_array.Items.Length == 0)
                {
                    writeData.@sampler_assign_array = null;
                }
            }
            if (this.@ShaderParams.Count > 0)
            {
                writeData.@shader_param_array = new shader_param_arrayType();
                writeData.@shader_param_array.Items = this.@shaderParams.Select(x => x.CreateSerializableData()).Where(x => x != null).ToArray();
                if (writeData.@shader_param_array.Items.Length == 0)
                {
                    writeData.@shader_param_array = null;
                }
            }
            if (this.@AttribAssigns.Count > 0)
            {
                writeData.@attrib_assign_array = new attrib_assign_arrayType();
                writeData.@attrib_assign_array.Items = this.@attribAssigns.Select(x => x.CreateSerializableData()).Where(x => x != null).ToArray();
                if (writeData.@attrib_assign_array.Items.Length == 0)
                {
                    writeData.@attrib_assign_array = null;
                }
            }
            writeData.@shader_archive = this.ShaderArchive;
            writeData.@shading_model = this.ShadingModel;
            writeData.@revision = this.Revision;
            return writeData;
        }

        /// <summary>
        /// 現在のインスタンスをディープコピーで複製した新規インスタンスを返します。
        /// </summary>
        ShaderAssign IDeepCopyable<ShaderAssign>.DeepCopy()
        {
            return new ShaderAssign(this);
        }

        /// <summary>
        /// 入力ファイルからディープコピーします。
        /// </summary>
        /// <param name="source">コピー元となる入力ファイルです。</param>
        public void DeepCopyFrom(ShaderAssign source)
        {
            CopyElements(source.@renderInfos, this.@renderInfos);
            CopyElements(source.@shaderOptions, this.@shaderOptions);
            CopyElements(source.@samplerAssigns, this.@samplerAssigns);
            CopyElements(source.@shaderParams, this.@shaderParams);
            CopyElements(source.@attribAssigns, this.@attribAssigns);
            this.@shaderArchive = source.@shaderArchive;
            this.@shadingModel = source.@shadingModel;
            this.@revision = source.@revision;
        }

        /// <summary>
        /// エンティティの CRC を作成します。（内部処理用）
        /// </summary>
        /// <returns>CRC の値を返します。</returns>
        protected override uint CreateCRCInternal()
        {
            CRC32 crc = new CRC32();
            List<byte> buffers = new List<byte>();
            foreach (var elem in this.RenderInfos)
            {
                buffers.AddRange(BitConverter.GetBytes(elem.HashValue));
            }
            foreach (var elem in this.ShaderOptions)
            {
                buffers.AddRange(BitConverter.GetBytes(elem.HashValue));
            }
            foreach (var elem in this.SamplerAssigns)
            {
                buffers.AddRange(BitConverter.GetBytes(elem.HashValue));
            }
            foreach (var elem in this.ShaderParams)
            {
                buffers.AddRange(BitConverter.GetBytes(elem.HashValue));
            }
            foreach (var elem in this.AttribAssigns)
            {
                buffers.AddRange(BitConverter.GetBytes(elem.HashValue));
            }
            buffers.AddRange(BitConverter.GetBytes(crc.ComputeHashUInt32(this.ShaderArchive)));
            buffers.AddRange(BitConverter.GetBytes(crc.ComputeHashUInt32(this.ShadingModel)));
            buffers.AddRange(BitConverter.GetBytes(crc.ComputeHashUInt32(this.Revision)));
            return crc.ComputeHashUInt32(buffers.ToArray());
        }

        /// <summary>
        /// 自動計算フラグを設定します。(内部処理用）
        /// </summary>
        protected override void SetAutoCalcFlagInternal()
        {
            foreach (var elem in this.@renderInfos)
            {
                elem.AutoCalc = this.AutoCalc;
            }
            foreach (var elem in this.@shaderOptions)
            {
                elem.AutoCalc = this.AutoCalc;
            }
            foreach (var elem in this.@samplerAssigns)
            {
                elem.AutoCalc = this.AutoCalc;
            }
            foreach (var elem in this.@shaderParams)
            {
                elem.AutoCalc = this.AutoCalc;
            }
            foreach (var elem in this.@attribAssigns)
            {
                elem.AutoCalc = this.AutoCalc;
            }
        }

        /// <summary>
        /// エンティティの状態をリセットします。(内部処理用）
        /// </summary>
        protected override void ResetInternal()
        {
            foreach (var elem in this.@renderInfos)
            {
                elem.Reset();
            }
            foreach (var elem in this.@shaderOptions)
            {
                elem.Reset();
            }
            foreach (var elem in this.@samplerAssigns)
            {
                elem.Reset();
            }
            foreach (var elem in this.@shaderParams)
            {
                elem.Reset();
            }
            foreach (var elem in this.@attribAssigns)
            {
                elem.Reset();
            }
        }

        /// <summary>
        /// エンティティの状態を更新します。(内部処理用）
        /// </summary>
        protected override void RefreshInternal()
        {
            foreach (var elem in this.@renderInfos)
            {
                elem.Refresh();
            }
            foreach (var elem in this.@shaderOptions)
            {
                elem.Refresh();
            }
            foreach (var elem in this.@samplerAssigns)
            {
                elem.Refresh();
            }
            foreach (var elem in this.@shaderParams)
            {
                elem.Refresh();
            }
            foreach (var elem in this.@attribAssigns)
            {
                elem.Refresh();
            }
        }

        private void OnPropertyChanged(object sender, PropertyChangedEventArgs e)
        {
            if (e.PropertyName == "HashValue")
            {
                this.CalcCRC();
            }
        }
    }
}
