﻿// --------------------------------------------------------------------------------
// <copyright>
// Copyright (C)Nintendo. All rights reserved.
//
// These coded instructions, statements, and computer programs contain proprietary
// information of Nintendo and/or its licensed developers and are protected by
// national and international copyright laws. They may not be disclosed to third
// parties or copied or duplicated in any form, in whole or in part, without the
// prior written consent of Nintendo.
//
// The content herein is highly confidential and should be handled accordingly.
// </copyright>
// --------------------------------------------------------------------------------

using System;
using System.Collections.Generic;
using System.Collections.ObjectModel;
using System.ComponentModel;
using System.Linq;
using System.Text;
using System.Threading.Tasks;
using System.Xml;
using nw.g3d.nw4f_3dif;
using Opal.Security.Cryptography;
using Nintendo.ToolFoundation.Collections;
using Nintendo.ToolFoundation.Contracts;
using Nintendo.G3dTool.Entities.Internal;

namespace Nintendo.G3dTool.Entities
{
    public class SceneAnimInfo : ObservableEntity<scene_anim_infoType>, IDeepCopyable<SceneAnimInfo>, IDeepCopyFrom<SceneAnimInfo>
    {
        private int @frameResolution = 1;
        private string @dccPreset = string.Empty;
        private float @dccMagnify = 1;
        private int @dccStartFrame;
        private int @dccEndFrame;
        private float @dccFps = 1;
        private bool @bakeAll;
        private float @bakeToleranceRotate;
        private float @bakeToleranceTranslate;
        private float @bakeToleranceColor;
        private float @quantizeToleranceRotate;
        private float @quantizeToleranceTranslate;

        /// <summary>
        /// コンストラクタです。
        /// </summary>
        public SceneAnimInfo()
        {
        }

        /// <summary>
        /// コピーコンストラクタです。
        /// </summary>
        /// <param name="source">設定するデータです。</param>
        public SceneAnimInfo(SceneAnimInfo source)
            : this()
        {
            this.DeepCopyFrom(source);
        }

        /// <summary>
        /// コンストラクタです。
        /// </summary>
        /// <param name="source">設定するデータです。</param>
        public SceneAnimInfo(scene_anim_infoType source)
        {
            this.@frameResolution = source.@frame_resolution;
            this.@dccPreset = source.@dcc_preset;
            this.@dccMagnify = source.@dcc_magnify;
            this.@dccStartFrame = source.@dcc_start_frame;
            this.@dccEndFrame = source.@dcc_end_frame;
            this.@dccFps = source.@dcc_fps;
            this.@bakeAll = source.@bake_all;
            this.@bakeToleranceRotate = source.@bake_tolerance_rotate;
            this.@bakeToleranceTranslate = source.@bake_tolerance_translate;
            this.@bakeToleranceColor = source.@bake_tolerance_color;
            this.@quantizeToleranceRotate = source.@quantize_tolerance_rotate;
            this.@quantizeToleranceTranslate = source.@quantize_tolerance_translate;
        }

        public int FrameResolution
        {
            get
            {
                return this.@frameResolution;
            }

            set
            {
                if (this.@frameResolution == value)
                {
                    return;
                }

                this.SetProperty(ref this.@frameResolution, value, () => this.CalcCRC());
            }
        }

        public string DccPreset
        {
            get
            {
                return this.@dccPreset;
            }

            set
            {
                if (this.@dccPreset == value)
                {
                    return;
                }

                Ensure.Argument.NotNull(value);
                this.SetProperty(ref this.@dccPreset, value, () => this.CalcCRC());
            }
        }

        public float DccMagnify
        {
            get
            {
                return this.@dccMagnify;
            }

            set
            {
                if (this.@dccMagnify == value)
                {
                    return;
                }

                this.SetProperty(ref this.@dccMagnify, value, () => this.CalcCRC());
            }
        }

        public int DccStartFrame
        {
            get
            {
                return this.@dccStartFrame;
            }

            set
            {
                if (this.@dccStartFrame == value)
                {
                    return;
                }

                this.SetProperty(ref this.@dccStartFrame, value, () => this.CalcCRC());
            }
        }

        public int DccEndFrame
        {
            get
            {
                return this.@dccEndFrame;
            }

            set
            {
                if (this.@dccEndFrame == value)
                {
                    return;
                }

                this.SetProperty(ref this.@dccEndFrame, value, () => this.CalcCRC());
            }
        }

        public float DccFps
        {
            get
            {
                return this.@dccFps;
            }

            set
            {
                if (this.@dccFps == value)
                {
                    return;
                }

                this.SetProperty(ref this.@dccFps, value, () => this.CalcCRC());
            }
        }

        public bool BakeAll
        {
            get
            {
                return this.@bakeAll;
            }

            set
            {
                if (this.@bakeAll == value)
                {
                    return;
                }

                this.SetProperty(ref this.@bakeAll, value, () => this.CalcCRC());
            }
        }

        public float BakeToleranceRotate
        {
            get
            {
                return this.@bakeToleranceRotate;
            }

            set
            {
                if (this.@bakeToleranceRotate == value)
                {
                    return;
                }

                this.SetProperty(ref this.@bakeToleranceRotate, value, () => this.CalcCRC());
            }
        }

        public float BakeToleranceTranslate
        {
            get
            {
                return this.@bakeToleranceTranslate;
            }

            set
            {
                if (this.@bakeToleranceTranslate == value)
                {
                    return;
                }

                this.SetProperty(ref this.@bakeToleranceTranslate, value, () => this.CalcCRC());
            }
        }

        public float BakeToleranceColor
        {
            get
            {
                return this.@bakeToleranceColor;
            }

            set
            {
                if (this.@bakeToleranceColor == value)
                {
                    return;
                }

                this.SetProperty(ref this.@bakeToleranceColor, value, () => this.CalcCRC());
            }
        }

        public float QuantizeToleranceRotate
        {
            get
            {
                return this.@quantizeToleranceRotate;
            }

            set
            {
                if (this.@quantizeToleranceRotate == value)
                {
                    return;
                }

                this.SetProperty(ref this.@quantizeToleranceRotate, value, () => this.CalcCRC());
            }
        }

        public float QuantizeToleranceTranslate
        {
            get
            {
                return this.@quantizeToleranceTranslate;
            }

            set
            {
                if (this.@quantizeToleranceTranslate == value)
                {
                    return;
                }

                this.SetProperty(ref this.@quantizeToleranceTranslate, value, () => this.CalcCRC());
            }
        }

        /// <summary>
        /// 出力データを作成します。
        /// </summary>
        /// <returns>出力データのインスタンスを返します。</returns>
        public override scene_anim_infoType CreateSerializableData()
        {
            var writeData = new scene_anim_infoType();
            writeData.@frame_resolution = this.FrameResolution;
            writeData.@dcc_preset = this.DccPreset;
            writeData.@dcc_magnify = this.DccMagnify;
            writeData.@dcc_start_frame = this.DccStartFrame;
            writeData.@dcc_end_frame = this.DccEndFrame;
            writeData.@dcc_fps = this.DccFps;
            writeData.@bake_all = this.BakeAll;
            writeData.@bake_tolerance_rotate = this.BakeToleranceRotate;
            writeData.@bake_tolerance_translate = this.BakeToleranceTranslate;
            writeData.@bake_tolerance_color = this.BakeToleranceColor;
            writeData.@quantize_tolerance_rotate = this.QuantizeToleranceRotate;
            writeData.@quantize_tolerance_translate = this.QuantizeToleranceTranslate;
            return writeData;
        }

        /// <summary>
        /// 現在のインスタンスをディープコピーで複製した新規インスタンスを返します。
        /// </summary>
        SceneAnimInfo IDeepCopyable<SceneAnimInfo>.DeepCopy()
        {
            return new SceneAnimInfo(this);
        }

        /// <summary>
        /// 入力ファイルからディープコピーします。
        /// </summary>
        /// <param name="source">コピー元となる入力ファイルです。</param>
        public void DeepCopyFrom(SceneAnimInfo source)
        {
            this.@frameResolution = source.@frameResolution;
            this.@dccPreset = source.@dccPreset;
            this.@dccMagnify = source.@dccMagnify;
            this.@dccStartFrame = source.@dccStartFrame;
            this.@dccEndFrame = source.@dccEndFrame;
            this.@dccFps = source.@dccFps;
            this.@bakeAll = source.@bakeAll;
            this.@bakeToleranceRotate = source.@bakeToleranceRotate;
            this.@bakeToleranceTranslate = source.@bakeToleranceTranslate;
            this.@bakeToleranceColor = source.@bakeToleranceColor;
            this.@quantizeToleranceRotate = source.@quantizeToleranceRotate;
            this.@quantizeToleranceTranslate = source.@quantizeToleranceTranslate;
        }

        /// <summary>
        /// エンティティの CRC を作成します。（内部処理用）
        /// </summary>
        /// <returns>CRC の値を返します。</returns>
        protected override uint CreateCRCInternal()
        {
            CRC32 crc = new CRC32();
            List<byte> buffers = new List<byte>();
            buffers.AddRange(BitConverter.GetBytes(crc.ComputeHashUInt32(this.FrameResolution)));
            buffers.AddRange(BitConverter.GetBytes(crc.ComputeHashUInt32(this.DccPreset)));
            buffers.AddRange(BitConverter.GetBytes(crc.ComputeHashUInt32(this.DccMagnify)));
            buffers.AddRange(BitConverter.GetBytes(crc.ComputeHashUInt32(this.DccStartFrame)));
            buffers.AddRange(BitConverter.GetBytes(crc.ComputeHashUInt32(this.DccEndFrame)));
            buffers.AddRange(BitConverter.GetBytes(crc.ComputeHashUInt32(this.DccFps)));
            buffers.AddRange(BitConverter.GetBytes(crc.ComputeHashUInt32(this.BakeAll)));
            buffers.AddRange(BitConverter.GetBytes(crc.ComputeHashUInt32(this.BakeToleranceRotate)));
            buffers.AddRange(BitConverter.GetBytes(crc.ComputeHashUInt32(this.BakeToleranceTranslate)));
            buffers.AddRange(BitConverter.GetBytes(crc.ComputeHashUInt32(this.BakeToleranceColor)));
            buffers.AddRange(BitConverter.GetBytes(crc.ComputeHashUInt32(this.QuantizeToleranceRotate)));
            buffers.AddRange(BitConverter.GetBytes(crc.ComputeHashUInt32(this.QuantizeToleranceTranslate)));
            return crc.ComputeHashUInt32(buffers.ToArray());
        }
    }
}
