﻿// --------------------------------------------------------------------------------
// <copyright>
// Copyright (C)Nintendo. All rights reserved.
//
// These coded instructions, statements, and computer programs contain proprietary
// information of Nintendo and/or its licensed developers and are protected by
// national and international copyright laws. They may not be disclosed to third
// parties or copied or duplicated in any form, in whole or in part, without the
// prior written consent of Nintendo.
//
// The content herein is highly confidential and should be handled accordingly.
// </copyright>
// --------------------------------------------------------------------------------

using System;
using System.Collections.Generic;
using System.Collections.ObjectModel;
using System.ComponentModel;
using System.Linq;
using System.Text;
using System.Threading.Tasks;
using System.Xml;
using nw.g3d.nw4f_3dif;
using Opal.Security.Cryptography;
using Nintendo.ToolFoundation.Collections;
using Nintendo.ToolFoundation.Contracts;
using Nintendo.G3dTool.Entities.Internal;

namespace Nintendo.G3dTool.Entities
{
    public class Sampler : ObservableEntity<samplerType>, IDeepCopyable<Sampler>, IDeepCopyFrom<Sampler>
    {
        private readonly Wrap @wrap = new Wrap();
        private readonly Filter @filter = new Filter();
        private readonly Lod @lod = new Lod();
        private int @samplerIndex;
        private string @name = string.Empty;
        private string @hint = string.Empty;
        private string @texName = string.Empty;

        /// <summary>
        /// コンストラクタです。
        /// </summary>
        public Sampler()
        {
            this.@wrap.PropertyChanged += this.OnPropertyChanged;
            this.@filter.PropertyChanged += this.OnPropertyChanged;
            this.@lod.PropertyChanged += this.OnPropertyChanged;
        }

        /// <summary>
        /// コピーコンストラクタです。
        /// </summary>
        /// <param name="source">設定するデータです。</param>
        public Sampler(Sampler source)
            : this()
        {
            this.DeepCopyFrom(source);
        }

        /// <summary>
        /// コンストラクタです。
        /// </summary>
        /// <param name="source">設定するデータです。</param>
        public Sampler(samplerType source)
        {
            if (source.@wrap != null)
            {
                this.@wrap = new Wrap(source.@wrap);
            }
            else
            {
                this.@wrap = new Wrap();
            }
            if (source.@filter != null)
            {
                this.@filter = new Filter(source.@filter);
            }
            else
            {
                this.@filter = new Filter();
            }
            if (source.@lod != null)
            {
                this.@lod = new Lod(source.@lod);
            }
            else
            {
                this.@lod = new Lod();
            }
            this.@samplerIndex = source.@sampler_index;
            this.@name = source.@name;
            this.@hint = source.@hint;
            this.@texName = source.@tex_name;
            this.@wrap.PropertyChanged += this.OnPropertyChanged;
            this.@filter.PropertyChanged += this.OnPropertyChanged;
            this.@lod.PropertyChanged += this.OnPropertyChanged;
        }

        public Wrap Wrap
        {
            get
            {
                return this.@wrap;
            }
        }

        public Filter Filter
        {
            get
            {
                return this.@filter;
            }
        }

        public Lod Lod
        {
            get
            {
                return this.@lod;
            }
        }

        public int SamplerIndex
        {
            get
            {
                return this.@samplerIndex;
            }

            set
            {
                if (this.@samplerIndex == value)
                {
                    return;
                }

                this.SetProperty(ref this.@samplerIndex, value, () => this.CalcCRC());
            }
        }

        public string Name
        {
            get
            {
                return this.@name;
            }

            set
            {
                if (this.@name == value)
                {
                    return;
                }

                Ensure.Argument.NotNull(value);
                this.SetProperty(ref this.@name, value, () => this.CalcCRC());
            }
        }

        public string Hint
        {
            get
            {
                return this.@hint;
            }

            set
            {
                if (this.@hint == value)
                {
                    return;
                }

                Ensure.Argument.NotNull(value);
                this.SetProperty(ref this.@hint, value, () => this.CalcCRC());
            }
        }

        public string TexName
        {
            get
            {
                return this.@texName;
            }

            set
            {
                if (this.@texName == value)
                {
                    return;
                }

                Ensure.Argument.NotNull(value);
                this.SetProperty(ref this.@texName, value, () => this.CalcCRC());
            }
        }

        /// <summary>
        /// 出力データを作成します。
        /// </summary>
        /// <returns>出力データのインスタンスを返します。</returns>
        public override samplerType CreateSerializableData()
        {
            var writeData = new samplerType();
            writeData.@wrap = this.@wrap.CreateSerializableData() as wrapType;
            writeData.@filter = this.@filter.CreateSerializableData() as filterType;
            writeData.@lod = this.@lod.CreateSerializableData() as lodType;
            writeData.@sampler_index = this.SamplerIndex;
            writeData.@name = this.Name;
            writeData.@hint = this.Hint;
            writeData.@tex_name = this.TexName;
            return writeData;
        }

        /// <summary>
        /// 現在のインスタンスをディープコピーで複製した新規インスタンスを返します。
        /// </summary>
        Sampler IDeepCopyable<Sampler>.DeepCopy()
        {
            return new Sampler(this);
        }

        /// <summary>
        /// 入力ファイルからディープコピーします。
        /// </summary>
        /// <param name="source">コピー元となる入力ファイルです。</param>
        public void DeepCopyFrom(Sampler source)
        {
            this.@wrap.DeepCopyFrom(source.@wrap);
            this.@filter.DeepCopyFrom(source.@filter);
            this.@lod.DeepCopyFrom(source.@lod);
            this.@samplerIndex = source.@samplerIndex;
            this.@name = source.@name;
            this.@hint = source.@hint;
            this.@texName = source.@texName;
        }

        /// <summary>
        /// エンティティの CRC を作成します。（内部処理用）
        /// </summary>
        /// <returns>CRC の値を返します。</returns>
        protected override uint CreateCRCInternal()
        {
            CRC32 crc = new CRC32();
            List<byte> buffers = new List<byte>();
            buffers.AddRange(BitConverter.GetBytes(crc.ComputeHashUInt32(this.Wrap.HashValue)));
            buffers.AddRange(BitConverter.GetBytes(crc.ComputeHashUInt32(this.Filter.HashValue)));
            buffers.AddRange(BitConverter.GetBytes(crc.ComputeHashUInt32(this.Lod.HashValue)));
            buffers.AddRange(BitConverter.GetBytes(crc.ComputeHashUInt32(this.SamplerIndex)));
            buffers.AddRange(BitConverter.GetBytes(crc.ComputeHashUInt32(this.Name)));
            buffers.AddRange(BitConverter.GetBytes(crc.ComputeHashUInt32(this.Hint)));
            buffers.AddRange(BitConverter.GetBytes(crc.ComputeHashUInt32(this.TexName)));
            return crc.ComputeHashUInt32(buffers.ToArray());
        }

        /// <summary>
        /// 自動計算フラグを設定します。(内部処理用）
        /// </summary>
        protected override void SetAutoCalcFlagInternal()
        {
            this.@wrap.AutoCalc = this.AutoCalc;
            this.@filter.AutoCalc = this.AutoCalc;
            this.@lod.AutoCalc = this.AutoCalc;
        }

        /// <summary>
        /// エンティティの状態をリセットします。(内部処理用）
        /// </summary>
        protected override void ResetInternal()
        {
            this.@wrap.Reset();
            this.@filter.Reset();
            this.@lod.Reset();
        }

        /// <summary>
        /// エンティティの状態を更新します。(内部処理用）
        /// </summary>
        protected override void RefreshInternal()
        {
            this.@wrap.Refresh();
            this.@filter.Refresh();
            this.@lod.Refresh();
        }

        private void OnPropertyChanged(object sender, PropertyChangedEventArgs e)
        {
            if (e.PropertyName == "HashValue")
            {
                this.CalcCRC();
            }
        }
    }
}
