﻿// --------------------------------------------------------------------------------
// <copyright>
// Copyright (C)Nintendo. All rights reserved.
//
// These coded instructions, statements, and computer programs contain proprietary
// information of Nintendo and/or its licensed developers and are protected by
// national and international copyright laws. They may not be disclosed to third
// parties or copied or duplicated in any form, in whole or in part, without the
// prior written consent of Nintendo.
//
// The content herein is highly confidential and should be handled accordingly.
// </copyright>
// --------------------------------------------------------------------------------

using System;
using System.Collections.Generic;
using System.Collections.ObjectModel;
using System.ComponentModel;
using System.Linq;
using System.Text;
using System.Threading.Tasks;
using System.Xml;
using nw.g3d.nw4f_3dif;
using Opal.Security.Cryptography;
using Nintendo.ToolFoundation.Collections;
using Nintendo.ToolFoundation.Contracts;
using Nintendo.G3dTool.Entities.Internal;

namespace Nintendo.G3dTool.Entities
{
    public class RenderInfoSlotInt : RenderInfoSlot, IDeepCopyable<RenderInfoSlotInt>, IDeepCopyFrom<RenderInfoSlotInt>
    {
        private readonly RenderInfoIntChoice @choice = new RenderInfoIntChoice();
        private readonly ObservableList<int> @default = new ObservableList<int>();

        /// <summary>
        /// コンストラクタです。
        /// </summary>
        public RenderInfoSlotInt()
        {
            this.@choice.PropertyChanged += this.OnPropertyChanged;
            this.@default.CollectionChanged += OnValueCollectionChanged;
        }

        /// <summary>
        /// コピーコンストラクタです。
        /// </summary>
        /// <param name="source">設定するデータです。</param>
        public RenderInfoSlotInt(RenderInfoSlotInt source)
            : this()
        {
            this.DeepCopyFrom(source);
        }

        /// <summary>
        /// コンストラクタです。
        /// </summary>
        /// <param name="source">設定するデータです。</param>
        public RenderInfoSlotInt(render_info_slotType source)
            : base(source)
        {
            this.@default.CollectionChanged += OnValueCollectionChanged;
            Ensure.Argument.AreEqual(render_info_slot_typeType.@int, source.type);
            this.@choice = new RenderInfoIntChoice(source.choice);
            if (source.count > 0)
            {
                this.Default.Add(StringUtility.ConvertTextToValueArray<int>(source.@default));
            }
            this.@choice.PropertyChanged += this.OnPropertyChanged;
        }

        public override render_info_slot_typeType Type
        {
            get
            {
                return render_info_slot_typeType.@int;
            }
        }

        public RenderInfoIntChoice Choice
        {
            get
            {
                return this.@choice;
            }
        }

        public ObservableList<int> Default
        {
            get
            {
                return this.@default;
            }
        }

        /// <summary>
        /// 出力データを作成します。
        /// </summary>
        /// <returns>出力データのインスタンスを返します。</returns>
        public override render_info_slotType CreateSerializableData()
        {
            var writeData = base.CreateSerializableData();
            writeData.@type = this.Type;
            writeData.@choice = this.@choice.CreateSerializableData() as String;
            if (this.Count > 0)
            {
                StringBuilder valuesAsText = new StringBuilder();
                foreach (var value in this.Default)
                {
                    valuesAsText.Append($"{value} ");
                }
                writeData.@default = valuesAsText.ToString().Trim();
            }
            else
            {
                writeData.@default = null;
            }
            return writeData;
        }

        /// <summary>
        /// 現在のインスタンスをディープコピーで複製した新規インスタンスを返します。
        /// </summary>
        RenderInfoSlotInt IDeepCopyable<RenderInfoSlotInt>.DeepCopy()
        {
            return new RenderInfoSlotInt(this);
        }

        /// <summary>
        /// 現在のインスタンスをディープコピーで複製した新規インスタンスを返します。
        /// </summary>
        private protected override RenderInfoSlot DeepCopy()
        {
            return new RenderInfoSlotInt(this);
        }

        /// <summary>
        /// 入力ファイルからディープコピーします。
        /// </summary>
        /// <param name="source">コピー元となる入力ファイルです。</param>
        public void DeepCopyFrom(RenderInfoSlotInt source)
        {
            base.DeepCopyFrom(source);
            this.@choice.DeepCopyFrom(source.@choice);
            CopyValues(source.@default, this.@default);
        }

        /// <summary>
        /// エンティティの CRC を作成します。（内部処理用）
        /// </summary>
        /// <returns>CRC の値を返します。</returns>
        protected override uint CreateCRCInternal()
        {
            CRC32 crc = new CRC32();
            List<byte> buffers = new List<byte>();
            buffers.AddRange(BitConverter.GetBytes(base.CreateCRCInternal()));
            buffers.AddRange(BitConverter.GetBytes(crc.ComputeHashUInt32(this.Choice.HashValue)));
            foreach (var elem in this.Default)
            {
                buffers.AddRange(BitConverter.GetBytes(crc.ComputeHashUInt32(elem)));
            }
            return crc.ComputeHashUInt32(buffers.ToArray());
        }

        /// <summary>
        /// 自動計算フラグを設定します。(内部処理用）
        /// </summary>
        protected override void SetAutoCalcFlagInternal()
        {
            base.SetAutoCalcFlagInternal();
            this.@choice.AutoCalc = this.AutoCalc;
        }

        /// <summary>
        /// エンティティの状態をリセットします。(内部処理用）
        /// </summary>
        protected override void ResetInternal()
        {
            base.ResetInternal();
            this.@choice.Reset();
        }

        /// <summary>
        /// エンティティの状態を更新します。(内部処理用）
        /// </summary>
        protected override void RefreshInternal()
        {
            base.RefreshInternal();
            this.@choice.Refresh();
        }

        private void OnPropertyChanged(object sender, PropertyChangedEventArgs e)
        {
            if (e.PropertyName == "HashValue")
            {
                this.CalcCRC();
            }
        }
    }
}
