﻿// --------------------------------------------------------------------------------
// <copyright>
// Copyright (C)Nintendo. All rights reserved.
//
// These coded instructions, statements, and computer programs contain proprietary
// information of Nintendo and/or its licensed developers and are protected by
// national and international copyright laws. They may not be disclosed to third
// parties or copied or duplicated in any form, in whole or in part, without the
// prior written consent of Nintendo.
//
// The content herein is highly confidential and should be handled accordingly.
// </copyright>
// --------------------------------------------------------------------------------

using System;
using System.Collections.Generic;
using System.Collections.ObjectModel;
using System.ComponentModel;
using System.Linq;
using System.Text;
using System.Threading.Tasks;
using System.Xml;
using nw.g3d.nw4f_3dif;
using Opal.Security.Cryptography;
using Nintendo.ToolFoundation.Collections;
using Nintendo.ToolFoundation.Contracts;
using Nintendo.G3dTool.Entities.Internal;

namespace Nintendo.G3dTool.Entities
{
    public class PatternAnim : ObservableEntity<pattern_anim_targetType>, IChildEntity, IAnimTarget, IDeepCopyable<PatternAnim>, IDeepCopyFrom<PatternAnim>, IDisposable
    {
        private StepCurve @curve = null;
        private string @samplerName = string.Empty;
        private string @hint = string.Empty;
        private float @baseValue;
        private PerMaterialAnim parent = null;

        /// <summary>
        /// コンストラクタです。
        /// </summary>
        public PatternAnim()
        {
        }

        /// <summary>
        /// コピーコンストラクタです。
        /// </summary>
        /// <param name="source">設定するデータです。</param>
        public PatternAnim(PatternAnim source)
            : this()
        {
            this.DeepCopyFrom(source);
        }

        /// <summary>
        /// コンストラクタです。
        /// </summary>
        /// <param name="source">設定するデータです。</param>
        public PatternAnim(pattern_anim_targetType source)
        {
            if (source.step_curve != null)
            {
                this.Curve = DataModelConverter.ConvertCurve(source.step_curve) as ICurve;
            }
            else
            {
                this.Curve = null;
            }
            this.@samplerName = source.@sampler_name;
            this.@hint = source.@hint;
            this.@baseValue = source.@base_value;
        }

        public void Dispose()
        {
            if (this.@curve != null)
            {
                this.@curve.PropertyChanged -= this.OnPropertyChanged;
            }
        }

        public ICurve Curve
        {
            get
            {
                return this.@curve as ICurve;
            }

            set
            {
                if (this.@curve == value)
                {
                    return;
                }

                Ensure.Argument.True(value != null ? typeof(StepCurve).IsAssignableFrom(value.GetType()) : true);
                if (this.@curve != null)
                {
                    (this.@curve as IChildEntity).Parent = null;
                }

                if (value != null)
                {
                    (value as IChildEntity).Parent = this;
                }

                if (this.@curve != null)
                {
                    this.@curve.PropertyChanged -= this.OnPropertyChanged;
                }

                if (value != null)
                {
                    value.PropertyChanged += this.OnPropertyChanged;
                }

                this.SetProperty(ref this.@curve, value as StepCurve, () => this.CalcCRC());
            }
        }

        public string SamplerName
        {
            get
            {
                return this.@samplerName;
            }

            set
            {
                if (this.@samplerName == value)
                {
                    return;
                }

                Ensure.Argument.NotNull(value);
                this.SetProperty(ref this.@samplerName, value, () => this.CalcCRC());
            }
        }

        public string Hint
        {
            get
            {
                return this.@hint;
            }

            set
            {
                if (this.@hint == value)
                {
                    return;
                }

                Ensure.Argument.NotNull(value);
                this.SetProperty(ref this.@hint, value, () => this.CalcCRC());
            }
        }

        public float BaseValue
        {
            get
            {
                return this.@baseValue;
            }

            set
            {
                if (this.@baseValue == value)
                {
                    return;
                }

                this.SetProperty(ref this.@baseValue, value, () => this.CalcCRC());
            }
        }

        IEntity IChildEntity.Parent
        {
            get
            {
                return this.parent as IEntity;
            }

            set
            {
                if (this.parent == value)
                {
                    return;
                }

                Ensure.Argument.True(value != null ? value is PerMaterialAnim : true);
                this.SetProperty(ref this.parent, value as PerMaterialAnim, () => this.CalcCRC());
            }
        }

        public PerMaterialAnim Parent
        {
            get
            {
                return this.parent;
            }
        }

        Type IChildEntity.ParentType
        {
            get
            {
                return typeof(PerMaterialAnim);
            }
        }

        /// <summary>
        /// 出力データを作成します。
        /// </summary>
        /// <returns>出力データのインスタンスを返します。</returns>
        public override pattern_anim_targetType CreateSerializableData()
        {
            var writeData = new pattern_anim_targetType();
            if (this.@curve != null)
            {
                writeData.@step_curve = this.Curve.CreateSerializableData() as step_curveType;
            }
            writeData.@sampler_name = this.SamplerName;
            writeData.@hint = this.Hint;
            writeData.@base_value = this.BaseValue;
            return writeData;
        }

        /// <summary>
        /// 現在のインスタンスをディープコピーで複製した新規インスタンスを返します。
        /// </summary>
        PatternAnim IDeepCopyable<PatternAnim>.DeepCopy()
        {
            return new PatternAnim(this);
        }

        /// <summary>
        /// 入力ファイルからディープコピーします。
        /// </summary>
        /// <param name="source">コピー元となる入力ファイルです。</param>
        public void DeepCopyFrom(PatternAnim source)
        {
            if (source.@curve == null)
            {
                this.Curve = null;
            }
            else
            {
                this.Curve.DeepCopyFrom(source.@curve);
            }
            this.@samplerName = source.@samplerName;
            this.@hint = source.@hint;
            this.@baseValue = source.@baseValue;
        }

        /// <summary>
        /// エンティティの CRC を作成します。（内部処理用）
        /// </summary>
        /// <returns>CRC の値を返します。</returns>
        protected override uint CreateCRCInternal()
        {
            CRC32 crc = new CRC32();
            List<byte> buffers = new List<byte>();
            if (this.Curve != null)
            {
                buffers.AddRange(BitConverter.GetBytes(crc.ComputeHashUInt32(this.Curve.HashValue)));
            }
            buffers.AddRange(BitConverter.GetBytes(crc.ComputeHashUInt32(this.SamplerName)));
            buffers.AddRange(BitConverter.GetBytes(crc.ComputeHashUInt32(this.Hint)));
            buffers.AddRange(BitConverter.GetBytes(crc.ComputeHashUInt32(this.BaseValue)));
            return crc.ComputeHashUInt32(buffers.ToArray());
        }

        /// <summary>
        /// 自動計算フラグを設定します。(内部処理用）
        /// </summary>
        protected override void SetAutoCalcFlagInternal()
        {
            if (this.@curve != null)
            {
                this.@curve.AutoCalc = this.AutoCalc;
            }
        }

        /// <summary>
        /// エンティティの状態をリセットします。(内部処理用）
        /// </summary>
        protected override void ResetInternal()
        {
            if (this.@curve != null)
            {
                this.@curve.Reset();
            }
        }

        /// <summary>
        /// エンティティの状態を更新します。(内部処理用）
        /// </summary>
        protected override void RefreshInternal()
        {
            if (this.@curve != null)
            {
                this.@curve.Refresh();
            }
        }

        private void OnPropertyChanged(object sender, PropertyChangedEventArgs e)
        {
            if (e.PropertyName == "HashValue")
            {
                this.CalcCRC();
            }
        }
    }
}
