﻿// --------------------------------------------------------------------------------
// <copyright>
// Copyright (C)Nintendo. All rights reserved.
//
// These coded instructions, statements, and computer programs contain proprietary
// information of Nintendo and/or its licensed developers and are protected by
// national and international copyright laws. They may not be disclosed to third
// parties or copied or duplicated in any form, in whole or in part, without the
// prior written consent of Nintendo.
//
// The content herein is highly confidential and should be handled accordingly.
// </copyright>
// --------------------------------------------------------------------------------

using System;
using System.Collections.Generic;
using System.Collections.ObjectModel;
using System.ComponentModel;
using System.Linq;
using System.Text;
using System.Threading.Tasks;
using System.Xml;
using nw.g3d.nw4f_3dif;
using Opal.Security.Cryptography;
using Nintendo.ToolFoundation.Collections;
using Nintendo.ToolFoundation.Contracts;
using Nintendo.G3dTool.Entities.Internal;

namespace Nintendo.G3dTool.Entities
{
    public class Mesh : ObservableEntity<meshType>, IChildEntity, IStreamIndex, IDeepCopyable<Mesh>, IDeepCopyFrom<Mesh>
    {
        private readonly ObservableList<Submesh> @submeshes = new ObservableList<Submesh>();
        private mesh_modeType @mode = mesh_modeType.triangles;
        private mesh_quantize_typeType @quantizeType = mesh_quantize_typeType.none;
        private int @count;
        private int @streamIndex;
        private Shape parent = null;
        private StreamInt @indexStream = null;

        /// <summary>
        /// コンストラクタです。
        /// </summary>
        public Mesh()
        {
            this.@submeshes.CollectionChanged += OnCollectionChanged<Submesh>;
        }

        /// <summary>
        /// コピーコンストラクタです。
        /// </summary>
        /// <param name="source">設定するデータです。</param>
        public Mesh(Mesh source)
            : this()
        {
            this.DeepCopyFrom(source);
        }

        /// <summary>
        /// コンストラクタです。
        /// </summary>
        /// <param name="source">設定するデータです。</param>
        public Mesh(meshType source)
        {
            this.@submeshes.CollectionChanged += OnCollectionChanged<Submesh>;
            if (source.@submesh_array != null)
            {
                this.@submeshes.Clear();
                foreach (var elem in source.@submesh_array.Items)
                {
                    this.@submeshes.Add(new Submesh(elem));
                }
            }
            this.@mode = source.@mode;
            this.@quantizeType = source.@quantize_type;
            this.@count = source.@count;
            this.@streamIndex = source.@stream_index;
        }

        public ObservableList<Submesh> Submeshes
        {
            get
            {
                return this.@submeshes;
            }
        }

        public mesh_modeType Mode
        {
            get
            {
                return this.@mode;
            }

            set
            {
                if (this.@mode == value)
                {
                    return;
                }

                this.SetProperty(ref this.@mode, value, () => this.CalcCRC());
            }
        }

        public mesh_quantize_typeType QuantizeType
        {
            get
            {
                return this.@quantizeType;
            }

            set
            {
                if (this.@quantizeType == value)
                {
                    return;
                }

                this.SetProperty(ref this.@quantizeType, value, () => this.CalcCRC());
            }
        }

        public int Count
        {
            get
            {
                if (this.indexStream != null)
                {
                    return this.indexStream.Values.Count;
                }
                return this.@count;
            }
        }

        int IStreamIndex.StreamIndex
        {
            get
            {
                if (this.IndexStream != null)
                {
                    // 親が見つからない場合はインデックスを求められないので -1 を返す
                    IRootEntity rootEntity = Utility.FindRootEntity((this as IChildEntity).Parent);
                    if (rootEntity == null)
                    {
                        return -1;
                    }

                    var sourceProp = rootEntity.GetType().GetProperty("Streams").GetValue(rootEntity) as ObservableList<Stream>;
                    return sourceProp.IndexOf(this.IndexStream);
                }
                return this.@streamIndex;
            }
        }

        IEntity IChildEntity.Parent
        {
            get
            {
                return this.parent as IEntity;
            }

            set
            {
                if (this.parent == value)
                {
                    return;
                }

                Ensure.Argument.True(value != null ? value is Shape : true);
                this.SetProperty(ref this.parent, value as Shape, () => this.CalcCRC());
            }
        }

        public Shape Parent
        {
            get
            {
                return this.parent;
            }
        }

        Type IChildEntity.ParentType
        {
            get
            {
                return typeof(Shape);
            }
        }

        public StreamInt IndexStream
        {
            get
            {
                return this.@indexStream;
            }

            set
            {
                if (this.@indexStream == value)
                {
                    return;
                }

                Ensure.Argument.NotNull(value);
                this.SetProperty(ref this.@indexStream, value, () => this.CalcCRC());
            }
        }

        /// <summary>
        /// 出力データを作成します。
        /// </summary>
        /// <returns>出力データのインスタンスを返します。</returns>
        public override meshType CreateSerializableData()
        {
            var writeData = new meshType();
            writeData.submesh_array = new submesh_arrayType();
            if (this.@submeshes.Count > 0)
            {
                writeData.submesh_array.Items = this.@submeshes.Select(x => x.CreateSerializableData()).ToArray();
            }
            else
            {
                // ひとつもないときはひとつ作成
                writeData.submesh_array.Items = new submeshType[]
                {
                    new submeshType()
                    {
                        offset = 0,
                        count = this.IndexStream != null ? this.IndexStream.Values.Count : this.Count,
                    }
                };
            }
            writeData.@mode = this.Mode;
            writeData.@quantize_type = this.QuantizeType;
            writeData.@count = this.Count;
            {
                IRootEntity rootEntity = Utility.FindRootEntity((this as IChildEntity).Parent);
                Ensure.Operation.NotNull(rootEntity);
                var sourceProp = rootEntity.GetType().GetProperty("Streams").GetValue(rootEntity) as ObservableList<Stream>;
                // 読み取り高速化のために不要なデータを読み飛ばす場合があるので、参照先が存在しない場合を許容します
                Ensure.Operation.True(indexStream != null ? sourceProp.Contains(IndexStream) : sourceProp.Count == 0);
                writeData.stream_index = (this as IStreamIndex).StreamIndex;
            }
            for (int submeshIndex = 0; submeshIndex < writeData.submesh_array.Items.Length; ++submeshIndex)
            {
                var submesh = writeData.submesh_array.Items[submeshIndex];
                submesh.submesh_index = submeshIndex;
            }
            return writeData;
        }

        /// <summary>
        /// 現在のインスタンスをディープコピーで複製した新規インスタンスを返します。
        /// </summary>
        Mesh IDeepCopyable<Mesh>.DeepCopy()
        {
            return new Mesh(this);
        }

        /// <summary>
        /// 入力ファイルからディープコピーします。
        /// </summary>
        /// <param name="source">コピー元となる入力ファイルです。</param>
        public void DeepCopyFrom(Mesh source)
        {
            CopyElements(source.@submeshes, this.@submeshes);
            this.@mode = source.@mode;
            this.@quantizeType = source.@quantizeType;
            this.@count = source.@count;
            this.@streamIndex = source.@streamIndex;
        }

        /// <summary>
        /// エンティティの CRC を作成します。（内部処理用）
        /// </summary>
        /// <returns>CRC の値を返します。</returns>
        protected override uint CreateCRCInternal()
        {
            CRC32 crc = new CRC32();
            List<byte> buffers = new List<byte>();
            foreach (var elem in this.Submeshes)
            {
                buffers.AddRange(BitConverter.GetBytes(elem.HashValue));
            }
            buffers.AddRange(BitConverter.GetBytes(crc.ComputeHashUInt32(this.Mode)));
            buffers.AddRange(BitConverter.GetBytes(crc.ComputeHashUInt32(this.QuantizeType)));
            buffers.AddRange(BitConverter.GetBytes(crc.ComputeHashUInt32(this.Count)));
            buffers.AddRange(BitConverter.GetBytes(crc.ComputeHashUInt32((this as IStreamIndex).StreamIndex)));
            return crc.ComputeHashUInt32(buffers.ToArray());
        }

        /// <summary>
        /// 自動計算フラグを設定します。(内部処理用）
        /// </summary>
        protected override void SetAutoCalcFlagInternal()
        {
            foreach (var elem in this.@submeshes)
            {
                elem.AutoCalc = this.AutoCalc;
            }
        }

        /// <summary>
        /// エンティティの状態をリセットします。(内部処理用）
        /// </summary>
        protected override void ResetInternal()
        {
            foreach (var elem in this.@submeshes)
            {
                elem.Reset();
            }
        }

        /// <summary>
        /// エンティティの状態を更新します。(内部処理用）
        /// </summary>
        protected override void RefreshInternal()
        {
            foreach (var elem in this.@submeshes)
            {
                elem.Refresh();
            }
        }

        private void OnPropertyChanged(object sender, PropertyChangedEventArgs e)
        {
            if (e.PropertyName == "HashValue")
            {
                this.CalcCRC();
            }
        }
    }
}
