﻿// --------------------------------------------------------------------------------
// <copyright>
// Copyright (C)Nintendo. All rights reserved.
//
// These coded instructions, statements, and computer programs contain proprietary
// information of Nintendo and/or its licensed developers and are protected by
// national and international copyright laws. They may not be disclosed to third
// parties or copied or duplicated in any form, in whole or in part, without the
// prior written consent of Nintendo.
//
// The content herein is highly confidential and should be handled accordingly.
// </copyright>
// --------------------------------------------------------------------------------

using System;
using System.Collections.Generic;
using System.Collections.ObjectModel;
using System.ComponentModel;
using System.Linq;
using System.Text;
using System.Threading.Tasks;
using System.Xml;
using nw.g3d.nw4f_3dif;
using Opal.Security.Cryptography;
using Nintendo.ToolFoundation.Collections;
using Nintendo.ToolFoundation.Contracts;
using Nintendo.G3dTool.Entities.Internal;

namespace Nintendo.G3dTool.Entities
{
    public class Filter : ObservableEntity<filterType>, IDeepCopyable<Filter>, IDeepCopyFrom<Filter>
    {
        private filter_mag_minType @mag = filter_mag_minType.linear;
        private filter_mag_minType @min = filter_mag_minType.linear;
        private filter_mipType @mip = filter_mipType.linear;
        private filter_max_anisoType @maxAniso = filter_max_anisoType.aniso_1;

        /// <summary>
        /// コンストラクタです。
        /// </summary>
        public Filter()
        {
        }

        /// <summary>
        /// コピーコンストラクタです。
        /// </summary>
        /// <param name="source">設定するデータです。</param>
        public Filter(Filter source)
            : this()
        {
            this.DeepCopyFrom(source);
        }

        /// <summary>
        /// コンストラクタです。
        /// </summary>
        /// <param name="source">設定するデータです。</param>
        public Filter(filterType source)
        {
            this.@mag = source.@mag;
            this.@min = source.@min;
            this.@mip = source.@mip;
            this.@maxAniso = source.@max_aniso;
        }

        public filter_mag_minType Mag
        {
            get
            {
                return this.@mag;
            }

            set
            {
                if (this.@mag == value)
                {
                    return;
                }

                this.SetProperty(ref this.@mag, value, () => this.CalcCRC());
            }
        }

        public filter_mag_minType Min
        {
            get
            {
                return this.@min;
            }

            set
            {
                if (this.@min == value)
                {
                    return;
                }

                this.SetProperty(ref this.@min, value, () => this.CalcCRC());
            }
        }

        public filter_mipType Mip
        {
            get
            {
                return this.@mip;
            }

            set
            {
                if (this.@mip == value)
                {
                    return;
                }

                this.SetProperty(ref this.@mip, value, () => this.CalcCRC());
            }
        }

        public filter_max_anisoType MaxAniso
        {
            get
            {
                return this.@maxAniso;
            }

            set
            {
                if (this.@maxAniso == value)
                {
                    return;
                }

                this.SetProperty(ref this.@maxAniso, value, () => this.CalcCRC());
            }
        }

        /// <summary>
        /// 出力データを作成します。
        /// </summary>
        /// <returns>出力データのインスタンスを返します。</returns>
        public override filterType CreateSerializableData()
        {
            var writeData = new filterType();
            writeData.@mag = this.Mag;
            writeData.@min = this.Min;
            writeData.@mip = this.Mip;
            writeData.@max_aniso = this.MaxAniso;
            return writeData;
        }

        /// <summary>
        /// 現在のインスタンスをディープコピーで複製した新規インスタンスを返します。
        /// </summary>
        Filter IDeepCopyable<Filter>.DeepCopy()
        {
            return new Filter(this);
        }

        /// <summary>
        /// 入力ファイルからディープコピーします。
        /// </summary>
        /// <param name="source">コピー元となる入力ファイルです。</param>
        public void DeepCopyFrom(Filter source)
        {
            this.@mag = source.@mag;
            this.@min = source.@min;
            this.@mip = source.@mip;
            this.@maxAniso = source.@maxAniso;
        }

        /// <summary>
        /// エンティティの CRC を作成します。（内部処理用）
        /// </summary>
        /// <returns>CRC の値を返します。</returns>
        protected override uint CreateCRCInternal()
        {
            CRC32 crc = new CRC32();
            List<byte> buffers = new List<byte>();
            buffers.AddRange(BitConverter.GetBytes(crc.ComputeHashUInt32(this.Mag)));
            buffers.AddRange(BitConverter.GetBytes(crc.ComputeHashUInt32(this.Min)));
            buffers.AddRange(BitConverter.GetBytes(crc.ComputeHashUInt32(this.Mip)));
            buffers.AddRange(BitConverter.GetBytes(crc.ComputeHashUInt32(this.MaxAniso)));
            return crc.ComputeHashUInt32(buffers.ToArray());
        }
    }
}
