﻿// --------------------------------------------------------------------------------
// <copyright>
// Copyright (C)Nintendo. All rights reserved.
//
// These coded instructions, statements, and computer programs contain proprietary
// information of Nintendo and/or its licensed developers and are protected by
// national and international copyright laws. They may not be disclosed to third
// parties or copied or duplicated in any form, in whole or in part, without the
// prior written consent of Nintendo.
//
// The content herein is highly confidential and should be handled accordingly.
// </copyright>
// --------------------------------------------------------------------------------

using System;
using System.Collections.Generic;
using System.Collections.ObjectModel;
using System.ComponentModel;
using System.Linq;
using System.Text;
using System.Threading.Tasks;
using System.Xml;
using nw.g3d.nw4f_3dif;
using Opal.Security.Cryptography;
using Nintendo.ToolFoundation.Collections;
using Nintendo.ToolFoundation.Contracts;
using Nintendo.G3dTool.Entities.Internal;

namespace Nintendo.G3dTool.Entities
{
    public class FileInfo : ObservableEntity<file_infoType>, IDeepCopyable<FileInfo>, IDeepCopyFrom<FileInfo>
    {
        private readonly Create @create = new Create();
        private readonly Modify @modify = new Modify();

        /// <summary>
        /// コンストラクタです。
        /// </summary>
        public FileInfo()
        {
            this.@create.PropertyChanged += this.OnPropertyChanged;
            this.@modify.PropertyChanged += this.OnPropertyChanged;
        }

        /// <summary>
        /// コピーコンストラクタです。
        /// </summary>
        /// <param name="source">設定するデータです。</param>
        public FileInfo(FileInfo source)
            : this()
        {
            this.DeepCopyFrom(source);
        }

        /// <summary>
        /// コンストラクタです。
        /// </summary>
        /// <param name="source">設定するデータです。</param>
        public FileInfo(file_infoType source)
        {
            if (source.@create != null)
            {
                this.@create = new Create(source.@create);
            }
            else
            {
                this.@create = new Create();
            }
            if (source.@modify != null)
            {
                this.@modify = new Modify(source.@modify);
            }
            else
            {
                this.@modify = new Modify();
            }
            this.@create.PropertyChanged += this.OnPropertyChanged;
            this.@modify.PropertyChanged += this.OnPropertyChanged;
        }

        public Create Create
        {
            get
            {
                return this.@create;
            }
        }

        public Modify Modify
        {
            get
            {
                return this.@modify;
            }
        }

        /// <summary>
        /// 出力データを作成します。
        /// </summary>
        /// <returns>出力データのインスタンスを返します。</returns>
        public override file_infoType CreateSerializableData()
        {
            var writeData = new file_infoType();
            writeData.@create = this.@create.CreateSerializableData() as createType;
            writeData.@modify = this.@modify.CreateSerializableData() as modifyType;
            bool isEmpty = true;
            isEmpty &= writeData.create == null;
            isEmpty &= writeData.modify == null;
            if (isEmpty)
            {
                return null;
            }
            return writeData;
        }

        /// <summary>
        /// 現在のインスタンスをディープコピーで複製した新規インスタンスを返します。
        /// </summary>
        FileInfo IDeepCopyable<FileInfo>.DeepCopy()
        {
            return new FileInfo(this);
        }

        /// <summary>
        /// 入力ファイルからディープコピーします。
        /// </summary>
        /// <param name="source">コピー元となる入力ファイルです。</param>
        public void DeepCopyFrom(FileInfo source)
        {
            this.@create.DeepCopyFrom(source.@create);
            this.@modify.DeepCopyFrom(source.@modify);
        }

        /// <summary>
        /// エンティティの CRC を作成します。（内部処理用）
        /// </summary>
        /// <returns>CRC の値を返します。</returns>
        protected override uint CreateCRCInternal()
        {
            CRC32 crc = new CRC32();
            List<byte> buffers = new List<byte>();
            buffers.AddRange(BitConverter.GetBytes(crc.ComputeHashUInt32(this.Create.HashValue)));
            buffers.AddRange(BitConverter.GetBytes(crc.ComputeHashUInt32(this.Modify.HashValue)));
            return crc.ComputeHashUInt32(buffers.ToArray());
        }

        /// <summary>
        /// 自動計算フラグを設定します。(内部処理用）
        /// </summary>
        protected override void SetAutoCalcFlagInternal()
        {
            this.@create.AutoCalc = this.AutoCalc;
            this.@modify.AutoCalc = this.AutoCalc;
        }

        /// <summary>
        /// エンティティの状態をリセットします。(内部処理用）
        /// </summary>
        protected override void ResetInternal()
        {
            this.@create.Reset();
            this.@modify.Reset();
        }

        /// <summary>
        /// エンティティの状態を更新します。(内部処理用）
        /// </summary>
        protected override void RefreshInternal()
        {
            this.@create.Refresh();
            this.@modify.Refresh();
        }

        private void OnPropertyChanged(object sender, PropertyChangedEventArgs e)
        {
            if (e.PropertyName == "HashValue")
            {
                this.CalcCRC();
            }
        }
    }
}
