﻿// --------------------------------------------------------------------------------
// <copyright>
// Copyright (C)Nintendo. All rights reserved.
//
// These coded instructions, statements, and computer programs contain proprietary
// information of Nintendo and/or its licensed developers and are protected by
// national and international copyright laws. They may not be disclosed to third
// parties or copied or duplicated in any form, in whole or in part, without the
// prior written consent of Nintendo.
//
// The content herein is highly confidential and should be handled accordingly.
// </copyright>
// --------------------------------------------------------------------------------

using System;
using System.Collections.Generic;
using System.Collections.ObjectModel;
using System.ComponentModel;
using System.Linq;
using System.Text;
using System.Threading.Tasks;
using System.Xml;
using nw.g3d.nw4f_3dif;
using Opal.Security.Cryptography;
using Nintendo.ToolFoundation.Collections;
using Nintendo.ToolFoundation.Contracts;
using Nintendo.G3dTool.Entities.Internal;

namespace Nintendo.G3dTool.Entities
{
    public class CameraAnim : ObservableEntity<camera_animType>, IChildEntity, IDeepCopyable<CameraAnim>, IDeepCopyFrom<CameraAnim>
    {
        private readonly ObservableDictionary<camera_anim_target_targetType, CameraAnimTarget> @cameraAnimTargets = new ObservableDictionary<camera_anim_target_targetType, CameraAnimTarget>();
        private readonly ObservableList<UserData> @userDatas = new ObservableList<UserData>();
        private readonly Comment @comment = new Comment();
        private readonly ToolData @toolData = new ToolData();
        private readonly UserToolData @userToolData = new UserToolData();
        private string @cameraName = string.Empty;
        private int @frameCount;
        private bool @loop;
        private camera_anim_rotate_modeType @rotateMode;
        private camera_anim_projection_modeType @projectionMode;
        private SceneAnim parent = null;

        /// <summary>
        /// コンストラクタです。
        /// </summary>
        public CameraAnim()
        {
            this.@cameraAnimTargets.CollectionChanged += OnDictionaryChanged<camera_anim_target_targetType, CameraAnimTarget>;
            this.@cameraAnimTargets.Add(camera_anim_target_targetType.position_x, new CameraAnimTarget()
            {
                Target = camera_anim_target_targetType.position_x
            });
            this.@cameraAnimTargets.Add(camera_anim_target_targetType.position_y, new CameraAnimTarget()
            {
                Target = camera_anim_target_targetType.position_y
            });
            this.@cameraAnimTargets.Add(camera_anim_target_targetType.position_z, new CameraAnimTarget()
            {
                Target = camera_anim_target_targetType.position_z
            });
            this.@cameraAnimTargets.Add(camera_anim_target_targetType.aim_x, new CameraAnimTarget()
            {
                Target = camera_anim_target_targetType.aim_x
            });
            this.@cameraAnimTargets.Add(camera_anim_target_targetType.aim_y, new CameraAnimTarget()
            {
                Target = camera_anim_target_targetType.aim_y
            });
            this.@cameraAnimTargets.Add(camera_anim_target_targetType.aim_z, new CameraAnimTarget()
            {
                Target = camera_anim_target_targetType.aim_z
            });
            this.@cameraAnimTargets.Add(camera_anim_target_targetType.twist, new CameraAnimTarget()
            {
                Target = camera_anim_target_targetType.twist
            });
            this.@cameraAnimTargets.Add(camera_anim_target_targetType.rotate_x, new CameraAnimTarget()
            {
                Target = camera_anim_target_targetType.rotate_x
            });
            this.@cameraAnimTargets.Add(camera_anim_target_targetType.rotate_y, new CameraAnimTarget()
            {
                Target = camera_anim_target_targetType.rotate_y
            });
            this.@cameraAnimTargets.Add(camera_anim_target_targetType.rotate_z, new CameraAnimTarget()
            {
                Target = camera_anim_target_targetType.rotate_z
            });
            this.@cameraAnimTargets.Add(camera_anim_target_targetType.aspect, new CameraAnimTarget()
            {
                Target = camera_anim_target_targetType.aspect
            });
            this.@cameraAnimTargets.Add(camera_anim_target_targetType.near, new CameraAnimTarget()
            {
                Target = camera_anim_target_targetType.near
            });
            this.@cameraAnimTargets.Add(camera_anim_target_targetType.far, new CameraAnimTarget()
            {
                Target = camera_anim_target_targetType.far
            });
            this.@cameraAnimTargets.Add(camera_anim_target_targetType.ortho_height, new CameraAnimTarget()
            {
                Target = camera_anim_target_targetType.ortho_height
            });
            this.@cameraAnimTargets.Add(camera_anim_target_targetType.persp_fovy, new CameraAnimTarget()
            {
                Target = camera_anim_target_targetType.persp_fovy
            });
            this.@userDatas.CollectionChanged += OnCollectionChanged<UserData>;
            this.@comment.PropertyChanged += this.OnPropertyChanged;
            this.@toolData.PropertyChanged += this.OnPropertyChanged;
            this.@userToolData.PropertyChanged += this.OnPropertyChanged;
        }

        /// <summary>
        /// コピーコンストラクタです。
        /// </summary>
        /// <param name="source">設定するデータです。</param>
        public CameraAnim(CameraAnim source)
            : this()
        {
            this.DeepCopyFrom(source);
        }

        /// <summary>
        /// コンストラクタです。
        /// </summary>
        /// <param name="source">設定するデータです。</param>
        public CameraAnim(camera_animType source)
        {
            this.@cameraAnimTargets.CollectionChanged += OnDictionaryChanged<camera_anim_target_targetType, CameraAnimTarget>;
            this.@cameraAnimTargets.Add(camera_anim_target_targetType.position_x, new CameraAnimTarget()
            {
                Target = camera_anim_target_targetType.position_x
            });
            this.@cameraAnimTargets.Add(camera_anim_target_targetType.position_y, new CameraAnimTarget()
            {
                Target = camera_anim_target_targetType.position_y
            });
            this.@cameraAnimTargets.Add(camera_anim_target_targetType.position_z, new CameraAnimTarget()
            {
                Target = camera_anim_target_targetType.position_z
            });
            this.@cameraAnimTargets.Add(camera_anim_target_targetType.aim_x, new CameraAnimTarget()
            {
                Target = camera_anim_target_targetType.aim_x
            });
            this.@cameraAnimTargets.Add(camera_anim_target_targetType.aim_y, new CameraAnimTarget()
            {
                Target = camera_anim_target_targetType.aim_y
            });
            this.@cameraAnimTargets.Add(camera_anim_target_targetType.aim_z, new CameraAnimTarget()
            {
                Target = camera_anim_target_targetType.aim_z
            });
            this.@cameraAnimTargets.Add(camera_anim_target_targetType.twist, new CameraAnimTarget()
            {
                Target = camera_anim_target_targetType.twist
            });
            this.@cameraAnimTargets.Add(camera_anim_target_targetType.rotate_x, new CameraAnimTarget()
            {
                Target = camera_anim_target_targetType.rotate_x
            });
            this.@cameraAnimTargets.Add(camera_anim_target_targetType.rotate_y, new CameraAnimTarget()
            {
                Target = camera_anim_target_targetType.rotate_y
            });
            this.@cameraAnimTargets.Add(camera_anim_target_targetType.rotate_z, new CameraAnimTarget()
            {
                Target = camera_anim_target_targetType.rotate_z
            });
            this.@cameraAnimTargets.Add(camera_anim_target_targetType.aspect, new CameraAnimTarget()
            {
                Target = camera_anim_target_targetType.aspect
            });
            this.@cameraAnimTargets.Add(camera_anim_target_targetType.near, new CameraAnimTarget()
            {
                Target = camera_anim_target_targetType.near
            });
            this.@cameraAnimTargets.Add(camera_anim_target_targetType.far, new CameraAnimTarget()
            {
                Target = camera_anim_target_targetType.far
            });
            this.@cameraAnimTargets.Add(camera_anim_target_targetType.ortho_height, new CameraAnimTarget()
            {
                Target = camera_anim_target_targetType.ortho_height
            });
            this.@cameraAnimTargets.Add(camera_anim_target_targetType.persp_fovy, new CameraAnimTarget()
            {
                Target = camera_anim_target_targetType.persp_fovy
            });
            this.@userDatas.CollectionChanged += OnCollectionChanged<UserData>;
            if (source.camera_anim_target != null)
            {
                foreach (var elem in source.camera_anim_target)
                {
                    this.@cameraAnimTargets[elem.target] = new CameraAnimTarget(elem);
                }
            }
            if (source.@user_data_array != null)
            {
                this.@userDatas.Clear();
                foreach (var elem in source.@user_data_array.Items)
                {
                    this.@userDatas.Add(DataModelConverter.Convert(elem));
                }
            }
            if (source.@comment != null)
            {
                this.@comment = new Comment(source.@comment);
            }
            else
            {
                this.@comment = new Comment();
            }
            if (source.@tool_data != null)
            {
                this.@toolData = new ToolData(source.@tool_data);
            }
            else
            {
                this.@toolData = new ToolData();
            }
            if (source.@user_tool_data != null)
            {
                this.@userToolData = new UserToolData(source.@user_tool_data);
            }
            else
            {
                this.@userToolData = new UserToolData();
            }
            this.@cameraName = source.@camera_name;
            this.@frameCount = source.@frame_count;
            this.@loop = source.@loop;
            this.@rotateMode = source.@rotate_mode;
            this.@projectionMode = source.@projection_mode;
            this.@comment.PropertyChanged += this.OnPropertyChanged;
            this.@toolData.PropertyChanged += this.OnPropertyChanged;
            this.@userToolData.PropertyChanged += this.OnPropertyChanged;
        }

        public IReadOnlyDictionary<camera_anim_target_targetType, CameraAnimTarget> CameraAnimTargets
        {
            get
            {
                return this.@cameraAnimTargets as IReadOnlyDictionary<camera_anim_target_targetType, CameraAnimTarget>;
            }
        }

        public ObservableList<UserData> UserDatas
        {
            get
            {
                return this.@userDatas;
            }
        }

        public Comment Comment
        {
            get
            {
                return this.@comment;
            }
        }

        public ToolData ToolData
        {
            get
            {
                return this.@toolData;
            }
        }

        public UserToolData UserToolData
        {
            get
            {
                return this.@userToolData;
            }
        }

        public string CameraName
        {
            get
            {
                return this.@cameraName;
            }

            set
            {
                if (this.@cameraName == value)
                {
                    return;
                }

                Ensure.Argument.NotNull(value);
                this.SetProperty(ref this.@cameraName, value, () => this.CalcCRC());
            }
        }

        public int FrameCount
        {
            get
            {
                return this.@frameCount;
            }

            set
            {
                if (this.@frameCount == value)
                {
                    return;
                }

                this.SetProperty(ref this.@frameCount, value, () => this.CalcCRC());
            }
        }

        public bool Loop
        {
            get
            {
                return this.@loop;
            }

            set
            {
                if (this.@loop == value)
                {
                    return;
                }

                this.SetProperty(ref this.@loop, value, () => this.CalcCRC());
            }
        }

        public camera_anim_rotate_modeType RotateMode
        {
            get
            {
                return this.@rotateMode;
            }

            set
            {
                if (this.@rotateMode == value)
                {
                    return;
                }

                this.SetProperty(ref this.@rotateMode, value, () => this.CalcCRC());
            }
        }

        public camera_anim_projection_modeType ProjectionMode
        {
            get
            {
                return this.@projectionMode;
            }

            set
            {
                if (this.@projectionMode == value)
                {
                    return;
                }

                this.SetProperty(ref this.@projectionMode, value, () => this.CalcCRC());
            }
        }

        IEntity IChildEntity.Parent
        {
            get
            {
                return this.parent as IEntity;
            }

            set
            {
                if (this.parent == value)
                {
                    return;
                }

                Ensure.Argument.True(value != null ? value is SceneAnim : true);
                this.SetProperty(ref this.parent, value as SceneAnim, () => this.CalcCRC());
            }
        }

        public SceneAnim Parent
        {
            get
            {
                return this.parent;
            }
        }

        Type IChildEntity.ParentType
        {
            get
            {
                return typeof(SceneAnim);
            }
        }

        /// <summary>
        /// 出力データを作成します。
        /// </summary>
        /// <returns>出力データのインスタンスを返します。</returns>
        public override camera_animType CreateSerializableData()
        {
            var writeData = new camera_animType();
            {
                var targets = new List<CameraAnimTarget>();
                targets.Add(this.CameraAnimTargets[camera_anim_target_targetType.position_x]);
                targets.Add(this.CameraAnimTargets[camera_anim_target_targetType.position_y]);
                targets.Add(this.CameraAnimTargets[camera_anim_target_targetType.position_z]);
                targets.Add(this.CameraAnimTargets[camera_anim_target_targetType.aspect]);
                targets.Add(this.CameraAnimTargets[camera_anim_target_targetType.near]);
                targets.Add(this.CameraAnimTargets[camera_anim_target_targetType.far]);
                switch (this.RotateMode)
                {
                case camera_anim_rotate_modeType.aim:
                    {
                        targets.Add(this.CameraAnimTargets[camera_anim_target_targetType.aim_x]);
                        targets.Add(this.CameraAnimTargets[camera_anim_target_targetType.aim_y]);
                        targets.Add(this.CameraAnimTargets[camera_anim_target_targetType.aim_z]);
                        targets.Add(this.CameraAnimTargets[camera_anim_target_targetType.twist]);
                    }
                    break;
                case camera_anim_rotate_modeType.euler_zxy:
                    {
                        targets.Add(this.CameraAnimTargets[camera_anim_target_targetType.rotate_x]);
                        targets.Add(this.CameraAnimTargets[camera_anim_target_targetType.rotate_y]);
                        targets.Add(this.CameraAnimTargets[camera_anim_target_targetType.rotate_z ]);
                    }
                    break;
                default:
                    throw new Exception($"Unexpected rotate_mode. {this.RotateMode}");
                }

                switch (this.ProjectionMode)
                {
                case camera_anim_projection_modeType.ortho:
                    {
                        targets.Add(this.CameraAnimTargets[camera_anim_target_targetType.ortho_height ]);
                    }
                    break;
                case camera_anim_projection_modeType.persp:
                    {
                        targets.Add(this.CameraAnimTargets[camera_anim_target_targetType.persp_fovy]);
                    }
                    break;
                default:
                    throw new Exception($"Unexpected projection_mode. {this.ProjectionMode}");
                }
                writeData.camera_anim_target = targets
                    .Where(x => x.Curve != null)
                    .Select(x => x.CreateSerializableData())
                    .ToArray();
            }
            if (this.@UserDatas.Count > 0)
            {
                writeData.@user_data_array = new user_data_arrayType();
                writeData.@user_data_array.Items = this.@userDatas.Select(x => x.CreateSerializableData()).Where(x => x != null).ToArray();
                if (writeData.@user_data_array.Items.Length == 0)
                {
                    writeData.@user_data_array = null;
                }
            }
            writeData.@comment = this.@comment.CreateSerializableData() as commentType;
            writeData.@tool_data = this.@toolData.CreateSerializableData() as tool_dataType;
            writeData.@user_tool_data = this.@userToolData.CreateSerializableData() as user_tool_dataType;
            writeData.@camera_name = this.CameraName;
            writeData.@frame_count = this.FrameCount;
            writeData.@loop = this.Loop;
            writeData.@rotate_mode = this.RotateMode;
            writeData.@projection_mode = this.ProjectionMode;
            return writeData;
        }

        /// <summary>
        /// 現在のインスタンスをディープコピーで複製した新規インスタンスを返します。
        /// </summary>
        CameraAnim IDeepCopyable<CameraAnim>.DeepCopy()
        {
            return new CameraAnim(this);
        }

        /// <summary>
        /// 入力ファイルからディープコピーします。
        /// </summary>
        /// <param name="source">コピー元となる入力ファイルです。</param>
        public void DeepCopyFrom(CameraAnim source)
        {
            CopyElements(source.@cameraAnimTargets, this.@cameraAnimTargets);
            CopyElements(source.@userDatas, this.@userDatas);
            this.@comment.DeepCopyFrom(source.@comment);
            this.@toolData.DeepCopyFrom(source.@toolData);
            this.@userToolData.DeepCopyFrom(source.@userToolData);
            this.@cameraName = source.@cameraName;
            this.@frameCount = source.@frameCount;
            this.@loop = source.@loop;
            this.@rotateMode = source.@rotateMode;
            this.@projectionMode = source.@projectionMode;
        }

        /// <summary>
        /// エンティティの CRC を作成します。（内部処理用）
        /// </summary>
        /// <returns>CRC の値を返します。</returns>
        protected override uint CreateCRCInternal()
        {
            CRC32 crc = new CRC32();
            List<byte> buffers = new List<byte>();
            foreach (var elem in this.CameraAnimTargets)
            {
                buffers.AddRange(BitConverter.GetBytes(elem.Value.HashValue));
            }
            foreach (var elem in this.UserDatas)
            {
                buffers.AddRange(BitConverter.GetBytes(elem.HashValue));
            }
            buffers.AddRange(BitConverter.GetBytes(crc.ComputeHashUInt32(this.Comment.HashValue)));
            buffers.AddRange(BitConverter.GetBytes(crc.ComputeHashUInt32(this.ToolData.HashValue)));
            buffers.AddRange(BitConverter.GetBytes(crc.ComputeHashUInt32(this.UserToolData.HashValue)));
            buffers.AddRange(BitConverter.GetBytes(crc.ComputeHashUInt32(this.CameraName)));
            buffers.AddRange(BitConverter.GetBytes(crc.ComputeHashUInt32(this.FrameCount)));
            buffers.AddRange(BitConverter.GetBytes(crc.ComputeHashUInt32(this.Loop)));
            buffers.AddRange(BitConverter.GetBytes(crc.ComputeHashUInt32(this.RotateMode)));
            buffers.AddRange(BitConverter.GetBytes(crc.ComputeHashUInt32(this.ProjectionMode)));
            return crc.ComputeHashUInt32(buffers.ToArray());
        }

        /// <summary>
        /// 自動計算フラグを設定します。(内部処理用）
        /// </summary>
        protected override void SetAutoCalcFlagInternal()
        {
            foreach (var elem in this.@cameraAnimTargets)
            {
                elem.Value.AutoCalc = this.AutoCalc;
            }
            foreach (var elem in this.@userDatas)
            {
                elem.AutoCalc = this.AutoCalc;
            }
            this.@comment.AutoCalc = this.AutoCalc;
            this.@toolData.AutoCalc = this.AutoCalc;
            this.@userToolData.AutoCalc = this.AutoCalc;
        }

        /// <summary>
        /// エンティティの状態をリセットします。(内部処理用）
        /// </summary>
        protected override void ResetInternal()
        {
            foreach (var elem in this.@cameraAnimTargets)
            {
                elem.Value.Reset();
            }
            foreach (var elem in this.@userDatas)
            {
                elem.Reset();
            }
            this.@comment.Reset();
            this.@toolData.Reset();
            this.@userToolData.Reset();
        }

        /// <summary>
        /// エンティティの状態を更新します。(内部処理用）
        /// </summary>
        protected override void RefreshInternal()
        {
            foreach (var elem in this.@cameraAnimTargets)
            {
                elem.Value.Refresh();
            }
            foreach (var elem in this.@userDatas)
            {
                elem.Refresh();
            }
            this.@comment.Refresh();
            this.@toolData.Refresh();
            this.@userToolData.Refresh();
        }

        private void OnPropertyChanged(object sender, PropertyChangedEventArgs e)
        {
            if (e.PropertyName == "HashValue")
            {
                this.CalcCRC();
            }
        }
    }
}
