﻿// --------------------------------------------------------------------------------
// <copyright>
// Copyright (C)Nintendo. All rights reserved.
//
// These coded instructions, statements, and computer programs contain proprietary
// information of Nintendo and/or its licensed developers and are protected by
// national and international copyright laws. They may not be disclosed to third
// parties or copied or duplicated in any form, in whole or in part, without the
// prior written consent of Nintendo.
//
// The content herein is highly confidential and should be handled accordingly.
// </copyright>
// --------------------------------------------------------------------------------
using System;
using System.Collections.Generic;
using System.ComponentModel;
using System.Diagnostics;
using System.Drawing;
using System.Windows.Forms;
using App.Controls;

namespace App
{

    // 強制的に無効状態にする
    class ForceDisableUi
    {
        public static ForceDisableUi Instance = new ForceDisableUi();

        private ForceDisableUi()
        {
        }
    }

    #region MenuCommandDesc
    /// <summary>
    /// メニューコマンド記述子クラス。
    /// </summary>
    public sealed class MenuCommandDesc<TCommandID> where TCommandID : struct
    {
        // コマンドＩＤ
        private readonly TCommandID _commandID;
        // コマンドハンドラ
        private readonly MenuCommandHandler _commandHandler;
        // コマンドデータ
        private object _commandData = null;
        // メニューテキストＩＤ
        private string _MenuText = null;
        // ツールチップテキストＩＤ
        private string _ToolTipText = null;
        // メッセージＩＤ
        private string _message = null;
        // イメージ
        private Image _image = null;

        /// <summary>
        /// コンストラクタ。
        /// </summary>
        public MenuCommandDesc(TCommandID commandID, MenuCommandHandler commandHandler)
        {
            _commandID      = commandID;
            _commandHandler = commandHandler;
        }

        /// <summary>
        /// コマンドＩＤ。
        /// </summary>
        public TCommandID CommandID
        {
            get { return _commandID; }
        }

        /// <summary>
        /// コマンドハンドラ。
        /// </summary>
        public MenuCommandHandler CommandHandler
        {
            get { return _commandHandler; }
        }

        /// <summary>
        /// コマンドデータ。
        /// </summary>
        public object CommandData
        {
            get { return _commandData; }
            set { _commandData = value; }
        }

        /// <summary>
        /// メニューテキストＩＤ。
        /// </summary>
        public string MenuText
        {
            get { return _MenuText; }
            set { _MenuText = value; }
        }

        /// <summary>
        /// ツールチップテキストＩＤ。
        /// </summary>
        public string ToolTipText
        {
            get { return _ToolTipText; }
            set { _ToolTipText = value; }
        }

        /// <summary>
        /// メッセージＩＤ。
        /// </summary>
        public string Message
        {
            get { return _message; }
            set { _message = value; }
        }

        /// <summary>
        /// イメージ。
        /// </summary>
        public Image Image
        {
            get { return _image; }
            set { _image = value; }
        }
    }
    #endregion

    #region MenuCommandUI
    /// <summary>
    /// メニューコマンドＵＩクラス。
    /// </summary>
    public sealed class MenuCommandUI
    {
        // 対象項目
        private readonly ToolStripItem _targetItem;

        /// <summary>
        /// コンストラクタ。
        /// </summary>
        public MenuCommandUI(ToolStripItem targetItem)
        {
            _targetItem = targetItem;
        }

        /// <summary>
        /// 対象項目。
        /// </summary>
        public ToolStripItem TargetItem
        {
            get { return _targetItem; }
        }

        /// <summary>
        /// 有効状態。
        /// </summary>
        public bool Enabled
        {
            get {
                if (_targetItem.Tag is ForceDisableUi)
                {
                    return false;
                }
                else
                {
                    return _targetItem.Enabled;
                }
            }
            set {
                if (_targetItem.Tag is ForceDisableUi)
                {
                    _targetItem.Enabled = false;
                }
                else
                {
                    _targetItem.Enabled = value;
                }
            }
        }

        /// <summary>
        /// チェック状態。
        /// </summary>
        public bool Checked
        {
            get
            {
                // メニュー項目
                ToolStripMenuItem menuItem = _targetItem as ToolStripMenuItem;
                if (menuItem != null)
                {
                    return menuItem.Checked;
                }
                // ボタン
                ToolStripButton button = _targetItem as ToolStripButton;
                if (button != null)
                {
                    return button.Checked;
                }
                // その他
                return false;
            }
            set
            {
                // メニュー項目
                ToolStripMenuItem menuItem = _targetItem as ToolStripMenuItem;
                if (menuItem != null)
                {
                    menuItem.Checked = value;
                    return;
                }
                // ボタン
                ToolStripButton button = _targetItem as ToolStripButton;
                if (button != null)
                {
                    button.Checked = value;
                    return;
                }
                // その他は何もしない
            }
        }
    }
    #endregion

    #region MenuCommandArgs
    /// <summary>
    /// メニューコマンド引数クラス。
    /// </summary>
    public sealed class MenuCommandArgs
    {
        // 更新要求フラグ
        private readonly bool _requireUpdate;
        // コマンドＵＩ
        private readonly MenuCommandUI _commandUI;
        // コマンドデータ 暫定もしかしたら必要か
        //private readonly object _commandData;

        /// <summary>
        /// コンストラクタ。
        /// </summary>
        public MenuCommandArgs(bool requireUpdate, MenuCommandUI commandUI)
        {
            _requireUpdate = requireUpdate;
            _commandUI = commandUI;
            //_commandData = commandData;
        }

        /// <summary>
        /// 更新要求フラグ。
        /// </summary>
        public bool RequireUpdate
        {
            get { return _requireUpdate; }
        }

        /// <summary>
        /// 実行要求フラグ。
        /// </summary>
        public bool RequireExecute
        {
            get { return !_requireUpdate; }
        }

        /// <summary>
        /// コマンドＵＩ。
        /// </summary>
        public MenuCommandUI CommandUI
        {
            get { return _commandUI; }
        }

        /*
        /// <summary>
        /// コマンドデータ。
        /// </summary>
        public object CommandData
        {
            get { return _commandData; }
        }*/
    }
    #endregion

    #region MenuCommandHost

    /// <summary>
    /// メニューコマンドホストクラス。
    /// </summary>
    public sealed class MenuCommandHost
    {
        // コマンドイメージリスト
        private readonly ImageList _commandImageList = new ImageList();
        // コマンド記述子テーブル
        //private readonly Dictionary<TCommandID, MenuCommandDesc<TCommandID>> _commandDescTable = new Dictionary<TCommandID, MenuCommandDesc<TCommandID>>();
        // メッセージハンドラ
        private MenuCommandMessageHandler _messageHandler = null;
        // アイドル時更新項目リスト
        private readonly List<ToolStripItem> _idleUpdateItems = new List<ToolStripItem>();
        // メッセージ更新中フラグ
        private bool _messageUpdating = false;
        // ドロップダウンレベル
        private int _dropDownLevel = 0;
        // コンテキストメニュー
        private UIContextMenuStrip contextMenuStrip;
        // 関連動作
        private readonly Dictionary<ToolStripItem, UIToolStripMenuItem> _bindTable = new Dictionary<ToolStripItem, UIToolStripMenuItem>();

        /// <summary>
        /// コンストラクタ。
        /// </summary>
        public MenuCommandHost()
        {
            _commandImageList.ImageSize = new Size(16, 16);
            _commandImageList.ColorDepth = ColorDepth.Depth24Bit;
            _commandImageList.TransparentColor = Color.Magenta;
            contextMenuStrip = new UIContextMenuStrip();
            contextMenuStrip.Opening += this.Event_ContextMenu_Opening;
            contextMenuStrip.Closed += this.Event_ContextMenu_Closed;
        }

        /// <summary>
        /// メッセージハンドラ。
        /// </summary>
        public MenuCommandMessageHandler MessageHandler
        {
            get { return _messageHandler; }
            set { _messageHandler = value; }
        }

        /// <summary>
        /// コマンド発行。
        /// </summary>
        public void InvokeCommand(ToolStripItem item)
        {
            // 実行ハンドラ呼び出し
            CallCommandHandler(item);
        }

        /// <summary>
        /// アイドル時更新処理。
        /// </summary>
        public void OnIdleUpdate()
        {
            // ドロップダウン中じゃなければ
            if (_dropDownLevel == 0)
            {
                // 更新ハンドラ呼び出し
                foreach (ToolStripItem item in _idleUpdateItems)
                {
                    CallUpdateHandler(item);
                }

                // コマンドメッセージクリア
                if (_messageHandler != null && !_messageUpdating)
                {
                    _messageHandler(string.Empty);
                }
            }
        }

        /// <summary>
        /// 汎用コンテキストメニュー作成。
        /// </summary>
        public UIContextMenuStrip CreateGeneralContextMenu()
        {
            contextMenuStrip.Items.Clear();
            return contextMenuStrip;
        }

        /// <summary>
        /// 汎用セパレータ作成。
        /// </summary>
        public UIToolStripSeparator CreateGeneralSeparator()
        {
            UIToolStripSeparator separator = new UIToolStripSeparator();
            separator.MouseEnter += this.Event_ToolStripItem_MouseEnter;
            return separator;
        }

        //----------------------------------------------------------------------
        // セットアップ
        //----------------------------------------------------------------------
        /// <summary>
        /// セットアップ処理。
        /// </summary>
        public void Setup(ToolStrip toolStrip)
        {
            // マウスイベント
            toolStrip.MouseEnter += new EventHandler(Event_ToolStrip_MouseEnter);
            toolStrip.MouseLeave += new EventHandler(Event_ToolStrip_MouseLeave);

            // 項目をセットアップ
            foreach (ToolStripItem item in toolStrip.Items)
            {
                Setup(item);
            }

            // ツールバーなら
            if (toolStrip is UIToolStrip)
            {
                // 最上位項目をアイドル時更新対象にする
                foreach (ToolStripItem item in toolStrip.Items)
                {
                    if (!_idleUpdateItems.Contains(item))
                    {
                        _idleUpdateItems.Add(item);
                    }
                }

                // 無効項目の選択タイミング取得用
                ((UIToolStrip)toolStrip).BeforeMouseMove += new MouseEventHandler(Event_UIToolStrip_BeforeMouseMove);
            }
        }

        /// <summary>
        /// セットアップ処理。
        /// </summary>
        private void Setup(ToolStripItem item)
        {
            // 選択時イベント（全項目に設定）
            item.MouseEnter += new EventHandler(Event_ToolStripItem_MouseEnter);

            // ドロップダウン項目
            ToolStripDropDownItem dropDownItem = item as ToolStripDropDownItem;
            if (dropDownItem != null)
            {
                // 子項目なし
                if (dropDownItem.DropDownItems.Count == 0)
                {
                    // クリック時イベント
                    dropDownItem.Click += new EventHandler(Event_ToolStripItem_Click);

                    // ペイントイベント（無効状態の選択タイミング取得用）
                    dropDownItem.Paint += new PaintEventHandler(Event_DropDownItem_Paint);

                    // ショートカット処理前イベント
                    UIToolStripMenuItem menuItem = dropDownItem as UIToolStripMenuItem;
                    if (menuItem != null && menuItem.ShortcutKeys != Keys.None)
                    {
                        menuItem.BeforeShortcutKeyProcessing += new EventHandler(Event_ToolStripMenuItem_BeforeShortcutKeyProcessing);
                    }
                }
                // 子項目あり
                else
                {
                    // ドロップダウン時イベント
                    dropDownItem.DropDownOpening += new EventHandler(Event_ToolStripDropDownItem_DropDownOpening);
                    dropDownItem.DropDownClosed  += new EventHandler(Event_ToolStripDropDownItem_DropDownClosed);

                    // 子項目の設定
                    foreach (ToolStripItem subItem in dropDownItem.DropDownItems)
                    {
                        Setup(subItem);
                    }
                }
                return;
            }

            // ボタン項目
            ToolStripButton button = item as ToolStripButton;
            if (button != null)
            {
                // クリック時イベント
                button.Click += new EventHandler(Event_ToolStripItem_Click);
                return;
            }
        }

        //----------------------------------------------------------------------
        // コマンド記述子
        //----------------------------------------------------------------------
        public void BindCommandItem(UIToolStripMenuItem menuItem, params ToolStripItem[] commandItems)
        {
            foreach (var item in commandItems)
            {
                _bindTable[item] = menuItem;
            }
        }
#if false
        /// <summary>
        /// コマンド記述子登録。
        /// </summary>
        public void RegisterCommandDesc(TCommandID commandID, MenuCommandHandler commandHandler, string message, params ToolStripItem[] commandItems)
        {
            RegisterCommandDesc(commandID, commandHandler, null, message, commandItems);
        }

        /// <summary>
        /// コマンド記述子登録。
        /// </summary>
        public void RegisterCommandDesc(TCommandID commandID, MenuCommandHandler commandHandler, object commandData, string message, params ToolStripItem[] commandItems)
        {
            // 記述子作成
            MenuCommandDesc<TCommandID> desc = new MenuCommandDesc<TCommandID>(commandID, commandHandler);
            desc.CommandData = commandData;
            desc.Message = message;

            // 記述子登録
            Debug.Assert(!_commandDescTable.ContainsKey(commandID));
            _commandDescTable.Add(commandID, desc);

            // コマンド項目に記述子をバインド
            foreach (ToolStripItem item in commandItems)
            {
                BindCommandItemToDesc(commandID, item);
            }
        }

        /// <summary>
        /// コマンド記述子取得。
        /// </summary>
        public MenuCommandDesc<TCommandID> GetCommandDesc(TCommandID commandID)
        {
            Debug.Assert(_commandDescTable.ContainsKey(commandID));
            return _commandDescTable[commandID];
        }

        /// <summary>
        /// コマンド記述子取得。
        /// </summary>
        private MenuCommandDesc<TCommandID> GetCommandDesc(ToolStripItem item)
        {
            // タグデータあり
            if (item.Tag != null)
            {
                // 記述子型の場合のみ
                MenuCommandDesc<TCommandID> desc = item.Tag as MenuCommandDesc<TCommandID>;
                if (desc != null)
                {
                    return desc;
                }
            }
            return null;
        }

        /// <summary>
        /// コマンド記述子にコマンドデータを設定。
        /// </summary>
        public void SetCommandData(TCommandID commandID, object commandData)
        {
            // 対応する記述子を取得
            MenuCommandDesc<TCommandID> desc = GetCommandDesc(commandID);
            // コマンド記述子にデータを設定
            desc.CommandData = commandData;
        }

        /// <summary>
        /// コマンド項目を記述子にバインド。
        /// </summary>
        public void BindCommandItemToDesc(TCommandID commandID, ToolStripItem item)
        {
            // 対応する記述子を取得
            MenuCommandDesc<TCommandID> desc = GetCommandDesc(commandID);
            // タグにコマンド記述子を設定
            item.Tag = desc;
        }
#endif
        //----------------------------------------------------------------------
        // ハンドラ呼び出し
        //----------------------------------------------------------------------
        /// <summary>
        /// 更新ハンドラ呼び出し。
        /// </summary>
        private void CallUpdateHandler(ToolStripItem item)
        {
            UIToolStripMenuItem menuItem = item as UIToolStripMenuItem;
            if (menuItem == null && _bindTable.ContainsKey(item))
            {
                menuItem = _bindTable[item];
            }

            if (menuItem != null)
            {
                MenuCommandArgs args = new MenuCommandArgs(true, new MenuCommandUI(item));
                menuItem.InvokeMenuCommand(args);
            }
        }

        /// <summary>
        /// コマンドハンドラ呼び出し。
        /// </summary>
        private void CallCommandHandler(ToolStripItem item)
        {
            UIToolStripMenuItem menuItem = item as UIToolStripMenuItem;
            if (menuItem == null && _bindTable.ContainsKey(item))
            {
                menuItem = _bindTable[item];
            }

            if (menuItem != null)
            {
                MenuCommandArgs args = new MenuCommandArgs(false, new MenuCommandUI(item));
                menuItem.InvokeMenuCommand(args);
            }
        }

        /// <summary>
        /// メッセージハンドラ呼び出し。
        /// </summary>
        private void CallMessageHandler(ToolStripItem item)
        {
            if (_messageHandler != null)
            {
                UIToolStripMenuItem menuItem = item as UIToolStripMenuItem;
                if (menuItem == null && _bindTable.ContainsKey(item))
                {
                    menuItem = _bindTable[item];
                }

                if (menuItem != null && menuItem.CaptionText != null)
                {
                    _messageHandler(menuItem.CaptionText);
                }
                else
                {
                    _messageHandler(string.Empty);
                }
            }
        }

        #region イベントハンドラ
        /// <summary>
        /// イベントハンドラ。
        /// </summary>
        private void Event_ToolStrip_MouseEnter(object sender, EventArgs e)
        {
            // メッセージ更新状態開始
            _messageUpdating = true;
        }

        /// <summary>
        /// イベントハンドラ。
        /// </summary>
        private void Event_ToolStrip_MouseLeave(object sender, EventArgs e)
        {
            // メッセージ更新状態終了
            _messageUpdating = false;
        }

        /// <summary>
        /// イベントハンドラ。
        /// </summary>
        private void Event_UIToolStrip_BeforeMouseMove(object sender, MouseEventArgs e)
        {
            UIToolStrip toolStrip = (UIToolStrip)sender;

            // マウス上の無効状態項目を取得
            ToolStripItem item = toolStrip.GetItemAt(e.Location);
            if (item != null && !item.Enabled)
            {
                // メッセージハンドラ呼び出し
                CallMessageHandler(item);
            }
        }

        /// <summary>
        /// イベントハンドラ。
        /// </summary>
        private void Event_ToolStripItem_MouseEnter(object sender, EventArgs e)
        {
            // メッセージハンドラ呼び出し
            CallMessageHandler((ToolStripItem)sender);
        }

        /// <summary>
        /// イベントハンドラ。
        /// </summary>
        private void Event_ToolStripItem_Click(object sender, EventArgs e)
        {
            // 長時間処理コマンドになる前の描画更新
            Application.DoEvents();

            // 実行ハンドラ呼び出し
            CallCommandHandler((ToolStripItem)sender);
        }

        /// <summary>
        /// イベントハンドラ。
        /// </summary>
        private void Event_DropDownItem_Paint(object sender, PaintEventArgs e)
        {
            ToolStripDropDownItem item = (ToolStripDropDownItem)sender;

            // 無効状態の場合のみ
            if (!item.Enabled)
            {
                if (item.Owner != null)
                {
                    // マウスが領域内にあれば
                    Rectangle rect = item.Owner.RectangleToScreen(item.Bounds);
                    if (rect.Contains(Control.MousePosition))
                    {
                        // メッセージハンドラ呼び出し
                        CallMessageHandler(item);
                    }
                }
            }
        }

        /// <summary>
        /// イベントハンドラ。
        /// </summary>
        private void Event_ToolStripDropDownItem_DropDownOpening(object sender, EventArgs e)
        {
            // ドロップダウンレベルを上げる
            _dropDownLevel++;

            // 更新ハンドラ呼び出し
            ToolStripDropDownItem item = (ToolStripDropDownItem)sender;
            foreach (ToolStripItem subItem in item.DropDownItems)
            {
                CallUpdateHandler(subItem);
            }
        }

        /// <summary>
        /// イベントハンドラ。
        /// </summary>
        private void Event_ToolStripDropDownItem_DropDownClosed(object sender, EventArgs e)
        {
            // ドロップダウンレベルを下げる
            _dropDownLevel--;
            Debug.Assert(_dropDownLevel >= 0);
        }

        /// <summary>
        /// イベントハンドラ。
        /// </summary>
        void Event_ContextMenu_Opening(object sender, CancelEventArgs e)
        {
            // ドロップダウンレベルを上げる
            _dropDownLevel++;

            // 更新ハンドラ呼び出し
            ToolStripDropDown dropDown = (ToolStripDropDown)sender;
            foreach (ToolStripItem subItem in dropDown.Items)
            {
                CallUpdateHandler(subItem);
            }
        }

        /// <summary>
        /// イベントハンドラ。
        /// </summary>
        void Event_ContextMenu_Closed(object sender, ToolStripDropDownClosedEventArgs e)
        {
            // ドロップダウンレベルを下げる
            _dropDownLevel--;
            Debug.Assert(_dropDownLevel >= 0);
        }

        /// <summary>
        /// イベントハンドラ。
        /// </summary>
        private void Event_ToolStripMenuItem_BeforeShortcutKeyProcessing(object sender, EventArgs e)
        {
            // 更新ハンドラ呼び出し
            CallUpdateHandler((ToolStripItem)sender);
        }
        #endregion
    }
    #endregion

    /// <summary>
    /// メニューコマンドハンドラデリゲート。
    /// </summary>
    public delegate void MenuCommandHandler(MenuCommandArgs args);

    /// <summary>
    /// メニューコマンドメッセージハンドラデリゲート。
    /// </summary>
    public delegate void MenuCommandMessageHandler(string message);
}
