﻿// --------------------------------------------------------------------------------
// <copyright>
// Copyright (C)Nintendo. All rights reserved.
//
// These coded instructions, statements, and computer programs contain proprietary
// information of Nintendo and/or its licensed developers and are protected by
// national and international copyright laws. They may not be disclosed to third
// parties or copied or duplicated in any form, in whole or in part, without the
// prior written consent of Nintendo.
//
// The content herein is highly confidential and should be handled accordingly.
// </copyright>
// --------------------------------------------------------------------------------
using System;
using System.Collections.Generic;
using System.ComponentModel;
using System.Diagnostics;
using System.Drawing;
using System.Windows.Forms;
using System.Windows.Forms.Design;
using App.Data;

namespace App.Controls
{
    /// <summary>
    /// 低いテキストボックスクラス。
    /// </summary>
    [ToolboxItem(true)]
    public sealed partial class LowTextBox : UIUserControl
    {
        [DesignerSerializationVisibility(DesignerSerializationVisibility.Visible)]
        public bool AutoHint
        {
            get
            {
                return textBox.AutoHint;
            }

            set
            {
                textBox.AutoHint = value;
            }
        }

        [DesignerSerializationVisibility(DesignerSerializationVisibility.Visible)]
        public bool ReadOnly
        {
            get
            {
                return textBox.ReadOnly;
            }

            set
            {
                textBox.ReadOnly = value;
            }
        }

        [DesignerSerializationVisibility(DesignerSerializationVisibility.Visible)]
        public override string Text
        {
            get
            {
                return textBox.Text;
            }

            set
            {
                textBox.Text = value;
            }
        }

        [DesignerSerializationVisibility(DesignerSerializationVisibility.Visible)]
        public HorizontalAlignment TextAlign
        {
            get
            {
                return textBox.TextAlign;
            }

            set
            {
                textBox.TextAlign = value;
            }
        }

        public override Color ForeColor
        {
            get
            {
                return textBox.ForeColor;
            }

            set
            {
                textBox.ForeColor = value;
            }
        }

        public override Color BackColor
        {
            get
            {
                return textBox.BackColor;
            }

            set
            {
                textBox.BackColor = value;
            }
        }

        /// <summary>
        /// コンストラクタ。
        /// </summary>
        public LowTextBox()
        {
            InitializeComponent();

            if (TextBoxRenderer.IsSupported == false)
            {
                BorderStyle = System.Windows.Forms.BorderStyle.Fixed3D;
                textBox.Left --;
                textBox.Top --;
            }

            // コンテキストメニューを設定する
            {
                textBox.OverrideContextMenu = true;

                ContextMenuStrip cms = new ContextMenuStrip();
                {
                    ToolStripMenuItem menuItemCopy = new ToolStripMenuItem(App.res.Strings.Menu_Copy);
                    {
                        menuItemCopy.Click += (sender, e) => textBox.Copy();
                        cms.Items.Add(menuItemCopy);
                    }

                    ToolStripMenuItem menuItemSelectAll = new ToolStripMenuItem(App.res.Strings.Menu_SelectAll);
                    {
                        menuItemSelectAll.Click += (sender, e) => textBox.SelectAll();
                        cms.Items.Add(menuItemSelectAll);
                    }

                    cms.Opening += delegate
                    {
                        menuItemCopy.Enabled = textBox.SelectionLength > 0;
                        textBox.Focus();
                    };
                }
                textBox.ContextMenuStrip = cms;
                textBox.Paint += (o, e) =>
                    {
                        e.Graphics.DrawString(textBox.Text, textBox.Font, new SolidBrush(textBox.ForeColor), 0, 0);
                    };
            }
        }

        protected override void OnPaint(PaintEventArgs e)
        {
            base.OnPaint(e);

            if (TextBoxRenderer.IsSupported)
            {
                TextBoxRenderer.DrawTextBox(
                    e.Graphics,
                    new Rectangle(0, 0, Width, Height),
                    ReadOnly ?
                        System.Windows.Forms.VisualStyles.TextBoxState.Readonly :
                        System.Windows.Forms.VisualStyles.TextBoxState.Normal
                );

                // TODO:textBoxRenderer.DrawTextBox() の TextBoxState の指定でReadonlyを指定しても無視される？ので
                e.Graphics.FillRectangle(
                    ReadOnly ?
                        SystemBrushes.Control :
                        SystemBrushes.Window,
                    1,
                    1,
                    Width - 2,
                    Height - 2
                );
            }
        }

        protected override void OnResize(EventArgs e)
        {
            Invalidate();
            base.OnResize(e);
        }

        private void textBox_KeyDown(object sender, KeyEventArgs e)
        {
            if (ReadOnly)
            {
                if ((e.Modifiers == Keys.Control) && (e.KeyCode == Keys.A))
                {
                    textBox.SelectAll();
                }
            }
        }
    }
}
