﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#ifndef __ShellExtension_Texture_H__
#define __ShellExtension_Texture_H__

#include "ShellExtension_Type.h"

#define GAMMA_CORRECTION_POWER    2.2f
#include <vector>

//==============================================================================
//
// NWTexture Class
//
// - Contains information about given texture, and binary data
//
//==============================================================================
class CNWTexture
{
public :
    CNWTexture();
    virtual ~CNWTexture();

    //--------------------------------------------------------------------------
    // Load from image file
    //
    // - If bLoadPreview is true, create preview image for icon
    //   If bLoadPreview is false, just load information ( format, size etc )
    //--------------------------------------------------------------------------
    bool Load( const WCHAR *szFilePath,
               bool bLoadPreview );

    //--------------------------------------------------------------------------
    // Set texture description
    //
    // - If cubemap, width and height are each face's width and height
    //--------------------------------------------------------------------------
    void SetDescription( NW_IMG_TYPE texType, NW_IMG_FMT format, NW_IMG_FMT paletteFormat,
                         int width, int height, int depth, int numMipmaps );
    void SetDescription( NW_IMG_TYPE texType, NW_IMG_FMT format, NW_IMG_FMT paletteFormat,
                         int width, int height, int previewWidth, int previewHeight,
                         int depth, int numMipmaps );

    //--------------------------------------------------------------------------
    // Create preview
    //--------------------------------------------------------------------------
    bool CreatePreviewImage();
    void Destroy();

    //--------------------------------------------------------------------------
    // Create Color and Alpha Preview Map from PreviewMap
    //--------------------------------------------------------------------------
    bool CreateColorMap( Gdiplus::BitmapData &srcBmpData, OriginalImageFormat originalFmt, Gdiplus::Rect drawRect );
    bool CreateAlphaMap( Gdiplus::BitmapData &srcBmpData, OriginalImageFormat originalFmt, Gdiplus::Rect drawRect );
    bool CreateColorAndAlphaMap( Gdiplus::BitmapData &srcBmpData, OriginalImageFormat originalFmt );
    bool CreateColorSmallMap( Gdiplus::BitmapData &srcBmpData, OriginalImageFormat originalFmt, Gdiplus::Rect drawRect );
    bool CreateAlphaSmallMap( Gdiplus::BitmapData &srcBmpData, OriginalImageFormat originalFmt, Gdiplus::Rect drawRect );
    bool CreateColorAndAlphaSmallMap( Gdiplus::BitmapData &srcBmpData, OriginalImageFormat originalFmt );

    bool CreatePreviewBitmap(Gdiplus::Bitmap* srcBmp, Gdiplus::Rect drawRect, OriginalImageFormat originalFmt, bool isSmall, bool isColor, bool isAlpha);

    void SetPreviewBitmap(Gdiplus::Bitmap* bitmap) { m_pPreviewBitmap = bitmap; }

    //--------------------------------------------------------------------------
    // Information and accessor
    //--------------------------------------------------------------------------
    int  GetWidth()  const { return m_width; }
    int  GetHeight() const { return m_height; }
    int  GetDepth() const  { return m_depth; }
    int  GetCount() const  { return m_count; }
    int  GetNumMipmaps() const { return m_numMipmaps; };

    bool IsUsingAllMipmaps() const { return m_bUseAllMipmaps; }
    bool HasMipmaps() const;

    NW_IMG_TYPE  GetTextureType() const { return m_textureType; }
    const std::wstring& GetTextureTypeName() const { return m_typeName; };

    bool HasAlpha() const { return m_bHasAlpha; };
    bool MakeHasAlphaFromAlphaImageWhenBC1(void);

    int  GetPreviewWidth()  const { return m_previewWidth; };
    int  GetPreviewHeight() const { return m_previewHeight; };

    NW_IMG_FMT GetFormat() const { return m_format; };
    const WCHAR* GetFormatName() const { return m_formatName.c_str(); };

    NW_IMG_FMT GetPaletteFormat() const { return m_paletteFormat; };
    const std::wstring& GetPaletteFormatName() const { return m_paletteFormatName; };

    const OriginalImageField* GetOriginalImageFieldInfo() const { return m_pOriginalImageFieldInfo.data(); };
    void SetOriginalImageFieldInfo(std::vector<OriginalImageField>& info);

    bool OnlyShowImage() {
#if defined NW_FOR_ESET || defined NW_FOR_FLYT || defined NW_FOR_THUMBS || defined NW_FOR_AI || defined NW_FOR_EPS
        return true;
#else
        return false;
#endif
    }

    //--------------------------------------------------------------------------
    // Byte count of this texture ( including mipmaps )
    //--------------------------------------------------------------------------
    void SetTextureDataSize( __int64 dataSize ) { m_textureDataSize = dataSize; };
    __int64 GetTextureDataSize() const { return m_textureDataSize; };

    //--------------------------------------------------------------------------
    // Comment
    //--------------------------------------------------------------------------
    void SetComment( const WCHAR* comment );
    const std::wstring&  GetComment() const { return m_commentText; }

    //--------------------------------------------------------------------------
    // Hint
    //--------------------------------------------------------------------------
    void SetHint( const WCHAR* hint ) { m_hintText = (hint != NULL) ? hint : L""; }
    const std::wstring& GetHint() const { return m_hintText; }

    //--------------------------------------------------------------------------
    // CompressType
    //--------------------------------------------------------------------------
    void SetCompressTypeText( const WCHAR* compressType );
    const std::wstring& GetCompressTypeText() const { return m_compressTypeText; }
    const std::wstring& GetCompressTypeTextShort() const { return m_compressTypeTextShort; }

    //--------------------------------------------------------------------------
    // output folder
    //--------------------------------------------------------------------------
    void SetOutputFolder( const WCHAR* folder ) { m_outputFolder = (folder != NULL) ? folder : L""; }
    const std::wstring& GetOutputFolder() const { return m_outputFolder; }

    void SetTgaFolder( const WCHAR* folder ) { m_tgaFolder = (folder != NULL) ? folder : L""; }
    const std::wstring& GetTgaFolder() const { return m_tgaFolder; }

    void SetCtexFolder( const WCHAR* folder ) { m_ctexFolder = (folder != NULL) ? folder : L""; }
    const std::wstring& GetCtexFolder() const { return m_ctexFolder; }

    //--------------------------------------------------------------------------
    // TGA file
    //--------------------------------------------------------------------------
    void SetAsTGAFile( bool bTGA, NW_TGA_Type TGAType );

    bool IsTGA() const   { return m_bTGA; };
    NW_TGA_Type TGAType() const { return m_TGAType; };

    //--------------------------------------------------------------------------
    // PSD file
    //--------------------------------------------------------------------------
    void SetAsPSDFile( bool bPSD, bool hasExpSet );

    bool IsPSD() const   { return m_bPSD; };
    bool HasPSDExportSetting() const { return m_bHasExportSetting; }

    //--------------------------------------------------------------------------
    // Bitmap for Preview
    //--------------------------------------------------------------------------
    Gdiplus::Bitmap*       GetPreviewBitmap() { return m_pPreviewBitmap; };
    const Gdiplus::Bitmap* GetPreviewBitmap() const { return m_pPreviewBitmap; };

    Gdiplus::Bitmap*       GetPreviewColorBitmap() { return m_pPreviewColorBitmap; };
    const Gdiplus::Bitmap* GetPreviewColorBitmap() const { return m_pPreviewColorBitmap; };

    Gdiplus::Bitmap*       GetPreviewAlphaBitmap() { return m_pPreviewAlphaBitmap; };
    const Gdiplus::Bitmap* GetPreviewAlphaBitmap() const { return m_pPreviewAlphaBitmap; };

    //--------------------------------------------------------------------------
    // Bitmap for Small Icon
    //--------------------------------------------------------------------------
    Gdiplus::Bitmap*       GetPreviewSmallBitmap() { return m_pPreviewSmallBitmap; };
    const Gdiplus::Bitmap* GetPreviewSmallBitmap() const { return m_pPreviewSmallBitmap; };

    Gdiplus::Bitmap*       GetPreviewColorSmallBitmap() { return m_pPreviewColorSmallBitmap; };
    const Gdiplus::Bitmap* GetPreviewColorSmallBitmap() const { return m_pPreviewColorSmallBitmap; };

    Gdiplus::Bitmap*       GetPreviewAlphaSmallBitmap() { return m_pPreviewAlphaSmallBitmap; };
    const Gdiplus::Bitmap* GetPreviewAlphaSmallBitmap() const { return m_pPreviewAlphaSmallBitmap; };

    //--------------------------------------------------------------------------
    // Check if Preview is loaded
    //--------------------------------------------------------------------------
    bool HasPreviewImage() const
    {
        if (m_pPreviewBitmap==NULL)
            return false;

        if (m_bLoadPreviewFailed)
            return false;

        return true;
    }; // End of HasPreviewImage

    //--------------------------------------------------------------------------
    // Compute ImageDataSize
    //--------------------------------------------------------------------------
    int  ComputeTextureFaceDataSize( int mipLevel );
    void ComputeImageDataSize();

    //--------------------------------------------------------------------------
    // Linear flags
    //--------------------------------------------------------------------------
    void SetLinearFlags( bool redCh, bool greenCh, bool blueCh, bool alphaCh, bool twoChannelsAlphaFlag );
    const bool GetLinearFlag( NW_COMP_SEL_ELEMENT channel );
    const bool GetUILinearFlag( NW_COMP_SEL_ELEMENT channel );

    //--------------------------------------------------------------------------
    // Component selector
    //--------------------------------------------------------------------------
    void SetCompSel( const NW_COMP_SEL &sel );
    const NW_COMP_SEL& GetCompSel() const { return m_compSel; };
    bool HasCompSel() const;

    //--------------------------------------------------------------------------
    // Check if force treating this as normal map
    //--------------------------------------------------------------------------
    bool IsForceNormalMapOn() const { return m_bForceNormalMap; };

    void SetHintNormalMap( bool bHint );
    bool IsHintNormalMap() const { return m_bHintNormalMap; };

    //--------------------------------------------------------------------------
    // Set Layer Comps
    //--------------------------------------------------------------------------
    void SetLayerComps( const PSD_LAYER_COMPS* layerComps, int size);
    int GetLayerCompsSize() const { return m_layerCompsSize; }
    const PSD_LAYER_COMPS* GetLayerComps() const { return m_layerComps; }

    //--------------------------------------------------------------------------
    // Flag to identify if this texture is for NW4F/4C
    //--------------------------------------------------------------------------
    bool IsForNW4F() const { return m_bForNW4F; };
    bool IsForNW4C() const { return m_bForNW4C; };

    void SetForNW4F( bool bSet ) { m_bForNW4F = bSet; };
    void SetForNW4C( bool bSet ) { m_bForNW4C = bSet; };

    //--------------------------------------------------------------------------
    // Flag to show mipmap count
    //--------------------------------------------------------------------------
    bool IsShowMipmapCountEnabled() const { return m_bShowMipmapCount; };

    void SetShowMipmapCount( bool bShow ) { m_bShowMipmapCount = bShow; };

private:
    //--------------------------------------------------------------------------
    // Private methods
    //--------------------------------------------------------------------------
    void InitializeGammaLookupTable();
private :
    int          m_width;      // Size of face
    int          m_height;     // Size of face
    int          m_depth;      // Size of face
    int          m_count;      // Size of count
    int          m_numMipmaps;
    bool         m_bUseAllMipmaps;
    bool         m_bHasAlpha;
    NW_IMG_TYPE  m_textureType;
    std::wstring m_typeName;
    NW_IMG_FMT   m_format;
    std::wstring m_formatName;
    NW_IMG_FMT   m_paletteFormat;
    std::wstring m_paletteFormatName;
    int          m_previewWidth;
    int          m_previewHeight;
    __int64      m_textureDataSize;
    bool         m_bTGA;
    NW_TGA_Type  m_TGAType;
    bool         m_bPSD;
    bool         m_bHasExportSetting;

    bool         m_bLoadPreviewFailed;


    //bool         m_bSignedFormat;

    bool         m_bForceNormalMap;
    bool         m_bForceUseComputedZ;
    bool         m_bHintNormalMap;
    bool         m_linearFlags[4];
    bool         m_UILinearFlags[4];

    std::wstring m_outputFolder;
    std::wstring m_tgaFolder;
    std::wstring m_ctexFolder;
    std::wstring m_commentText;
    std::wstring m_hintText;
    std::wstring m_compressTypeText;
    std::wstring m_compressTypeTextShort;

    NW_COMP_SEL  m_compSel;

    PSD_LAYER_COMPS*    m_layerComps;
    int                 m_layerCompsSize;

    bool                m_bForNW4F;
    bool                m_bForNW4C;
    bool                m_bShowMipmapCount;

    std::vector<OriginalImageField> m_pOriginalImageFieldInfo;

    // For preview and icon
    // Note : Order of bytes are B,G,R,A from low to high
    Gdiplus::Bitmap *m_pPreviewBitmap;
    Gdiplus::Bitmap *m_pPreviewColorBitmap;
    Gdiplus::Bitmap *m_pPreviewAlphaBitmap;

    // For small icon
    // Note : Order of bytes are B,G,R,A from low to high
    Gdiplus::Bitmap *m_pPreviewSmallBitmap;
    Gdiplus::Bitmap *m_pPreviewColorSmallBitmap;
    Gdiplus::Bitmap *m_pPreviewAlphaSmallBitmap;

private:
    //--------------------------------------------------------------------------
    // Create Color and Alpha Preview Map from PreviewMap
    //--------------------------------------------------------------------------
    bool CreateColorMapImpl( Gdiplus::Bitmap** ppPreviewColorBitmap,Gdiplus::BitmapData &srcBmpData, Gdiplus::Rect drawRect, OriginalImageFormat originalFmt )const;
    bool CreateAlphaMapImpl( Gdiplus::Bitmap** ppPreviewAlphaBitmap,Gdiplus::BitmapData &srcBmpData, Gdiplus::Rect drawRect, OriginalImageFormat originalFmt )const;
    bool CreateColorAndAlphaMapImpl( Gdiplus::Bitmap** ppPreviewColorBitmap,Gdiplus::BitmapData &srcBmpData, OriginalImageFormat originalFmt )const;

    bool DrawFillRect(Gdiplus::BitmapData  bmpData, Gdiplus::Color color,  int stX, int stY, int edX, int edY)const;

}; // End of CNWTexture

#endif // End of __ShellExtension_Texture_H__

