﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include "ShellExtension_PCH.h"

#include "ShellExtension_Resource.h"
#include "ShellExtension_Type.h"
#include "ShellExtension_SyncObject.h"
#include "ShellExtension_Texture.h"
#include "ShellExtension_FileData.h"
#include "ShellExtension_Manager.h"
#include "ShellExtension_Utility.h"

#include "Service/ExtractImage/ShellExtension_ExtractImage.h"

//==============================================================================
//
// CShellExtensionExtractImage Implementation
//
//==============================================================================
CShellExtensionExtractImage::CShellExtensionExtractImage() :
    m_pFileData(NULL)
{
    m_desiredSize.cx = 0;
    m_desiredSize.cy = 0;
    memset(&m_fileTime,0,sizeof(m_fileTime));

    NWShellLockModule();
} // End of Constructor for CShellExtensionExtractImage


//------------------------------------------------------------------------------
CShellExtensionExtractImage::~CShellExtensionExtractImage()
{
    if (m_pFileData!=NULL)
    {
        m_pFileData->Release(L"ExtractImage");
        m_pFileData = NULL;
    } // End if

    NWShellUnlockModule();
} // End of Destructor for CShellExtensionExtractImage


//------------------------------------------------------------------------------
// GetExtractImageLocation
//------------------------------------------------------------------------------
STDMETHODIMP CShellExtensionExtractImage::GetLocation( LPWSTR pszPathBuffer,
                                                     DWORD cchMax,
                                                     DWORD *pdwPriority,
                                                     const SIZE *prgSize,
                                                     DWORD dwRecClrDepth,
                                                     DWORD *pdwFlags )
{
    NW_USE_VAR(pszPathBuffer);
    NW_USE_VAR(cchMax);
    NW_USE_VAR(pdwPriority);
    NW_USE_VAR(prgSize);
    NW_USE_VAR(dwRecClrDepth);
    NW_USE_VAR(pdwFlags);

    if (pszPathBuffer!=NULL)
    {
        if (m_pFileData!=NULL)
        {
            wcscpy_s(pszPathBuffer,cchMax,m_pFileData->GetFileName().c_str());
        } // End if
        else
        {
            pszPathBuffer[0] = (WCHAR)0;
        } // End else
    } // End if

    if (pdwPriority!=NULL)
    {
        *pdwPriority = IEIT_PRIORITY_NORMAL;
    } // End if

    if (prgSize!=NULL)
    {
        m_desiredSize = *prgSize;
    } // End if

    return S_OK;
} // End of GetExtractImageLocation for CShellExtensionExtractImage


//------------------------------------------------------------------------------
// Extract
//------------------------------------------------------------------------------
STDMETHODIMP CShellExtensionExtractImage::Extract( HBITMAP *phBmpImage )
{
    if (m_pFileData==NULL)
    {
        *phBmpImage = NULL;
        return E_FAIL;
    } // End if

    if ( (m_desiredSize.cx<=0) ||
         (m_desiredSize.cy<=0) )
    {
        return E_FAIL;
    } // Ed if

    CNWSingleLock lock(m_pFileData->GetCriticalSection());

    if ( (m_pFileData->GetTexture()==NULL) ||
         (m_pFileData->GetTexture()->GetPreviewBitmap()==NULL) )
    {
        if (m_pFileData->UpdateImage(true)==false)
            return S_FALSE;

        SYSTEMTIME st;

        GetSystemTime(&st);                      // Gets the current system time
        SystemTimeToFileTime(&st, &m_fileTime);  // Converts the current system time to file time format
    } // End if

    if ( (m_desiredSize.cx>=96) ||
         (m_desiredSize.cy>=96) )
    {
        *phBmpImage = m_pFileData->GetBitmap(m_desiredSize.cx,m_desiredSize.cy,true);
    } // End if
    else
    {
        *phBmpImage = m_pFileData->GetBitmap(m_desiredSize.cx,m_desiredSize.cy,false);
    } // End else

    return S_OK;
} // End of Extract for CShellExtensionExtractImage


//------------------------------------------------------------------------------
// IExtractImage2
//------------------------------------------------------------------------------
STDMETHODIMP CShellExtensionExtractImage::GetDateStamp( FILETIME *pDateStamp )
{
    if (pDateStamp!=NULL)
    {
        memcpy(pDateStamp,&m_fileTime,sizeof(FILETIME));
    } // End if

    return S_OK;
} // End of GetDateStamp for CShellExtensionExtractImage


//------------------------------------------------------------------------------
// Load
//------------------------------------------------------------------------------
STDMETHODIMP CShellExtensionExtractImage::Load( LPCOLESTR wszFile, DWORD dwMode )
{
    if (m_pFileData!=NULL)
    {
        m_pFileData->Release(L"ExtractImage::Load");
        m_pFileData = NULL;
    } // End if

    NW_USE_VAR(dwMode);

    USES_CONVERSION;
    const WCHAR* szFileName = W2CT(wszFile);
    m_pFileData = CShellExtensionManager::Instance()->GetFileData(szFileName,L"ExtractImage");
    if (m_pFileData==NULL)
        return E_FAIL;

    m_fileName = szFileName;

    return S_OK;
} // End of Load for CShellExtensionExtractImage


//------------------------------------------------------------------------------
// Register
//------------------------------------------------------------------------------
bool CShellExtensionExtractImage::RegisterService( REGISTRATION_PARAM &param )
{
    WCHAR szGUIDStr[128];
    WCHAR szValStr[128];

    _snwprintf_s(szGUIDStr,127,_TRUNCATE,L"{%s}", NWCreateUUIDString(CLSID_ShellExtensionExtractImage).c_str());

    //--------------------------------------------------------------------------
    // CLSID
    //--------------------------------------------------------------------------
    CRegKey CLSIDKey;
    if (CLSIDKey.Open( HKEY_CLASSES_ROOT,
                       _T("CLSID"),
                       KEY_SET_VALUE )==ERROR_SUCCESS)
    {
        //----------------------------------------------------------------------
        // My Interface ID
        //----------------------------------------------------------------------
        _snwprintf_s(szValStr,127,_TRUNCATE,_T("NintendoWare %s ExtractImage Shell Extension"), param.fileTypeName.c_str() );

        CRegKey GUIDKey;
        if (GUIDKey.Create(CLSIDKey.m_hKey,szGUIDStr)==ERROR_SUCCESS)
        {
            GUIDKey.SetStringValue(NULL,szValStr);

            //------------------------------------------------------------------
            // Inproc server
            //------------------------------------------------------------------
            CRegKey InpProc32Key;
            if (InpProc32Key.Create(GUIDKey.m_hKey,L"InprocServer32")==ERROR_SUCCESS)
            {
                InpProc32Key.SetStringValue(NULL,param.moduleName.c_str());
                InpProc32Key.SetStringValue(L"ThreadingModel", L"Apartment");
                InpProc32Key.Close();
            } // End if

            GUIDKey.Close();
        } // End if

        CLSIDKey.Close();
    } // End if

    //--------------------------------------------------------------------------
    // ProgID
    //--------------------------------------------------------------------------
    int i;
    for (i=0;i<(int)param.supportedExtensions.size();i++)
    {
        CRegKey progIDKey;
        if (progIDKey.Open( HKEY_CLASSES_ROOT,
                            param.progIDNames[i].c_str(),
                            KEY_SET_VALUE )==ERROR_SUCCESS)
        {
            //------------------------------------------------------------------
            // ShellEx Icon
            //------------------------------------------------------------------
            CRegKey shellExKey;
            if (shellExKey.Open(progIDKey.m_hKey,
                                L"ShellEx",
                                KEY_SET_VALUE)==ERROR_SUCCESS)
            {
                //--------------------------------------------------------------
                // ShellEx Icon
                //--------------------------------------------------------------
                CRegKey handlerKey;
                if (handlerKey.Create(shellExKey.m_hKey,L"{BB2E617C-0920-11d1-9A0B-00C04FC2D6C1}")==ERROR_SUCCESS)
                {
                    handlerKey.SetStringValue(NULL,szGUIDStr);
                    handlerKey.Close();
                } // End if

                shellExKey.Close();
            } // End if

            progIDKey.Close();
        } // End if
    } // End for

    return true;
} // End of RegisterService for CShellExtensionExtractImage


//------------------------------------------------------------------------------
// Unregister
//------------------------------------------------------------------------------
bool CShellExtensionExtractImage::UnregisterService( REGISTRATION_PARAM &param )
{
    NW_USE_VAR(param.supportedExtensions.size());

    WCHAR szGUIDStr[128];
    _snwprintf_s(szGUIDStr,127,_TRUNCATE,L"{%s}", NWCreateUUIDString(CLSID_ShellExtensionExtractImage).c_str());

    //--------------------------------------------------------------------------
    // CLSID
    //--------------------------------------------------------------------------
    CRegKey CLSIDKey;
    if (CLSIDKey.Open ( HKEY_CLASSES_ROOT,
                        _T("CLSID"),
                        KEY_SET_VALUE )==ERROR_SUCCESS)
    {
        //----------------------------------------------------------------------
        // My Interface ID
        //----------------------------------------------------------------------
        CLSIDKey.RecurseDeleteKey(szGUIDStr);

        CLSIDKey.Close();
    } // End if

    return true;
} // End of UnregisterService for CShellExtensionExtractImage
