﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include "ShellExtension_PCH.h"
#include "ShellExtension_Utility.h"
#include "ShellExtension_Type.h"
#include "ShellExtension_Texture.h"
#include "ImageLoader/ShellExtension_ImageLoader.h"
#include "ImageLoader/ShellExtension_CTexLoader.h"

//==============================================================================
//
// Names used in XML for cubemap
//
//==============================================================================
static const char* s_CubeImagesNames[6] =
{
    "PositiveXImages",
    "NegativeXImages",
    "PositiveYImages",
    "NegativeYImages",
    "PositiveZImages",
    "NegativeZImages",
}; // End of s_CubeImagesNames

static const char* s_CubeImageCtrNames[6] =
{
    "PositiveXImageCtr",
    "NegativeXImageCtr",
    "PositiveYImageCtr",
    "NegativeYImageCtr",
    "PositiveZImageCtr",
    "NegativeZImageCtr",
}; // End of s_CubeImageCtrNames

static int s_3BitsDeltaTable[8] =
{
    0,
    1,
    2,
    3,
    -4,
    -3,
    -2,
    -1
}; // End of s_3BitsDeltaTable

struct CodeWordModifier
{
    CodeWordModifier( int v1, int v2, int v3, int v4 )
    {
        values[0] = v1;
        values[1] = v2;
        values[2] = v3;
        values[3] = v4;
    }; // End of Constructor

    int values[4];
}; // End of CodeWordModifier

static CodeWordModifier s_codeWordModifierTable[8] =
{
    CodeWordModifier(  -8,  -2,  2,   8),
    CodeWordModifier( -17,  -5,  5,  17),
    CodeWordModifier( -29,  -9,  9,  29),
    CodeWordModifier( -42, -13, 13,  42),
    CodeWordModifier( -60, -18, 18,  60),
    CodeWordModifier( -80, -24, 24,  80),
    CodeWordModifier(-106, -33, 33, 106),
    CodeWordModifier(-183, -47, 47, 183)
}; // End of s_codeWordModifierTable

static int s_pixelIndexModifierIndex[4] =
{
    2,
    3,
    1,
    0
}; // End of s_pixelIndexModifierIndex

//==============================================================================
//
// Implementation of CNWCTexLoader
//
//==============================================================================
CNWCTexLoader::CNWCTexLoader() :
    CNWImageLoader()
{
} // End of Constructor for CNWCTexLoader


//------------------------------------------------------------------------------
CNWCTexLoader::~CNWCTexLoader()
{
} // End of Destructor for CNWCTexLoader


//------------------------------------------------------------------------------
// Decode one integer value
//------------------------------------------------------------------------------
int CNWCTexLoader::GetBase64Value( int c )
{
    if (c >= (int)'A' && c <= (int)'Z')
    {
        return c - (int)'A';
    } // End if
    else if (c >= (int)'a' && c <= (int)'z')
    {
        return c - (int)'a' + 26;
    } // End else if
    else if (c >= (int)'0' && c <= (int)'9')
    {
        return c - (int)'0' + 52;
    } // End else if
    else if ((int)'+' == c)
    {
        return 62;
    } // End else if
    else if ((int)'/' == c)
    {
        return 63;
    } // End else if
    else if ((int)'=' == c)
    {
        return 0;
    } // End else if

    return -1;
} // End of GetBase64Value for CNWCTexLoader


//------------------------------------------------------------------------------
// Decode text into images
//------------------------------------------------------------------------------
std::unique_ptr<unsigned char[]> CNWCTexLoader::DecodeBase64( const unsigned char* srcBuf, int srcSize,
                                            int& dstSize)
{
    //--------------------------------------------------------------------------
    // alloc dst buffer
    //--------------------------------------------------------------------------
    int allocSize = srcSize;
    if (allocSize<=0)
        return NULL;

    std::unique_ptr<unsigned char[]> dstBuf(new unsigned char[allocSize]);
    if (dstBuf==NULL)
        return NULL;

    //--------------------------------------------------------------------------
    // decode
    //--------------------------------------------------------------------------
    const unsigned char* src = srcBuf;
    const unsigned char* srcEnd = srcBuf + srcSize;
    unsigned char* dst = dstBuf.get();
    while (src < srcEnd)
    {
        int ichar = 0;
        unsigned int base64 = 0;
        while (src < srcEnd)
        {
            int c = *src++;
            if (c == '=')
            {
                break;
            } /// End if

            int val = GetBase64Value(c);
            if (val != -1)
            {
                base64 = (base64 << 6) | val;
                ++ichar;
                if (ichar >= 4)
                {
                    break;
                } // End if
            } // End if
        } // End while

        if (ichar == 0)
        {
            break;
        } // End if

        int outSize = 3;
        if (ichar < 4)
        {
            base64 <<= (4 - ichar) * 6;
            outSize = (ichar == 2) ? 1 : 2;
        } // End if

        for (int idst = 0; idst < outSize; ++idst)
        {
            *dst++ = static_cast<unsigned char>((base64 >> ((2 - idst) * 8)) & 0xff);
        } // End for
    } // End while

    //--------------------------------------------------------------------------
    // End
    //--------------------------------------------------------------------------
    dstSize = static_cast<int>(dst - dstBuf.get());

    return dstBuf;
} // End of DecodeBase64 for CNWCTexLoader


//------------------------------------------------------------------------------
// Load from image file onto pTexture
//
// - If bLoadPreview is true, create preview image for icon
//   If bLoadPreview is false, just load information ( format, size etc )
//------------------------------------------------------------------------------
bool CNWCTexLoader::Load( CNWTexture *pTexture,
                          const WCHAR *szFilePath,
                          bool bLoadPreview )
{
    pTexture->SetAsTGAFile(false, NW_TGA_NONE);

    //--------------------------------------------------------------------------
    // Create document
    //--------------------------------------------------------------------------
    MSXML2::IXMLDOMDocument2Ptr pDoc;
    HRESULT hr = pDoc.CreateInstance(__uuidof(MSXML2::DOMDocument30));
    if (FAILED(hr))
    {
        return false;
    } // End if

    pDoc->async = VARIANT_FALSE; // (default = TRUE)
    pDoc->validateOnParse = VARIANT_FALSE; // disable schema check (default = TRUE)
    pDoc->resolveExternals = VARIANT_FALSE; // (default = TRUE)
    VARIANT_BOOL result = pDoc->load(szFilePath);
    if (result != VARIANT_TRUE)
    {
        return false;
    } // End if

    //--------------------------------------------------------------------------
    // Get texture element
    //--------------------------------------------------------------------------
    MSXML2::IXMLDOMNodePtr rootNode = NWXMLFindChildNode(pDoc, "NintendoWareIntermediateFile");
    if (rootNode == NULL)
        return false;

    MSXML2::IXMLDOMNodePtr contentNode = NWXMLFindChildNode(rootNode, "GraphicsContentCtr");
    if (contentNode == NULL)
        return false;

    MSXML2::IXMLDOMNodePtr texturesNode = NWXMLFindChildNode(contentNode, "Textures");
    if (texturesNode == NULL)
        return false;

    MSXML2::IXMLDOMNodePtr textureNode = NWXMLFindChildNode(texturesNode, "CubeTextureCtr");
    NW_IMG_TYPE textureType = NW_IMG_TYPE_2D;

    if (textureNode != NULL)
    {
        textureType = NW_IMG_TYPE_CUBE;
    } // End if
    else
    {
        textureType = NW_IMG_TYPE_2D;

        textureNode = NWXMLFindChildNode(texturesNode, "ImageTextureCtr");
        if (texturesNode == NULL)
            return false;
    } // End else

    std::wstring compressTypeText = ReadCompressTypeText(contentNode);
    if (compressTypeText.size()>0)
    {
        pTexture->SetCompressTypeText(compressTypeText.c_str());
    } // End if

    // Get information
    int faceW = atol(NWXMLGetAttrValue(textureNode, "Width").c_str());
    int faceH = atol(NWXMLGetAttrValue(textureNode, "Height").c_str());
    int numMipmaps = atol(NWXMLGetAttrValue(textureNode, "MipmapSize").c_str());
    numMipmaps = max(1,numMipmaps);

    std::string formatStr = NWXMLGetAttrValue(textureNode,"Format");
    NW_IMG_FMT format = NWGetTextureFormatFromString(formatStr.c_str());

    int width  = faceW;
    int height = 0;
    if (textureType == NW_IMG_TYPE_CUBE)
    {
        height = width;
    } // End if
    else
    {
        height = faceH;
    } // End else

    pTexture->SetDescription(textureType,format,NW_IMG_FMT_NONE,width,height,0,numMipmaps);

    if (bLoadPreview)
    {
        if (textureType == NW_IMG_TYPE_CUBE)
        {
            if (LoadCubeTexture(pTexture,textureNode)==false)
                return false;
        } // End if
        else
        {
            if (LoadImageTexture(pTexture,textureNode)==false)
                return false;
        } // End else
    } // End if

    pTexture->ComputeImageDataSize();

    return true;
} // End of Load for CNWCTexLoader


//------------------------------------------------------------------------------
// Get compress type text
//------------------------------------------------------------------------------
std::wstring CNWCTexLoader::ReadCompressTypeText( MSXML2::IXMLDOMNodePtr contentNode )
{
    std::string str = "";

    MSXML2::IXMLDOMNodePtr editDataNode = NWXMLFindChildNode(contentNode, "EditData");
    if (editDataNode!=NULL)
    {
        MSXML2::IXMLDOMNodePtr contentSummaryMetaDataNode = NWXMLFindChildNode(editDataNode, "ContentSummaryMetaData");
        if (contentSummaryMetaDataNode!=NULL)
        {
            MSXML2::IXMLDOMNodePtr valuesNode = NWXMLFindChildNode(contentSummaryMetaDataNode, "Values");
            if (valuesNode!=NULL)
            {
                MSXML2::IXMLDOMNodeListPtr children = valuesNode->childNodes;
                for (;;)
                {
                    MSXML2::IXMLDOMNodePtr child = children->nextNode();
                    if (child == NULL)
                    {
                        break;
                    } // End if

                    if (strcmp(child->nodeName,"ContentSummary")==0)
                    {
                        std::string contentTypeStr = NWXMLGetAttrValue(child,"ContentTypeName");
                        if (contentTypeStr=="GraphicsContent")
                        {
                            MSXML2::IXMLDOMNodePtr objectSummariesNode = NWXMLFindChildNode(child, "ObjectSummaries");
                            if (objectSummariesNode!=NULL)
                            {
                                MSXML2::IXMLDOMNodeListPtr objSummaries = objectSummariesNode->childNodes;
                                for (;;)
                                {
                                    MSXML2::IXMLDOMNodePtr objSummaryNode = objSummaries->nextNode();
                                    if (objSummaryNode == NULL)
                                    {
                                        break;
                                    } // End if

                                    if (strcmp(objSummaryNode->nodeName,"ObjectSummary")==0)
                                    {
                                        std::string typeName = NWXMLGetAttrValue(objSummaryNode,"TypeName");
                                        if (typeName=="ImageTexture")
                                        {
                                            MSXML2::IXMLDOMNodePtr notesList = NWXMLFindChildNode(objSummaryNode, "Notes");
                                            if (notesList!=NULL)
                                            {
                                                MSXML2::IXMLDOMNodeListPtr notes = notesList->childNodes;
                                                for (;;)
                                                {
                                                    MSXML2::IXMLDOMNodePtr noteNode = notes->nextNode();
                                                    if (noteNode == NULL)
                                                    {
                                                        break;
                                                    } // End if

                                                    if (strcmp(noteNode->nodeName,"Note")==0)
                                                    {
                                                        std::string nameVal = NWXMLGetAttrValue(noteNode,"Name");
                                                        if (nameVal=="EtcEncoding")
                                                        {
                                                            str = NWXMLGetAttrValue(noteNode,"Value");
                                                        } // End if
                                                    } // End if
                                                } // End while
                                            } // End if
                                        } // End if
                                    } // End if
                                } // End if
                            } // End if
                        } // End if
                    } // End if
                } // End for
            } // End if
        } // End if
    } // End if

    return NWStringToWString(str);
} // End of ReadCompressTypeText for CNWCTexLoader


//------------------------------------------------------------------------------
// Load ImageTexture
//------------------------------------------------------------------------------
bool CNWCTexLoader::LoadImageTexture( CNWTexture *pTexture,
                                      MSXML2::IXMLDOMNodePtr textureNode )
{
    if (pTexture->GetWidth()<=0)
        return false;

    if (pTexture->GetHeight()<=0)
        return false;

    MSXML2::IXMLDOMNodePtr imagesNode = NWXMLFindChildNode(textureNode, "Images");
    if (imagesNode==NULL)
        return false;

    if (pTexture->CreatePreviewImage()==false)
        return false;

    Gdiplus::Bitmap *pPreviewBitmap = pTexture->GetPreviewBitmap();
    if (pPreviewBitmap==NULL)
    {
        return false;
    } // End if

    bool bLocked = false;
    Gdiplus::BitmapData bmpData;

    try
    {
        // Lock bits to prepare preview image
        Gdiplus::Rect lockRect(0,0,pPreviewBitmap->GetWidth(),pPreviewBitmap->GetHeight());
        if (pPreviewBitmap->LockBits(&lockRect,Gdiplus::ImageLockModeWrite,
                                     pPreviewBitmap->GetPixelFormat(),
                                     &bmpData)!=Gdiplus::Ok)
        {
            return false;
        } // End if

        bLocked = true;

        // Start loading each faces
        int level = 0;
        MSXML2::IXMLDOMNodeListPtr childs = imagesNode->childNodes;
        for (;;)
        {
            MSXML2::IXMLDOMNodePtr child = childs->nextNode();
            if (child == NULL)
            {
                break;
            } // End if

            if (strcmp(child->nodeName,"PixelBasedImageCtr")==0)
            {
                if (level < pTexture->GetNumMipmaps())
                {
                    // For this COM control, we just need first mip level
                    if (level==0)
                    {
                        std::string imageText = child->text;
                        int decodedSize = 0;
                        int imageSize = (int)imageText.size();
                        std::unique_ptr<const unsigned char[]> decodedBuf = DecodeBase64((const unsigned char*)imageText.c_str(),
                                                                       imageSize, decodedSize);
                        if (decodedBuf == nullptr)
                        {
                            pPreviewBitmap->UnlockBits(&bmpData);
                            return false;
                        } // End if

                        // Face enum will be ignored for 2D image
                        ConvertToPreviewImage(pTexture,decodedBuf.get(),decodedSize,NULL,NW_CUBE_FACE_POS_X,bmpData);
                    } // End if

                    level++;
                } // End if
            } // End if
        } // End for

        Gdiplus::Rect drawRect(0,0,pPreviewBitmap->GetWidth(),pPreviewBitmap->GetHeight());

        pTexture->CreateColorMap(bmpData, ORIGINAL_FORMAT_NOT_USED, drawRect);

        if (pTexture->HasAlpha())
        {
            pTexture->CreateAlphaMap(bmpData, ORIGINAL_FORMAT_NOT_USED, drawRect);
        } // End if

        // Finish locking
        pPreviewBitmap->UnlockBits(&bmpData);
        bLocked = false;
    }

    catch(...)
    {
        if (bLocked)
        {
            pPreviewBitmap->UnlockBits(&bmpData);
        }
        return false;
    } // End of except

    return true;
} // End of LoadImageTexture for CNWCTexLoader


//------------------------------------------------------------------------------
// Load CubeFace
//------------------------------------------------------------------------------
bool CNWCTexLoader::LoadCubeFace( CNWTexture *pTexture,
                                    MSXML2::IXMLDOMNodePtr imagesNode,
                                    const char* szImageCtrName,
                                    NW_CUBE_FACE cubeFace,
                                    Gdiplus::BitmapData &bmpData )
{
    // Start loading each faces
    int level = 0;
    MSXML2::IXMLDOMNodeListPtr childs = imagesNode->childNodes;
    for (;;)
    {
        MSXML2::IXMLDOMNodePtr child = childs->nextNode();
        if (child == NULL)
        {
            break;
        } // End if

        if (strcmp(child->nodeName,szImageCtrName)==0)
        {
            if (level < pTexture->GetNumMipmaps())
            {
                // For this COM control, we just need first mip level
                if (level==0)
                {
                    std::string imageText = child->text;
                    int imageSize   = (int)imageText.size();
                    int decodedSize = 0;
                    std::unique_ptr<const unsigned char[]> decodedBuf(DecodeBase64((const unsigned char*)imageText.c_str(),
                                                                    imageSize, decodedSize));
                    if (decodedBuf==NULL)
                    {
                        return false;
                    } // End if

                    // Face enum will be ignored for 2D image
                    ConvertToPreviewImage(pTexture,decodedBuf.get(),decodedSize,NULL,cubeFace,bmpData);
                } // End if

                level++;
            } // End if
        } // End if
    } // End for

    return true;
} // End of LoadCubeFace for CNWCTexLoader


//------------------------------------------------------------------------------
// Load CubeTexture
//------------------------------------------------------------------------------
bool CNWCTexLoader::LoadCubeTexture( CNWTexture *pTexture,
                                       MSXML2::IXMLDOMNodePtr textureNode )
{
    if (pTexture->GetWidth()<=0)
        return false;

    if (pTexture->GetHeight()<=0)
        return false;

    if (pTexture->CreatePreviewImage()==false)
        return false;

    Gdiplus::Bitmap *pPreviewBitmap = pTexture->GetPreviewBitmap();
    if (pPreviewBitmap==NULL)
    {
        return false;
    } // End if

    // Lock bits to prepare preview image
    bool bLocked = false;
    Gdiplus::BitmapData bmpData;

    try
    {
        Gdiplus::Rect lockRect(0,0,pPreviewBitmap->GetWidth(),pPreviewBitmap->GetHeight());
        if (pPreviewBitmap->LockBits(&lockRect,Gdiplus::ImageLockModeWrite,
                                     pPreviewBitmap->GetPixelFormat(),
                                     &bmpData)!=Gdiplus::Ok)
        {
            return false;
        } // End if
        bLocked = true;

        int previewImgByteCount = pPreviewBitmap->GetHeight() * bmpData.Stride;
        memset(bmpData.Scan0,0,previewImgByteCount);

        // Start loading each faces
        MSXML2::IXMLDOMNodeListPtr childs = textureNode->childNodes;
        for (;;)
        {
            MSXML2::IXMLDOMNodePtr child = childs->nextNode();
            if (child == NULL)
            {
                break;
            } // End if

            int iFace;
            for (iFace=0;iFace<6;iFace++)
            {
                if (strcmp(child->nodeName,s_CubeImagesNames[iFace])==0)
                {
                    LoadCubeFace(pTexture,child,s_CubeImageCtrNames[iFace],
                                 (NW_CUBE_FACE)iFace, bmpData);
                } // End if
            } // End for
        } // End for

        Gdiplus::Rect drawRect(0,0,pPreviewBitmap->GetWidth(),pPreviewBitmap->GetHeight());

        pTexture->CreateColorMap(bmpData, ORIGINAL_FORMAT_NOT_USED, drawRect);

        if (pTexture->HasAlpha())
        {
            pTexture->CreateAlphaMap(bmpData, ORIGINAL_FORMAT_NOT_USED, drawRect);
        } // End if

        // Finish locking
        pPreviewBitmap->UnlockBits(&bmpData);
        bLocked = false;
    }
    catch(...)
    {
        if (bLocked)
        {
            pPreviewBitmap->UnlockBits(&bmpData);
        }
        return false;
    } // End of except

    return true;
} // End of LoadCubeTexture for CNWCTexLoader


// This S3TC decoding function is already defined if NW_FOR_FTX is defined

struct RGBA
{
    unsigned char b_;
    unsigned char g_;
    unsigned char r_;
    unsigned char a_;
};

const unsigned char Table1F[0x20] =
{
    (unsigned char)(float(0xFF) * float(0x00) / float(0x1F)),
    (unsigned char)(float(0xFF) * float(0x01) / float(0x1F)),
    (unsigned char)(float(0xFF) * float(0x02) / float(0x1F)),
    (unsigned char)(float(0xFF) * float(0x03) / float(0x1F)),
    (unsigned char)(float(0xFF) * float(0x04) / float(0x1F)),
    (unsigned char)(float(0xFF) * float(0x05) / float(0x1F)),
    (unsigned char)(float(0xFF) * float(0x06) / float(0x1F)),
    (unsigned char)(float(0xFF) * float(0x07) / float(0x1F)),
    (unsigned char)(float(0xFF) * float(0x08) / float(0x1F)),
    (unsigned char)(float(0xFF) * float(0x09) / float(0x1F)),
    (unsigned char)(float(0xFF) * float(0x0A) / float(0x1F)),
    (unsigned char)(float(0xFF) * float(0x0B) / float(0x1F)),
    (unsigned char)(float(0xFF) * float(0x0C) / float(0x1F)),
    (unsigned char)(float(0xFF) * float(0x0D) / float(0x1F)),
    (unsigned char)(float(0xFF) * float(0x0E) / float(0x1F)),
    (unsigned char)(float(0xFF) * float(0x0F) / float(0x1F)),
    (unsigned char)(float(0xFF) * float(0x10) / float(0x1F)),
    (unsigned char)(float(0xFF) * float(0x11) / float(0x1F)),
    (unsigned char)(float(0xFF) * float(0x12) / float(0x1F)),
    (unsigned char)(float(0xFF) * float(0x13) / float(0x1F)),
    (unsigned char)(float(0xFF) * float(0x14) / float(0x1F)),
    (unsigned char)(float(0xFF) * float(0x15) / float(0x1F)),
    (unsigned char)(float(0xFF) * float(0x16) / float(0x1F)),
    (unsigned char)(float(0xFF) * float(0x17) / float(0x1F)),
    (unsigned char)(float(0xFF) * float(0x18) / float(0x1F)),
    (unsigned char)(float(0xFF) * float(0x19) / float(0x1F)),
    (unsigned char)(float(0xFF) * float(0x1A) / float(0x1F)),
    (unsigned char)(float(0xFF) * float(0x1B) / float(0x1F)),
    (unsigned char)(float(0xFF) * float(0x1C) / float(0x1F)),
    (unsigned char)(float(0xFF) * float(0x1D) / float(0x1F)),
    (unsigned char)(float(0xFF) * float(0x1E) / float(0x1F)),
    (unsigned char)(float(0xFF) * float(0x1F) / float(0x1F)),
};

const unsigned char Table3F[0x40] =
{
    (unsigned char)(float(0xFF) * float(0x00) / float(0x3F)),
    (unsigned char)(float(0xFF) * float(0x01) / float(0x3F)),
    (unsigned char)(float(0xFF) * float(0x02) / float(0x3F)),
    (unsigned char)(float(0xFF) * float(0x03) / float(0x3F)),
    (unsigned char)(float(0xFF) * float(0x04) / float(0x3F)),
    (unsigned char)(float(0xFF) * float(0x05) / float(0x3F)),
    (unsigned char)(float(0xFF) * float(0x06) / float(0x3F)),
    (unsigned char)(float(0xFF) * float(0x07) / float(0x3F)),
    (unsigned char)(float(0xFF) * float(0x08) / float(0x3F)),
    (unsigned char)(float(0xFF) * float(0x09) / float(0x3F)),
    (unsigned char)(float(0xFF) * float(0x0A) / float(0x3F)),
    (unsigned char)(float(0xFF) * float(0x0B) / float(0x3F)),
    (unsigned char)(float(0xFF) * float(0x0C) / float(0x3F)),
    (unsigned char)(float(0xFF) * float(0x0D) / float(0x3F)),
    (unsigned char)(float(0xFF) * float(0x0E) / float(0x3F)),
    (unsigned char)(float(0xFF) * float(0x0F) / float(0x3F)),
    (unsigned char)(float(0xFF) * float(0x10) / float(0x3F)),
    (unsigned char)(float(0xFF) * float(0x11) / float(0x3F)),
    (unsigned char)(float(0xFF) * float(0x12) / float(0x3F)),
    (unsigned char)(float(0xFF) * float(0x13) / float(0x3F)),
    (unsigned char)(float(0xFF) * float(0x14) / float(0x3F)),
    (unsigned char)(float(0xFF) * float(0x15) / float(0x3F)),
    (unsigned char)(float(0xFF) * float(0x16) / float(0x3F)),
    (unsigned char)(float(0xFF) * float(0x17) / float(0x3F)),
    (unsigned char)(float(0xFF) * float(0x18) / float(0x3F)),
    (unsigned char)(float(0xFF) * float(0x19) / float(0x3F)),
    (unsigned char)(float(0xFF) * float(0x1A) / float(0x3F)),
    (unsigned char)(float(0xFF) * float(0x1B) / float(0x3F)),
    (unsigned char)(float(0xFF) * float(0x1C) / float(0x3F)),
    (unsigned char)(float(0xFF) * float(0x1D) / float(0x3F)),
    (unsigned char)(float(0xFF) * float(0x1E) / float(0x3F)),
    (unsigned char)(float(0xFF) * float(0x1F) / float(0x3F)),
    (unsigned char)(float(0xFF) * float(0x20) / float(0x3F)),
    (unsigned char)(float(0xFF) * float(0x21) / float(0x3F)),
    (unsigned char)(float(0xFF) * float(0x22) / float(0x3F)),
    (unsigned char)(float(0xFF) * float(0x23) / float(0x3F)),
    (unsigned char)(float(0xFF) * float(0x24) / float(0x3F)),
    (unsigned char)(float(0xFF) * float(0x25) / float(0x3F)),
    (unsigned char)(float(0xFF) * float(0x26) / float(0x3F)),
    (unsigned char)(float(0xFF) * float(0x27) / float(0x3F)),
    (unsigned char)(float(0xFF) * float(0x28) / float(0x3F)),
    (unsigned char)(float(0xFF) * float(0x29) / float(0x3F)),
    (unsigned char)(float(0xFF) * float(0x2A) / float(0x3F)),
    (unsigned char)(float(0xFF) * float(0x2B) / float(0x3F)),
    (unsigned char)(float(0xFF) * float(0x2C) / float(0x3F)),
    (unsigned char)(float(0xFF) * float(0x2D) / float(0x3F)),
    (unsigned char)(float(0xFF) * float(0x2E) / float(0x3F)),
    (unsigned char)(float(0xFF) * float(0x2F) / float(0x3F)),
    (unsigned char)(float(0xFF) * float(0x30) / float(0x3F)),
    (unsigned char)(float(0xFF) * float(0x31) / float(0x3F)),
    (unsigned char)(float(0xFF) * float(0x32) / float(0x3F)),
    (unsigned char)(float(0xFF) * float(0x33) / float(0x3F)),
    (unsigned char)(float(0xFF) * float(0x34) / float(0x3F)),
    (unsigned char)(float(0xFF) * float(0x35) / float(0x3F)),
    (unsigned char)(float(0xFF) * float(0x36) / float(0x3F)),
    (unsigned char)(float(0xFF) * float(0x37) / float(0x3F)),
    (unsigned char)(float(0xFF) * float(0x38) / float(0x3F)),
    (unsigned char)(float(0xFF) * float(0x39) / float(0x3F)),
    (unsigned char)(float(0xFF) * float(0x3A) / float(0x3F)),
    (unsigned char)(float(0xFF) * float(0x3B) / float(0x3F)),
    (unsigned char)(float(0xFF) * float(0x3C) / float(0x3F)),
    (unsigned char)(float(0xFF) * float(0x3D) / float(0x3F)),
    (unsigned char)(float(0xFF) * float(0x3E) / float(0x3F)),
    (unsigned char)(float(0xFF) * float(0x3F) / float(0x3F)),
};

void Decode_unorm_bc1(unsigned char *dst, const unsigned char *src, int x, int y, int dstStride)
{
    assert(((x & 3) == 0) && ((y & 3) == 0));

    RGBA srcRgba[4];
    {
        unsigned int color0 = (src[0] << 8) | (src[1] << 0); // Data comes in big endian
        unsigned int color1 = (src[2] << 8) | (src[3] << 0); // so we swap bytes

        srcRgba[0].r_ = Table1F[(color0 >> (5+6)) & 0x1F];
        srcRgba[0].g_ = Table3F[(color0 >> (5  )) & 0x3F];
        srcRgba[0].b_ = Table1F[(color0 >> (0  )) & 0x1F];
        srcRgba[0].a_ = 0xFF;

        srcRgba[1].r_ = Table1F[(color1 >> (5+6)) & 0x1F];
        srcRgba[1].g_ = Table3F[(color1 >> (5  )) & 0x3F];
        srcRgba[1].b_ = Table1F[(color1 >> (0  )) & 0x1F];
        srcRgba[1].a_ = 0xFF;

        if (color0 > color1)
        {
            srcRgba[2].r_ = (unsigned char)((srcRgba[0].r_ * 2 + srcRgba[1].r_) / 3);
            srcRgba[2].g_ = (unsigned char)((srcRgba[0].g_ * 2 + srcRgba[1].g_) / 3);
            srcRgba[2].b_ = (unsigned char)((srcRgba[0].b_ * 2 + srcRgba[1].b_) / 3);
            srcRgba[2].a_ = 0xFF;

            srcRgba[3].r_ = (unsigned char)((srcRgba[0].r_ + srcRgba[1].r_ * 2) / 3);
            srcRgba[3].g_ = (unsigned char)((srcRgba[0].g_ + srcRgba[1].g_ * 2) / 3);
            srcRgba[3].b_ = (unsigned char)((srcRgba[0].b_ + srcRgba[1].b_ * 2) / 3);
            srcRgba[3].a_ = 0xFF;
        }
        else
        {
            srcRgba[2].r_ = (unsigned char)((srcRgba[0].r_ + srcRgba[1].r_) / 2);
            srcRgba[2].g_ = (unsigned char)((srcRgba[0].g_ + srcRgba[1].g_) / 2);
            srcRgba[2].b_ = (unsigned char)((srcRgba[0].b_ + srcRgba[1].b_) / 2);
            srcRgba[2].a_ = 0xFF;

            srcRgba[3].r_ = 0x00;
            srcRgba[3].g_ = 0x00;
            srcRgba[3].b_ = 0x00;
            srcRgba[3].a_ = 0x00;
        }
    }

    {
        RGBA *dstRgba = reinterpret_cast<RGBA *>(dst + x * sizeof(RGBA) + y * dstStride);

        int colorRgbaStride = dstStride / sizeof(RGBA);

        dstRgba[colorRgbaStride * 0 + 3] = srcRgba[(src[4] >> (2*0)) & 3];
        dstRgba[colorRgbaStride * 0 + 2] = srcRgba[(src[4] >> (2*1)) & 3];
        dstRgba[colorRgbaStride * 0 + 1] = srcRgba[(src[4] >> (2*2)) & 3];
        dstRgba[colorRgbaStride * 0 + 0] = srcRgba[(src[4] >> (2*3)) & 3];

        dstRgba[colorRgbaStride * 1 + 3] = srcRgba[(src[5] >> (2*0)) & 3];
        dstRgba[colorRgbaStride * 1 + 2] = srcRgba[(src[5] >> (2*1)) & 3];
        dstRgba[colorRgbaStride * 1 + 1] = srcRgba[(src[5] >> (2*2)) & 3];
        dstRgba[colorRgbaStride * 1 + 0] = srcRgba[(src[5] >> (2*3)) & 3];

        dstRgba[colorRgbaStride * 2 + 3] = srcRgba[(src[6] >> (2*0)) & 3];
        dstRgba[colorRgbaStride * 2 + 2] = srcRgba[(src[6] >> (2*1)) & 3];
        dstRgba[colorRgbaStride * 2 + 1] = srcRgba[(src[6] >> (2*2)) & 3];
        dstRgba[colorRgbaStride * 2 + 0] = srcRgba[(src[6] >> (2*3)) & 3];

        dstRgba[colorRgbaStride * 3 + 3] = srcRgba[(src[7] >> (2*0)) & 3];
        dstRgba[colorRgbaStride * 3 + 2] = srcRgba[(src[7] >> (2*1)) & 3];
        dstRgba[colorRgbaStride * 3 + 1] = srcRgba[(src[7] >> (2*2)) & 3];
        dstRgba[colorRgbaStride * 3 + 0] = srcRgba[(src[7] >> (2*3)) & 3];
    }
}


//------------------------------------------------------------------------------
// Convert ctex binary data to Preview-possible image ( linear format )
// Made to be static so that it can be called from outside
//------------------------------------------------------------------------------
bool CNWCTexLoader::ConvertToPreviewImage( CNWTexture *pTexture,
                                           const unsigned char* srcBuf,
                                           int srcBufSize,
                                           const unsigned short * palette,
                                           NW_CUBE_FACE cubeFace,
                                           Gdiplus::BitmapData &bmpData )
{
    Gdiplus::Bitmap *pPreviewBitmap = pTexture->GetPreviewBitmap();
    if (pPreviewBitmap==NULL)
    {
        return false;
    } // End if

    int xStartOffset = 0;
    int yStartOffset = 0;
    bool bFlipUV = false;

    if (pTexture->GetTextureType()==NW_IMG_TYPE_CUBE)
    {
        switch (cubeFace)
        {
            case NW_CUBE_FACE_POS_X :
                xStartOffset = pTexture->GetWidth() * 2;
                yStartOffset = pTexture->GetHeight();
                bFlipUV = true;
                break;

            case NW_CUBE_FACE_NEG_X :
                xStartOffset = 0;
                yStartOffset = pTexture->GetHeight();
                bFlipUV = true;
                break;

            case NW_CUBE_FACE_POS_Y :
                xStartOffset = pTexture->GetWidth();
                yStartOffset = 0;
                bFlipUV = false;
                break;

            case NW_CUBE_FACE_NEG_Y :
                xStartOffset = pTexture->GetWidth();
                yStartOffset = pTexture->GetHeight()*2;
                bFlipUV = false;
                break;

            case NW_CUBE_FACE_POS_Z :
                xStartOffset = pTexture->GetWidth() * 3;
                yStartOffset = pTexture->GetHeight();
                bFlipUV = true;
                break;

            case NW_CUBE_FACE_NEG_Z :
                xStartOffset = pTexture->GetWidth();
                yStartOffset = pTexture->GetHeight();
                bFlipUV = true;
                break;
        } // End switch
    } // End if

    // Destination is always RGBA32
    int numBytesPerPixel = 4;

    // Get source and destination buffer
    const unsigned char *pSrcBuffer  = srcBuf;
    unsigned char *pDestBuffer = (unsigned char*)bmpData.Scan0 +
                                  xStartOffset * numBytesPerPixel +
                                  yStartOffset * bmpData.Stride;

    // Start outputting
    if (pTexture->GetFormat()==NW_IMG_FMT_R_IA8 ||
        pTexture->GetFormat()==NW_IMG_FMT_R_C14 ||
        // pTexture->GetFormat()==NW_IMG_FMT_R_R5G6B5 || // This case will not happen because of name collision
        (pTexture->GetFormat()==NW_IMG_FMT_RGB565 && pTexture->TGAType()==NW_TGA_4R) ||
        pTexture->GetFormat()==NW_IMG_FMT_R_RGB5A3 ||
        // pTexture->GetFormat()==NW_IMG_FMT_R_RGBA8 || // This case will not happen because of name collision
        (pTexture->GetFormat()==NW_IMG_FMT_RGBA8 && pTexture->TGAType()==NW_TGA_4R))
    {
        int numBlocksX = pTexture->GetWidth()  / 4;
        int numBlocksY = pTexture->GetHeight() / 4;
        int numBlocks = numBlocksX * numBlocksY;
        int srcBytesPerBlock = NWGetTextureFormatBitsPerPixel(pTexture->GetFormat()) * 4 * 4 / 8;

        int srcBufSizeExpected = srcBytesPerBlock * numBlocks;
        if (srcBufSize<srcBufSizeExpected)
        {
            return false;
        }

        int i;

        //#pragma omp parallel for
        for (i=0;i<numBlocks;i++)
        {
            const unsigned char *pSrcBlock = pSrcBuffer + i*srcBytesPerBlock;
            ConvertToPreviewImageBlock_4x4(pTexture,pSrcBlock,pDestBuffer,palette,
                                           i,numBlocksX,numBlocksY,bFlipUV,bmpData);

            Sleep(0);
        }
    }
    else if (pTexture->GetFormat()==NW_IMG_FMT_R_I8 ||
             pTexture->GetFormat()==NW_IMG_FMT_R_IA4 ||
             pTexture->GetFormat()==NW_IMG_FMT_R_C8)
    {
        int numBlocksX = pTexture->GetWidth()  / 8;
        int numBlocksY = pTexture->GetHeight() / 4;
        int numBlocks = numBlocksX * numBlocksY;
        int srcBytesPerBlock = NWGetTextureFormatBitsPerPixel(pTexture->GetFormat()) * 8 * 4 / 8;

        int i;

        int srcBufSizeExpected = srcBytesPerBlock * numBlocks;
        if (srcBufSize<srcBufSizeExpected)
        {
            return false;
        }

        //#pragma omp parallel for
        for (i=0;i<numBlocks;i++)
        {
            const unsigned char *pSrcBlock = pSrcBuffer + i*srcBytesPerBlock;
            ConvertToPreviewImageBlock_8x4(pTexture,pSrcBlock,pDestBuffer,palette,
                                           i,numBlocksX,numBlocksY,bFlipUV,bmpData);

            Sleep(0);
        }
    }
    else if (pTexture->GetFormat()==NW_IMG_FMT_R_CMPR)
    {
        int blockWidth  = 4;
        int blockHeight = 4;
        int numBlocksX  = (pTexture->GetWidth()+3)  / blockWidth;
        int numBlocksY  = (pTexture->GetHeight()+3) / blockHeight;
        int    blockSize   = (blockWidth * blockHeight) / 2;

        int srcBufSizeExpected = blockSize * numBlocksX * numBlocksY;
        if (srcBufSize<srcBufSizeExpected)
        {
            return false;
        }

        const unsigned char * pSrcBlock = pSrcBuffer;

        for (int j = 0; j < numBlocksY; j += 2)
        {
            for (int i = 0; i < numBlocksX; i += 2)
            {
                Decode_unorm_bc1(pDestBuffer, pSrcBlock, blockWidth * (i    ), blockHeight * (j    ), bmpData.Stride); pSrcBlock += blockSize;

                if ((i+1)<numBlocksX)
                {
                    Decode_unorm_bc1(pDestBuffer, pSrcBlock, blockWidth * (i + 1), blockHeight * (j    ), bmpData.Stride); pSrcBlock += blockSize;
                }

                if ((j+1)<numBlocksY)
                {
                    Decode_unorm_bc1(pDestBuffer, pSrcBlock, blockWidth * (i    ), blockHeight * (j + 1), bmpData.Stride); pSrcBlock += blockSize;

                    if ((i+1)<numBlocksX)
                    {
                        Decode_unorm_bc1(pDestBuffer, pSrcBlock, blockWidth * (i + 1), blockHeight * (j + 1), bmpData.Stride); pSrcBlock += blockSize;
                    }
                }
            }

            Sleep(0);
        }
    }
    else if (pTexture->GetFormat()==NW_IMG_FMT_R_I4 ||
             pTexture->GetFormat()==NW_IMG_FMT_R_C4 ||
             ((pTexture->GetFormat()!=NW_IMG_FMT_ETC1) &&
              (pTexture->GetFormat()!=NW_IMG_FMT_ETCA)))
    {
        // Non-compressed format
        int numBlocksX = pTexture->GetWidth()  / 8;
        int numBlocksY = pTexture->GetHeight() / 8;

        int numBlocks = numBlocksX * numBlocksY;

        // Number of blocks must be at least 2x2
        if (numBlocksX<=1)
            numBlocks *= 2;

        if (numBlocksY<=1)
            numBlocks *= 2;

        int srcBytesPerBlock = NWGetTextureFormatBitsPerPixel(pTexture->GetFormat()) * 8 * 8 / 8;

        int srcBufSizeExpected = srcBytesPerBlock * numBlocks;
        if (srcBufSize<srcBufSizeExpected)
        {
            return false;
        }

        int i;

        //#pragma omp parallel for
        for (i=0;i<numBlocks;i++)
        {
            const unsigned char *pSrcBlock = pSrcBuffer + i*srcBytesPerBlock;
            ConvertToPreviewImageBlock_8x8(pTexture,pSrcBlock,pDestBuffer,palette,
                                           i,numBlocksX,numBlocksY,bFlipUV,bmpData);

            Sleep(0);
        }
    }
    else if (pTexture->GetFormat()==NW_IMG_FMT_ETC1)
    {
        int numBlocksX = pTexture->GetWidth()  / 4;
        int numBlocksY = pTexture->GetHeight() / 4;

        int srcBytesPerBlock = 8;

        int numBlocks = numBlocksX * numBlocksY;

        // Number of blocks must be at least 2x2
        if (numBlocksX<=1)
            numBlocks *= 2;

        if (numBlocksY<=1)
            numBlocks *= 2;

        int srcBufSizeExpected = srcBytesPerBlock * numBlocks;
        if (srcBufSize<srcBufSizeExpected)
        {
            return false;
        }

        int i;

        //#pragma omp parallel for
        for (i=0;i<numBlocks;i++)
        {
            const unsigned char *pSrcBlock = pSrcBuffer + i*srcBytesPerBlock;
            ConvertToPreviewImageBlock_4x4_Etc1(pTexture,pSrcBlock,pDestBuffer,
                                                i,numBlocksX,numBlocksY,
                                                bFlipUV,false,bmpData);

            Sleep(0);
        }
    }
    else if (pTexture->GetFormat()==NW_IMG_FMT_ETCA)
    {
        int numBlocksX = pTexture->GetWidth()  / 4;
        int numBlocksY = pTexture->GetHeight() / 4;

        int numBlocks = numBlocksX * numBlocksY;

        // Number of blocks must be at least 2x2
        if (numBlocksX<=1)
            numBlocks *= 2;

        if (numBlocksY<=1)
            numBlocks *= 2;

        int srcBytesPerBlock = 16;

        int srcBufSizeExpected = srcBytesPerBlock * numBlocks;
        if (srcBufSize<srcBufSizeExpected)
        {
            return false;
        }

        int i;

        //#pragma omp parallel for
        for (i=0;i<numBlocks;i++)
        {
            const unsigned char *pSrcBlock = pSrcBuffer + i*srcBytesPerBlock;
            ConvertToPreviewImageBlock_4x4_Etc1(pTexture,pSrcBlock,pDestBuffer,
                                                i,numBlocksX,numBlocksY,
                                                bFlipUV,true,bmpData);

            Sleep(0);
        }
    }

    return true;
} // End of ConvertToPreviewImage for CNWCTexLoader

//------------------------------------------------------------------------------
// Reads a pixel color according to the given format and saves it as ARGB in dest
// Destination pixels are organized in B,G,R,A order ( from low to high bytes )
//------------------------------------------------------------------------------
void convertI8ToARGB8(unsigned char * dest, const void * src)
{
    // [IIII.IIII]
    unsigned char a = 0xff;
    unsigned char x = *(unsigned char*)src;

    dest[0] = x;
    dest[1] = x;
    dest[2] = x;
    dest[3] = a;
}

void convertIA4ToARGB8(unsigned char * dest, const void * src)
{
    // [AAAA.IIII]
    unsigned char a = ((*(unsigned char*)src)&0xf0);
    unsigned char x = ((*(unsigned char*)src)<<4);

    // Fill bits
    x |= (x>>4);

    dest[0] = x;
    dest[1] = x;
    dest[2] = x;
    dest[3] = a;
}

void convertIA8ToARGB8(unsigned char * dest, const void * src)
{
    unsigned short value = NWSwapByteOrderWord(*(unsigned short*)src);

    // [AAAA.AAAA.IIII.IIII]
    unsigned char a = unsigned char((value&0xff00)>>8);
    unsigned char x = unsigned char(value&0x00ff);

    dest[0] = x;
    dest[1] = x;
    dest[2] = x;
    dest[3] = a;
}

void convertRGB565ToARGB8(unsigned char * dest, const void * src)
{
    unsigned short value = NWSwapByteOrderWord(*(unsigned short *)src);

    // [RRRR.RGGG.GGGB.BBBB]
    unsigned char a = 0xff;
    unsigned char r = unsigned char((value&0xf800)>>8);
    unsigned char g = unsigned char((value&0x07e0)>>3);
    unsigned char b = unsigned char((value&0x001f)<<3);

    // Fill bits
    r |= (r>>5);
    g |= (g>>6);
    b |= (b>>5);

    dest[0] = b;
    dest[1] = g;
    dest[2] = r;
    dest[3] = a;
}

void convertRGB5A3ToARGB8(unsigned char * dest, const void * src)
{
    unsigned char a = 0;
    unsigned char r = 0;
    unsigned char g = 0;
    unsigned char b = 0;

    unsigned short value = NWSwapByteOrderWord(*(unsigned short *)src);
    if ((value&0x8000) == 0)
    {
        // [0AAA.RRRR.GGGG.BBBB]
        a = unsigned char((value&0x7000)>>7);
        r = unsigned char((value&0x0f00)>>4);
        g = unsigned char((value&0x00f0));
        b = unsigned char((value&0x000f)<<4);

        // Fill bits
        a |= (a>>3) | (a>>6);
        r |= (r>>4);
        g |= (g>>4);
        b |= (b>>4);
    }
    else
    {
        // [1RRR.RRGG.GGGB.BBBB]
        a = 0xff;
        r = unsigned char((value&0x7c00)>>7);
        g = unsigned char((value&0x03e0)>>2);
        b = unsigned char((value&0x001f)<<3);

        // Fill bits
        r |= (r>>5);
        g |= (g>>5);
        b |= (b>>5);
    }

    dest[0] = b;
    dest[1] = g;
    dest[2] = r;
    dest[3] = a;
}

//------------------------------------------------------------------------------
// Convert nxm non-compressed block using the given converter
//------------------------------------------------------------------------------
void convertPixelBlock8b(unsigned char * pDestBlock,
                         const unsigned char * pSrcBuffer,
                         int width, int height,
                         Gdiplus::BitmapData &bmpData,
                         void (*converter)(unsigned char * dest, const void * src))
{
    const unsigned char *pSrcPixel = (unsigned char *)pSrcBuffer;

    for (int j=0; j < height; ++j)
        for (int i = 0; i < width; ++i)
        {
            unsigned char * pDestPixel = pDestBlock + 4 * i + bmpData.Stride * j;
            converter(pDestPixel, pSrcPixel);
            ++pSrcPixel;
        }
}

void convertPixelBlock16b(unsigned char * pDestBlock,
                          const unsigned char * pSrcBuffer,
                          int width, int height,
                          Gdiplus::BitmapData &bmpData,
                          void (*converter)(unsigned char * dest, const void * src))
{
    const unsigned short *pSrcPixel = (unsigned short *)pSrcBuffer;

    for (int j=0; j < height; ++j)
        for (int i = 0; i < width; ++i)
        {
            unsigned char * pDestPixel = pDestBlock + 4 * i + bmpData.Stride * j;
            converter(pDestPixel, pSrcPixel);
            ++pSrcPixel;
        }
}

void convertPixelBlock8b(unsigned char * pDestBlock,
                         const unsigned char * pSrcBuffer,
                         const unsigned short * palette,
                         int width, int height,
                         Gdiplus::BitmapData &bmpData,
                         void (*converter)(unsigned char * dest, const void * src))
{
    unsigned char *pSrcPixel = (unsigned char*)pSrcBuffer;

    for (int j=0; j < height; ++j)
        for (int i = 0; i < width; ++i)
        {
            unsigned char * pDestPixel = pDestBlock + 4 * i + bmpData.Stride * j;
            unsigned char index = *pSrcPixel;
            converter(pDestPixel, palette + index);
            ++pSrcPixel;
        }
}

void convertPixelBlock16b(unsigned char * pDestBlock,
                         const unsigned char * pSrcBuffer,
                         const unsigned short * palette,
                         int width, int height,
                         Gdiplus::BitmapData &bmpData,
                         void (*converter)(unsigned char * dest, const void * src))
{
    unsigned short *pSrcPixel = (unsigned short*)pSrcBuffer;

    for (int j=0; j < height; ++j)
        for (int i = 0; i < width; ++i)
        {
            unsigned char * pDestPixel = pDestBlock + 4 * i + bmpData.Stride * j;
            unsigned short index = NWSwapByteOrderWord(*pSrcPixel);
            converter(pDestPixel, palette + index);
            ++pSrcPixel;
        }
}

//------------------------------------------------------------------------------
// Convert 4x4 non-compressed block to linear PC displayable format
//------------------------------------------------------------------------------
bool CNWCTexLoader::ConvertToPreviewImageBlock_4x4( CNWTexture *pTexture,
                                                    const unsigned char* pSrcBuffer,
                                                    unsigned char *pDestBuffer,
                                                    const unsigned short * palette,
                                                    int blockIndex,
                                                    int numBlocksX,
                                                    int numBlocksY,
                                                    bool bFlipUV,
                                                    Gdiplus::BitmapData &bmpData )
{
//    PixelOffset *pPixelOffsetTable = s_blockToLinearPixelOffset;
    int destBlockX = blockIndex % numBlocksX;
    int destBlockY = blockIndex / numBlocksX;

    if (bFlipUV)
    {
        destBlockX = numBlocksX - destBlockX - 1;
        destBlockY = numBlocksY - destBlockY - 1;
//        pPixelOffsetTable   = s_blockToLinearPixelOffsetFlipUV;
    }

    unsigned char *pDestBlock = pDestBuffer +
                                destBlockX * 4 * 4 +
                                destBlockY * bmpData.Stride * 4;

    // Note : Destination pixels are organized in B,G,R,A order ( from low to high bytes )
    //
    // pDestPixel[0] is Blue
    // pDestPixel[1] is Green
    // pDestPixel[2] is Red
    // pDestPixel[3] is Alpha
    //

    switch (pTexture->GetFormat())
    {
        case NW_IMG_FMT_R_IA8:
            convertPixelBlock16b(pDestBlock, pSrcBuffer, 4, 4, bmpData, convertIA8ToARGB8);
            break;
        // case NW_IMG_FMT_R_R5G6B5: // This case will not happen because of name collision
        case NW_IMG_FMT_RGB565:
            convertPixelBlock16b(pDestBlock, pSrcBuffer, 4, 4, bmpData, convertRGB565ToARGB8);
            break;
        case NW_IMG_FMT_R_RGB5A3:
            convertPixelBlock16b(pDestBlock, pSrcBuffer, 4, 4, bmpData, convertRGB5A3ToARGB8);
            break;
        // case NW_IMG_FMT_R_RGBA8: // This case will not happen because of name collision
        case NW_IMG_FMT_RGBA8:
            {
                unsigned short *pSrcPixel = (unsigned short*)pSrcBuffer;

                for (int j=0;j<4;++j)
                    for (int i=0;i<4;++i)
                    {
                        unsigned char * pDestPixel = pDestBlock + 4 * i + bmpData.Stride * j;
                        unsigned short value = NWSwapByteOrderWord(*pSrcPixel);

                        // [AAAA.AAAA.RRRR.RRRR]
                        unsigned char a = unsigned char((value&0xff00)>>8);
                        unsigned char r = unsigned char(value&0x00ff);

                        pDestPixel[2] = r;
                        pDestPixel[3] = a;

                        ++pSrcPixel;
                    }
                for (int j=0;j<4;++j)
                    for (int i=0;i<4;++i)
                    {
                        unsigned char * pDestPixel = pDestBlock + 4 * i + bmpData.Stride * j;
                        unsigned short value = NWSwapByteOrderWord(*pSrcPixel);

                        // [GGGG.GGGG.BBBB.BBBB]
                        unsigned char g = unsigned char((value&0xff00)>>8);
                        unsigned char b = unsigned char(value&0x00ff);

                        pDestPixel[0] = b;
                        pDestPixel[1] = g;

                        ++pSrcPixel;
                    }
            }
            break;
        case NW_IMG_FMT_R_C14:
            {
                void (*converter)(unsigned char * dest, const void * src) = NULL;

                switch (pTexture->GetPaletteFormat())
                {
                case NW_IMG_FMT_R_IA8:
                    converter = convertIA8ToARGB8;
                    break;
                // case NW_IMG_FMT_R_R5G6B5: // This case will not happen because of name collision
                case NW_IMG_FMT_RGB565:
                    converter = convertRGB565ToARGB8;
                    break;
                case NW_IMG_FMT_R_RGB5A3:
                    converter = convertRGB5A3ToARGB8;
                    break;
                default:
                    return false;
                }
                convertPixelBlock16b(pDestBlock, pSrcBuffer, palette, 4, 4, bmpData, converter);
            }
            break;
    }
    return true;
}

//------------------------------------------------------------------------------
// Convert 8x4 non-compressed block to linear PC displayable format
//------------------------------------------------------------------------------
bool CNWCTexLoader::ConvertToPreviewImageBlock_8x4( CNWTexture *pTexture,
                                                    const unsigned char* pSrcBuffer,
                                                    unsigned char *pDestBuffer,
                                                    const unsigned short * palette,
                                                    int blockIndex,
                                                    int numBlocksX,
                                                    int numBlocksY,
                                                    bool bFlipUV,
                                                    Gdiplus::BitmapData &bmpData )
{
    int destBlockX = blockIndex % numBlocksX;
    int destBlockY = blockIndex / numBlocksX;

    if (bFlipUV)
    {
        destBlockX = numBlocksX - destBlockX - 1;
        destBlockY = numBlocksY - destBlockY - 1;
    }

    unsigned char *pDestBlock = pDestBuffer +
                                destBlockX * 4 * 8 +
                                destBlockY * bmpData.Stride * 4;

    // Note : Destination pixels are organized in B,G,R,A order ( from low to high bytes )
    //
    // pDestPixel[0] is Blue
    // pDestPixel[1] is Green
    // pDestPixel[2] is Red
    // pDestPixel[3] is Alpha
    //

    switch (pTexture->GetFormat())
    {
        case NW_IMG_FMT_R_I8:
            convertPixelBlock8b(pDestBlock, pSrcBuffer, 8, 4, bmpData, convertI8ToARGB8);
            break;
        case NW_IMG_FMT_R_IA4:
            convertPixelBlock8b(pDestBlock, pSrcBuffer, 8, 4, bmpData, convertIA4ToARGB8);
            break;
        case NW_IMG_FMT_R_C8:
            {
                void (*converter)(unsigned char * dest, const void * src) = NULL;

                switch (pTexture->GetPaletteFormat())
                {
                case NW_IMG_FMT_R_IA8:
                    converter = convertIA8ToARGB8;
                    break;
                // case NW_IMG_FMT_R_R5G6B5: // This case will not happen because of name collision
                case NW_IMG_FMT_RGB565:
                    converter = convertRGB565ToARGB8;
                    break;
                case NW_IMG_FMT_R_RGB5A3:
                    converter = convertRGB5A3ToARGB8;
                    break;
                default:
                    return false;
                }
                convertPixelBlock8b(pDestBlock, pSrcBuffer, palette, 8, 4, bmpData, converter);
            }
            break;
    }
    return true;
}

//------------------------------------------------------------------------------
// Convert 8x8 non-compressed block to linear PC displayable format
//------------------------------------------------------------------------------
bool CNWCTexLoader::ConvertToPreviewImageBlock_8x8( CNWTexture *pTexture,
                                                    const unsigned char* pSrcBuffer,
                                                    unsigned char *pDestBuffer,
                                                    const unsigned short * palette,
                                                    int blockIndex,
                                                    int numBlocksX,
                                                    int numBlocksY,
                                                    bool bFlipUV,
                                                    Gdiplus::BitmapData &bmpData )
{
    PixelOffset *pPixelOffsetTable = s_blockToLinearPixelOffset;
    int destBlockX = blockIndex % numBlocksX;
    int destBlockY = blockIndex / numBlocksX;

    if (bFlipUV)
    {
        destBlockX = numBlocksX - destBlockX - 1;
        destBlockY = numBlocksY - destBlockY - 1;
        pPixelOffsetTable   = s_blockToLinearPixelOffsetFlipUV;
    } // End if

    unsigned char *pDestBlock = pDestBuffer +
                                destBlockX * 4 * 8 +
                                destBlockY * bmpData.Stride * 8;
    unsigned char *pDestPixel = NULL;

    // Note : Source pixels are organized in B,G,R or A,B,G,R order ( from low to high bytes )

    // Note : Destination pixels are organized in B,G,R,A order ( from low to high bytes )
    //
    // pDestPixel[0] is Blue
    // pDestPixel[1] is Green
    // pDestPixel[2] is Red
    // pDestPixel[3] is Alpha
    //

    switch (pTexture->GetFormat())
    {
        case NW_IMG_FMT_L4 :
            {
                unsigned char *pSrcPixel = (unsigned char*)pSrcBuffer;

                for (int i=0;i<8*8;i++)
                {
                    pDestPixel = pDestBlock +
                                 pPixelOffsetTable[i].x * 4 +
                                 pPixelOffsetTable[i].y * bmpData.Stride;

                    unsigned char luminanceVal;
                    if ((i&0x01)==0)
                    {
                        luminanceVal = pSrcPixel[0] & 0x0F;
                    } // End if
                    else
                    {
                        luminanceVal = (pSrcPixel[0]>>4) & 0x0F;
                        pSrcPixel++;
                    } // End else

                    // Extend to 8 bits
                    luminanceVal = (luminanceVal<<4) + luminanceVal;

                    pDestPixel[0] = (unsigned char)luminanceVal;
                    pDestPixel[1] = (unsigned char)luminanceVal;
                    pDestPixel[2] = (unsigned char)luminanceVal;
                    pDestPixel[3] = (unsigned char)255;
                } // End for
            } // End case
            break;

        case NW_IMG_FMT_R_I4:
            {
                unsigned char *pSrcDoublePixel = (unsigned char*)pSrcBuffer;

                for (int j=0;j<8;++j)
                    for (int i=0;i<4;++i)
                    {
                        unsigned char * pDestPixel0 = pDestBlock + 4 * 2 * i + bmpData.Stride * j;
                        unsigned char * pDestPixel1 = pDestBlock + 4 * (2 * i + 1) + bmpData.Stride * j;

                        // [IIII.JJJJ]
                        unsigned char a = 0xff;
                        unsigned char i0 = ((*pSrcDoublePixel)&0xf0);
                        unsigned char i1 = ((*pSrcDoublePixel)<<4);

                        pDestPixel0[0] = i0;
                        pDestPixel0[1] = i0;
                        pDestPixel0[2] = i0;
                        pDestPixel0[3] = a;

                        pDestPixel1[0] = i1;
                        pDestPixel1[1] = i1;
                        pDestPixel1[2] = i1;
                        pDestPixel1[3] = a;

                        ++pSrcDoublePixel;
                    }
            }
            break;

        case NW_IMG_FMT_R_C4:
            {
                void (*converter)(unsigned char * dest, const void * src) = NULL;

                switch (pTexture->GetPaletteFormat())
                {
                case NW_IMG_FMT_R_IA8:
                    converter = convertIA8ToARGB8;
                    break;
                // case NW_IMG_FMT_R_R5G6B5: // This case will not happen because of name collision
                case NW_IMG_FMT_RGB565:
                    converter = convertRGB565ToARGB8;
                    break;
                case NW_IMG_FMT_R_RGB5A3:
                    converter = convertRGB5A3ToARGB8;
                    break;
                default:
                    return false;
                }

                unsigned char *pSrcDoublePixel = (unsigned char*)pSrcBuffer;

                for (int j=0;j<8;++j)
                    for (int i=0;i<4;++i)
                    {
                        unsigned char * pDestPixel0 = pDestBlock + 4 * 2 * i + bmpData.Stride * j;
                        unsigned char * pDestPixel1 = pDestBlock + 4 * (2 * i + 1) + bmpData.Stride * j;

                        // [IIII.JJJJ]
                        unsigned char i0 = ((*pSrcDoublePixel)>>4);
                        unsigned char i1 = ((*pSrcDoublePixel)&0x0f);

                        converter(pDestPixel0, palette + i0);
                        converter(pDestPixel1, palette + i1);

                        ++pSrcDoublePixel;
                    }
            }
            break;

        case NW_IMG_FMT_L8 :
            {
                unsigned char *pSrcPixel = (unsigned char*)pSrcBuffer;

                for (int i=0;i<8*8;i++)
                {
                    pDestPixel = pDestBlock +
                                 pPixelOffsetTable[i].x * 4 +
                                 pPixelOffsetTable[i].y * bmpData.Stride;

                    pDestPixel[0] = (unsigned char)pSrcPixel[0];
                    pDestPixel[1] = (unsigned char)pSrcPixel[0];
                    pDestPixel[2] = (unsigned char)pSrcPixel[0];
                    pDestPixel[3] = (unsigned char)255;

                    pSrcPixel++;
                } // End for
            } // End case
            break;

        case NW_IMG_FMT_A4 :
            {
                unsigned char *pSrcPixel = (unsigned char*)pSrcBuffer;

                for (int i=0;i<8*8;i++)
                {
                    pDestPixel = pDestBlock +
                                 pPixelOffsetTable[i].x * 4 +
                                 pPixelOffsetTable[i].y * bmpData.Stride;

                    unsigned char alphaVal;
                    if ((i&0x01)==0)
                    {
                        alphaVal = pSrcPixel[0] & 0x0F;
                    } // End if
                    else
                    {
                        alphaVal = (pSrcPixel[0]>>4) & 0x0F;
                        pSrcPixel++;
                    } // End else

                    // Extend to 8 bits
                    alphaVal = (alphaVal<<4) + alphaVal;

                    pDestPixel[0] = (unsigned char)alphaVal;
                    pDestPixel[1] = (unsigned char)alphaVal;
                    pDestPixel[2] = (unsigned char)alphaVal;
                    pDestPixel[3] = (unsigned char)alphaVal;
                } // End for
            } // End case
            break;
            break;

        case NW_IMG_FMT_A8 :
            {
                unsigned char *pSrcPixel = (unsigned char*)pSrcBuffer;

                for (int i=0;i<8*8;i++)
                {
                    pDestPixel = pDestBlock +
                                 pPixelOffsetTable[i].x * 4 +
                                 pPixelOffsetTable[i].y * bmpData.Stride;

                    pDestPixel[0] = (unsigned char)pSrcPixel[0];
                    pDestPixel[1] = (unsigned char)pSrcPixel[0];
                    pDestPixel[2] = (unsigned char)pSrcPixel[0];
                    pDestPixel[3] = (unsigned char)pSrcPixel[0];

                    pSrcPixel++;
                } // End for
            } // End case
            break;

        case NW_IMG_FMT_LA4 :
            {
                unsigned char *pSrcPixel = (unsigned char*)pSrcBuffer;

                for (int i=0;i<8*8;i++)
                {
                    pDestPixel = pDestBlock +
                                 pPixelOffsetTable[i].x * 4 +
                                 pPixelOffsetTable[i].y * bmpData.Stride;

                    unsigned char alphaVal     = (pSrcPixel[0]>>0) & 0x0F;
                    unsigned char luminanceVal = (pSrcPixel[0]>>4) & 0x0F;

                    // Extend to 8 bits
                    luminanceVal = (luminanceVal<<4) + luminanceVal;
                    alphaVal     = (alphaVal<<4) + alphaVal;

                    pDestPixel[0] = (unsigned char)luminanceVal;
                    pDestPixel[1] = (unsigned char)luminanceVal;
                    pDestPixel[2] = (unsigned char)luminanceVal;
                    pDestPixel[3] = (unsigned char)alphaVal;

                    pSrcPixel++;
                } // End for
            } // End case
            break;

        case NW_IMG_FMT_LA8 :
            {
                unsigned char *pSrcPixel = (unsigned char*)pSrcBuffer;

                for (int i=0;i<8*8;i++)
                {
                    pDestPixel = pDestBlock +
                                 pPixelOffsetTable[i].x * 4 +
                                 pPixelOffsetTable[i].y * bmpData.Stride;

                    unsigned char alphaVal     = pSrcPixel[0];
                    unsigned char luminanceVal = pSrcPixel[1];

                    pDestPixel[0] = (unsigned char)luminanceVal;
                    pDestPixel[1] = (unsigned char)luminanceVal;
                    pDestPixel[2] = (unsigned char)luminanceVal;
                    pDestPixel[3] = alphaVal;

                    pSrcPixel+=2;
                } // End for
            } // End case
            break;

        case NW_IMG_FMT_HILO8 :
            {
                unsigned char *pSrcPixel = (unsigned char*)pSrcBuffer;

                for (int i=0;i<8*8;i++)
                {
                    pDestPixel = pDestBlock +
                                 pPixelOffsetTable[i].x * 4 +
                                 pPixelOffsetTable[i].y * bmpData.Stride;

                    unsigned char loVal = pSrcPixel[0];
                    unsigned char hiVal = pSrcPixel[1];

                    pDestPixel[0] = (unsigned char)0;
                    pDestPixel[1] = (unsigned char)loVal;
                    pDestPixel[2] = (unsigned char)hiVal;
                    pDestPixel[3] = 255;

                    pSrcPixel+=2;
                } // End for
            } // End case
            break;

        case NW_IMG_FMT_RGB565 :
            {
                unsigned short *pSrcPixel = (unsigned short*)pSrcBuffer;

                for (int i=0;i<8*8;i++)
                {
                    pDestPixel = pDestBlock +
                                 pPixelOffsetTable[i].x * 4 +
                                 pPixelOffsetTable[i].y * bmpData.Stride;

                    unsigned int srcPixel = (unsigned int)(*pSrcPixel);
                    unsigned int b = srcPixel & 0x1F;
                    unsigned int g = ( srcPixel >> 5 )  & 0x3F;
                    unsigned int r = ( srcPixel >> 11 ) & 0x1F;
                    unsigned int a = 0xFF;

                    float fR = (float)r / (float)0x1F;
                    float fG = (float)g / (float)0x3F;
                    float fB = (float)b / (float)0x1F;

                    r = (int)(fR * 255.0f);
                    g = (int)(fG * 255.0f);
                    b = (int)(fB * 255.0f);

                    pDestPixel[0] = (unsigned char)b;
                    pDestPixel[1] = (unsigned char)g;
                    pDestPixel[2] = (unsigned char)r;
                    pDestPixel[3] = (unsigned char)a;

                    pSrcPixel++;
                } // End for
            } // End case
            break;

        case NW_IMG_FMT_RGB8 :
            {
                unsigned char *pSrcPixel = (unsigned char*)pSrcBuffer;

                for (int i=0;i<8*8;i++)
                {
                    pDestPixel = pDestBlock +
                                 pPixelOffsetTable[i].x * 4 +
                                 pPixelOffsetTable[i].y * bmpData.Stride;

                    unsigned char b = (unsigned char)pSrcPixel[0];
                    unsigned char g = (unsigned char)pSrcPixel[1];
                    unsigned char r = (unsigned char)pSrcPixel[2];

                    pDestPixel[0] = (unsigned char)b;
                    pDestPixel[1] = (unsigned char)g;
                    pDestPixel[2] = (unsigned char)r;
                    pDestPixel[3] = (unsigned char)0xFF;

                    pSrcPixel += 3;
                } // End for
            } // End case
            break;

        case NW_IMG_FMT_RGB5_A1 :
            {
                unsigned short *pSrcPixel = (unsigned short*)pSrcBuffer;

                for (int i=0;i<8*8;i++)
                {
                    pDestPixel = pDestBlock +
                                 pPixelOffsetTable[i].x * 4 +
                                 pPixelOffsetTable[i].y * bmpData.Stride;

                    unsigned short srcPixel = (unsigned short)(*pSrcPixel);

                    unsigned int a = srcPixel & 0x01;
                    unsigned int b = ( srcPixel >> 1 )  & 0x1F;
                    unsigned int g = ( srcPixel >> 6 )  & 0x1F;
                    unsigned int r = ( srcPixel >> 11 ) & 0x1F;

                    float fR = (float)r / (float)0x1F;
                    float fG = (float)g / (float)0x1F;
                    float fB = (float)b / (float)0x1F;

                    r = (int)(fR * 255.0f);
                    g = (int)(fG * 255.0f);
                    b = (int)(fB * 255.0f);
                    if (a!=0)
                        a = 0xFF;

                    pDestPixel[0] = (unsigned char)b;
                    pDestPixel[1] = (unsigned char)g;
                    pDestPixel[2] = (unsigned char)r;
                    pDestPixel[3] = (unsigned char)a;

                    pSrcPixel++;
                } // End for
            } // End case
            break;

        case NW_IMG_FMT_RGBA4 :
            {
                unsigned char *pSrcPixel = (unsigned char*)pSrcBuffer;

                for (int i=0;i<8*8;i++)
                {
                    pDestPixel = pDestBlock +
                                 pPixelOffsetTable[i].x * 4 +
                                 pPixelOffsetTable[i].y * bmpData.Stride;

                    unsigned char a = pSrcPixel[0] & 0x0F;
                    unsigned char b = (pSrcPixel[0]>>4) & 0x0F;
                    unsigned char g = pSrcPixel[1] & 0x0F;
                    unsigned char r = (pSrcPixel[1]>>4) & 0x0F;

                    // Extend to 8 bits
                    r     = (r<<4) + r;
                    g     = (g<<4) + g;
                    b     = (b<<4) + b;
                    a     = (a<<4) + a;

                    pDestPixel[0] = (unsigned char)b;
                    pDestPixel[1] = (unsigned char)g;
                    pDestPixel[2] = (unsigned char)r;
                    pDestPixel[3] = (unsigned char)a;

                    pSrcPixel+=2;
                } // End for
            } // End case
            break;

        case NW_IMG_FMT_RGBA8 :
            {
                unsigned char *pSrcPixel = (unsigned char*)pSrcBuffer;

                for (int i=0;i<8*8;i++)
                {
                    pDestPixel = pDestBlock +
                                 pPixelOffsetTable[i].x * 4 +
                                 pPixelOffsetTable[i].y * bmpData.Stride;

                    unsigned char a = pSrcPixel[0];
                    unsigned char b = pSrcPixel[1];
                    unsigned char g = pSrcPixel[2];
                    unsigned char r = pSrcPixel[3];

                    pDestPixel[0] = (unsigned char)b;
                    pDestPixel[1] = (unsigned char)g;
                    pDestPixel[2] = (unsigned char)r;
                    pDestPixel[3] = (unsigned char)a;

                    pSrcPixel += 4;
                } // End for
            } // End case
            break;
    } // End switch

    return true;
} // End of ConvertToPreviewImageBlock_8x8 for CNWCTexLoader


//------------------------------------------------------------------------------
// Convert 8x8 etc1 compressed block to linear PC displayable format
// If bHasAlpha = true, format is etc1_a4
//------------------------------------------------------------------------------
bool CNWCTexLoader::ConvertToPreviewImageBlock_4x4_Etc1( CNWTexture *pTexture,
                                                         const unsigned char* pSrcBuffer,
                                                         unsigned char *pDestBuffer,
                                                         int blockIndex,
                                                         int numBlocksX,
                                                         int numBlocksY,
                                                         bool bFlipUV,
                                                         bool bHasAlpha,
                                                         Gdiplus::BitmapData &bmpData )
{
    NW_USE_VAR(pTexture);

    int i,j;

    int num2x2TilesPerRow = max(1,(numBlocksX/2));
    int tile2x2Loc = (blockIndex/4);
    int tile2x2Row = tile2x2Loc / (num2x2TilesPerRow);
    int tile2x2Col = tile2x2Loc - (tile2x2Row * num2x2TilesPerRow);
    int tileLocIn2x2 = blockIndex & 0x03;
    int tileLocIn2x2_x = tileLocIn2x2 & 0x01;
    int tileLocIn2x2_y = tileLocIn2x2 >> 1;

    int destBlockX = tile2x2Col * 2 + tileLocIn2x2_x;
    int destBlockY = tile2x2Row * 2 + tileLocIn2x2_y;
    if (bFlipUV)
    {
        destBlockX = numBlocksX - destBlockX - 1;
        destBlockY = numBlocksY - destBlockY - 1;
    } // End if

    if (destBlockX>=numBlocksX)
        return true;

    if (destBlockY>=numBlocksY)
        return true;

    unsigned char *pDestBlock = pDestBuffer +
                                destBlockX * 4 * 4 +
                                destBlockY * bmpData.Stride * 4;

    unsigned char *pDestPixel = NULL;

    unsigned int  *pEtcBlock      = NULL;
    unsigned char alphaTable[4][4];

    if (bHasAlpha)
    {
        unsigned char *pEtcBlockAlpha = (unsigned char*)(pSrcBuffer+0);
        int alphaIndex = 0;
        for (i=0;i<4;i++)
        {
            for (j=0;j<4;j++)
            {
                unsigned char alphaVal = pEtcBlockAlpha[alphaIndex];

                if ((j&0x01)==0)
                {
                    // Take first 4 bits
                    alphaVal &= 0x0F;

                    // Flip row and column
                    // Extend 4 bits to 8 bits
                    alphaTable[j][i] = (alphaVal<<4) + alphaVal;
                } // End if
                else
                {
                    // Take second 4 bits
                    alphaVal = (alphaVal>>4) & 0x0F;

                    // Flip row and column
                    // Extend 4 bits to 8 bits
                    alphaTable[j][i] = (alphaVal<<4) + alphaVal;

                    // Advance to next byte
                    alphaIndex++;
                } // End else
            } // End for
        } // End for

        pEtcBlock      = (unsigned int*)(pSrcBuffer+8);
    } // End if
    else
    {
        pEtcBlock      = (unsigned int*)(pSrcBuffer+0);
    } // End else

    unsigned int dwDwordLo  = pEtcBlock[0];
    unsigned int dwDwordHi  = pEtcBlock[1];

    bool bFlipBitOn = (dwDwordHi & 0x01) != 0;
    bool bDiffBitOn = (dwDwordHi & 0x02) != 0;

    // Codeword
    int cw1 = ( dwDwordHi >> 5 ) & 0x07;
    int cw2 = ( dwDwordHi >> 2 ) & 0x07;

    // Base colors
    unsigned int r1,r2;
    unsigned int g1,g2;
    unsigned int b1,b2;

    // Pixel Indices
    unsigned int pixelIndices[4][4];

    int shiftIndex = 0;
    for (i=0;i<4;i++)
    {
        for (j=0;j<4;j++)
        {
            unsigned int val = (((dwDwordLo>>(16+shiftIndex))&0x01)<<1) + ((dwDwordLo>>shiftIndex)&0x01);

            // i and j are flipped because Ext pixels are organized in column major
            pixelIndices[j][i] = s_pixelIndexModifierIndex[val];

            shiftIndex++;
        } // End for
    } // End for

    if (bDiffBitOn==false)
    {
        r1 = ( dwDwordHi & 0xF0000000 ) >> 28;
        r2 = ( dwDwordHi & 0x0F000000 ) >> 24;
        g1 = ( dwDwordHi & 0x00F00000 ) >> 20;
        g2 = ( dwDwordHi & 0x000F0000 ) >> 16;
        b1 = ( dwDwordHi & 0x0000F000 ) >> 12;
        b2 = ( dwDwordHi & 0x00000F00 ) >>  8;

        // Extend low 4 bits by copying high 4 bits
        // Same as duplicating 4 bits to make 8 bits
        r1 = r1 + (r1<<4);
        r2 = r2 + (r2<<4);
        g1 = g1 + (g1<<4);
        g2 = g2 + (g2<<4);
        b1 = b1 + (b1<<4);
        b2 = b2 + (b2<<4);
    } // End if
    else // If diff bit is on
    {
        int baseR1;
        int dG2;
        int baseG1;
        int dB2;
        int baseB1;
        int dR2;

        baseR1  = (dwDwordHi>>27) & 0x1F;
        dR2     = (dwDwordHi>>24) & 0x07;
        baseG1  = (dwDwordHi>>19) & 0x1F;
        dG2     = (dwDwordHi>>16) & 0x07;
        baseB1  = (dwDwordHi>>11) & 0x1F;
        dB2     = (dwDwordHi>>8)  & 0x07;

        // Convert to actual delta value from bits
        dR2 = s_3BitsDeltaTable[dR2];
        dG2 = s_3BitsDeltaTable[dG2];
        dB2 = s_3BitsDeltaTable[dB2];

        // Convert from 5 bits to 8 bits and fill low 3 bits by high 3 bits
        r1 = (baseR1<<3) | (baseR1>>2);
        g1 = (baseG1<<3) | (baseG1>>2);
        b1 = (baseB1<<3) | (baseB1>>2);

        // Add differential
        r2 = baseR1 + dR2;
        g2 = baseG1 + dG2;
        b2 = baseB1 + dB2;

        // Necessary? already calculated at encoding time?
        r2 = min(31,max(0,r2));
        g2 = min(31,max(0,g2));
        b2 = min(31,max(0,b2));

        // Now convert from 5 bits to 8 bits
        r2 = (r2<<3) | (r2>>2);
        g2 = (g2<<3) | (g2>>2);
        b2 = (b2<<3) | (b2>>2);
    } // End else

    // Now compute color for each 4x4 pixels
    CodeWordModifier &codeWordModifier1 = s_codeWordModifierTable[cw1];
    CodeWordModifier &codeWordModifier2 = s_codeWordModifierTable[cw2];

    if (bFlipBitOn==false) // 2x4 block
    {
        for (i=0;i<4;i++)
        {
            for (j=0;j<4;j++)
            {
                CodeWordModifier *pCodeModifier;
                int r,g,b;

                if (j<2)
                {
                    r = r1;
                    g = g1;
                    b = b1;

                    pCodeModifier = &codeWordModifier1;
                } // End if
                else
                {
                    r = r2;
                    g = g2;
                    b = b2;

                    pCodeModifier = &codeWordModifier2;
                } // End else

                if (bFlipUV==false)
                {
                    pDestPixel = pDestBlock +
                                 j * 4 +
                                 i * bmpData.Stride;
                } // End if
                else
                {
                    int x = 4 - j - 1;
                    int y = 4 - i - 1;
                    pDestPixel = pDestBlock +
                                 x * 4 +
                                 y * bmpData.Stride;
                } // End else

                int modifierIndex = pixelIndices[i][j];
                int modifier = pCodeModifier->values[modifierIndex];

                int finalR = r + modifier;
                int finalG = g + modifier;
                int finalB = b + modifier;

                finalR = min(255,max(0,finalR));
                finalG = min(255,max(0,finalG));
                finalB = min(255,max(0,finalB));

                pDestPixel[0] = (unsigned char)(finalB);
                pDestPixel[1] = (unsigned char)(finalG);
                pDestPixel[2] = (unsigned char)(finalR);

                if (bHasAlpha)
                    pDestPixel[3] = alphaTable[i][j];
                else
                    pDestPixel[3] = 255;
            } // End for
        } // End for
    } // End if
    else // 4x2 block
    {
        for (i=0;i<4;i++)
        {
            CodeWordModifier *pCodeModifier;

            int r,g,b;

            if (i<2)
            {
                r = r1;
                g = g1;
                b = b1;

                pCodeModifier = &codeWordModifier1;
            } // End if
            else
            {
                r = r2;
                g = g2;
                b = b2;

                pCodeModifier = &codeWordModifier2;
            } // End else

            for (j=0;j<4;j++)
            {
                if (bFlipUV==false)
                {
                    pDestPixel = pDestBlock +
                                 j * 4 +
                                 i * bmpData.Stride;
                } // End if
                else
                {
                    int x = 4 - j - 1;
                    int y = 4 - i - 1;
                    pDestPixel = pDestBlock +
                                 x * 4 +
                                 y * bmpData.Stride;
                } // End else

                int modifierIndex = pixelIndices[i][j];
                int modifier = pCodeModifier->values[modifierIndex];

                int finalR = r + modifier;
                int finalG = g + modifier;
                int finalB = b + modifier;

                finalR = min(255,max(0,finalR));
                finalG = min(255,max(0,finalG));
                finalB = min(255,max(0,finalB));

                pDestPixel[0] = (unsigned char)(finalB);
                pDestPixel[1] = (unsigned char)(finalG);
                pDestPixel[2] = (unsigned char)(finalR);

                if (bHasAlpha)
                    pDestPixel[3] = alphaTable[i][j];
                else
                    pDestPixel[3] = 255;
            } // End for
        } // End for
    } // End else

    return true;
} // End of ConvertToPreviewImageBlock_4x4_Etc1 for CNWCTexLoader


//------------------------------------------------------------------------------
// Construct table to convert block to linear offset
//------------------------------------------------------------------------------
CNWCTexLoader::PixelOffset CNWCTexLoader::s_blockToLinearPixelOffset[64];
CNWCTexLoader::PixelOffset CNWCTexLoader::s_blockToLinearPixelOffsetFlipUV[64];

void CNWCTexLoader::InitBlockToLinearTable()
{
    PixelOffset blockOffset[4] =
    {
        PixelOffset( 0, 0 ),
        PixelOffset( 1, 0 ),
        PixelOffset( 0, 1 ),
        PixelOffset( 1, 1 )
    }; // End of blockOffset

    int topLevelLoop;
    int midLevelLoop;
    int bottomLevelLoop;
    int index = 0;

    // Top-level loop
    for (topLevelLoop=0;topLevelLoop<4;topLevelLoop++)
    {
        const PixelOffset &topLevelOffset = blockOffset[topLevelLoop];

        for (midLevelLoop=0;midLevelLoop<4;midLevelLoop++)
        {
            const PixelOffset &midLevelOffset = blockOffset[midLevelLoop];

            for (bottomLevelLoop=0;bottomLevelLoop<4;bottomLevelLoop++)
            {
                const PixelOffset &bottomLevelOffset = blockOffset[bottomLevelLoop];

                // Normal Case
                PixelOffset &pixelOffset = s_blockToLinearPixelOffset[index];
                pixelOffset.x = topLevelOffset.x * 4 +
                                midLevelOffset.x * 2 +
                                bottomLevelOffset.x;

                pixelOffset.y = topLevelOffset.y * 4 +
                                midLevelOffset.y * 2 +
                                bottomLevelOffset.y;

                index++;
            } // End for
        } // End for
    } // End for

    // Create flipped case
    int i;
    for (i=0;i<8*8;i++)
    {
        PixelOffset &pixelOffset = s_blockToLinearPixelOffset[i];
        s_blockToLinearPixelOffsetFlipUV[i].x = 8 - pixelOffset.x - 1;
        s_blockToLinearPixelOffsetFlipUV[i].y = 8 - pixelOffset.y - 1;
    } // End for
} // End of InitBlockToLinearTable


