﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/
#ifdef NW_FOR_FTX
// ConvertSimpleFormat
// EncodeFromUnorm8 EncodeFromSnorm8 EncodeFromFloat32
// DecodeToUnorm8   DecodeToSnorm8   DecodeToFloat32
// EncodeFloat111110 DecodeFloat111110

//=============================================================================
// include
//=============================================================================
//#include "ShellExtension_PCH.h"
#include "SimpleFormat.h"

using namespace std;
using namespace nn::gfx::tool::texenc;

namespace
{

//-----------------------------------------------------------------------------
//! @brief プレーンな RGBA フォーマットから RGBA 成分の一部または全部をコピーします。
//!
//! @param[out] pDst 変換後のデータを格納します。
//! @param[in] pSrc 変換前のデータです。
//! @param[in] dstFormatStr 変換後のフォーマット文字列です。
//! @param[in] pixCount ピクセル数です。
//-----------------------------------------------------------------------------
void CopyFromPlainRgba(
    void* pDst,
    const void* pSrc,
    const std::string& dstFormatStr,
    const size_t pixCount
)
{
    const bool isFloat = IsFloatFormat(dstFormatStr);
    const size_t srcColBytes = (isFloat) ? EncRgbaFloatBytes : EncRgbaBytes;
    const size_t copyBytes = GetBitsPerPixel(dstFormatStr) / 8;
    uint8_t* pDstU8 = reinterpret_cast<uint8_t*>(pDst);
    const uint8_t* pSrcU8 = reinterpret_cast<const uint8_t*>(pSrc);
    for (size_t iPix = 0; iPix < pixCount; ++iPix)
    {
        memcpy(pDstU8, pSrcU8, copyBytes);
        pDstU8 += copyBytes;
        pSrcU8 += srcColBytes;
    }
}

//-----------------------------------------------------------------------------
//! @brief RGBA 成分の一部または全部をコピーしてプレーンな RGBA フォーマットに変換します。
//!
//! @param[out] pDst 変換後のデータを格納します。
//! @param[in] pSrc 変換前のデータです。
//! @param[in] srcFormatStr 変換前のフォーマット文字列です。
//! @param[in] pixCount ピクセル数です。
//-----------------------------------------------------------------------------
void CopyToPlainRgba(
    void* pDst,
    const void* pSrc,
    const std::string& srcFormatStr,
    const size_t pixCount
)
{
    const bool isFloat = IsFloatFormat(srcFormatStr);
    const size_t copyBytes = GetBitsPerPixel(srcFormatStr) / 8;
    const uint8_t* pSrcU8 = reinterpret_cast<const uint8_t*>(pSrc);
    if (!isFloat)
    {
        const uint8_t defaultA = IsSignedFormat(srcFormatStr) ? 0x7f : 0xff;
        uint8_t* pDstU8 = reinterpret_cast<uint8_t*>(pDst);
        for (size_t iPix = 0; iPix < pixCount; ++iPix)
        {
            pDstU8[1] = 0x00;
            pDstU8[2] = 0x00;
            pDstU8[3] = defaultA;
            memcpy(pDstU8, pSrcU8, copyBytes);
            pDstU8 += EncRgbaCount;
            pSrcU8 += copyBytes;
        }
    }
    else
    {
        float* pDstF32 = reinterpret_cast<float*>(pDst);
        for (size_t iPix = 0; iPix < pixCount; ++iPix)
        {
            pDstF32[1] = 0.0f;
            pDstF32[2] = 0.0f;
            pDstF32[3] = 1.0f;
            memcpy(pDstF32, pSrcU8, copyBytes);
            pDstF32 += EncRgbaCount;
            pSrcU8 += copyBytes;
        }
    }
}

//-----------------------------------------------------------------------------
//! @brief unorm_4_4 にエンコードします。
//!
//! @param[out] pDst 変換後のデータを格納します。
//! @param[in] pSrc 変換前のデータです（unorm_8_8_8_8）。
//! @param[in] pixCount ピクセル数です。
//! @param[in] isRev 上位ビットから RGB の順に格納するなら false、BGR の順に格納するなら true です。
//-----------------------------------------------------------------------------
void EncodeUnorm44(
    void* pDst,
    const void* pSrc,
    const size_t pixCount,
    const bool isRev
)
{
    uint8_t* pDstU8 = reinterpret_cast<uint8_t*>(pDst);
    const uint8_t* pSrcU8 = reinterpret_cast<const uint8_t*>(pSrc);
    for (size_t iPix = 0; iPix < pixCount; ++iPix)
    {
        *pDstU8++ = (!isRev) ?
            static_cast<uint8_t>(((pSrcU8[0] >> 4) << 4) | (pSrcU8[1] >> 4)) :
            static_cast<uint8_t>(((pSrcU8[1] >> 4) << 4) | (pSrcU8[0] >> 4));
        pSrcU8 += EncRgbaCount;
    }
}

//-----------------------------------------------------------------------------
//! @brief unorm_4_4 をデコードします。
//!
//! @param[out] pDst 変換後のデータを格納します。
//! @param[in] pSrc 変換前のデータです。
//! @param[in] pixCount ピクセル数です。
//! @param[in] isRev 上位ビットから RGB の順に格納されているなら false、BGR の順に格納されているなら true です。
//-----------------------------------------------------------------------------
void DecodeUnorm44(
    void* pDst,
    const void* pSrc,
    const size_t pixCount,
    const bool isRev
)
{
    uint8_t* pDstU8 = reinterpret_cast<uint8_t*>(pDst);
    const uint8_t* pSrcU8 = reinterpret_cast<const uint8_t*>(pSrc);
    for (size_t iPix = 0; iPix < pixCount; ++iPix)
    {
        const uint8_t src = *pSrcU8++;
        const int r = (!isRev) ? (src >> 4) & 0x0f : (src     ) & 0x0f;
        const int g = (!isRev) ? (src     ) & 0x0f : (src >> 4) & 0x0f;
        pDstU8[0] = static_cast<uint8_t>((r << 4) | r);
        pDstU8[1] = static_cast<uint8_t>((g << 4) | g);
        pDstU8[2] = 0x00;
        pDstU8[3] = 0xff;
        pDstU8 += EncRgbaCount;
    }
}

//-----------------------------------------------------------------------------
//! @brief unorm_5_6_5 にエンコードします。
//!
//! @param[out] pDst 変換後のデータを格納します。
//! @param[in] pSrc 変換前のデータです（unorm_8_8_8_8）。
//! @param[in] pixCount ピクセル数です。
//! @param[in] isRev 上位ビットから RGB の順に格納するなら false、BGR の順に格納するなら true です。
//-----------------------------------------------------------------------------
void EncodeUnorm565(
    void* pDst,
    const void* pSrc,
    const size_t pixCount,
    const bool isRev
)
{
    uint16_t* pDstU16 = reinterpret_cast<uint16_t*>(pDst);
    const uint8_t* pSrcU8 = reinterpret_cast<const uint8_t*>(pSrc);
    for (size_t iPix = 0; iPix < pixCount; ++iPix)
    {
        if (!isRev)
        {
            *pDstU16++ = static_cast<uint16_t>(
                ((pSrcU8[0] >> 3) << 11) |
                ((pSrcU8[1] >> 2) <<  5) |
                ((pSrcU8[2] >> 3)      ));
        }
        else
        {
            *pDstU16++ = static_cast<uint16_t>(
                ((pSrcU8[2] >> 3) << 11) |
                ((pSrcU8[1] >> 2) <<  5) |
                ((pSrcU8[0] >> 3)      ));
        }
        pSrcU8 += EncRgbaCount;
    }
}

//-----------------------------------------------------------------------------
//! @brief unorm_5_6_5 をデコードします。
//!
//! @param[out] pDst 変換後のデータを格納します。
//! @param[in] pSrc 変換前のデータです。
//! @param[in] pixCount ピクセル数です。
//! @param[in] isRev 上位ビットから RGB の順に格納されているなら false、BGR の順に格納されているなら true です。
//-----------------------------------------------------------------------------
void DecodeUnorm565(
    void* pDst,
    const void* pSrc,
    const size_t pixCount,
    const bool isRev
)
{
    uint8_t* pDstU8 = reinterpret_cast<uint8_t*>(pDst);
    const uint16_t* pSrcU16 = reinterpret_cast<const uint16_t*>(pSrc);
    for (size_t iPix = 0; iPix < pixCount; ++iPix)
    {
        const uint16_t src = *pSrcU16++;
        int r = 0;
        int g = 0;
        int b = 0;
        if (!isRev)
        {
            r = (src >> 11) & 0x1f;
            g = (src >>  5) & 0x3f;
            b = (src      ) & 0x1f;
        }
        else
        {
            r = (src      ) & 0x1f;
            g = (src >>  5) & 0x3f;
            b = (src >> 11) & 0x1f;
        }
        pDstU8[0] = static_cast<uint8_t>((r << 3) | (r >> 2));
        pDstU8[1] = static_cast<uint8_t>((g << 2) | (g >> 4));
        pDstU8[2] = static_cast<uint8_t>((b << 3) | (b >> 2));
        pDstU8[3] = 0xff;
        pDstU8 += EncRgbaCount;
    }
}

//-----------------------------------------------------------------------------
//! @brief unorm_5_5_5_1 にエンコードします。
//!
//! @param[out] pDst 変換後のデータを格納します。
//! @param[in] pSrc 変換前のデータです（unorm_8_8_8_8）。
//! @param[in] pixCount ピクセル数です。
//! @param[in] isRev 上位ビットから RGB の順に格納するなら false、BGR の順に格納するなら true です。
//-----------------------------------------------------------------------------
void EncodeUnorm5551(
    void* pDst,
    const void* pSrc,
    const size_t pixCount,
    const bool isRev
)
{
    uint16_t* pDstU16 = reinterpret_cast<uint16_t*>(pDst);
    const uint8_t* pSrcU8 = reinterpret_cast<const uint8_t*>(pSrc);
    for (size_t iPix = 0; iPix < pixCount; ++iPix)
    {
        const int alpha = (pSrcU8[3] >= EncOpaAlphaBorder) ? 1 : 0;
        if (!isRev)
        {
            *pDstU16++ = static_cast<uint16_t>(
                ((pSrcU8[0] >> 3) << 11) |
                ((pSrcU8[1] >> 3) <<  6) |
                ((pSrcU8[2] >> 3) <<  1) |
                ((alpha         )      ));
        }
        else
        {
            *pDstU16++ = static_cast<uint16_t>(
                ((alpha         ) << 15) |
                ((pSrcU8[2] >> 3) << 10) |
                ((pSrcU8[1] >> 3) <<  5) |
                ((pSrcU8[0] >> 3)      ));
        }
        pSrcU8 += EncRgbaCount;
    }
}

//-----------------------------------------------------------------------------
//! @brief unorm_5_5_5_1 をデコードします。
//!
//! @param[out] pDst 変換後のデータを格納します。
//! @param[in] pSrc 変換前のデータです。
//! @param[in] pixCount ピクセル数です。
//! @param[in] isRev 上位ビットから RGB の順に格納されているなら false、BGR の順に格納されているなら true です。
//-----------------------------------------------------------------------------
void DecodeUnorm5551(
    void* pDst,
    const void* pSrc,
    const size_t pixCount,
    const bool isRev
)
{
    uint8_t* pDstU8 = reinterpret_cast<uint8_t*>(pDst);
    const uint16_t* pSrcU16 = reinterpret_cast<const uint16_t*>(pSrc);
    for (size_t iPix = 0; iPix < pixCount; ++iPix)
    {
        const uint16_t src = *pSrcU16++;
        int r = 0;
        int g = 0;
        int b = 0;
        int a = 0;
        if (!isRev)
        {
            r = (src >> 11) & 0x1f;
            g = (src >>  6) & 0x1f;
            b = (src >>  1) & 0x1f;
            a = (src      ) & 0x01;
        }
        else
        {
            r = (src      ) & 0x1f;
            g = (src >>  5) & 0x1f;
            b = (src >> 10) & 0x1f;
            a = (src >> 15) & 0x01;
        }
        pDstU8[0] = static_cast<uint8_t>((r << 3) | (r >> 2));
        pDstU8[1] = static_cast<uint8_t>((g << 3) | (g >> 2));
        pDstU8[2] = static_cast<uint8_t>((b << 3) | (b >> 2));
        pDstU8[3] = (a != 0) ? 0xff : 0x00;
        pDstU8 += EncRgbaCount;
    }
}

//-----------------------------------------------------------------------------
//! @brief unorm_4_4_4_4 にエンコードします。
//!
//! @param[out] pDst 変換後のデータを格納します。
//! @param[in] pSrc 変換前のデータです（unorm_8_8_8_8）。
//! @param[in] pixCount ピクセル数です。
//! @param[in] isRev 上位ビットから RGB の順に格納するなら false、BGR の順に格納するなら true です。
//-----------------------------------------------------------------------------
void EncodeUnorm4444(
    void* pDst,
    const void* pSrc,
    const size_t pixCount,
    const bool isRev
)
{
    uint16_t* pDstU16 = reinterpret_cast<uint16_t*>(pDst);
    const uint8_t* pSrcU8 = reinterpret_cast<const uint8_t*>(pSrc);
    for (size_t iPix = 0; iPix < pixCount; ++iPix)
    {
        if (!isRev)
        {
            *pDstU16++ = static_cast<uint16_t>(
                ((pSrcU8[0] >> 4) << 12) |
                ((pSrcU8[1] >> 4) <<  8) |
                ((pSrcU8[2] >> 4) <<  4) |
                ((pSrcU8[3] >> 4)      ));
        }
        else
        {
            *pDstU16++ = static_cast<uint16_t>(
                ((pSrcU8[3] >> 4) << 12) |
                ((pSrcU8[2] >> 4) <<  8) |
                ((pSrcU8[1] >> 4) <<  4) |
                ((pSrcU8[0] >> 4)      ));
        }
        pSrcU8 += EncRgbaCount;
    }
}

//-----------------------------------------------------------------------------
//! @brief unorm_4_4_4_4 をデコードします。
//!
//! @param[out] pDst 変換後のデータを格納します。
//! @param[in] pSrc 変換前のデータです。
//! @param[in] pixCount ピクセル数です。
//! @param[in] isRev 上位ビットから RGB の順に格納されているなら false、BGR の順に格納されているなら true です。
//-----------------------------------------------------------------------------
void DecodeUnorm4444(
    void* pDst,
    const void* pSrc,
    const size_t pixCount,
    const bool isRev
)
{
    uint8_t* pDstU8 = reinterpret_cast<uint8_t*>(pDst);
    const uint16_t* pSrcU16 = reinterpret_cast<const uint16_t*>(pSrc);
    for (size_t iPix = 0; iPix < pixCount; ++iPix)
    {
        const uint16_t src = *pSrcU16++;
        int r = 0;
        int g = 0;
        int b = 0;
        int a = 0;
        if (!isRev)
        {
            r = (src >> 12) & 0x0f;
            g = (src >>  8) & 0x0f;
            b = (src >>  4) & 0x0f;
            a = (src      ) & 0x0f;
        }
        else
        {
            r = (src      ) & 0x0f;
            g = (src >>  4) & 0x0f;
            b = (src >>  8) & 0x0f;
            a = (src >> 12) & 0x0f;
        }
        pDstU8[0] = static_cast<uint8_t>((r << 4) | r);
        pDstU8[1] = static_cast<uint8_t>((g << 4) | g);
        pDstU8[2] = static_cast<uint8_t>((b << 4) | b);
        pDstU8[3] = static_cast<uint8_t>((a << 4) | a);
        pDstU8 += EncRgbaCount;
    }
}

//-----------------------------------------------------------------------------
//! @brief unorm_8_8_8_8 から float_32_32_32_32 に変換します。
//!
//! @param[out] pDst 変換後のデータを格納します。
//! @param[in] pSrc 変換前のデータです。
//! @param[in] pixCount ピクセル数です。
//-----------------------------------------------------------------------------
void ConvertUnorm8ToFloat32(
    void* pDst,
    const void* pSrc,
    const size_t pixCount
)
{
    float* pDstF32 = reinterpret_cast<float*>(pDst);
    const uint8_t* pSrcU8 = reinterpret_cast<const uint8_t*>(pSrc);
    const size_t valueCount = pixCount * EncRgbaCount;
    for (size_t iVal = 0; iVal < valueCount; ++iVal)
    {
        *pDstF32++ = static_cast<float>(*pSrcU8++) / 0xff;
    }
}

//-----------------------------------------------------------------------------
//! @brief float_32_32_32_32 から unorm_8_8_8_8 に変換します。
//!
//! @param[out] pDst 変換後のデータを格納します。
//! @param[in] pSrc 変換前のデータです。
//! @param[in] pixCount ピクセル数です。
//-----------------------------------------------------------------------------
void ConvertFloat32ToUnorm8(
    void* pDst,
    const void* pSrc,
    const size_t pixCount
)
{
    uint8_t* pDstU8 = reinterpret_cast<uint8_t*>(pDst);
    const float* pSrcF32 = reinterpret_cast<const float*>(pSrc);
    const size_t valueCount = pixCount * EncRgbaCount;
    for (size_t iVal = 0; iVal < valueCount; ++iVal)
    {
        *pDstU8++ = GetUint8Value(*pSrcF32++ * 0xff);
    }
}

//-----------------------------------------------------------------------------
//! @brief snorm_8_8_8_8 から float_32_32_32_32 に変換します。
//!
//! @param[out] pDst 変換後のデータを格納します。
//! @param[in] pSrc 変換前のデータです。
//! @param[in] pixCount ピクセル数です。
//-----------------------------------------------------------------------------
void ConvertSnorm8ToFloat32(
    void* pDst,
    const void* pSrc,
    const size_t pixCount
)
{
    float* pDstF32 = reinterpret_cast<float*>(pDst);
    const int8_t* pSrcS8 = reinterpret_cast<const int8_t*>(pSrc);
    const size_t valueCount = pixCount * EncRgbaCount;
    for (size_t iVal = 0; iVal < valueCount; ++iVal)
    {
        *pDstF32++ = (*pSrcS8 == -128) ? -1.0f : static_cast<float>(*pSrcS8) / 0x7f;
        ++pSrcS8;
    }
}

//-----------------------------------------------------------------------------
//! @brief float_32_32_32_32 から snorm_8_8_8_8 に変換します。
//!
//! @param[out] pDst 変換後のデータを格納します。
//! @param[in] pSrc 変換前のデータです。
//! @param[in] pixCount ピクセル数です。
//-----------------------------------------------------------------------------
void ConvertFloat32ToSnorm8(
    void* pDst,
    const void* pSrc,
    const size_t pixCount
)
{
    int8_t* pDstS8 = reinterpret_cast<int8_t*>(pDst);
    const float* pSrcF32 = reinterpret_cast<const float*>(pSrc);
    const size_t valueCount = pixCount * EncRgbaCount;
    for (size_t iVal = 0; iVal < valueCount; ++iVal)
    {
        *pDstS8++ = GetInt8Value(*pSrcF32++ * 0x7f);
    }
}

//-----------------------------------------------------------------------------
//! @brief unorm_16 / unorm_16_16 / unorm_16_16_16_16 にエンコードします。
//!
//! @param[out] pDst 変換後のデータを格納します。
//! @param[in] pSrc 変換前のデータです（float_32_32_32_32）。
//! @param[in] pixCount ピクセル数です。
//! @param[in] compCount 成分数です。
//-----------------------------------------------------------------------------
void EncodeUnorm16(
    void* pDst,
    const void* pSrc,
    const size_t pixCount,
    const int compCount
)
{
    uint16_t* pDstU16 = reinterpret_cast<uint16_t*>(pDst);
    const float* pSrcF32 = reinterpret_cast<const float*>(pSrc);
    for (size_t iPix = 0; iPix < pixCount; ++iPix)
    {
        for (int iRgba = 0; iRgba < compCount; ++iRgba)
        {
            *pDstU16++ = GetUint16Value(pSrcF32[iRgba] * 0xffff);
        }
        pSrcF32 += EncRgbaCount;
    }
}

//-----------------------------------------------------------------------------
//! @brief unorm_16 / unorm_16_16 / unorm_16_16_16_16 をデコードします。
//!
//! @param[out] pDst 変換後のデータを格納します。
//! @param[in] pSrc 変換前のデータです。
//! @param[in] dstFormatStr 変換後のフォーマット文字列です（unorm_8_8_8_8 / snorm_8_8_8_8 / float_32_32_32_32）。
//! @param[in] pixCount ピクセル数です。
//! @param[in] compCount 成分数です。
//-----------------------------------------------------------------------------
void DecodeUnorm16(
    void* pDst,
    const void* pSrc,
    const std::string& dstFormatStr,
    const size_t pixCount,
    const int compCount
)
{
    const float FloatScale = 1.0f / 0xffff;
    const uint16_t* pSrcU16 = reinterpret_cast<const uint16_t*>(pSrc);
    if (!IsFloatFormat(dstFormatStr))
    {
        const bool isToSigned = IsSignedFormat(dstFormatStr);
        const uint8_t defaultA = (isToSigned) ? 0x7f : 0xff;
        uint8_t* pDstU8 = reinterpret_cast<uint8_t*>(pDst);
        for (size_t iPix = 0; iPix < pixCount; ++iPix)
        {
            pDstU8[1] = 0x00;
            pDstU8[2] = 0x00;
            pDstU8[3] = defaultA;
            for (int iRgba = 0; iRgba < compCount; ++iRgba)
            {
                const float fv = (*pSrcU16++) * FloatScale;
                if (!isToSigned)
                {
                    pDstU8[iRgba] = GetUint8Value(fv * 0xff);
                }
                else
                {
                    reinterpret_cast<int8_t*>(pDstU8)[iRgba] = GetInt8Value(fv * 0x7f);
                }
            }
            pDstU8 += EncRgbaCount;
        }
    }
    else
    {
        float* pDstF32 = reinterpret_cast<float*>(pDst);
        for (size_t iPix = 0; iPix < pixCount; ++iPix)
        {
            pDstF32[1] = 0.0f;
            pDstF32[2] = 0.0f;
            pDstF32[3] = 1.0f;
            for (int iRgba = 0; iRgba < compCount; ++iRgba)
            {
                pDstF32[iRgba] = (*pSrcU16++) * FloatScale;
            }
            pDstF32 += EncRgbaCount;
        }
    }
}

//-----------------------------------------------------------------------------
//! @brief snorm_16 / snorm_16_16 / snorm_16_16_16_16 にエンコードします。
//!
//! @param[out] pDst 変換後のデータを格納します。
//! @param[in] pSrc 変換前のデータです（float_32_32_32_32）。
//! @param[in] pixCount ピクセル数です。
//! @param[in] compCount 成分数です。
//-----------------------------------------------------------------------------
void EncodeSnorm16(
    void* pDst,
    const void* pSrc,
    const size_t pixCount,
    const int compCount
)
{
    int16_t* pDstS16 = reinterpret_cast<int16_t*>(pDst);
    const float* pSrcF32 = reinterpret_cast<const float*>(pSrc);
    for (size_t iPix = 0; iPix < pixCount; ++iPix)
    {
        for (int iRgba = 0; iRgba < compCount; ++iRgba)
        {
            *pDstS16++ = GetInt16Value(pSrcF32[iRgba] * 0x7fff);
        }
        pSrcF32 += EncRgbaCount;
    }
}

//-----------------------------------------------------------------------------
//! @brief snorm_16 / snorm_16_16 / snorm_16_16_16_16 をデコードします。
//!
//! @param[out] pDst 変換後のデータを格納します。
//! @param[in] pSrc 変換前のデータです。
//! @param[in] dstFormatStr 変換後のフォーマット文字列です（unorm_8_8_8_8 / snorm_8_8_8_8 / float_32_32_32_32）。
//! @param[in] pixCount ピクセル数です。
//! @param[in] compCount 成分数です。
//-----------------------------------------------------------------------------
void DecodeSnorm16(
    void* pDst,
    const void* pSrc,
    const std::string& dstFormatStr,
    const size_t pixCount,
    const int compCount
)
{
    const float FloatScale = 1.0f / 0x7fff;
    const int16_t* pSrcS16 = reinterpret_cast<const int16_t*>(pSrc);
    if (!IsFloatFormat(dstFormatStr))
    {
        const bool isToSigned = IsSignedFormat(dstFormatStr);
        const uint8_t defaultA = (isToSigned) ? 0x7f : 0xff;
        uint8_t* pDstU8 = reinterpret_cast<uint8_t*>(pDst);
        for (size_t iPix = 0; iPix < pixCount; ++iPix)
        {
            pDstU8[1] = 0x00;
            pDstU8[2] = 0x00;
            pDstU8[3] = defaultA;
            for (int iRgba = 0; iRgba < compCount; ++iRgba)
            {
                const float fv = (*pSrcS16++) * FloatScale;
                if (!isToSigned)
                {
                    pDstU8[iRgba] = GetUint8Value(fv * 0xff);
                }
                else
                {
                    reinterpret_cast<int8_t*>(pDstU8)[iRgba] = GetInt8Value(fv * 0x7f);
                }
            }
            pDstU8 += EncRgbaCount;
        }
    }
    else
    {
        float* pDstF32 = reinterpret_cast<float*>(pDst);
        for (size_t iPix = 0; iPix < pixCount; ++iPix)
        {
            pDstF32[1] = 0.0f;
            pDstF32[2] = 0.0f;
            pDstF32[3] = 1.0f;
            for (int iRgba = 0; iRgba < compCount; ++iRgba)
            {
                pDstF32[iRgba] = (*pSrcS16++) * FloatScale;
            }
            pDstF32 += EncRgbaCount;
        }
    }
}

//-----------------------------------------------------------------------------
//! @brief uint_16 / uint_16_16 / uint_16_16_16_16 にエンコードします。
//!
//! @param[out] pDst 変換後のデータを格納します。
//! @param[in] pSrc 変換前のデータです（float_32_32_32_32）。
//! @param[in] pixCount ピクセル数です。
//! @param[in] compCount 成分数です。
//-----------------------------------------------------------------------------
void EncodeUint16(
    void* pDst,
    const void* pSrc,
    const size_t pixCount,
    const int compCount
)
{
    uint16_t* pDstU16 = reinterpret_cast<uint16_t*>(pDst);
    const float* pSrcF32 = reinterpret_cast<const float*>(pSrc);
    for (size_t pixelIdx = 0; pixelIdx < pixCount; ++pixelIdx)
    {
        for (int rgbaIdx = 0; rgbaIdx < compCount; ++rgbaIdx)
        {
            *pDstU16++ = GetUint16Value(pSrcF32[rgbaIdx]);
        }
        pSrcF32 += EncRgbaCount;
    }
}

//-----------------------------------------------------------------------------
//! @brief uint_16 / uint_16_16 / uint_16_16_16_16 をデコードします。
//!
//! @param[out] pDst 変換後のデータを格納します。
//! @param[in] pSrc 変換前のデータです。
//! @param[in] dstFormatStr 変換後のフォーマット文字列です（unorm_8_8_8_8 / snorm_8_8_8_8 / float_32_32_32_32）。
//! @param[in] pixCount ピクセル数です。
//! @param[in] compCount 成分数です。
//-----------------------------------------------------------------------------
void DecodeUint16(
    void* pDst,
    const void* pSrc,
    const std::string& dstFormatStr,
    const size_t pixCount,
    const int compCount
)
{
    const uint16_t* pSrcU16 = reinterpret_cast<const uint16_t*>(pSrc);
    if (!IsFloatFormat(dstFormatStr))
    {
        const bool isToSigned = IsSignedFormat(dstFormatStr);
        const uint8_t defaultA = (isToSigned) ? 0x7f : 0xff;
        uint8_t* pDstU8 = reinterpret_cast<uint8_t*>(pDst);
        for (size_t pixelIdx = 0; pixelIdx < pixCount; ++pixelIdx)
        {
            pDstU8[1] = 0x00;
            pDstU8[2] = 0x00;
            pDstU8[3] = defaultA;
            for (int rgbaIdx = 0; rgbaIdx < compCount; ++rgbaIdx)
            {
                const int v = static_cast<int>(*pSrcU16++);
                if (!isToSigned)
                {
                    pDstU8[rgbaIdx] = static_cast<uint8_t>(ClampValue(v, 0x00, 0xff));
                }
                else
                {
                    pDstU8[rgbaIdx] = static_cast<uint8_t>(ClampValue(v, 0x00, 0x7f));
                }
            }
            pDstU8 += EncRgbaCount;
        }
    }
    else
    {
        float* pDstF32 = reinterpret_cast<float*>(pDst);
        for (size_t pixelIdx = 0; pixelIdx < pixCount; ++pixelIdx)
        {
            pDstF32[1] = 0.0f;
            pDstF32[2] = 0.0f;
            pDstF32[3] = 1.0f;
            for (int rgbaIdx = 0; rgbaIdx < compCount; ++rgbaIdx)
            {
                pDstF32[rgbaIdx] = static_cast<float>(*pSrcU16++);
            }
            pDstF32 += EncRgbaCount;
        }
    }
}

//-----------------------------------------------------------------------------
//! @brief sint_16 / sint_16_16 / sint_16_16_16_16 にエンコードします。
//!
//! @param[out] pDst 変換後のデータを格納します。
//! @param[in] pSrc 変換前のデータです（float_32_32_32_32）。
//! @param[in] pixCount ピクセル数です。
//! @param[in] compCount 成分数です。
//-----------------------------------------------------------------------------
void EncodeSint16(
    void* pDst,
    const void* pSrc,
    const size_t pixCount,
    const int compCount
)
{
    int16_t* pDstS16 = reinterpret_cast<int16_t*>(pDst);
    const float* pSrcF32 = reinterpret_cast<const float*>(pSrc);
    for (size_t pixelIdx = 0; pixelIdx < pixCount; ++pixelIdx)
    {
        for (int rgbaIdx = 0; rgbaIdx < compCount; ++rgbaIdx)
        {
            *pDstS16++ = GetInt16Value(pSrcF32[rgbaIdx]);
        }
        pSrcF32 += EncRgbaCount;
    }
}

//-----------------------------------------------------------------------------
//! @brief sint_16 / sint_16_16 / sint_16_16_16_16 をデコードします。
//!
//! @param[out] pDst 変換後のデータを格納します。
//! @param[in] pSrc 変換前のデータです。
//! @param[in] dstFormatStr 変換後のフォーマット文字列です（unorm_8_8_8_8 / snorm_8_8_8_8 / float_32_32_32_32）。
//! @param[in] pixCount ピクセル数です。
//! @param[in] compCount 成分数です。
//-----------------------------------------------------------------------------
void DecodeSint16(
    void* pDst,
    const void* pSrc,
    const std::string& dstFormatStr,
    const size_t pixCount,
    const int compCount
)
{
    const int16_t* pSrcS16 = reinterpret_cast<const int16_t*>(pSrc);
    if (!IsFloatFormat(dstFormatStr))
    {
        const bool isToSigned = IsSignedFormat(dstFormatStr);
        const uint8_t defaultA = (isToSigned) ? 0x7f : 0xff;
        uint8_t* pDstU8 = reinterpret_cast<uint8_t*>(pDst);
        for (size_t pixelIdx = 0; pixelIdx < pixCount; ++pixelIdx)
        {
            pDstU8[1] = 0x00;
            pDstU8[2] = 0x00;
            pDstU8[3] = defaultA;
            for (int rgbaIdx = 0; rgbaIdx < compCount; ++rgbaIdx)
            {
                const int v = static_cast<int>(*pSrcS16++);
                if (!isToSigned)
                {
                    pDstU8[rgbaIdx] = static_cast<uint8_t>(ClampValue(v, 0x00, 0xff));
                }
                else
                {
                    reinterpret_cast<int8_t*>(pDstU8)[rgbaIdx] =
                        static_cast<int8_t>(ClampValue(v, -0x7f, 0x7f));
                }
            }
            pDstU8 += EncRgbaCount;
        }
    }
    else
    {
        float* pDstF32 = reinterpret_cast<float*>(pDst);
        for (size_t pixelIdx = 0; pixelIdx < pixCount; ++pixelIdx)
        {
            pDstF32[1] = 0.0f;
            pDstF32[2] = 0.0f;
            pDstF32[3] = 1.0f;
            for (int rgbaIdx = 0; rgbaIdx < compCount; ++rgbaIdx)
            {
                pDstF32[rgbaIdx] = static_cast<float>(*pSrcS16++);
            }
            pDstF32 += EncRgbaCount;
        }
    }
}

//-----------------------------------------------------------------------------
//! @brief uint_32 / uint_32_32 / uint_32_32_32 / uint_32_32_32_32 にエンコードします。
//!
//! @param[out] pDst 変換後のデータを格納します。
//! @param[in] pSrc 変換前のデータです（float_32_32_32_32）。
//! @param[in] pixCount ピクセル数です。
//! @param[in] compCount 成分数です。
//-----------------------------------------------------------------------------
void EncodeUint32(
    void* pDst,
    const void* pSrc,
    const size_t pixCount,
    const int compCount
)
{
    uint32_t* pDstU32 = reinterpret_cast<uint32_t*>(pDst);
    const float* pSrcF32 = reinterpret_cast<const float*>(pSrc);
    for (size_t pixelIdx = 0; pixelIdx < pixCount; ++pixelIdx)
    {
        for (int rgbaIdx = 0; rgbaIdx < compCount; ++rgbaIdx)
        {
            *pDstU32++ = GetUint32Value(pSrcF32[rgbaIdx]);
        }
        pSrcF32 += EncRgbaCount;
    }
}

//-----------------------------------------------------------------------------
//! @brief uint_32 / uint_32_32 / uint_32_32_32 / uint_32_32_32_32 をデコードします。
//!
//! @param[out] pDst 変換後のデータを格納します。
//! @param[in] pSrc 変換前のデータです。
//! @param[in] dstFormatStr 変換後のフォーマット文字列です（unorm_8_8_8_8 / snorm_8_8_8_8 / float_32_32_32_32）。
//! @param[in] pixCount ピクセル数です。
//! @param[in] compCount 成分数です。
//-----------------------------------------------------------------------------
void DecodeUint32(
    void* pDst,
    const void* pSrc,
    const std::string& dstFormatStr,
    const size_t pixCount,
    const int compCount
)
{
    const uint32_t* pSrcU32 = reinterpret_cast<const uint32_t*>(pSrc);
    if (!IsFloatFormat(dstFormatStr))
    {
        const bool isToSigned = IsSignedFormat(dstFormatStr);
        const uint8_t defaultA = (isToSigned) ? 0x7f : 0xff;
        uint8_t* pDstU8 = reinterpret_cast<uint8_t*>(pDst);
        for (size_t pixelIdx = 0; pixelIdx < pixCount; ++pixelIdx)
        {
            pDstU8[1] = 0x00;
            pDstU8[2] = 0x00;
            pDstU8[3] = defaultA;
            for (int rgbaIdx = 0; rgbaIdx < compCount; ++rgbaIdx)
            {
                const uint32_t v = *pSrcU32++;
                if (!isToSigned)
                {
                    pDstU8[rgbaIdx] = static_cast<uint8_t>((v > 0xff) ? 0xff : v);
                }
                else
                {
                    pDstU8[rgbaIdx] = static_cast<uint8_t>((v > 0x7f) ? 0x7f : v);
                }
            }
            pDstU8 += EncRgbaCount;
        }
    }
    else
    {
        float* pDstF32 = reinterpret_cast<float*>(pDst);
        for (size_t pixelIdx = 0; pixelIdx < pixCount; ++pixelIdx)
        {
            pDstF32[1] = 0.0f;
            pDstF32[2] = 0.0f;
            pDstF32[3] = 1.0f;
            for (int rgbaIdx = 0; rgbaIdx < compCount; ++rgbaIdx)
            {
                pDstF32[rgbaIdx] = static_cast<float>(*pSrcU32++);
            }
            pDstF32 += EncRgbaCount;
        }
    }
}

//-----------------------------------------------------------------------------
//! @brief sint_32 / sint_32_32 / sint_32_32_32 / sint_32_32_32_32 にエンコードします。
//!
//! @param[out] pDst 変換後のデータを格納します。
//! @param[in] pSrc 変換前のデータです（float_32_32_32_32）。
//! @param[in] pixCount ピクセル数です。
//! @param[in] compCount 成分数です。
//-----------------------------------------------------------------------------
void EncodeSint32(
    void* pDst,
    const void* pSrc,
    const size_t pixCount,
    const int compCount
)
{
    int32_t* pDstS32 = reinterpret_cast<int32_t*>(pDst);
    const float* pSrcF32 = reinterpret_cast<const float*>(pSrc);
    for (size_t pixelIdx = 0; pixelIdx < pixCount; ++pixelIdx)
    {
        for (int rgbaIdx = 0; rgbaIdx < compCount; ++rgbaIdx)
        {
            *pDstS32++ = GetInt32Value(pSrcF32[rgbaIdx]);
        }
        pSrcF32 += EncRgbaCount;
    }
}

//-----------------------------------------------------------------------------
//! @brief sint_32 / sint_32_32 / sint_32_32_32 / sint_32_32_32_32 をデコードします。
//!
//! @param[out] pDst 変換後のデータを格納します。
//! @param[in] pSrc 変換前のデータです。
//! @param[in] dstFormatStr 変換後のフォーマット文字列です（unorm_8_8_8_8 / snorm_8_8_8_8 / float_32_32_32_32）。
//! @param[in] pixCount ピクセル数です。
//! @param[in] compCount 成分数です。
//-----------------------------------------------------------------------------
void DecodeSint32(
    void* pDst,
    const void* pSrc,
    const std::string& dstFormatStr,
    const size_t pixCount,
    const int compCount
)
{
    const int32_t* pSrcS32 = reinterpret_cast<const int32_t*>(pSrc);
    if (!IsFloatFormat(dstFormatStr))
    {
        const bool isToSigned = IsSignedFormat(dstFormatStr);
        const uint8_t defaultA = (isToSigned) ? 0x7f : 0xff;
        uint8_t* pDstU8 = reinterpret_cast<uint8_t*>(pDst);
        for (size_t pixelIdx = 0; pixelIdx < pixCount; ++pixelIdx)
        {
            pDstU8[1] = 0x00;
            pDstU8[2] = 0x00;
            pDstU8[3] = defaultA;
            for (int rgbaIdx = 0; rgbaIdx < compCount; ++rgbaIdx)
            {
                const int v = static_cast<int>(*pSrcS32++);
                if (!isToSigned)
                {
                    pDstU8[rgbaIdx] = static_cast<uint8_t>((v > 0xff) ? 0xff : v);
                }
                else
                {
                    reinterpret_cast<int8_t*>(pDstU8)[rgbaIdx] =
                        static_cast<int8_t>(ClampValue(v, -0x7f, 0x7f));
                }
            }
            pDstU8 += EncRgbaCount;
        }
    }
    else
    {
        float* pDstF32 = reinterpret_cast<float*>(pDst);
        for (size_t pixelIdx = 0; pixelIdx < pixCount; ++pixelIdx)
        {
            pDstF32[1] = 0.0f;
            pDstF32[2] = 0.0f;
            pDstF32[3] = 1.0f;
            for (int rgbaIdx = 0; rgbaIdx < compCount; ++rgbaIdx)
            {
                pDstF32[rgbaIdx] = static_cast<float>(*pSrcS32++);
            }
            pDstF32 += EncRgbaCount;
        }
    }
}

//-----------------------------------------------------------------------------
// unorm_10_10_10_2 / uint_10_10_10_2 用の定数です。
//-----------------------------------------------------------------------------
const int Shifts1010102[]    = { 22, 12,  2,  0 }; // 各成分のシフト値（上位ビットから RGBA）です。
const int Shifts1010102Rev[] = {  0, 10, 20, 30 }; // 各成分のシフト値（下位ビットから RGBA）です。
const uint8_t Alphas1010102[] = { 0x00, 0x55, 0xaa, 0xff }; // 2bit のアルファ値を 8bit に変換するテーブルです。

//-----------------------------------------------------------------------------
//! @brief unorm_10_10_10_2 にエンコードします。
//!
//! @param[out] pDst 変換後のデータを格納します。
//! @param[in] pSrc 変換前のデータです（float_32_32_32_32）。
//! @param[in] pixCount ピクセル数です。
//! @param[in] isRev 上位ビットから RGBA の順に格納されているなら false、ABGR の順に格納されているなら true です。
//-----------------------------------------------------------------------------
void EncodeUnorm1010102(
    void* pDst,
    const void* pSrc,
    const size_t pixCount,
    const bool isRev
)
{
    const int MaxU2  = 0x003;
    const int MaxU10 = 0x3ff;
    const int* shifts = (!isRev) ? Shifts1010102 : Shifts1010102Rev;
    uint32_t* pDstU32 = reinterpret_cast<uint32_t*>(pDst);
    const float* pSrcF32 = reinterpret_cast<const float*>(pSrc);
    for (size_t pixelIdx = 0; pixelIdx < pixCount; ++pixelIdx)
    {
        const uint32_t r = static_cast<uint32_t>(ClampValue(RoundValue(pSrcF32[0] * MaxU10), 0, MaxU10));
        const uint32_t g = static_cast<uint32_t>(ClampValue(RoundValue(pSrcF32[1] * MaxU10), 0, MaxU10));
        const uint32_t b = static_cast<uint32_t>(ClampValue(RoundValue(pSrcF32[2] * MaxU10), 0, MaxU10));
        const uint32_t a = static_cast<uint32_t>(ClampValue(RoundValue(pSrcF32[3] * MaxU2 ), 0, MaxU2 ));
        *pDstU32++ = (r << shifts[0]) | (g << shifts[1]) | (b << shifts[2]) | (a << shifts[3]);
        pSrcF32 += EncRgbaCount;
    }
}

//-----------------------------------------------------------------------------
//! @brief unorm_10_10_10_2 をデコードします。
//!
//! @param[out] pDst 変換後のデータを格納します。
//! @param[in] pSrc 変換前のデータです。
//! @param[in] dstFormatStr 変換後のフォーマット文字列です（unorm_8_8_8_8 / snorm_8_8_8_8 / float_32_32_32_32）。
//! @param[in] pixCount ピクセル数です。
//! @param[in] isRev 上位ビットから RGBA の順に格納されているなら false、ABGR の順に格納されているなら true です。
//-----------------------------------------------------------------------------
void DecodeUnorm1010102(
    void* pDst,
    const void* pSrc,
    const std::string& dstFormatStr,
    const size_t pixCount,
    const bool isRev
)
{
    const int MaxU2  = 0x003;
    const int MaxU10 = 0x3ff;
    const bool isFloat = IsFloatFormat(dstFormatStr);
    const bool isToSigned = IsSignedFormat(dstFormatStr);
    const int* shifts = (!isRev) ? Shifts1010102 : Shifts1010102Rev;
    const int dstShiftRgbU8 = (!isToSigned) ? 2 : 3;
    const int dstShiftAU8   = (!isToSigned) ? 0 : 1;
    uint8_t* pDstU8 = reinterpret_cast<uint8_t*>(pDst);
    float* pDstF32 = reinterpret_cast<float*>(pDst);
    const uint32_t* pSrcU32 = reinterpret_cast<const uint32_t*>(pSrc);
    for (size_t pixelIdx = 0; pixelIdx < pixCount; ++pixelIdx)
    {
        const uint32_t v = *pSrcU32++;
        const uint32_t r = (v >> shifts[0]) & MaxU10;
        const uint32_t g = (v >> shifts[1]) & MaxU10;
        const uint32_t b = (v >> shifts[2]) & MaxU10;
        const uint32_t a = (v >> shifts[3]) & MaxU2;
        if (!isFloat)
        {
            *pDstU8++ = static_cast<uint8_t>(r >> dstShiftRgbU8);
            *pDstU8++ = static_cast<uint8_t>(g >> dstShiftRgbU8);
            *pDstU8++ = static_cast<uint8_t>(b >> dstShiftRgbU8);
            *pDstU8++ = static_cast<uint8_t>(Alphas1010102[a] >> dstShiftAU8);
        }
        else
        {
            *pDstF32++ = static_cast<float>(r) / MaxU10;
            *pDstF32++ = static_cast<float>(g) / MaxU10;
            *pDstF32++ = static_cast<float>(b) / MaxU10;
            *pDstF32++ = static_cast<float>(a) / MaxU2;
        }
    }
}

//-----------------------------------------------------------------------------
//! @brief uint_10_10_10_2 にエンコードします。
//!
//! @param[out] pDst 変換後のデータを格納します。
//! @param[in] pSrc 変換前のデータです（float_32_32_32_32）。
//! @param[in] pixCount ピクセル数です。
//! @param[in] isRev 上位ビットから RGBA の順に格納されているなら false、ABGR の順に格納されているなら true です。
//-----------------------------------------------------------------------------
void EncodeUint1010102(
    void* pDst,
    const void* pSrc,
    const size_t pixCount,
    const bool isRev
)
{
    const int MaxU2  = 0x003;
    const int MaxU10 = 0x3ff;
    const int* shifts = (!isRev) ? Shifts1010102 : Shifts1010102Rev;
    uint32_t* pDstU32 = reinterpret_cast<uint32_t*>(pDst);
    const float* pSrcF32 = reinterpret_cast<const float*>(pSrc);
    for (size_t pixelIdx = 0; pixelIdx < pixCount; ++pixelIdx)
    {
        const uint32_t r = static_cast<uint32_t>(ClampValue(RoundValue(pSrcF32[0]), 0, MaxU10));
        const uint32_t g = static_cast<uint32_t>(ClampValue(RoundValue(pSrcF32[1]), 0, MaxU10));
        const uint32_t b = static_cast<uint32_t>(ClampValue(RoundValue(pSrcF32[2]), 0, MaxU10));
        const uint32_t a = static_cast<uint32_t>(ClampValue(RoundValue(pSrcF32[3]), 0, MaxU2 ));
        *pDstU32++ = (r << shifts[0]) | (g << shifts[1]) | (b << shifts[2]) | (a << shifts[3]);
        pSrcF32 += EncRgbaCount;
    }
}

//-----------------------------------------------------------------------------
//! @brief uint_10_10_10_2 をデコードします。
//!
//! @param[out] pDst 変換後のデータを格納します。
//! @param[in] pSrc 変換前のデータです。
//! @param[in] dstFormatStr 変換後のフォーマット文字列です（unorm_8_8_8_8 / snorm_8_8_8_8 / float_32_32_32_32）。
//! @param[in] pixCount ピクセル数です。
//! @param[in] isRev 上位ビットから RGBA の順に格納されているなら false、ABGR の順に格納されているなら true です。
//-----------------------------------------------------------------------------
void DecodeUint1010102(
    void* pDst,
    const void* pSrc,
    const std::string& dstFormatStr,
    const size_t pixCount,
    const bool isRev
)
{
    const int MaxU2  = 0x003;
    const int MaxU10 = 0x3ff;
    const bool isFloat = IsFloatFormat(dstFormatStr);
    const bool isToSigned = IsSignedFormat(dstFormatStr);
    const int* shifts = (!isRev) ? Shifts1010102 : Shifts1010102Rev;
    const uint8_t dstMaxRgbU8 = (!isToSigned) ? 0xff : 0x7f;
    uint8_t* pDstU8 = reinterpret_cast<uint8_t*>(pDst);
    float* pDstF32 = reinterpret_cast<float*>(pDst);
    const uint32_t* pSrcU32 = reinterpret_cast<const uint32_t*>(pSrc);
    for (size_t pixelIdx = 0; pixelIdx < pixCount; ++pixelIdx)
    {
        const uint32_t v = *pSrcU32++;
        const uint32_t r = (v >> shifts[0]) & MaxU10;
        const uint32_t g = (v >> shifts[1]) & MaxU10;
        const uint32_t b = (v >> shifts[2]) & MaxU10;
        const uint32_t a = (v >> shifts[3]) & MaxU2;
        if (!isFloat)
        {
            *pDstU8++ = static_cast<uint8_t>((r > dstMaxRgbU8) ? dstMaxRgbU8 : r);
            *pDstU8++ = static_cast<uint8_t>((g > dstMaxRgbU8) ? dstMaxRgbU8 : g);
            *pDstU8++ = static_cast<uint8_t>((b > dstMaxRgbU8) ? dstMaxRgbU8 : b);
            *pDstU8++ = static_cast<uint8_t>(a);
        }
        else
        {
            *pDstF32++ = static_cast<float>(r);
            *pDstF32++ = static_cast<float>(g);
            *pDstF32++ = static_cast<float>(b);
            *pDstF32++ = static_cast<float>(a);
        }
    }
}

//-----------------------------------------------------------------------------
//! @brief float_16 / float_16_16 / float_16_16_16_16 にエンコードします。
//!
//! @param[out] pDst 変換後のデータを格納します。
//! @param[in] pSrc 変換前のデータです（float_32_32_32_32）。
//! @param[in] pixCount ピクセル数です。
//! @param[in] compCount 成分数です。
//-----------------------------------------------------------------------------
void EncodeFloat16(
    void* pDst,
    const void* pSrc,
    const size_t pixCount,
    const int compCount
)
{
    uint16_t* pDstU16 = reinterpret_cast<uint16_t*>(pDst);
    const float* pSrcF32 = reinterpret_cast<const float*>(pSrc);
    for (size_t iPix = 0; iPix < pixCount; ++iPix)
    {
        for (int iRgba = 0; iRgba < compCount; ++iRgba)
        {
            *pDstU16++ = GetFloat16Value(pSrcF32[iRgba]);
        }
        pSrcF32 += EncRgbaCount;
    }
}

//-----------------------------------------------------------------------------
//! @brief float_16 / float_16_16 / float_16_16_16_16 をデコードします。
//!
//! @param[out] pDst 変換後のデータを格納します。
//! @param[in] pSrc 変換前のデータです。
//! @param[in] dstFormatStr 変換後のフォーマット文字列です（unorm_8_8_8_8 / snorm_8_8_8_8 / float_32_32_32_32）。
//! @param[in] pixCount ピクセル数です。
//! @param[in] compCount 成分数です。
//-----------------------------------------------------------------------------
void DecodeFloat16(
    void* pDst,
    const void* pSrc,
    const std::string& dstFormatStr,
    const size_t pixCount,
    const int compCount
)
{
    const uint16_t* pSrcU16 = reinterpret_cast<const uint16_t*>(pSrc);
    if (!IsFloatFormat(dstFormatStr))
    {
        const bool isToSigned = IsSignedFormat(dstFormatStr);
        const uint8_t defaultA = (isToSigned) ? 0x7f : 0xff;
        uint8_t* pDstU8 = reinterpret_cast<uint8_t*>(pDst);
        for (size_t iPix = 0; iPix < pixCount; ++iPix)
        {
            pDstU8[1] = 0x00;
            pDstU8[2] = 0x00;
            pDstU8[3] = defaultA;
            for (int iRgba = 0; iRgba < compCount; ++iRgba)
            {
                const float fv = GetFloat32Value(*pSrcU16++);
                if (!isToSigned)
                {
                    pDstU8[iRgba] = GetUint8Value(fv * 0xff);
                }
                else
                {
                    reinterpret_cast<int8_t*>(pDstU8)[iRgba] = GetInt8Value(fv * 0x7f);
                }
            }
            pDstU8 += EncRgbaCount;
        }
    }
    else
    {
        float* pDstF32 = reinterpret_cast<float*>(pDst);
        for (size_t iPix = 0; iPix < pixCount; ++iPix)
        {
            pDstF32[1] = 0.0f;
            pDstF32[2] = 0.0f;
            pDstF32[3] = 1.0f;
            for (int iRgba = 0; iRgba < compCount; ++iRgba)
            {
                pDstF32[iRgba] = GetFloat32Value(*pSrcU16++);
            }
            pDstF32 += EncRgbaCount;
        }
    }
}

//-----------------------------------------------------------------------------
//! @brief float_32 / float_32_32 / float_32_32_32 を unorm_8_8_8_8 / snorm_8_8_8_8 にデコードします。
//!
//! @param[out] pDst 変換後のデータを格納します。
//! @param[in] pSrc 変換前のデータです。
//! @param[in] dstFormatStr 変換後のフォーマット文字列です（unorm_8_8_8_8 / snorm_8_8_8_8）。
//! @param[in] pixCount ピクセル数です。
//! @param[in] compCount 成分数です。
//-----------------------------------------------------------------------------
void DecodeFloat32(
    void* pDst,
    const void* pSrc,
    const std::string& dstFormatStr,
    const size_t pixCount,
    const int compCount
)
{
    const bool isToSigned = IsSignedFormat(dstFormatStr);
    const uint8_t defaultA = (isToSigned) ? 0x7f : 0xff;
    const float* pSrcF32 = reinterpret_cast<const float*>(pSrc);
    uint8_t* pDstU8 = reinterpret_cast<uint8_t*>(pDst);
    for (size_t iPix = 0; iPix < pixCount; ++iPix)
    {
        pDstU8[1] = 0x00;
        pDstU8[2] = 0x00;
        pDstU8[3] = defaultA;
        for (int iRgba = 0; iRgba < compCount; ++iRgba)
        {
            const float fv = *pSrcF32++;
            if (!isToSigned)
            {
                pDstU8[iRgba] = GetUint8Value(fv * 0xff);
            }
            else
            {
                reinterpret_cast<int8_t*>(pDstU8)[iRgba] = GetInt8Value(fv * 0x7f);
            }
        }
        pDstU8 += EncRgbaCount;
    }
}

//-----------------------------------------------------------------------------
//! @brief float_11_11_10 にエンコードします。
//!
//! @param[out] pDst 変換後のデータを格納します。
//! @param[in] pSrc 変換前のデータです（float_32_32_32_32）。
//! @param[in] pixCount ピクセル数です。
//! @param[in] isRev 上位ビットから RGB の順に格納するなら false、BGR の順に格納するなら true です。
//-----------------------------------------------------------------------------
void EncodeFloat111110(
    void* pDst,
    const void* pSrc,
    const size_t pixCount,
    const bool isRev
)
{
    uint32_t* pDstU32 = reinterpret_cast<uint32_t*>(pDst);
    const float* pSrcF32 = reinterpret_cast<const float*>(pSrc);
    for (size_t iPix = 0; iPix < pixCount; ++iPix)
    {
        uint32_t r = GetFloat16Value(pSrcF32[0]);
        uint32_t g = GetFloat16Value(pSrcF32[1]);
        uint32_t b = GetFloat16Value(pSrcF32[2]);
        if (r & 0x8000)
        {
            r = 0;
        }
        if (g & 0x8000)
        {
            g = 0;
        }
        if (b & 0x8000)
        {
            b = 0;
        }
        if (!isRev)
        {
            *pDstU32++ =
                (r << 17) & 0xffe00000 |
                (g <<  6) & 0x001ffc00 |
                (b >>  5) & 0x000003ff;
        }
        else
        {
            *pDstU32++ =
                (r >>  4) & 0x000007ff |
                (g <<  7) & 0x003ff800 |
                (b << 17) & 0xffc00000;
        }
        pSrcF32 += EncRgbaCount;
    }
}

//-----------------------------------------------------------------------------
//! @brief float_11_11_10 をデコードします。
//!
//! @param[out] pDst 変換後のデータを格納します。
//! @param[in] pSrc 変換前のデータです。
//! @param[in] dstFormatStr 変換後のフォーマット文字列です（unorm_8_8_8_8 / float_32_32_32_32）。
//! @param[in] pixCount ピクセル数です。
//! @param[in] isRev 上位ビットから RGB の順に格納されているなら false、BGR の順に格納されているなら true です。
//-----------------------------------------------------------------------------
void DecodeFloat111110(
    void* pDst,
    const void* pSrc,
    const std::string& dstFormatStr,
    const size_t pixCount,
    const bool isRev
)
{
    const bool isToFloat = IsFloatFormat(dstFormatStr);
    const uint32_t* pSrcU32 = reinterpret_cast<const uint32_t*>(pSrc);
    float* pDstF32 = reinterpret_cast<float*>(pDst);
    uint8_t* pDstU8 = reinterpret_cast<uint8_t*>(pDst);
    for (size_t iPix = 0; iPix < pixCount; ++iPix)
    {
        uint32_t value = *pSrcU32++;
        uint32_t r = 0;
        uint32_t g = 0;
        uint32_t b = 0;
        if (!isRev)
        {
            r = (value & 0xffe00000) >> 17;
            g = (value & 0x001ffc00) >>  6;
            b = (value & 0x000003ff) <<  5;
        }
        else
        {
            r = (value & 0x000007ff) <<  4;
            g = (value & 0x003ff800) >>  7;
            b = (value & 0xffc00000) >> 17;
        }
        const float fr = GetFloat32Value(static_cast<uint16_t>(r));
        const float fg = GetFloat32Value(static_cast<uint16_t>(g));
        const float fb = GetFloat32Value(static_cast<uint16_t>(b));
        if (isToFloat)
        {
            pDstF32[0] = fr;
            pDstF32[1] = fg;
            pDstF32[2] = fb;
            pDstF32[3] = 1.0f;
            pDstF32 += EncRgbaCount;
        }
        else
        {
            pDstU8[0] = GetUint8Value(fr * 0xff);
            pDstU8[1] = GetUint8Value(fg * 0xff);
            pDstU8[2] = GetUint8Value(fb * 0xff);
            pDstU8[3] = 0xff;
            pDstU8 += EncRgbaCount;
        }
    }
}

//-----------------------------------------------------------------------------
//! @brief unorm_8_8_8_8 からエンコードします。
//!
//! @param[out] pDst 変換後のデータを格納します。
//! @param[in] pSrc 変換前のデータです。
//! @param[in] dstFormatStr 変換後のフォーマット文字列です。
//! @param[in] pixCount ピクセル数です。
//! @param[in] isRev 上位ビットから RGB の順に格納するなら false、BGR の順に格納するなら true です。
//!
//! @return 処理成功なら true を返します。
//-----------------------------------------------------------------------------
bool EncodeFromUnorm8(
    void* pDst,
    const void* pSrc,
    const std::string& dstFormatStr,
    const size_t pixCount,
    const bool isRev
)
{
    if (dstFormatStr == "unorm_4_4")
    {
        EncodeUnorm44(pDst, pSrc, pixCount, isRev);
    }
    else if (dstFormatStr == "unorm_5_6_5")
    {
        EncodeUnorm565(pDst, pSrc, pixCount, isRev);
    }
    else if (dstFormatStr == "unorm_5_5_5_1")
    {
        EncodeUnorm5551(pDst, pSrc, pixCount, isRev);
    }
    else if (dstFormatStr == "unorm_4_4_4_4")
    {
        EncodeUnorm4444(pDst, pSrc, pixCount, isRev);
    }
    else if (dstFormatStr == "float_32_32_32_32")
    {
        ConvertUnorm8ToFloat32(pDst, pSrc, pixCount);
    }
    else if (dstFormatStr == "unorm_8"       ||
             dstFormatStr == "uint_8"        ||
             dstFormatStr == "unorm_8_8"     ||
             dstFormatStr == "uint_8_8"      ||
             dstFormatStr == "unorm_8_8_8_8" ||
             dstFormatStr == "uint_8_8_8_8"  ||
             dstFormatStr == "srgb_8_8_8_8")
    {
        CopyFromPlainRgba(pDst, pSrc, dstFormatStr, pixCount);
    }
    else
    {
        return false;
    }
    return true;
}

//-----------------------------------------------------------------------------
//! @brief snorm_8_8_8_8 からエンコードします。
//!
//! @param[out] pDst 変換後のデータを格納します。
//! @param[in] pSrc 変換前のデータです。
//! @param[in] dstFormatStr 変換後のフォーマット文字列です。
//! @param[in] pixCount ピクセル数です。
//! @param[in] isRev 上位ビットから RGB の順に格納するなら false、BGR の順に格納するなら true です。
//!
//! @return 処理成功なら true を返します。
//-----------------------------------------------------------------------------
bool EncodeFromSnorm8(
    void* pDst,
    const void* pSrc,
    const std::string& dstFormatStr,
    const size_t pixCount,
    const bool isRev
)
{
    ENC_UNUSED_VARIABLE(isRev);

    if (dstFormatStr == "float_32_32_32_32")
    {
        ConvertSnorm8ToFloat32(pDst, pSrc, pixCount);
    }
    else if (dstFormatStr == "snorm_8"       ||
             dstFormatStr == "sint_8"        ||
             dstFormatStr == "snorm_8_8"     ||
             dstFormatStr == "sint_8_8"      ||
             dstFormatStr == "snorm_8_8_8_8" ||
             dstFormatStr == "sint_8_8_8_8")
    {
        CopyFromPlainRgba(pDst, pSrc, dstFormatStr, pixCount);
    }
    else
    {
        return false;
    }
    return true;
}

//-----------------------------------------------------------------------------
//! @brief float_32_32_32_32 からエンコードします。
//!
//! @param[out] pDst 変換後のデータを格納します。
//! @param[in] pSrc 変換前のデータです。
//! @param[in] dstFormatStr 変換後のフォーマット文字列です。
//! @param[in] pixCount ピクセル数です。
//! @param[in] isRev 上位ビットから RGB の順に格納するなら false、BGR の順に格納するなら true です。
//!
//! @return 処理成功なら true を返します。
//-----------------------------------------------------------------------------
bool EncodeFromFloat32(
    void* pDst,
    const void* pSrc,
    const std::string& dstFormatStr,
    const size_t pixCount,
    const bool isRev
)
{
    const int compCount = GetComponentCount(dstFormatStr);
    if (dstFormatStr == "unorm_16"          ||
        dstFormatStr == "unorm_16_16"       ||
        dstFormatStr == "unorm_16_16_16_16")
    {
        EncodeUnorm16(pDst, pSrc, pixCount, compCount);
    }
    else if (dstFormatStr == "snorm_16"          ||
             dstFormatStr == "snorm_16_16"       ||
             dstFormatStr == "snorm_16_16_16_16")
    {
        EncodeSnorm16(pDst, pSrc, pixCount, compCount);
    }
    else if (dstFormatStr == "uint_16"          ||
             dstFormatStr == "uint_16_16"       ||
             dstFormatStr == "uint_16_16_16_16")
    {
        EncodeUint16(pDst, pSrc, pixCount, compCount);
    }
    else if (dstFormatStr == "sint_16"          ||
             dstFormatStr == "sint_16_16"       ||
             dstFormatStr == "sint_16_16_16_16")
    {
        EncodeSint16(pDst, pSrc, pixCount, compCount);
    }
    else if (dstFormatStr == "uint_32"          ||
             dstFormatStr == "uint_32_32"       ||
             dstFormatStr == "uint_32_32_32"    ||
             dstFormatStr == "uint_32_32_32_32")
    {
        EncodeUint32(pDst, pSrc, pixCount, compCount);
    }
    else if (dstFormatStr == "sint_32"          ||
             dstFormatStr == "sint_32_32"       ||
             dstFormatStr == "sint_32_32_32"    ||
             dstFormatStr == "sint_32_32_32_32")
    {
        EncodeSint32(pDst, pSrc, pixCount, compCount);
    }
    else if (dstFormatStr == "unorm_10_10_10_2")
    {
        EncodeUnorm1010102(pDst, pSrc, pixCount, isRev);
    }
    else if (dstFormatStr == "uint_10_10_10_2")
    {
        EncodeUint1010102(pDst, pSrc, pixCount, isRev);
    }
    else if (dstFormatStr == "float_16"          ||
             dstFormatStr == "float_16_16"       ||
             dstFormatStr == "float_16_16_16_16")
    {
        EncodeFloat16(pDst, pSrc, pixCount, compCount);
    }
    else if (dstFormatStr == "float_11_11_10")
    {
        EncodeFloat111110(pDst, pSrc, pixCount, isRev);
    }
    else if (dstFormatStr == "unorm_8_8_8_8" ||
             dstFormatStr == "srgb_8_8_8_8")
    {
        ConvertFloat32ToUnorm8(pDst, pSrc, pixCount);
    }
    else if (dstFormatStr == "snorm_8_8_8_8")
    {
        ConvertFloat32ToSnorm8(pDst, pSrc, pixCount);
    }
    else if (dstFormatStr == "float_32"          ||
             dstFormatStr == "float_32_32"       ||
             dstFormatStr == "float_32_32_32"    ||
             dstFormatStr == "float_32_32_32_32")
    {
        CopyFromPlainRgba(pDst, pSrc, dstFormatStr, pixCount);
    }
    else
    {
        return false;
    }
    return true;
}

//-----------------------------------------------------------------------------
//! @brief unorm_8_8_8_8 へデコードします。
//!
//! @param[out] pDst 変換後のデータを格納します。
//! @param[in] pSrc 変換前のデータです。
//! @param[in] srcFormatStr 変換前のフォーマット文字列です。
//! @param[in] pixCount ピクセル数です。
//! @param[in] isRev 上位ビットから RGB の順に格納されているなら false、BGR の順に格納されているなら true です。
//!
//! @return 処理成功なら true を返します。
//-----------------------------------------------------------------------------
bool DecodeToUnorm8(
    void* pDst,
    const void* pSrc,
    const std::string& srcFormatStr,
    const size_t pixCount,
    const bool isRev
)
{
    const int compCount = GetComponentCount(srcFormatStr);
    if (srcFormatStr == "unorm_4_4")
    {
        DecodeUnorm44(pDst, pSrc, pixCount, isRev);
    }
    else if (srcFormatStr == "unorm_5_6_5")
    {
        DecodeUnorm565(pDst, pSrc, pixCount, isRev);
    }
    else if (srcFormatStr == "unorm_5_5_5_1")
    {
        DecodeUnorm5551(pDst, pSrc, pixCount, isRev);
    }
    else if (srcFormatStr == "unorm_4_4_4_4")
    {
        DecodeUnorm4444(pDst, pSrc, pixCount, isRev);
    }
    else if (srcFormatStr == "unorm_16"          ||
             srcFormatStr == "unorm_16_16"       ||
             srcFormatStr == "unorm_16_16_16_16")
    {
        DecodeUnorm16(pDst, pSrc, "unorm_8_8_8_8", pixCount, compCount);
    }
    else if (srcFormatStr == "uint_16"          ||
             srcFormatStr == "uint_16_16"       ||
             srcFormatStr == "uint_16_16_16_16")
    {
        DecodeUint16(pDst, pSrc, "unorm_8_8_8_8", pixCount, compCount);
    }
    else if (srcFormatStr == "uint_32"          ||
             srcFormatStr == "uint_32_32"       ||
             srcFormatStr == "uint_32_32_32"    ||
             srcFormatStr == "uint_32_32_32_32")
    {
        DecodeUint32(pDst, pSrc, "unorm_8_8_8_8", pixCount, compCount);
    }
    else if (srcFormatStr == "unorm_10_10_10_2")
    {
        DecodeUnorm1010102(pDst, pSrc, "unorm_8_8_8_8", pixCount, isRev);
    }
    else if (srcFormatStr == "uint_10_10_10_2")
    {
        DecodeUint1010102(pDst, pSrc, "unorm_8_8_8_8", pixCount, isRev);
    }
    else if (srcFormatStr == "float_16"          ||
             srcFormatStr == "float_16_16"       ||
             srcFormatStr == "float_16_16_16_16")
    {
        DecodeFloat16(pDst, pSrc, "unorm_8_8_8_8", pixCount, compCount);
    }
    else if (srcFormatStr == "float_32"       ||
             srcFormatStr == "float_32_32"    ||
             srcFormatStr == "float_32_32_32")
    {
        DecodeFloat32(pDst, pSrc, "unorm_8_8_8_8", pixCount, compCount);
    }
    else if (srcFormatStr == "float_11_11_10")
    {
        DecodeFloat111110(pDst, pSrc, "unorm_8_8_8_8", pixCount, isRev);
    }
    else if (srcFormatStr == "unorm_8"       ||
             srcFormatStr == "uint_8"        ||
             srcFormatStr == "unorm_8_8"     ||
             srcFormatStr == "uint_8_8"      ||
             srcFormatStr == "unorm_8_8_8_8" ||
             srcFormatStr == "uint_8_8_8_8"  ||
             srcFormatStr == "srgb_8_8_8_8")
    {
        CopyToPlainRgba(pDst, pSrc, srcFormatStr, pixCount);
    }
    else
    {
        return false;
    }
    return true;
}

//-----------------------------------------------------------------------------
//! @brief snorm_8_8_8_8 へデコードします。
//!
//! @param[out] pDst 変換後のデータを格納します。
//! @param[in] pSrc 変換前のデータです。
//! @param[in] srcFormatStr 変換前のフォーマット文字列です。
//! @param[in] pixCount ピクセル数です。
//! @param[in] isRev 上位ビットから RGB の順に格納されているなら false、BGR の順に格納されているなら true です。
//!
//! @return 処理成功なら true を返します。
//-----------------------------------------------------------------------------
bool DecodeToSnorm8(
    void* pDst,
    const void* pSrc,
    const std::string& srcFormatStr,
    const size_t pixCount,
    const bool isRev
)
{
    ENC_UNUSED_VARIABLE(isRev);

    const int compCount = GetComponentCount(srcFormatStr);
    if (srcFormatStr == "snorm_16"          ||
        srcFormatStr == "snorm_16_16"       ||
        srcFormatStr == "snorm_16_16_16_16")
    {
        DecodeSnorm16(pDst, pSrc, "snorm_8_8_8_8", pixCount, compCount);
    }
    else if (srcFormatStr == "sint_16"          ||
             srcFormatStr == "sint_16_16"       ||
             srcFormatStr == "sint_16_16_16_16")
    {
        DecodeSint16(pDst, pSrc, "snorm_8_8_8_8", pixCount, compCount);
    }
    else if (srcFormatStr == "sint_32"          ||
             srcFormatStr == "sint_32_32"       ||
             srcFormatStr == "sint_32_32_32"    ||
             srcFormatStr == "sint_32_32_32_32")
    {
        DecodeSint32(pDst, pSrc, "snorm_8_8_8_8", pixCount, compCount);
    }
    else if (srcFormatStr == "float_16"          ||
             srcFormatStr == "float_16_16"       ||
             srcFormatStr == "float_16_16_16_16")
    {
        DecodeFloat16(pDst, pSrc, "snorm_8_8_8_8", pixCount, compCount);
    }
    else if (srcFormatStr == "float_32"       ||
             srcFormatStr == "float_32_32"    ||
             srcFormatStr == "float_32_32_32")
    {
        DecodeFloat32(pDst, pSrc, "snorm_8_8_8_8", pixCount, compCount);
    }
    else if (srcFormatStr == "snorm_8"       ||
             srcFormatStr == "sint_8"        ||
             srcFormatStr == "snorm_8_8"     ||
             srcFormatStr == "sint_8_8"      ||
             srcFormatStr == "snorm_8_8_8_8" ||
             srcFormatStr == "sint_8_8_8_8" )
    {
        CopyToPlainRgba(pDst, pSrc, srcFormatStr, pixCount);
    }
    else
    {
        return false;
    }
    return true;
}

//-----------------------------------------------------------------------------
//! @brief float_32_32_32_32 へデコードします。
//!
//! @param[out] pDst 変換後のデータを格納します。
//! @param[in] pSrc 変換前のデータです。
//! @param[in] srcFormatStr 変換前のフォーマット文字列です。
//! @param[in] pixCount ピクセル数です。
//! @param[in] isRev 上位ビットから RGB の順に格納されているなら false、BGR の順に格納されているなら true です。
//!
//! @return 処理成功なら true を返します。
//-----------------------------------------------------------------------------
bool DecodeToFloat32(
    void* pDst,
    const void* pSrc,
    const std::string& srcFormatStr,
    const size_t pixCount,
    const bool isRev
)
{
    const int compCount = GetComponentCount(srcFormatStr);
    if (srcFormatStr == "unorm_16"          ||
        srcFormatStr == "unorm_16_16"       ||
        srcFormatStr == "unorm_16_16_16_16")
    {
        DecodeUnorm16(pDst, pSrc, "float_32_32_32_32", pixCount, compCount);
    }
    else if (srcFormatStr == "snorm_16"          ||
             srcFormatStr == "snorm_16_16"       ||
             srcFormatStr == "snorm_16_16_16_16")
    {
        DecodeSnorm16(pDst, pSrc, "float_32_32_32_32", pixCount, compCount);
    }
    else if (srcFormatStr == "uint_16"          ||
             srcFormatStr == "uint_16_16"       ||
             srcFormatStr == "uint_16_16_16_16")
    {
        DecodeUint16(pDst, pSrc, "float_32_32_32_32", pixCount, compCount);
    }
    else if (srcFormatStr == "sint_16"          ||
             srcFormatStr == "sint_16_16"       ||
             srcFormatStr == "sint_16_16_16_16")
    {
        DecodeSint16(pDst, pSrc, "float_32_32_32_32", pixCount, compCount);
    }
    else if (srcFormatStr == "uint_32"          ||
             srcFormatStr == "uint_32_32"       ||
             srcFormatStr == "uint_32_32_32"    ||
             srcFormatStr == "uint_32_32_32_32")
    {
        DecodeUint32(pDst, pSrc, "float_32_32_32_32", pixCount, compCount);
    }
    else if (srcFormatStr == "sint_32"          ||
             srcFormatStr == "sint_32_32"       ||
             srcFormatStr == "sint_32_32_32"    ||
             srcFormatStr == "sint_32_32_32_32")
    {
        DecodeSint32(pDst, pSrc, "float_32_32_32_32", pixCount, compCount);
    }
    else if (srcFormatStr == "unorm_10_10_10_2")
    {
        DecodeUnorm1010102(pDst, pSrc, "float_32_32_32_32", pixCount, isRev);
    }
    else if (srcFormatStr == "uint_10_10_10_2")
    {
        DecodeUint1010102(pDst, pSrc, "float_32_32_32_32", pixCount, isRev);
    }
    else if (srcFormatStr == "float_16"          ||
             srcFormatStr == "float_16_16"       ||
             srcFormatStr == "float_16_16_16_16")
    {
        DecodeFloat16(pDst, pSrc, "float_32_32_32_32", pixCount, compCount);
    }
    else if (srcFormatStr == "float_11_11_10")
    {
        DecodeFloat111110(pDst, pSrc, "float_32_32_32_32", pixCount, isRev);
    }
    else if (srcFormatStr == "float_32"          ||
             srcFormatStr == "float_32_32"       ||
             srcFormatStr == "float_32_32_32"    ||
             srcFormatStr == "float_32_32_32_32")
    {
        CopyToPlainRgba(pDst, pSrc, srcFormatStr, pixCount);
    }
    else
    {
        return false;
    }
    return true;
}

} // unnamed namespace

//-----------------------------------------------------------------------------
//! @brief シンプルなフォーマットを変換します。
//!
//! @param[out] pDst 変換後のデータを格納します。
//! @param[in] pSrc 変換前のデータです。
//! @param[in] dstFormatStr 変換後のフォーマット文字列です。
//! @param[in] srcFormatStr 変換前のフォーマット文字列です。
//! @param[in] encodeFlag エンコードフラグです。
//! @param[in] imageW 画像の幅です。
//! @param[in] imageH 画像の高さです。
//! @param[in] imageD 画像の奥行きです。
//!
//! @return 処理成功なら true を返します。
//-----------------------------------------------------------------------------
bool ConvertSimpleFormat(
    void* pDst,
    const void* pSrc,
    const std::string& dstFormatStr,
    const std::string& srcFormatStr,
    const int encodeFlag,
    const int imageW,
    const int imageH,
    const int imageD
)
{
    const bool isRev = (encodeFlag & EncodeFlag_ReverseRgba) != 0;
    const size_t pixCount = static_cast<size_t>(imageW) * imageH * imageD;

    if (srcFormatStr == "unorm_8_8_8_8" ||
        srcFormatStr == "srgb_8_8_8_8")
    {
        return EncodeFromUnorm8(pDst, pSrc, dstFormatStr, pixCount, isRev);
    }
    else if (srcFormatStr == "snorm_8_8_8_8")
    {
        return EncodeFromSnorm8(pDst, pSrc, dstFormatStr, pixCount, isRev);
    }
    else if (srcFormatStr == "float_32_32_32_32")
    {
        return EncodeFromFloat32(pDst, pSrc, dstFormatStr, pixCount, isRev);
    }
    else if (dstFormatStr == "unorm_8_8_8_8" ||
             dstFormatStr == "srgb_8_8_8_8")
    {
        return DecodeToUnorm8(pDst, pSrc, srcFormatStr, pixCount, isRev);
    }
    else if (dstFormatStr == "snorm_8_8_8_8")
    {
        return DecodeToSnorm8(pDst, pSrc, srcFormatStr, pixCount, isRev);
    }
    else if (dstFormatStr == "float_32_32_32_32")
    {
        return DecodeToFloat32(pDst, pSrc, srcFormatStr, pixCount, isRev);
    }
    else
    {
        return false;
    }
}
#endif

