﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/
#ifdef NW_FOR_FTX

// ConvertByAstcEvaluationCodec
// SetImageData SetErrorWeightingParams
// DecodeAstcData

//=============================================================================
// include
//=============================================================================
#include "AstcFormat.h"

#undef IGNORE
// ↑マクロ再定義の warning C4005 を回避
//   C:\Program Files (x86)\Windows Kits\8.0\include\um\winbase.h(697): #define IGNORE 0
//   astc_codec_internals.h(34): #define IGNORE(param) ((void)&param)

#include <AstcEvaluationCodec.h>

using namespace std;
using namespace nn::gfx::tool::texenc;

namespace
{

//-----------------------------------------------------------------------------
//! @brief Astc Evaluation Codec に渡す品質を取得します。
//!
//! @param[in] qualityStr エンコード品質です。
//!
//! @return 品質を返します。
//-----------------------------------------------------------------------------
std::string GetAstcQuality(const std::string& qualityStr)
{
    if (qualityStr == "veryfast"   ||
        qualityStr == "fast"       ||
        qualityStr == "medium"     ||
        qualityStr == "thorough"   ||
        qualityStr == "exhaustive")
    {
        return qualityStr;
    }
    else
    {
        const int qualityLevel = GetQualityLevel(qualityStr);
        return
            (qualityLevel == 0) ? "fast"       :
            (qualityLevel == 1) ? "medium"     :
            (qualityLevel == 2) ? "thorough"   :
            (qualityLevel >= 3) ? "exhaustive" :
            "medium";
    }
}

//-----------------------------------------------------------------------------
//! @brief デコードモードを取得します。
//!
//! @param[in] formatStr 中間ファイルのフォーマット文字列です。
//! @param[in] supportsHdr HDR に対応するなら true です。
//!
//! @return デコードモードを返します。
//-----------------------------------------------------------------------------
astc_decode_mode GetDecodeMode(
    const std::string& formatStr,
    const bool supportsHdr
)
{
    return
        (supportsHdr && IsFloatFormat(formatStr)) ? DECODE_HDR      :
        (IsSrgbFetchFormat(formatStr)           ) ? DECODE_LDR_SRGB :
        DECODE_LDR;
}

//-----------------------------------------------------------------------------
//! @brief 誤差の重み付けパラメータを設定します。
//!
//! @param[out] pEwp 誤差の重み付けパラメータを格納します。
//! @param[in] xdim ブロックの幅です。
//! @param[in] ydim ブロックの高さです。
//! @param[in] zdim ブロックの奥行きです。
//! @param[in] quality エンコード品質文字列です。
//! @param[in] forcesHdr HDR を強制するなら true です。
//! @param[in] isNormalMapLa 法線マップ用のエンコーディングなら true です。
//-----------------------------------------------------------------------------
void SetErrorWeightingParams(
    error_weighting_params* pEwp,
    const int xdim,
    const int ydim,
    const int zdim,
    const std::string& quality,
    const bool forcesHdr,
    const bool isNormalMapLa
)
{
    //-----------------------------------------------------------------------------
    // パラメータを初期化します。
    error_weighting_params& ewp = *pEwp;

    ewp.rgb_power          = 1.0f;
    ewp.rgb_base_weight    = 1.0f;
    ewp.rgb_mean_weight    = 0.0f;
    ewp.rgb_stdev_weight   = 0.0f;

    ewp.alpha_power        = 1.0f;
    ewp.alpha_base_weight  = 1.0f;
    ewp.alpha_mean_weight  = 0.0f;
    ewp.alpha_stdev_weight = 0.0f;

    ewp.rgb_mean_and_stdev_mixing = 0.0f;
    ewp.mean_stdev_radius = 0;
    ewp.enable_rgb_scale_with_alpha = 0;
    ewp.alpha_radius = 0;
    ewp.ra_normal_angular_scale = 0;
    ewp.block_artifact_suppression = 0.0f;

    ewp.rgba_weights[0] = 1.0f;
    ewp.rgba_weights[1] = 1.0f;
    ewp.rgba_weights[2] = 1.0f;
    ewp.rgba_weights[3] = 1.0f;

    //-----------------------------------------------------------------------------
    // 品質によって異なるパラメータを設定します。
    int plimit_autoset = -1;
    float oplimit_autoset = 0.0f;
    float dblimit_autoset_3d = 0.0f;
    float mincorrel_autoset = 0.0f;
    float bmc_autoset = 0.0f;
    int maxiters_autoset = 0;

    const float log10_texels_3d = log((float)(xdim * ydim * zdim)) / log(10.0f);

    if (quality == "veryfast")
    {
        plimit_autoset = 2;
        oplimit_autoset = 1.0;
        dblimit_autoset_3d = MAX(70 - 35 * log10_texels_3d, 53 - 19 * log10_texels_3d);
        bmc_autoset = 25;
        mincorrel_autoset = 0.5;
        maxiters_autoset = 1;
    }
    else if (quality == "fast")
    {
        plimit_autoset = 4;
        oplimit_autoset = 1.0;
        mincorrel_autoset = 0.5;
        dblimit_autoset_3d = MAX(85 - 35 * log10_texels_3d, 63 - 19 * log10_texels_3d);
        bmc_autoset = 50;
        maxiters_autoset = 1;
    }
    else if (quality == "medium")
    {
        plimit_autoset = 25;
        oplimit_autoset = 1.2f;
        mincorrel_autoset = 0.75f;
        dblimit_autoset_3d = MAX(95 - 35 * log10_texels_3d, 70 - 19 * log10_texels_3d);
        bmc_autoset = 75;
        maxiters_autoset = 2;
    }
    else if (quality == "thorough")
    {
        plimit_autoset = 100;
        oplimit_autoset = 2.5f;
        mincorrel_autoset = 0.95f;
        dblimit_autoset_3d = MAX(105 - 35 * log10_texels_3d, 77 - 19 * log10_texels_3d);
        bmc_autoset = 95;
        maxiters_autoset = 4;
    }
    else // exhaustive
    {
        plimit_autoset = PARTITION_COUNT;
        oplimit_autoset = 1000.0f;
        mincorrel_autoset = 0.99f;
        dblimit_autoset_3d = 999.0f;
        bmc_autoset = 100;
        maxiters_autoset = 4;
    }

    //-----------------------------------------------------------------------------
    // 法線マップ用のパラメータを設定します。
    const float SmallWeight = 0.001f;
    if (isNormalMapLa)
    {
        ewp.rgba_weights[1] = SmallWeight;
        ewp.rgba_weights[2] = SmallWeight;
        //ewp.rgba_weights[2] = SmallWeight;
        //ewp.rgba_weights[3] = SmallWeight;

        ewp.ra_normal_angular_scale = 1;

        // -normal_psnr
        oplimit_autoset = 1000.0f;
        mincorrel_autoset = 0.99f;

        // -normal_percep
        if (quality == "thorough" || quality == "exhaustive")
        {
            dblimit_autoset_3d = 999.0f;
            ewp.block_artifact_suppression = 1.8f;
            ewp.mean_stdev_radius = 3;
            ewp.rgb_mean_weight = 0;
            ewp.rgb_stdev_weight = 50;
            ewp.rgb_mean_and_stdev_mixing = 0.0;
            ewp.alpha_mean_weight = 0;
            ewp.alpha_stdev_weight = 50;
        }
    }

    //-----------------------------------------------------------------------------
    // パラメータを計算します。
    const int partitions_to_test = plimit_autoset;
    ewp.partition_search_limit = std::min(std::max(partitions_to_test, 1), PARTITION_COUNT);
    ewp.block_mode_cutoff = bmc_autoset / 100.0f;
    ewp.texel_avg_error_limit = (!forcesHdr) ?
        pow(0.1f, dblimit_autoset_3d * 0.1f) * 65535.0f * 65535.0f : 0.0f;
    ewp.partition_1_to_2_limit = oplimit_autoset;
    ewp.lowest_correlation_cutoff = mincorrel_autoset;
    ewp.max_refinement_iters = maxiters_autoset;

    expand_block_artifact_suppression(xdim, ydim, zdim, &ewp);
} // NOLINT(readability/fn_size)

//-----------------------------------------------------------------------------
//! @brief 画像データを設定します。
//!
//! @param[in,out] pImage 画像です。
//! @param[in] pBitmap ビットマップデータです。
//!                    1 ピクセルあたり 4 または 16 バイトで、RGBA の順に格納します。画像の左上が先頭です。
//-----------------------------------------------------------------------------
void SetImageData(astc_codec_image* pImage, const void* pBitmap)
{
    //-----------------------------------------------------------------------------
    // ビットマップをコピーします。
    const int lineValueCount = pImage->xsize * 4;
    if (pImage->imagedata8 != NULL) // 8bit
    {
        const int lineBytes = lineValueCount * sizeof(uint8_t);
        const uint8_t* pSrc = reinterpret_cast<const uint8_t*>(pBitmap);
        for (int z = 0; z < pImage->zsize; ++z)
        {
            const int dstZ = (pImage->zsize == 1) ? z : pImage->padding + z;
            for (int y = 0; y < pImage->ysize; ++y)
            {
                const int dstY = pImage->padding + y; // 画像の左上を先頭に
                //const int dstY = pImage->padding + (pImage->ysize - 1 - y); // 画像の左下を先頭に
                memcpy(pImage->imagedata8[dstZ][dstY] + pImage->padding * 4, pSrc, lineBytes);
                pSrc += lineBytes;
            }
        }
    }
    else // 16bit
    {
        const float* pSrc = reinterpret_cast<const float*>(pBitmap);
        for (int z = 0; z < pImage->zsize; ++z)
        {
            const int dstZ = (pImage->zsize == 1) ? z : pImage->padding + z;
            for (int y = 0; y < pImage->ysize; ++y)
            {
                const int dstY = pImage->padding + y; // 画像の左上を先頭に
                //const int dstY = pImage->padding + (pImage->ysize - 1 - y); // 画像の左下を先頭に
                uint16_t* pDst = pImage->imagedata16[dstZ][dstY] + pImage->padding * 4;
                for (int iValue = 0; iValue < lineValueCount; ++iValue)
                {
                    *pDst++ = GetFloat16Value(*pSrc++);
                }
            }
        }
    }

    //-----------------------------------------------------------------------------
    // パディングエリアを設定します。
    fill_image_padding_area(pImage);
}

//-----------------------------------------------------------------------------
//! @brief 画像ブロックをビットマップにコピーします。
//!
//! @param[out] pDst ビットマップデータを格納します。
//! @param[in] pBlock 画像ブロックへのポインタです。
//! @param[in] decode_mode デコードモードです。
//! @param[in] imageW 画像の幅です。
//! @param[in] imageH 画像の高さです。
//! @param[in] imageD 画像の奥行きです。
//! @param[in] xdim ブロックの幅です。
//! @param[in] ydim ブロックの高さです。
//! @param[in] zdim ブロックの奥行きです。
//-----------------------------------------------------------------------------
void CopyImageBlock(
    void* pDst,
    const imageblock* pBlock,
    const astc_decode_mode decode_mode,
    const int imageW,
    const int imageH,
    const int imageD,
    const int xdim,
    const int ydim,
    const int zdim
)
{
    const bool isFloat = (decode_mode == DECODE_HDR);
    const size_t colBytes = (isFloat) ? EncRgbaFloatBytes : EncRgbaBytes;
    const size_t lineBytes = colBytes * imageW;
    const size_t faceBytes = lineBytes * imageH;

    const float* pSrc = pBlock->orig_data;
    const uint8_t* pNan = pBlock->nan_texel;
    const int endZ = std::min(zdim, imageD - pBlock->zpos);
    for (int z = 0; z < endZ; ++z)
    {
        uint8_t* pDstU8 = reinterpret_cast<uint8_t*>(pDst) +
            (pBlock->zpos + z) * faceBytes + pBlock->ypos * lineBytes;
        for (int y = 0; y < ydim; ++y)
        {
            const int dstY = (pBlock->ypos + y);
            for (int x = 0; x < xdim; ++x)
            {
                const int dstX = (pBlock->xpos + x);
                if (dstX < imageW && dstY < imageH)
                {
                    float r = 1.0f;
                    float g = 1.0f;
                    float b = 1.0f;
                    float a = 1.0f;
                    if (*pNan == 0)
                    {
                        r = pSrc[0];
                        g = pSrc[1];
                        b = pSrc[2];
                        a = pSrc[3];
                    }

                    if (!isFloat)
                    {
                        uint8_t* pU = pDstU8 + dstX * EncRgbaCount;
                        pU[0] = static_cast<uint8_t>(ClampValue(RoundValue(r * 0xff), 0x00, 0xff));
                        pU[1] = static_cast<uint8_t>(ClampValue(RoundValue(g * 0xff), 0x00, 0xff));
                        pU[2] = static_cast<uint8_t>(ClampValue(RoundValue(b * 0xff), 0x00, 0xff));
                        pU[3] = static_cast<uint8_t>(ClampValue(RoundValue(a * 0xff), 0x00, 0xff));
                    }
                    else
                    {
                        float* pF = reinterpret_cast<float*>(pDstU8) + dstX * EncRgbaCount;
                        pF[0] = r;
                        pF[1] = g;
                        pF[2] = b;
                        pF[3] = a;
                    }
                }
                pSrc += EncRgbaCount;
                ++pNan;
            }
            pDstU8 += lineBytes;
        }
    }
}

//-----------------------------------------------------------------------------
//! @brief ASTC フォーマットにエンコードしたデータをデコードします。
//!
//! @param[out] pDst ビットマップデータを格納します。
//! @param[in] pSrc ASTC フォーマットにエンコードしたデータへのポインタです。
//! @param[in] decode_mode デコードモードです。
//! @param[in] imageW 画像の幅です。
//! @param[in] imageH 画像の高さです。
//! @param[in] imageD 画像の奥行きです。
//! @param[in] xdim ブロックの幅です。
//! @param[in] ydim ブロックの高さです。
//! @param[in] zdim ブロックの奥行きです。
//-----------------------------------------------------------------------------
void DecodeAstcData(
    void* pDst,
    const void* pSrc,
    const astc_decode_mode decode_mode,
    const int imageW,
    const int imageH,
    const int imageD,
    const int xdim,
    const int ydim,
    const int zdim
)
{
    const int xblocks = (imageW + xdim - 1) / xdim;
    const int yblocks = (imageH + ydim - 1) / ydim;
    const int zblocks = (imageD + zdim - 1) / zdim;

    const uint8_t* pS = reinterpret_cast<const uint8_t*>(pSrc);
    imageblock pb;
    for (int z = 0; z < zblocks; ++z)
    {
        const int posZ = z * zdim;
        for (int y = 0; y < yblocks; ++y)
        {
            const int posY = y * ydim;
            for (int x = 0; x < xblocks; ++x)
            {
                const int posX = x * xdim;
                physical_compressed_block pcb = *reinterpret_cast<const physical_compressed_block*>(pS);
                symbolic_compressed_block scb;
                physical_to_symbolic(xdim, ydim, zdim, pcb, &scb);
                decompress_symbolic_block(decode_mode, xdim, ydim, zdim, posX, posY, posZ, &scb, &pb);
                CopyImageBlock(pDst, &pb, decode_mode, imageW, imageH, imageD, xdim, ydim, zdim);
                pS += 16;
            }
        }
    }
}

} // unnamed namespace

//-----------------------------------------------------------------------------
//! @brief Astc Evaluation Codec でフォーマットを変換します。
//!
//! @param[out] pDst 変換後のデータを格納します。
//! @param[in] pSrc 変換前のデータです。
//! @param[in] dstFormatStr 変換後のフォーマット文字列です。
//! @param[in] srcFormatStr 変換前のフォーマット文字列です。
//! @param[in] qualityStr エンコード品質です。
//! @param[in] encodeFlag エンコードフラグです。
//! @param[in] imageW 画像の幅です。
//! @param[in] imageH 画像の高さです。
//! @param[in] imageD 画像の奥行きです。
//!
//! @return 処理成功なら true を返します。
//-----------------------------------------------------------------------------
bool ConvertByAstcEvaluationCodec(
    void* pDst,
    const void* pSrc,
    const std::string& dstFormatStr,
    const std::string& srcFormatStr,
    const std::string& qualityStr,
    const int encodeFlag,
    const int imageW,
    const int imageH,
    const int imageD
)
{
    //-----------------------------------------------------------------------------
    // Astc Evaluation Codec のグローバル変数を初期化します。
    suppress_progress_counter = 1; // 0 なら進捗状況を標準出力に表示します。

    static bool s_IsTableInitialized = false;
    if (!s_IsTableInitialized)
    {
        //cerr << "Initialize astc table" << endl;
        prepare_angular_tables();
        build_quantization_mode_table();
        s_IsTableInitialized = true;
    }

    //-----------------------------------------------------------------------------
    // フォーマットを変換します。
    int xdim;
    int ydim;
    int zdim;
    if (IsAstcFormat(dstFormatStr))
    {
        //-----------------------------------------------------------------------------
        // エンコードします。
        //-----------------------------------------------------------------------------

        //-----------------------------------------------------------------------------
        // 変換オプションを設定します。
        if (!IsPlainRgbaFormat(srcFormatStr) ||
            !GetAstcBlockSize(&xdim, &ydim, &zdim, dstFormatStr))
        {
            return false;
        }
        const bool isNormalMapLa = (encodeFlag & EncodeFlag_NormalMapLa) != 0;

        const astc_decode_mode decodeMode = GetDecodeMode(srcFormatStr, false);
        // ↑現在は変換前のフォーマットが実数型であっても LDR としてエンコードします。
        rgb_force_use_of_hdr = alpha_force_use_of_hdr = (decodeMode == DECODE_HDR) ? 1 : 0;
        const bool forcesHdr = (rgb_force_use_of_hdr != 0);

        const std::string quality = GetAstcQuality(qualityStr);
        error_weighting_params ewp;
        SetErrorWeightingParams(&ewp, xdim, ydim, zdim, quality, forcesHdr, isNormalMapLa);

        const swizzlepattern SwizzleRgba = { 0, 1, 2, 3 };
        const swizzlepattern SwizzleRrrg = { 0, 0, 0, 1 };
        const swizzlepattern swizzleEncode = (!isNormalMapLa) ? SwizzleRgba : SwizzleRrrg;
        const swizzlepattern swizzleDecode = SwizzleRgba;

        //const int threadCount = 1;
        const int threadCount = GetProcessorCount();

        //-----------------------------------------------------------------------------
        // 画像を作成します。
        const int bitness = (IsFloatFormat(srcFormatStr)) ? 16 : 8;
        const int padding = MAX(ewp.mean_stdev_radius, ewp.alpha_radius);
        astc_codec_image* pSrcImage = allocate_image(bitness, imageW, imageH, imageD, padding);
        SetImageData(pSrcImage, pSrc);

        //-----------------------------------------------------------------------------
        // 近傍テクセルの平均と分散を計算します。
        //cout << "Encoding ... " << dstFormatStr << ", " << threadCount << " threads, quailty = " << quality << ", flag = " << encodeFlag << endl;
        if (padding > 0 ||
            ewp.rgb_mean_weight != 0.0f ||
            ewp.rgb_stdev_weight != 0.0f ||
            ewp.alpha_mean_weight != 0.0f ||
            ewp.alpha_stdev_weight != 0.0f)
        {
            compute_averages_and_variances(pSrcImage, ewp.rgb_power, ewp.alpha_power,
                ewp.mean_stdev_radius, ewp.alpha_radius, swizzleEncode);
        }

        //-----------------------------------------------------------------------------
        // エンコードします。
        const int packAndUnpack = 0;
        encode_astc_image(pSrcImage, NULL, xdim, ydim, zdim, &ewp, decodeMode, swizzleEncode, swizzleDecode,
            reinterpret_cast<uint8_t*>(pDst), packAndUnpack, threadCount);
        destroy_image(pSrcImage);

        return true;
    }
    else
    {
        //-----------------------------------------------------------------------------
        // デコードします。
        //-----------------------------------------------------------------------------
        if (IsPlainRgbaFormat(dstFormatStr) &&
            GetAstcBlockSize(&xdim, &ydim, &zdim, srcFormatStr))
        {
            DecodeAstcData(pDst, pSrc, GetDecodeMode(dstFormatStr, true),
                imageW, imageH, imageD, xdim, ydim, zdim);
            return true;
        }
        else
        {
            return false;
        }
    }
}
#endif

