﻿// --------------------------------------------------------------------------------
// <copyright>
// Copyright (C)Nintendo. All rights reserved.
//
// These coded instructions, statements, and computer programs contain proprietary
// information of Nintendo and/or its licensed developers and are protected by
// national and international copyright laws. They may not be disclosed to third
// parties or copied or duplicated in any form, in whole or in part, without the
// prior written consent of Nintendo.
//
// The content herein is highly confidential and should be handled accordingly.
// </copyright>
// --------------------------------------------------------------------------------
using System;
using System.Collections.Generic;
using System.Diagnostics;
using System.Linq;
using System.Windows.Forms;
using System.Reflection;
using System.Runtime.InteropServices;

using System.Collections;
using System.IO;
using Shell32;
using SHDocVw;

namespace NintendoWare.ExtensionManager
{
    //==========================================================================
    /// <summary>
    /// Main Application
    /// </summary>
    //==========================================================================
    public class MainApp : IDisposable
    {
        //----------------------------------------------------------------------
        /// <summary>
        /// Constructor
        /// </summary>
        //----------------------------------------------------------------------
        public MainApp()
        {
            Debug.Assert(g_singleton == null);
            g_singleton = this;
        }

        //----------------------------------------------------------------------
        /// <summary>
        /// Dispose
        /// </summary>
        //----------------------------------------------------------------------
        virtual public void Dispose()
        {
            Debug.Assert(g_singleton == this);
            g_singleton = null;
        }

        #region Singleton
        //----------------------------------------------------------------------
        /// <summary>
        /// Private Singleton
        /// </summary>
        //----------------------------------------------------------------------
        static private MainApp g_singleton = null;

        //----------------------------------------------------------------------
        /// <summary>
        /// Singleton Accessor
        /// </summary>
        //----------------------------------------------------------------------
        static public MainApp Instance
        {
            get { return g_singleton; }
        }
        #endregion

        private bool RemoveUnspecifiedAdditionalExtensionsRegistry { get; set; }
        private List<string> AdditionalExtensions { get; set; }

        #region Initialization Code
        //----------------------------------------------------------------------
        /// <summary>
        /// Initialize App
        /// </summary>
        //----------------------------------------------------------------------
        public bool Init(bool useUnInstall, bool useEasy, bool removeUnspecifiedAdditionalExtensionsRegistry, List<string> additionalExtensions)
        {
            RemoveUnspecifiedAdditionalExtensionsRegistry = removeUnspecifiedAdditionalExtensionsRegistry;
            AdditionalExtensions = additionalExtensions;

            InitDirectory();

            OpenLogFile();

            InitLocalization();

            if (InitConfigManager() == false)
                return false;

            if (InitRegistrationManager() == false)
                return false;

            if (InitShellPluginManager() == false)
                return false;

            // オプション:簡易インストール/アンインストール
            if (useEasy)
            {
                // オプション:アンインストール
                if (useUnInstall)
                {
                    // 簡易アンインストール
                    m_registrationManager.LoadFromEasyUnInstall();
                }
                else
                {
                    // 簡易インストール
                    m_registrationManager.LoadFromEasyInstall();
                }
            }
            else
            {
                // オプション:GUIセットアップ表示
                // Load registration
                m_registrationManager.LoadFromRegistry();
            }

            return true;
        }

        //----------------------------------------------------------------------
        /// <summary>
        /// Uninitialize App
        /// </summary>
        //----------------------------------------------------------------------
        public void Deinit()
        {
            DeinitShellPluginManager();
            DeinitRegistrationManager();
            DeinitConfigManager();

            CloseLogFile();
        }

        //----------------------------------------------------------------------
        /// <summary>
        /// Initialize for GUI App
        /// </summary>
        ///
        /// <returns>True on success</returns>
        //----------------------------------------------------------------------
        public bool InitGUIApp()
        {
            if (InitUIManager() == false)
                return false;

            return true;
        }

        //----------------------------------------------------------------------
        /// <summary>
        /// Deinitialize for GUI App
        /// </summary>
        ///
        /// <returns>True on success</returns>
        //----------------------------------------------------------------------
        public void DeinitGUIApp()
        {
            DeinitUIManager();
        }

        //----------------------------------------------------------------------
        /// <summary>
        /// initialize localization
        /// </summary>
        //----------------------------------------------------------------------
        private void InitLocalization()
        {
            System.Globalization.CultureInfo uiCultureInfo = null;

            try
            {
                // ＵＩ言語初期化
                if (System.IO.File.Exists(m_executableDirectory + "\\en\\ExtensionManager.resources.dll"))
                {
                    uiCultureInfo = new System.Globalization.CultureInfo("en", false);
                }

            }

            catch (System.Exception e)
            {
                // To shut up compiler
                if (e.Data != null)
                {
                }

                uiCultureInfo = null;
            }

            if (uiCultureInfo != null)
            {
                System.Threading.Thread.CurrentThread.CurrentUICulture = uiCultureInfo;
            }
        }
        #endregion

        #region Directory
        //----------------------------------------------------------------------
        /// <summary>
        /// Initialize directory settings. Override this to customize your directory settings
        /// </summary>
        //----------------------------------------------------------------------
        private bool InitDirectory()
        {
            // Assuming installation will place app on [productDir] directory
            // Modify current directory
            System.String curDir = System.IO.Directory.GetCurrentDirectory();

            Assembly mainAssembly = Assembly.GetEntryAssembly();
            m_executableDirectory = System.IO.Path.GetDirectoryName(mainAssembly.Location);
            m_executableDirectory = m_executableDirectory.Replace("/", "\\"); // Fix slash

            m_workingDirectory = System.IO.Directory.GetCurrentDirectory();
            m_workingDirectory = m_workingDirectory.Replace("/", "\\"); // Fix slash

            m_configDirectory = m_executableDirectory;

            m_appdataDirectory = System.IO.Path.Combine(Environment.GetFolderPath(Environment.SpecialFolder.LocalApplicationData), "Nintendo", "ThumbnailExplorerExtensions");
            if(! System.IO.Directory.Exists(m_appdataDirectory))
            {
                System.IO.Directory.CreateDirectory(m_appdataDirectory);
            }

            return true;
        }

        //----------------------------------------------------------------------
        /// <summary>
        /// Application name
        /// </summary>
        //----------------------------------------------------------------------
        private System.String m_appName;

        //----------------------------------------------------------------------
        /// <summary>
        /// Accessor for application name
        /// </summary>
        //----------------------------------------------------------------------
        public System.String ApplicationName
        {
            set { m_appName = value; }
            get { return m_appName; }
        }

        //----------------------------------------------------------------------
        /// <summary>
        /// Accessor for executable directory
        /// </summary>
        //----------------------------------------------------------------------
        public System.String ExecutableDirectory
        {
            set { m_executableDirectory = value; }
            get { return m_executableDirectory; }
        }

        private System.String m_executableDirectory = "";

        //----------------------------------------------------------------------
        /// <summary>
        /// Accessor for working directory
        /// </summary>
        //----------------------------------------------------------------------
        public System.String WorkingDirectory
        {
            set { m_workingDirectory = value; }
            get { return m_workingDirectory; }
        }

        private System.String m_workingDirectory = "";

        //----------------------------------------------------------------------
        /// <summary>
        /// Accessor for Config directory
        /// </summary>
        //----------------------------------------------------------------------
        public System.String ConfigDirectory
        {
            set { m_configDirectory = value; }
            get { return m_configDirectory; }
        }

        private System.String m_configDirectory = "";

        //----------------------------------------------------------------------
        /// <summary>
        /// Accessor for AppData directory
        /// </summary>
        //----------------------------------------------------------------------
        public System.String AppDataDirectory
        {
            set { m_appdataDirectory = value; }
            get { return m_appdataDirectory; }
        }

        private System.String m_appdataDirectory = "";

        #endregion

        #region SubManagers
        //----------------------------------------------------------------------
        /// <summary>
        /// SubManagers
        /// </summary>
        //----------------------------------------------------------------------
        private List<SubManager> m_subManagers = new List<SubManager>();

        //----------------------------------------------------------------------
        /// <summary>
        /// Accessors to SubManagers
        /// </summary>
        //----------------------------------------------------------------------
        public List<SubManager> SubManagers
        {
            get { return m_subManagers; }
        }
        #endregion

        #region ConfigManager
        private NintendoWare.ExtensionManager.ConfigManager m_configManager;

        //----------------------------------------------------------------------
        /// <summary>
        /// ConfigManager Accessor
        /// </summary>
        //----------------------------------------------------------------------
        public NintendoWare.ExtensionManager.ConfigManager ConfigManager
        {
            get { return m_configManager; }
        }

        //----------------------------------------------------------------------
        /// <summary>
        /// Init ConfigManager
        /// </summary>
        ///
        /// <returns>True on success</returns>
        //----------------------------------------------------------------------
        private bool InitConfigManager()
        {
            m_configManager = new ConfigManager();
            if (m_configManager == null)
                return false;

            if (m_configManager.Init() == false)
            {
                m_configManager.Dispose();
                m_configManager = null;
                return false;
            }

            m_subManagers.Add(m_configManager);

            m_configManager.LoadConfig(ConfigDirectory);

            return true;
        }

        //----------------------------------------------------------------------
        /// <summary>
        /// Deinit ConfigManager
        /// </summary>
        //----------------------------------------------------------------------
        private void DeinitConfigManager()
        {
            if (m_configManager != null)
            {
                m_subManagers.Remove(m_configManager);
                m_configManager.Dispose();
                m_configManager = null;
            }
        }
        #endregion

        #region UIManager
        private NintendoWare.ExtensionManager.UIManager m_UIManager;

        //----------------------------------------------------------------------
        /// <summary>
        /// UIManager Accessor
        /// </summary>
        //----------------------------------------------------------------------
        public NintendoWare.ExtensionManager.UIManager UIManager
        {
            get { return m_UIManager; }
        }

        //----------------------------------------------------------------------
        /// <summary>
        /// Init UIManager
        /// </summary>
        ///
        /// <returns>True on success</returns>
        //----------------------------------------------------------------------
        private bool InitUIManager()
        {
            m_UIManager = new UIManager();
            if (m_UIManager == null)
                return false;

            if (m_UIManager.Init() == false)
            {
                m_UIManager.Dispose();
                m_UIManager = null;
                return false;
            }

            m_subManagers.Add(m_UIManager);

            // Create Main
            m_UIManager.CreateMainForm();

            return true;
        }

        //----------------------------------------------------------------------
        /// <summary>
        /// Deinit UIManager
        /// </summary>
        //----------------------------------------------------------------------
        private void DeinitUIManager()
        {
            if (m_UIManager != null)
            {
                m_subManagers.Remove(m_UIManager);
                m_UIManager.Dispose();
                m_UIManager = null;
            }
        }
        #endregion

        #region RegistrationManager
        private NintendoWare.ExtensionManager.RegistrationManager m_registrationManager;

        //----------------------------------------------------------------------
        /// <summary>
        /// RegistrationManager Accessor
        /// </summary>
        //----------------------------------------------------------------------
        public NintendoWare.ExtensionManager.RegistrationManager RegistrationManager
        {
            get { return m_registrationManager; }
        }

        //----------------------------------------------------------------------
        /// <summary>
        /// Init RegistrationManager
        /// </summary>
        ///
        /// <returns>True on success</returns>
        //----------------------------------------------------------------------
        private bool InitRegistrationManager()
        {
            m_registrationManager = new RegistrationManager(
                m_bUseLog,
                m_appdataDirectory,
                AdditionalExtensions,
                RemoveUnspecifiedAdditionalExtensionsRegistry);

            if (m_registrationManager == null)
                return false;

            if (m_registrationManager.Init() == false)
            {
                m_registrationManager.Dispose();
                m_registrationManager = null;
                return false;
            }

            m_subManagers.Add(m_registrationManager);

            return true;
        }

        //----------------------------------------------------------------------
        /// <summary>
        /// Deinit RegistrationManager
        /// </summary>
        //----------------------------------------------------------------------
        private void DeinitRegistrationManager()
        {
            if (m_registrationManager != null)
            {
                m_subManagers.Remove(m_registrationManager);
                m_registrationManager.Dispose();
                m_registrationManager = null;
            }
        }

        //----------------------------------------------------------------------
        /// <summary>
        /// Setup　Registraition
        /// </summary>
        //----------------------------------------------------------------------
        public ReturnCode SetupRegistraition(bool useEasy, bool isSilent, SetupType setupType, IWin32Window dialogOwner = null)
        {
            MainApp.Instance.WriteToLog(String.Format("SetupRegistraition -- Started"));

            if (m_registrationManager == null)
                return ReturnCode.Error;

            ReturnCode rtCode = ReturnCode.Success;
            DateTime t2;
            DateTime t1;

            // Window点滅
            FLASHWINFO fInfo = new FLASHWINFO();
            fInfo.cbSize = Convert.ToUInt32(Marshal.SizeOf(fInfo));
            fInfo.dwFlags = FLASHW_ALL;
            fInfo.uCount = 5; // 点滅回数
            fInfo.dwTimeout = 500;

            //if (m_registrationManager.IsSavingRequired())
            {
                MainApp.Instance.WriteToLog(String.Format("TurnOffServicesIfReset -- Started"));

                if (!isSilent && m_registrationManager.ShouldWarnAboutResetting())
                {
                    if (MessageBox.Show(Properties.Resources.TEXT_RESET_WARNING,
                                        "Warning", MessageBoxButtons.YesNo) != DialogResult.Yes)
                    {
                        return ReturnCode.Cancel;
                    }

                    m_registrationManager.TurnOffServicesIfReset();
                }

                t1 = System.DateTime.Now;
                MainApp.Instance.WriteToLog(String.Format("lockApp -- Started {0}", t1.TimeOfDay));

                while (true)
                {
                    var lockApps = m_registrationManager.LockApps;

                    bool ExceptExplorer = false;
                    bool Explorer = false;
                    foreach (var lockApp in lockApps)
                    {
                        t2 = System.DateTime.Now;
                        MainApp.Instance.WriteToLog(String.Format("lockApp -- Path  {0} ( {1} )", lockApp.LockingPath, t2.TimeOfDay));

                        // エクスプローラー関係以外のexeがDLLを握っているか確認
                        if ((lockApp.LockingPath.EndsWith("explorer.exe") == true) ||
                                (lockApp.LockingPath.EndsWith("prevhost.exe") == true))
                        {
                            Explorer = true;
                            continue;
                        }

                        ExceptExplorer = true;
                        break;
                    }

                    if (isSilent)
                    {
                        if (Explorer || ExceptExplorer)
                        {
                            MainApp.Instance.WriteToLog("Installation canceled silently");

                            // サイレンとモード時に dll が握られていたらキャンセル扱いにする
                            return ReturnCode.Cancel;
                        }
                    }

                    // エクスプローラー関係のexeのみがDLLを握っている場合
                    if (ExceptExplorer == false)
                    {
                        break;
                    }

                    t2 = System.DateTime.Now;
                    MainApp.Instance.WriteToLog(String.Format("lockApp -- Path  create LockAppsDialog {0} ", t2.TimeOfDay));

                    // エクスプローラー関係以外のexeがDLLを握っているときは、ダイアログを表示
                    using (var dialog = new LockAppsDialog(lockApps))
                    {
                        if (useEasy)
                        {
                            // 一番上に表示
                            dialog.TopMost = true;
                            // 点滅
                            fInfo.hwnd = dialog.Handle;
                            FlashWindowEx(ref fInfo);
                        }

                        dialog.ShowDialog(dialogOwner);

                        if (dialog.DialogResult == DialogResult.Cancel)
                        {
                            return ReturnCode.Cancel;
                        }
                    }
                }
                t2 = System.DateTime.Now;
                MainApp.Instance.WriteToLog(String.Format("lockApp -- Ended   {0} ( {1} Sec )", t2.TimeOfDay, (t2 - t1).TotalSeconds));

                // サイレントモード時はエクスプローラの再起動を行わない
                if (isSilent)
                {
                    // レジストリの更新
                    return m_registrationManager.Apply(setupType == SetupType.Uninstall) ? ReturnCode.Success : ReturnCode.Error;
                }

                // エクスプローラをシャットダウンしてよいか確認する
                TerminateExplorerDialog terminateExplorerDialog = new TerminateExplorerDialog();
                terminateExplorerDialog.SetGUILabel();
                terminateExplorerDialog.ShowDialog(dialogOwner);

                if (terminateExplorerDialog.DialogResult != DialogResult.OK)
                {
                    return ReturnCode.Cancel;
                }

                ReturnCode registerResult = ReturnCode.Success;

                Action registerMethod = () =>
                {
                    SHDocVw.ShellWindows windows = new SHDocVw.ShellWindows();
                    ArrayList openFolderPaths = new ArrayList();
                    foreach (IWebBrowser2 win in windows)
                    {
                        object sb;
                        IServiceProvider sp = win as IServiceProvider;
                        sp.QueryService(SID_STopLevelBrowser, typeof(IShellBrowser).GUID, out sb);
                        IShellBrowser shellBrowser = (IShellBrowser)sb;
                        object sv;
                        shellBrowser.QueryActiveShellView(out sv);

                        //Console.WriteLine(win.LocationURL + " " + win.LocationName);

                        // 特殊フォルダのclsidを取得
                        IFolderView fv = sv as IFolderView;
                        string specialFolder = "";
                        if (fv != null)
                        {
                            // only folder implementation support this
                            object pf;
                            fv.GetFolder(typeof(IPersistFolder2).GUID, out pf);
                            IPersistFolder2 persistFolder = (IPersistFolder2)pf;

                            // get folder class
                            Guid clsid;
                            persistFolder.GetClassID(out clsid);
                            //Console.WriteLine(" clsid:" + clsid);
                            specialFolder = clsid.ToString();

                        }

                        //エクスプローラのみ(IEを除外)
                        if (Path.GetFileName(win.FullName).ToUpper() == "EXPLORER.EXE")
                        {
                            // 通常フォルダ
                            if (String.IsNullOrEmpty(win.LocationURL) == false)
                            {
                                //リストに追加
                                string path = win.LocationURL.Replace("file:///", "");

                                openFolderPaths.Add(path);
                                //MessageBox.Show(path + " : " + win.LocationName);
                            } // 特殊フォルダ
                            else if (String.IsNullOrEmpty(specialFolder) == false)
                            {

                                openFolderPaths.Add("shell:::{" + specialFolder + "}");
                                //MessageBox.Show(path + " : " + win.LocationName);
                            } // end else if
                        }

                    }

                    // レジストリの更新
                    if (!m_registrationManager.Apply(setupType == SetupType.Uninstall))
                    {
                        // 簡易インストールの時は表示しない
                        if (!useEasy)
                        {
                            FailedDialog dlg = new FailedDialog();
                            dlg.ShowDialog(dialogOwner);
                        }
                        registerResult = ReturnCode.Error;
                        return;
                    }

                    System.IntPtr handle;
                    string key = Guid.NewGuid().ToString();

                    t1 = System.DateTime.Now;
                    MainApp.Instance.WriteToLog(String.Format("RestartManager Session -- Started {0}", t1.TimeOfDay));

                    RmStartSession(out handle, 0, key);

                    t2 = System.DateTime.Now;
                    MainApp.Instance.WriteToLog(String.Format("RestartManager Session -- Ended   {0} ( {1} Sec )", t2.TimeOfDay, (t2 - t1).TotalSeconds));

                    t1 = System.DateTime.Now;
                    MainApp.Instance.WriteToLog(String.Format("RestartManager Explorer Shutdown -- Started {0}", t1.TimeOfDay));
                    // エクスプローラーを停止する
                    RM_UNIQUE_PROCESS[] processes = GetProcesses("explorer");
                    RmRegisterResources(handle,
                                        0, null,
                                        (uint)processes.Length, processes,
                                        0, null);

                    // レジストリの設定にもよるが、30 秒以内にシャットダウンできないと強制終了される。
                    // 強制終了されるとコピー中のファイルが壊れる場合がある。
                    // Process.kill を使うとコピーの問題に加えてアイコンの位置が変わる場合がある。
                    int errorCode = RmShutdown(handle, RM_SHUTDOWN_TYPE.RmForceShutdown, null);

                    MainApp.Instance.WriteToLog(String.Format("RmShutdown return value: {0}", errorCode));

                    t2 = System.DateTime.Now;
                    MainApp.Instance.WriteToLog(String.Format("RestartManager Explorer Shutdown -- Ended   {0} ( {1} sec )", t2.TimeOfDay, (t2 - t1).TotalSeconds));

                    // エクスプローラーを起動させる
                    t1 = System.DateTime.Now;
                    MainApp.Instance.WriteToLog(String.Format("RestartManager Explorer Start -- Started {0}", t1.TimeOfDay));

                    RmRestart(handle, 0, null);

                    t2 = System.DateTime.Now;
                    MainApp.Instance.WriteToLog(String.Format("RestartManager Explorer Start -- Ended   {0} ( {1} Sec )", t2.TimeOfDay, (t2 - t1).TotalSeconds));

                    t1 = System.DateTime.Now;
                    MainApp.Instance.WriteToLog(String.Format("RestartManager Session end -- Started {0}", t1.TimeOfDay));

                    RmEndSession(handle);

                    t2 = System.DateTime.Now;
                    MainApp.Instance.WriteToLog(String.Format("RestartManager Session end -- Ended   {0} ( {1} Sec )", t2.TimeOfDay, (t2 - t1).TotalSeconds));

                    // Vista Windows7のとき
                    if (Environment.OSVersion.Version.Major == 6 && Environment.OSVersion.Version.Minor <= 1 && openFolderPaths.Count >= 1)
                    {
                        t1 = System.DateTime.Now;
                        MainApp.Instance.WriteToLog(String.Format("Open folders started -- Started {0}", t1.TimeOfDay));

                        // 開いていたフォルダを再度開く
                        foreach (string path in openFolderPaths)
                        {
                            // スペース置換
                            string sppath = path.Replace("%20", " ");
                            try
                            {
                                // file:// で始まる場合があるので、Path.IsPathRooted や Directory.Exists は呼ばない。

                                System.Diagnostics.Process.Start(sppath);
                            }
                            catch (Exception e)
                            {
                                MainApp.Instance.WriteToLog(String.Format("Failed to start process : {0}", sppath));
                                MainApp.Instance.WriteToLog(e.ToString());
                            }
                        }

                        t2 = System.DateTime.Now;
                        MainApp.Instance.WriteToLog(String.Format("Open folders end -- Ended   {0} ( {1} Sec )", t2.TimeOfDay, (t2 - t1).TotalSeconds));
                    }
                };

                // プログレスバーを表示して再起動処理を実行
                ProgressDialog progressDialog = new ProgressDialog(Properties.Resources.TEXT_EXPLORER_RESTARTING, "Thumbnail Explorer Extensions", registerMethod, 30.0f);
                progressDialog.ShowAndRun(dialogOwner);

                if (registerResult != ReturnCode.Success)
                {
                    return registerResult;
                }

                // 簡易インストールの時は表示しない
                if (!useEasy || isSilent)
                {
                    SuccessDialog dlg = new SuccessDialog();
                    dlg.ShowDialog(dialogOwner);
                }
                rtCode = ReturnCode.Success;
            }

            return rtCode;
        }
        #endregion

        #region ShellPluginManager
        private NintendoWare.ExtensionManager.ShellPluginManager m_shellPluginManager;

        //----------------------------------------------------------------------
        /// <summary>
        /// ShellPluginManager Accessor
        /// </summary>
        //----------------------------------------------------------------------
        public NintendoWare.ExtensionManager.ShellPluginManager ShellPluginManager
        {
            get { return m_shellPluginManager; }
        }

        //----------------------------------------------------------------------
        /// <summary>
        /// Init ShellPluginManager
        /// </summary>
        ///
        /// <returns>True on success</returns>
        //----------------------------------------------------------------------
        private bool InitShellPluginManager()
        {
            String rootDir = System.IO.Path.GetFullPath(ExecutableDirectory + "\\..");
            m_shellPluginManager = new ShellPluginManager(rootDir);
            if (m_shellPluginManager == null)
                return false;

            if (m_shellPluginManager.Init() == false)
            {
                m_shellPluginManager.Dispose();
                m_shellPluginManager = null;
                return false;
            }

            m_subManagers.Add(m_shellPluginManager);

            return true;
        }

        //----------------------------------------------------------------------
        /// <summary>
        /// Deinit ShellPluginManager
        /// </summary>
        //----------------------------------------------------------------------
        private void DeinitShellPluginManager()
        {
            if (m_shellPluginManager != null)
            {
                m_subManagers.Remove(m_shellPluginManager);
                m_shellPluginManager.Dispose();
                m_shellPluginManager = null;
            }
        }
        #endregion

        #region Event Handling
        //----------------------------------------------------------------------
        /// <summary>
        /// Called on Exiting App
        /// </summary>
        ///
        /// <returns>True on success</returns>
        //----------------------------------------------------------------------
        public void OnExitingApp()
        {
            if (ConfigManager != null)
            {
                ConfigManager.SaveConfig(ConfigDirectory);
            }
        }
        #endregion

        #region Log
        private bool m_bUseLog = true;
        private System.IO.StreamWriter m_logWriter = null;

        //----------------------------------------------------------------------
        /// <summary>
        /// Open the log file
        /// </summary>
        //----------------------------------------------------------------------
        private void OpenLogFile()
        {
            // ログ出力の度にファイルの保存を行いたいので、
            // WriteToLogでファイルの開閉を行うように変更。
            // 本関数内ではファイルを開かない

        }

        //----------------------------------------------------------------------
        /// <summary>
        /// Close the log file
        /// </summary>
        //----------------------------------------------------------------------
        private void CloseLogFile()
        {
            // ログ出力の度にファイルの保存を行いたいので、
            // WriteToLogでファイルの開閉を行うように変更。
            // 本関数内ではファイルを閉じない

        }

        //----------------------------------------------------------------------
        /// <summary>
        /// Write to log ( with return )
        /// </summary>
        //----------------------------------------------------------------------
        public void WriteToLog(String strLine)
        {
            try
            {
                // ログ出力の度にファイルを保存。
                if (m_bUseLog)
                {
                    m_logWriter = new System.IO.StreamWriter(m_appdataDirectory + "\\log_" + DateTime.Today.ToString("yyyyMMdd") + ".txt", true);
                    if (m_logWriter != null)
                    {
                        m_logWriter.WriteLine(strLine);

                        m_logWriter.Flush();
                        m_logWriter.Close();
                    }
                }
            }
            catch { }

        }
        #endregion

        #region RestartManager
        [StructLayout(LayoutKind.Sequential)]
        private struct RM_UNIQUE_PROCESS
        {
            public int dwProcessId;
            public System.Runtime.InteropServices.ComTypes.FILETIME ProcessStartTime;
        }

        [Flags]
        private enum RM_SHUTDOWN_TYPE : uint
        {
            RmForceShutdown = 0x1,
            RmShutdownOnlyRegistered = 0x10
        }

        delegate void RM_WRITE_STATUS_CALLBACK(UInt32 nPercentComplete);

        [DllImport("rstrtmgr.dll", CharSet = CharSet.Auto)]
        static extern int RmStartSession(out IntPtr pSessionHandle, int dwSessionFlags, string strSessionKey);
        [DllImport("rstrtmgr.dll")]
        static extern int RmEndSession(IntPtr pSessionHandle);
        [DllImport("rstrtmgr.dll", CharSet = CharSet.Auto)]
        static extern int RmRegisterResources(IntPtr pSessionHandle, UInt32 nFiles, string[] rgsFilenames, UInt32 nApplications, RM_UNIQUE_PROCESS[] rgApplications, UInt32 nServices, string[] rgsServiceNames);
        [DllImport("rstrtmgr.dll")]
        static extern int RmShutdown(IntPtr pSessionHandle, RM_SHUTDOWN_TYPE lActionFlags, RM_WRITE_STATUS_CALLBACK fnStatus);
        [DllImport("rstrtmgr.dll")]
        static extern int RmRestart(IntPtr pSessionHandle, int dwRestartFlags, RM_WRITE_STATUS_CALLBACK fnStatus);
        [DllImport("kernel32.dll")]
        static extern bool GetProcessTimes(IntPtr hProcess, out System.Runtime.InteropServices.ComTypes.FILETIME lpCreationTime,
                                            out System.Runtime.InteropServices.ComTypes.FILETIME lpExitTime,
                                            out System.Runtime.InteropServices.ComTypes.FILETIME lpKernelTime,
                                            out System.Runtime.InteropServices.ComTypes.FILETIME lpUserTime);

        static RM_UNIQUE_PROCESS[] GetProcesses(string name)
        {
            List<RM_UNIQUE_PROCESS> lst = new List<RM_UNIQUE_PROCESS>();
            foreach (Process p in Process.GetProcessesByName(name))
            {
                RM_UNIQUE_PROCESS rp = new RM_UNIQUE_PROCESS();
                rp.dwProcessId = p.Id;
                System.Runtime.InteropServices.ComTypes.FILETIME creationTime, exitTime, kernelTime, userTime;
                GetProcessTimes(p.Handle, out creationTime, out exitTime, out kernelTime, out userTime);
                rp.ProcessStartTime = creationTime;
                lst.Add(rp);
            }
            return lst.ToArray();
        }

        #endregion

        #region Get Foloder Path
            internal static readonly Guid SID_STopLevelBrowser = new Guid(0x4C96BE40, 0x915C, 0x11CF, 0x99, 0xD3, 0x00, 0xAA, 0x00, 0x4A, 0xE8, 0x37);
            [InterfaceType(ComInterfaceType.InterfaceIsIUnknown), Guid("6D5140C1-7436-11CE-8034-00AA006009FA")]
            internal interface IServiceProvider
            {
                void QueryService([MarshalAs(UnmanagedType.LPStruct)] Guid guidService, [MarshalAs(UnmanagedType.LPStruct)] Guid riid, [MarshalAs(UnmanagedType.IUnknown)] out object ppvObject);
            }

            [InterfaceType(ComInterfaceType.InterfaceIsIUnknown), Guid("1AC3D9F0-175C-11d1-95BE-00609797EA4F")]
            internal interface IPersistFolder2
            {
                void GetClassID(out Guid pClassID);
                void Initialize(IntPtr pidl);
                void GetCurFolder(out IntPtr pidl);
            }

            [InterfaceType(ComInterfaceType.InterfaceIsIUnknown), Guid("000214E2-0000-0000-C000-000000000046")]
            internal interface IShellBrowser
            {
                void _VtblGap0_12(); // skip 12 members
                void QueryActiveShellView([MarshalAs(UnmanagedType.IUnknown)] out object ppshv);
                // the rest is not defined
            }

            [InterfaceType(ComInterfaceType.InterfaceIsIUnknown), Guid("cde725b0-ccc9-4519-917e-325d72fab4ce")]
            internal interface IFolderView
            {
                void _VtblGap0_2(); // skip 2 members
                void GetFolder([MarshalAs(UnmanagedType.LPStruct)] Guid riid, [MarshalAs(UnmanagedType.IUnknown)] out object ppv);
                // the rest is not defined
            }
        #endregion

        #region FlashWindow
            public enum SetupType : int
            {
                Install = 0,
                Uninstall = 1,
                GUI = 2
            };
        #endregion

        #region FlashWindow
            [DllImport("user32.dll")]
            static extern Int32 FlashWindowEx(ref FLASHWINFO pwfi);

            [StructLayout(LayoutKind.Sequential)]
            public struct FLASHWINFO
            {
                public UInt32 cbSize;    // FLASHWINFO構造体のサイズ
                public IntPtr hwnd;      // 点滅対象のウィンドウ・ハンドル
                public UInt32 dwFlags;   // 以下の「FLASHW_XXX」のいずれか
                public UInt32 uCount;    // 点滅する回数
                public UInt32 dwTimeout; // 点滅する間隔（ミリ秒単位）
            }

            public const UInt32 FLASHW_STOP = 0;        // 点滅を止める
            public const UInt32 FLASHW_CAPTION = 1;     // タイトルバーを点滅させる
            public const UInt32 FLASHW_TRAY = 2;        // タスクバー・ボタンを点滅させる
            public const UInt32 FLASHW_ALL = 3;         // タスクバー・ボタンとタイトルバーを点滅させる
            public const UInt32 FLASHW_TIMER = 4;       // FLASHW_STOPが指定されるまでずっと点滅させる
            public const UInt32 FLASHW_TIMERNOFG = 12;  // ウィンドウが最前面に来るまでずっと点滅させる
        #endregion

    }
}


