﻿// --------------------------------------------------------------------------------
// <copyright>
// Copyright (C)Nintendo. All rights reserved.
//
// These coded instructions, statements, and computer programs contain proprietary
// information of Nintendo and/or its licensed developers and are protected by
// national and international copyright laws. They may not be disclosed to third
// parties or copied or duplicated in any form, in whole or in part, without the
// prior written consent of Nintendo.
//
// The content herein is highly confidential and should be handled accordingly.
// </copyright>
// --------------------------------------------------------------------------------
using System;
using System.Collections.Generic;
using System.Linq;
using System.Text;
using System.Windows.Forms;
using Microsoft.Win32;

namespace nw.g3d.bifedit
{
    /// <summary>
    /// Utility class
    /// </summary>
    public sealed class Utility
    {
        /// <summary>
        /// Find XML element's child by name
        /// </summary>
        ///
        /// <param name="doc">Parent document</param>
        /// <param name="name">Name to search</param>
        ///
        /// <returns>First child node with matching name. null if not found</returns>
        static public System.Xml.XmlElement XMLFindChildElement(System.Xml.XmlDocument doc,
                                                                String name)
        {
            if (doc == null)
                return null;

            foreach (System.Xml.XmlElement child in doc.ChildNodes.OfType<System.Xml.XmlElement>())
            {
                if (child == null)
                    continue;

                if (child.Name == name)
                    return child;
            } // End for

            return null;
        } // End of XMLFindChildElement

        /// <summary>
        /// Find XML element's child by name
        /// </summary>
        ///
        /// <param name="parent">Parent node</param>
        /// <param name="name">Name to search</param>
        ///
        /// <returns>First child node with matching name. null if not found</returns>
        static public System.Xml.XmlElement XMLFindChildElement(System.Xml.XmlElement parent,
                                                                String name)
        {
            if (parent == null)
                return null;

            foreach (System.Xml.XmlElement child in parent.ChildNodes.OfType<System.Xml.XmlElement>())
            {
                if (child == null)
                    continue;

                if (child.Name == name)
                    return child;
            } // End for

            return null;
        } // End of XMLFindChildElement

        /// <summary>
        /// Convert string to integer
        /// </summary>
        ///
        /// <param name="valStr">String containing value</param>
        /// <param name="defaultVal">Default value in case string is in invalid format</param>
        ///
        /// <returns>Integer value</returns>
        static public int ConvertStringToInt(String valStr, int defaultVal)
        {
            int val = defaultVal;

            try
            {
                System.Globalization.NumberStyles numStyle = System.Globalization.NumberStyles.Any;
                if (valStr.StartsWith("0x"))
                {
                    numStyle = System.Globalization.NumberStyles.HexNumber;
                    valStr = valStr.Remove(0, 2);
                    val = Int32.Parse(valStr, numStyle);
                    return val;
                } // End if

                double doubleVal = Double.Parse(valStr, numStyle);
                val = (int)doubleVal;
            }

            catch (System.Exception e)
            {
                if (e.Message == null)
                    return val;

                return val;
            }

            return val;
        } // End of ConvertStringToInt

        /// <summary>
        /// Convert string to float
        /// </summary>
        ///
        /// <param name="valStr">String containing value</param>
        /// <param name="defaultVal">Default value in case string is in invalid format</param>
        ///
        /// <returns>Float value</returns>
        static public float ConvertStringToFloat(String valStr, float defaultVal)
        {
            float val = defaultVal;

            try
            {
                System.Globalization.NumberStyles numStyle = System.Globalization.NumberStyles.Any;
                if (valStr.StartsWith("0x"))
                {
                    numStyle = System.Globalization.NumberStyles.HexNumber;
                    valStr = valStr.Remove(0, 2);
                    val = Int32.Parse(valStr, numStyle);
                    return val;
                } // End if

                double doubleVal = Double.Parse(valStr, numStyle);
                val = (float)doubleVal;
            }

            catch (System.Exception e)
            {
                if (e.Message == null)
                    return val;

                return val;
            }

            return val;
        } // End of ConvertStringToFloat

        /// <summary>
        /// Convert string to boolean
        /// </summary>
        ///
        /// <param name="valStr">String containing value</param>
        /// <param name="defaultVal">Default value in case string is in invalid format</param>
        ///
        /// <returns>Boolean value</returns>
        static public bool ConvertStringToBool(String valStr, bool defaultVal)
        {
            bool val = defaultVal;

            if (String.Equals(valStr, "True", StringComparison.OrdinalIgnoreCase))
                val = true;
            else if (String.Equals(valStr, "False", StringComparison.OrdinalIgnoreCase))
                val = false;

            return val;
        } // End of ConvertStringToBool

        /// <summary>
        /// Convert Xml to string
        /// </summary>
        ///
        /// <param name="doc">XML document</param>
        ///
        /// <returns>Converted string</returns>
        static public String ConvertXMLDocumentToString(System.Xml.XmlDocument doc)
        {
            System.IO.StringWriter stringWriter = new System.IO.StringWriter();

            System.Xml.XmlWriterSettings writerSettings = new System.Xml.XmlWriterSettings();
            writerSettings.Indent = true;
            writerSettings.IndentChars = "";
            writerSettings.NewLineOnAttributes = true;

            System.Xml.XmlWriter xmlTextWriter = System.Xml.XmlWriter.Create(stringWriter, writerSettings);
            doc.WriteTo(xmlTextWriter);
            xmlTextWriter.Flush();
            return stringWriter.GetStringBuilder().ToString();
        } // End of ConvertXMLDocumentToString
    } // End of Utility
} // End of nw.g3d.bifedit




