﻿// --------------------------------------------------------------------------------
// <copyright>
// Copyright (C)Nintendo. All rights reserved.
//
// These coded instructions, statements, and computer programs contain proprietary
// information of Nintendo and/or its licensed developers and are protected by
// national and international copyright laws. They may not be disclosed to third
// parties or copied or duplicated in any form, in whole or in part, without the
// prior written consent of Nintendo.
//
// The content herein is highly confidential and should be handled accordingly.
// </copyright>
// --------------------------------------------------------------------------------
using System;
using System.Collections.Generic;
using System.ComponentModel;
using System.Data;
using System.Drawing;
using System.Linq;
using System.ServiceModel;
using System.Text;
using System.Windows.Forms;
using System.Reflection;
using nw.g3d.bifedit.IPC;
using System.Threading;
using System.Diagnostics;

namespace nw.g3d.bifedit
{
    /// <summary>
    /// Main form
    /// </summary>
    public partial class MainForm : Form
    {
        /// <summary>
        /// Constructor
        /// </summary>
        public MainForm(bool show)
        {
            InitializeComponent();
            m_editorPathBox.TextChanged += (ss, ee) => { m_editorCmdLineBox.ReadOnly = m_editorPathBox.TextLength == 0; };
            m_difftoolPathBox.TextChanged += (ss, ee) => { m_difftoolCmdLineBox.ReadOnly = m_difftoolPathBox.TextLength == 0; };

            UpdateUI();

            if (show)
            {
                this.WindowState = FormWindowState.Normal;
            }
            else
            {
                this.WindowState = FormWindowState.Minimized;
                this.ShowInTaskbar = false;
            }
        }

        #region UI
        /// <summary>
        /// Update UI controls
        /// </summary>
        void UpdateUI()
        {
            PathManager pathManager = PathManager.Instance;

            if (m_editorChoiceBox == null)
                return;

            m_editorChoiceBox.Items.Clear();

            // Index zero is reserved for none
            m_editorChoiceBox.Items.Add("None");

            // Add app list
            foreach (AppInfo appInfo in pathManager.ApplicationList)
            {
                m_editorChoiceBox.Items.Add(appInfo.Name);
            }

            // Add custom
            m_editorChoiceBox.Items.Add("Custom");

            m_editorPathBrowse.Enabled = false;
            m_editorCmdLineBox.ReadOnly = true;

            // Set default
            int selectedIndex = 0;
            if ( (pathManager.EditorApplication==null) || (pathManager.EditorApplication.Length<=0) )
            {
                m_editorChoiceBox.SelectedIndex = 0; // None
            }
            else if ( pathManager.SelectedAppIndex>=0 )
            {
                // Find selected application
                selectedIndex = pathManager.SelectedAppIndex + 1; // Zero is reserved, so increment one
            }
            else if ( (pathManager.Config.CustomAppPath!=null) && (pathManager.Config.CustomAppPath.Length>0) )
            {
                m_editorPathBrowse.Enabled = true;
                m_editorPathBox.Text = pathManager.Config.CustomAppPath;
                m_editorCmdLineBox.Text = pathManager.Config.CustomAppCommandArguments;

                selectedIndex = pathManager.Config.ApplicationList.Count + 1;
            }

            m_editorChoiceBox.SelectedIndex = selectedIndex;

            m_difftoolPathBox.Text = pathManager.Config.DiffToolFilename;
            m_difftoolCmdLineBox.Text = pathManager.Config.DiffToolCommandArguments;
            m_difftoolPathBox.SelectionStart = m_difftoolPathBox.Text.Length;
        }
        #endregion

        void UpdateDataModel()
        {
            PathManager pathManager = PathManager.Instance;

            int selected = m_editorChoiceBox.SelectedIndex;
            if (selected >= m_editorChoiceBox.Items.Count - 1)
            {
                // User choose custom
                pathManager.Config.CustomAppPath = m_editorPathBox.Text;
                pathManager.Config.CustomAppCommandArguments = m_editorCmdLineBox.Text;
            }

            pathManager.SetSelectedEditor(selected - 1);

            pathManager.Config.DiffToolFilename = m_difftoolPathBox.Text;
            pathManager.Config.DiffToolCommandArguments = m_difftoolCmdLineBox.Text;
        }

        #region Event Handling
        /// <summary>
        /// Handle Ok Event
        /// </summary>
        private void OnOk(object sender, EventArgs e)
        {
            PathManager pathManager = PathManager.Instance;
            UpdateDataModel();
            pathManager.SaveStatusConfig();

            this.Hide();
        }

        /// <summary>
        /// Handle Cancel Event
        /// </summary>
        private void OnCancel(object sender, EventArgs e)
        {
            isCancelled = true;
            this.Close();
        }

        /// <summary>
        /// Handle EditApp ComboBox Selected
        /// </summary>
        private void OnEditAppSelected(object sender, EventArgs e)
        {
            PathManager pathManager = PathManager.Instance;

            int selected = m_editorChoiceBox.SelectedIndex-1;
            if (selected >= pathManager.Config.ApplicationList.Count)
            {
                // User choose custom
                m_editorPathBox.Text = pathManager.Config.CustomAppPath;
                m_editorCmdLineBox.Text = pathManager.Config.CustomAppCommandArguments;
                m_editorPathBrowse.Enabled = true;
                m_editorPathBox.ReadOnly = false;
                m_editorCmdLineBox.ReadOnly = m_editorPathBox.TextLength == 0;
            }
            else if (selected >= 0)
            {
                m_editorPathBox.Text = pathManager.Config.ApplicationList[selected].Path;
                m_editorCmdLineBox.Text = pathManager.Config.ApplicationList[selected].CommandArguments;
                m_editorPathBrowse.Enabled = false;
                m_editorPathBox.ReadOnly = true;
                m_editorCmdLineBox.ReadOnly = true;
            }
            else
            {
                m_editorPathBox.Text = "";
                m_editorCmdLineBox.Text = "";
                m_editorPathBrowse.Enabled = false;
                m_editorPathBox.ReadOnly = true;
                m_editorCmdLineBox.ReadOnly = true;
            }

            m_difftoolCmdLineBox.ReadOnly = m_difftoolPathBox.TextLength == 0;

            pathManager.SetSelectedEditor(selected);

            // update Open context menu availability
            m_menuStrip_Open.Enabled = string.IsNullOrWhiteSpace(pathManager.EditorApplication) == false;
        }

        /// <summary>
        /// Handle App browse
        /// </summary>
        private void OnEditAppBrowse(object sender, EventArgs e)
        {
            OpenFileDialog browseDialog = new OpenFileDialog();

            browseDialog.InitialDirectory = Environment.GetFolderPath(Environment.SpecialFolder.ProgramFiles);
            browseDialog.Title = "Open with...";
            browseDialog.Filter = "Programs (*.exe;*.pif;*.com;*.bat)|*.exe;*.pif;*.com;*.bat|All files (*.*)|*.*";
            browseDialog.FilterIndex = 1;
            browseDialog.RestoreDirectory = true;

            if (browseDialog.ShowDialog() == DialogResult.OK)
            {
                PathManager pathManager = PathManager.Instance;

                string shellCommand = browseDialog.FileName;

                m_editorPathBox.Text = shellCommand;

                int selected = m_editorChoiceBox.SelectedIndex;
                if (selected >= pathManager.Config.ApplicationList.Count)
                {
                    // User choose custom
                    pathManager.Config.CustomAppPath = shellCommand;
                    m_editorPathBrowse.Enabled = true;
                }
                else
                {
                    m_editorPathBrowse.Enabled = false;
                }

                pathManager.SetSelectedEditor(selected);

                // update Open context menu availability
                m_menuStrip_Open.Enabled = string.IsNullOrWhiteSpace(pathManager.EditorApplication) == false;

            } // End if
        }

        /// <summary>
        /// Handle DiffTool browse
        /// </summary>
        private void OnEditDiffToolBrowse(object sender, EventArgs e)
        {
            var browseDialog = new OpenFileDialog();

            browseDialog.InitialDirectory = Environment.GetFolderPath(Environment.SpecialFolder.ProgramFiles);
            browseDialog.Title = "Open with...";
            browseDialog.Filter = "Programs (*.exe;*.pif;*.com;*.bat)|*.exe;*.pif;*.com;*.bat|All files (*.*)|*.*";
            browseDialog.FilterIndex = 1;
            browseDialog.RestoreDirectory = true;

            if (browseDialog.ShowDialog() == DialogResult.OK)
            {
                var pathManager = PathManager.Instance;

                string shellCommand = browseDialog.FileName;

                m_difftoolPathBox.Text = shellCommand;
                pathManager.Config.DiffToolFilename = shellCommand;
            } // End if
        }

        /// <summary>
        /// Handle Open
        /// </summary>
        private void OnOpen(object sender, EventArgs e)
        {
            string[] extensions =
            {
                "fmd",
                "ftx",
                "fsk",
                "fsp",
                "fcl",
                "fts",
                "ftp",
                "fvb",
                "fvm",
                "fsh",
                "fsn",
                "fsc",
                "fsd",
            };

            var sb = new StringBuilder();

            sb.Append(Properties.Resources.NW4F_OPEN_FILE_ALL_IF_FILES + "|");
            sb.Append("*." + string.Join("b;*.", extensions) + "b");
            sb.Append("|");

            sb.Append(Properties.Resources.NW4F_OPEN_FILE_ALL_FILES + " (*.*)|*.*|");

            sb.AppendFormat(string.Join("|", extensions.Select(ext => string.Format(".{0}B Files|*.{1}b", ext.ToUpper(), ext))));

            var dlg = new OpenFileDialog
            {
                Title = Properties.Resources.NW4F_OPEN_FILE_TEXT,
                Filter = sb.ToString(),
                Multiselect = false,
            };

            if (dlg.ShowDialog() != DialogResult.OK)
                return;

            if (EditorProcess.Run(dlg.FileName) == false)
            {
                CustomMessageBox.Show(Properties.Resources.NW4F_EDIT_FILE_NO_EDITOR, Properties.Resources.NW4F_EDIT_FILE_NO_EDITOR_ERROR,
                    MessageBoxButtons.OK, MessageBoxIcon.Error);
            }
        }

        /// <summary>
        /// Handle Settings
        /// </summary>
        private void OnSettings(object sender, EventArgs e)
        {
            this.Show();
            if (this.WindowState == FormWindowState.Minimized)
                this.WindowState = FormWindowState.Normal;
        }

        /// <summary>
        /// Handle Help
        /// </summary>
        private void OnHelp(object sender, EventArgs e)
        {
            String dirName = System.IO.Path.GetDirectoryName(System.Reflection.Assembly.GetEntryAssembly().Location);
            String pathToHelpFile = dirName + "\\3dBinaryIntermediateFileEdit.html";

            if (System.IO.File.Exists(pathToHelpFile))
            {
                System.Diagnostics.Process p = new System.Diagnostics.Process();
                p.StartInfo.FileName = pathToHelpFile;
                p.Start();
            }
        }

        /// <summary>
        /// Handle About
        /// </summary>
        private void OnAbout(object sender, EventArgs e)
        {
            AboutBox aboutBox = new AboutBox();
            aboutBox.ShowDialog();
        }

        /// <summary>
        /// Handle Exit
        /// </summary>
        private void OnExit(object sender, EventArgs e)
        {
            this.Close();
        }

        /// <summary>
        /// Handle Show State change
        /// </summary>
        private void OnShown(object sender, EventArgs e)
        {
        }
        #endregion

        private bool isCancelled;

        #region On Closing
        /// <summary>
        /// On Closing
        /// </summary>
        virtual public void OnClosing(object sender, FormClosingEventArgs e)
        {
            if (isCancelled == false)
            {
                PathManager pathManager = PathManager.Instance;
                UpdateDataModel();
                pathManager.SaveStatusConfig();
            }

            if (m_notifyIcon != null)
            {
                m_notifyIcon.Dispose();
                m_notifyIcon = null;
            }
        }
        #endregion

        #region Process exit event handler
        /// <summary>
        /// Handle when editor existed
        /// </summary>
        private void OnEditorExited(object sender, EventArgs args)
        {
            this.Close();
        }
        #endregion

        private void OnCloseApp(object sender, EventArgs e)
        {
            this.Close();
        }

        public void ShowWindow()
        {
            WindowState = FormWindowState.Normal;
            Show();
        }
    }
} // End of nw.g3d.bifedit
