﻿// --------------------------------------------------------------------------------
// <copyright>
// Copyright (C)Nintendo. All rights reserved.
//
// These coded instructions, statements, and computer programs contain proprietary
// information of Nintendo and/or its licensed developers and are protected by
// national and international copyright laws. They may not be disclosed to third
// parties or copied or duplicated in any form, in whole or in part, without the
// prior written consent of Nintendo.
//
// The content herein is highly confidential and should be handled accordingly.
// </copyright>
// --------------------------------------------------------------------------------
using System;
using System.Collections.Generic;
using System.Text;
using System.Xml.Serialization;
using System.Diagnostics;
using System.IO;

/*
 * flyt のスキーマに対応するクラスにメンバを付加している。
 * 自動生成されたほうのクラス定義には、
 * System.Diagnostics.DebuggerStepThroughAttribute
 * が付加されているので、デバッグでステップインできない。
 * そのため、複雑なメソッドの定義は避ける。
 */

namespace NW4F.LayoutBinaryConverter.Schema.Flyt
{
    public class SystemExtData
    {
        public enum PaneSystemDataType
        {
            AABBMesh,
            OBBMesh,
            AlignmentInfo,
            Mask,
            DropShadow,
            LayoutAnimTag,
            ProceduralShape,
            // データコンバート時に埋め込めるシステム用拡張ユーザーデータはランタイムでのフラグのビット長により 16 個までに制限されています。
            // 拡張する際はこの enum の最大値が 16 を超えないようにしてください。
            Max
        }

        public enum LayoutSystemDataType
        {
            AnimTag,
            // データコンバート時に埋め込めるシステム用拡張ユーザーデータはランタイムでのフラグのビット長により 16 個までに制限されています。
            // 拡張する際はこの enum の最大値が 16 を超えないようにしてください。
            Max
        }

        public List<byte[]> BinaryDataList = new List<byte[]>();

        public void Add(byte[] data)
        {
            BinaryDataList.Add(data);
        }
    }

    public partial class Pane
    {
        [XmlIgnore]
        public List<Pane> ChildList;

        [XmlIgnore]
        public SystemExtData SystemExtData { get; private set; } = null;

        public Pane(string paneName, Vec2 size)
            : this()    // xsdで作成されるデフォルトコンストラクタを呼ぶ
        {
            name = paneName;
            kind = PaneKind.Null;

            basePositionType = new Position();
            basePositionType.x = HorizontalPosition.Center;
            basePositionType.y = VerticalPosition.Center;
            parentRelativePositionType = new Position();
            parentRelativePositionType.x = HorizontalPosition.Center;
            parentRelativePositionType.y = VerticalPosition.Center;

            translate = new Vec3(0, 0, 0);
            rotate = new Vec3(0, 0, 0);
            scale = new Vec2(1, 1);
            this.size = size;
        }

        public void AddSystemExtDataBlock(byte[] bytes)
        {
            if (SystemExtData == null)
            {
                SystemExtData = new SystemExtData();
            }

            SystemExtData.Add(bytes);
        }

        public bool IsMaskEnabled()
        {
            // 現在ランタイムで対応しているペインのみマスクが有効と判断する。
            if (kind == PaneKind.Picture ||
                kind == PaneKind.Window ||
                kind == PaneKind.TextBox)
            {
                // マスクが有効でもテクスチャが設定されていない場合は imageName に "none" が設定されている。
                if (mask != null &&
                    mask.maskEnabled &&
                    mask.maskTexMap.imageName != "none")
                {
                    return true;
                }
            }

            return false;
        }

        public bool IsDropShadowEnabled()
        {
            if (dropShadow != null &&
                (dropShadow.dropShadowEnabled ||
                 dropShadow.outerGlowEnabled ||
                 dropShadow.strokeEnabled))
            {
                return true;
            }

            return false;
        }

        public bool IsProceduralShapeEnabled()
        {
            if (kind == PaneKind.Picture)
            {
                Picture pict = (Picture)Item;
                if (pict.proceduralShapeParam != null &&
                    pict.proceduralShapeParam.proceduralShapeEnabled)
                {
                    return true;
                }
            }

            return false;
        }
    }

    /// <summary>
    /// 最適化メッシュ情報
    /// </summary>
    public class OptimizedMesh
    {
        /// <summary>
        /// 最適化メッシュのタイプ
        /// </summary>
        public ImageFitMesh.OptimizationType FillrateOptimizationType { get; set; } = ImageFitMesh.OptimizationType.None;

        /// <summary>
        /// 最適化されたメッシュの頂点情報
        /// </summary>
        public ImageFitMesh.Vertex2D[] OptimizedVertexList { get; set; } = null;

        /// <summary>
        /// OBB の場合のメッシュの軸
        /// </summary>
        public ImageFitMesh.Vertex2D[] Axis { get; set; } = null;
    }

    public partial class Picture
    {
        [XmlIgnore]
        public MaterialInfo MaterialInfo;

        [XmlIgnore]
        public ShapeInfo ShapeInfo;

        [XmlIgnore]
        public bool FillOptimizationEnabled = false;

        public Picture(Material material)
            : this()    // xsdで作成されるデフォルトコンストラクタを呼ぶ
        {
            vtxColLT = new Color4(255, 255, 255, 255);
            vtxColRT = new Color4(255, 255, 255, 255);
            vtxColLB = new Color4(255, 255, 255, 255);
            vtxColRB = new Color4(255, 255, 255, 255);

            this.material = material;
        }
    }

    public partial class TextBox
    {
        [XmlIgnore]
        public MaterialInfo MaterialInfo;
    }

    public partial class WindowFrame
    {
        [XmlIgnore]
        public MaterialInfo MaterialInfo;
    }

    public partial class WindowContent
    {
        [XmlIgnore]
        public MaterialInfo MaterialInfo;
    }

    public partial class WindowFrameSize
    {
        [XmlIgnore]
        public bool IsZero
        {
            get { return this.l == 0.0f && this.r == 0.0f && this.t == 0.0f && this.b == 0.0f; }
        }
    }

    public partial class Material
    {
        public Material(string name)
            : this()
        {
            blackColor = new BlackColor(0, 0, 0);
            whiteColor = new WhiteColor(255, 255, 255);
            this.name = name;
        }
    }

    public partial class TexCoord
    {
        public TexCoord() { }

        public TexCoord(float s, float t)
        {
            texLT = new TexVec2(0, 0);
            texRT = new TexVec2(s, 0);
            texLB = new TexVec2(0, t);
            texRB = new TexVec2(s, t);
        }
    }

    public partial class Vec2
    {
        public Vec2() { }

        public Vec2(float x, float y)
        {
            this.x = x;
            this.y = y;
        }
    }

    public partial class Vec3
    {
        public Vec3() {}

        public Vec3(float x, float y, float z)
        {
            this.x = x;
            this.y = y;
            this.z = z;
        }
    }

    public partial class TexVec2
    {
        public TexVec2() { }

        public TexVec2(float s, float t)
        {
            this.s = s;
            this.t = t;
        }
    }

    public partial class Color4
    {
        public Color4() { }

        public Color4(byte r, byte g, byte b, byte a)
        {
            this.r = r;
            this.g = g;
            this.b = b;
            this.a = a;
        }

        public Color4(BlackColor src)
        {
            this.r = src.r;
            this.g = src.g;
            this.b = src.b;
            this.a = src.a;
        }

        public Color4(WhiteColor src)
        {
            this.r = src.r;
            this.g = src.g;
            this.b = src.b;
            this.a = src.a;
        }
    }

    public partial class Color4f
    {
        public Color4f(float r, float g, float b, float a)
        {
            this.r = r;
            this.g = g;
            this.b = b;
            this.a = a;
        }

        public Color4f(Color4f src)
        {
            this.r = src.r;
            this.g = src.g;
            this.b = src.b;
            this.a = src.a;
            this.byte_convert = src.byte_convert;
        }

        public Color4f(BlackColorFloat src)
        {
            this.r = src.r;
            this.g = src.g;
            this.b = src.b;
            this.a = src.a;
            this.byte_convert = src.byte_convert;
        }

        public Color4f(WhiteColorFloat src)
        {
            this.r = src.r;
            this.g = src.g;
            this.b = src.b;
            this.a = src.a;
            this.byte_convert = src.byte_convert;
        }

        public Color4f(BlackColor src)
        {
            this.r = (float)src.r / 255.0f;
            this.g = (float)src.g / 255.0f;
            this.b = (float)src.b / 255.0f;
            this.a = (float)src.a / 255.0f;
            this.byte_convert = true;
        }

        public Color4f(WhiteColor src)
        {
            this.r = (float)src.r / 255.0f;
            this.g = (float)src.g / 255.0f;
            this.b = (float)src.b / 255.0f;
            this.a = (float)src.a / 255.0f;
            this.byte_convert = true;
        }

        public Color4f(Color4 src)
        {
            Set(src);
        }

        public void Set(Color4 color)
        {
            r = (float)color.r / 255.0f;
            g = (float)color.g / 255.0f;
            b = (float)color.b / 255.0f;
            a = (float)color.a / 255.0f;

            this.byte_convert = true;
        }
    }


#if false
    public partial class ColorS10_4
    {
        public ColorS10_4() { }

        public ColorS10_4(short r, short g, short b, short a)
        {
            this.r = r;
            this.g = g;
            this.b = b;
            this.a = a;
        }
    }
#endif

    public partial class BlackColor
    {
        public BlackColor(byte r, byte g, byte b)
            : this()    // xsdで作成されるデフォルトコンストラクタを呼ぶ
        {
            this.r = r;
            this.g = g;
            this.b = b;
        }
    }

    public partial class WhiteColor
    {
        public WhiteColor(byte r, byte g, byte b)
            : this()    // xsdで作成されるデフォルトコンストラクタを呼ぶ
        {
            this.r = r;
            this.g = g;
            this.b = b;
        }

        public WhiteColor(byte r, byte g, byte b, byte a)
        {
            this.r = r;
            this.g = g;
            this.b = b;
            this.a = a;
        }
    }

    public partial class TextureFile
    {
        bool _bIsIndirect;
        bool _bIsNoIndirect;
        bool _bIsCaptureTexture = false;

        public string GetName()
        {
            return Path.GetFileNameWithoutExtension(imagePath);
        }

        public static string GetImagePostfix(TexelFormat f, bool isIndirect)
        {
            // テクスチャ名の衝突を回避するために、用途やフォーマットによってファイル名が別になるようにする。
            // TexelFormat の値が、アルファベットの範囲をオーバーするので、大文字、小文字をつかいわける
            int code = (int)f;
            if (code < 26)
            {
                return (isIndirect ? "+" : "^") + Char.ConvertFromUtf32(code + 0x61 /* 'a' */);
            }
            else
            {
                // アルファベット大文字・小文字の区別しかないのは、誤判定の危険があるので、_A のようにする。
                return (isIndirect ? "+" : "^") + "_" + Char.ConvertFromUtf32(code - 26 + 0x41 /* 'A' */);
            }
        }

        public string GetConvertedFileName(bool isIndirect, bool isUseBntxTextureFormat)
        {
            // Bntx を利用する場合、テクスチャが一つのファイルに集約されるので、拡張子はつけない。
            string extention = isUseBntxTextureFormat ? "" : ".bflim";

            return Path.GetFileNameWithoutExtension(imagePath) + GetImagePostfix(format, isIndirect) + extention;
        }

        public bool IsIndirect
        {
            get { return _bIsIndirect; }
            set { _bIsIndirect = value; }
        }

        public bool IsNoIndirect
        {
            get { return _bIsNoIndirect; }
            set { _bIsNoIndirect = value; }
        }

        public bool IsCaptureTexture
        {
            get { return _bIsCaptureTexture; }
            set { _bIsCaptureTexture = value; }
        }

        public Vec2 ActualImageSize
        {
            get;
            set;
        }
    }

    public partial class FontFile
    {
        public string GetName()
        {
            if (name == null)
            {
                // name属性がない(0.13.0以前のflyt)場合は従来通り拡張子なしの名前を用いる
                return Path.GetFileNameWithoutExtension(path);
            }
            else
            {
                return name;
            }
        }

        // ResourceAccessorに登録されるときの名前を取得する
        public string GetRegisterName()
        {
            if (string.IsNullOrEmpty(path))
            {
                // 生成されるフォントファイル名を登録名として使う
                return name + ".bffnt";
            }

            string extName = Path.GetExtension(path).ToLowerInvariant();
            if (extName == ".ttf" || extName == ".otf" || extName == ".bfttf" || extName == ".bfotf")
            {
                // TTFの場合はname属性を使う
                return name;
            }
            else if (extName == ".ffnt")
            {
                // 変換設定の場合は bffnt に変える。
                return Path.GetFileNameWithoutExtension(path) + ".bffnt";
            }
            else
            {
                return Path.GetFileName(path);
            }
        }
    }

    public partial class Material_CTRCombinerUserShader
    {
        public string GetFileNameWithoutExtension()
        {
            return Path.GetFileNameWithoutExtension(fileName);
        }

        public bool ExistFileName()
        {
            if (string.IsNullOrWhiteSpace(fileName))
            {
                return false;
            }
            return true;
        }
    }

    public partial class TexCoordGen
    {
        public bool IsProjection()
        {
            if (srcParam == TexGenSrc.OrthogonalProjection ||
                srcParam == TexGenSrc.PaneBasedProjection ||
                srcParam == TexGenSrc.PerspectiveProjection ||
                srcParam == TexGenSrc.PaneBasedPerspectiveProjection)
            {
                return true;
            }

            return false;
        }

    }

    public class ArchiveShaderFile
    {
        public ushort key0;
        public ushort key1;
        public ushort firstBlend;
        public ushort secondBlend;
        public int texMapCount;
        public int useDetailedCombiner;
        public int useCombinerUserShader;
        public int useIndividualTable;
        public int stageCount;
        public UInt32[] stageBits;
        public bool perspectiveTextureProjection;
        public bool proceduralShape;
    }

    /// <summary>
    /// マテリアルに関する情報を含むクラス。
    /// このクラスはスキーマには無い。
    /// </summary>
    public class MaterialInfo
    {
        Pane _pane;
        Material _simple;
        Material_CTR _detail;
        bool _bOutTexture;
        bool _bDetailSetting;
        int _binaryIndex;
        Color4  _fontShadowBlackColor;
        Color4  _fontShadowWhiteColor;
        bool _bUseLowLevelCombinerSettings;
        bool _bCombinerUserShaderSettings;

        public Pane Pane { get { return _pane; } }

        public bool IsOutTexture { get { return _bOutTexture; } }

        public bool IsDetailSetting { get { return _bDetailSetting; } }

        public bool UseLowLevelCombinerSettings { get { return _bUseLowLevelCombinerSettings; } }

        public bool UseCombinerUserShaderSettings { get { return _bCombinerUserShaderSettings; } }

        public bool UseTextureOnly { get; set; }

        public bool UseThresholdingAlphaInterpolation { get; set; }

        public int BinaryIndex
        {
            get { return _binaryIndex; }
            set { _binaryIndex = value; }
        }

        public string name;

        public Color4f color;
        public Color4f[] konstColors;

        public Color4 FontShadowBlackColor
        {
            get { return _fontShadowBlackColor;}
        }

        public Color4 FontShadowWhiteColor
        {
            get { return _fontShadowWhiteColor;}
        }

        public TexMap[] texMap;
        public TexMatrix[] texMatrix;
        public TexCoordGen[] texCoordGen;

        public Material_CTRTevStage[] tevStage;
        public Material_CTRAlphaCompare alphaCompare;
        public Material_CTRBlendMode blendMode;
        public Material_CTRBlendMode blendModeAlpha;
        public MaterialHSBAdjustment hsbAdjustment;
        public Material_CTRCombinerUserShader combinerUserShader;

        public MaterialInfo(Pane pane, Material simple, Material_CTR detail, bool bOutTexture, bool bDetailSetting)
        {
            _pane = pane;
            _simple = simple;
            _detail = detail;
            _bOutTexture = bOutTexture;
            _bDetailSetting = bDetailSetting;
            _fontShadowBlackColor = null;
            _fontShadowWhiteColor = null;
            this.UseTextureOnly = false;
            _bUseLowLevelCombinerSettings = (detail != null && detail.useLowLevelCombinerSettings);
            _bCombinerUserShaderSettings = (detail != null && detail.useCombinerUserShaderSettings);
            if (detail.CombinerUserShader != null)
            {
                combinerUserShader = detail.CombinerUserShader;
            }
        }

        public int GetTextureCount()
        {
            return this.texMap != null ? this.texMap.Length : 0;
        }

        public bool HasIndirectParameter()
        {
            if (tevStage == null) return false;
            bool has_indirect = false;
            foreach (Material_CTRTevStage stage in tevStage)
            {
                if (stage.rgb.mode == TevMode.Indirect || stage.rgb.mode == TevMode.BlendIndirect || stage.rgb.mode == TevMode.EachIndirect)
                {
                    has_indirect = true;
                }
            }
            return has_indirect;
        }

        public UInt32 GetProjectionTexGenParameterNum()
        {
            UInt32 count = 0;
            foreach (TexCoordGen texGen in texCoordGen)
            {
                if (texGen.srcParam == TexGenSrc.OrthogonalProjection
                    || texGen.srcParam == TexGenSrc.PaneBasedProjection
                    || texGen.srcParam == TexGenSrc.PerspectiveProjection
                    || texGen.srcParam == TexGenSrc.PaneBasedPerspectiveProjection)
                {
                    count++;
                }
            }

            return count;
        }

        public bool HasFontShadowParameter()
        {
            // マテリアルがテキストボックスのものでかつ、カラー情報を持つ場合。
            // アニメーションを持たず、かつデフォルト値から変更がない場合は、中間ファイルに記載せず null が設定されます。
            if (this.FontShadowBlackColor != null && this.FontShadowWhiteColor != null)
            {
                // デフォルト値から変更がないかどうかはレイアウトエディタの方でチェックしているので、
                // ここではチェックする必要はない。FontShadowBlackColorとFontShadowWhiteColorがあれば
                // 出力してよい。
                return true;
            }

            return false;
        }

        public bool EqualsContent(MaterialInfo other)
        {
            return this._simple == other._simple
                && this._detail == other._detail
                && this._bOutTexture == other._bOutTexture
                && this._bDetailSetting == other._bDetailSetting
                && this.UseTextureOnly == other.UseTextureOnly;
        }

        public void SetFontShadowBlackWhiteColor(Color4 black, Color4 white)
        {
            _fontShadowBlackColor = black;
            _fontShadowWhiteColor = white;
        }
    }

    /// <summary>
    /// このクラスはスキーマにはない。
    /// </summary>
    public class ShapeInfo
    {
        /// <summary>
        /// シェイプ形状
        /// </summary>
        ShapeType _type;

        /// <summary>
        /// シェイプバイナリデータの ShapeInfoList 内でのインデックス。
        /// ランタイムでシェイプのバイナリ情報とピクチャペインを関連付けるために使用されます。
        /// </summary>
        uint _binaryIndex;

        /// <summary>
        /// nn::gfx::util::PrimitiveShape で生成されるシェイプの情報。
        /// すべての形状に関する情報を持っていますが、ShapeType で設定された形状の値のみ有効です。
        /// </summary>
        ShapeParam _gfxShapeParam;

        public ShapeType ShapeType
        {
            get { return _type; }
            set { _type = value; }
        }

        public ShapeParam GfxShapeParam
        {
            get { return _gfxShapeParam; }
            set { _gfxShapeParam = value; }
        }

        public uint BinaryIndex
        {
            get { return _binaryIndex; }
            set { _binaryIndex = value; }
        }

        /// <summary>
        /// ShapeInfoList 内で情報の一元化をする際の比較に使用されます。
        /// このメソッドで同じと判断された情報は一つにまとめられます。
        /// </summary>
        /// <param name="other">比較対象</param>
        /// <returns></returns>
        public bool EqualsContent(ShapeInfo other)
        {
            if (ShapeType == other.ShapeType)
            {
                switch (ShapeType)
                {
                    case ShapeType.GfxPrimitiveRoundRect:
                        return GfxShapeParam.roundRadius == other.GfxShapeParam.roundRadius &&
                            GfxShapeParam.roundSlice == other.GfxShapeParam.roundSlice;
                    case ShapeType.GfxPrimitiveCircle:
                        return GfxShapeParam.circleSlice == other.GfxShapeParam.circleSlice;
                }
            }

            return false;
        }
    }

    public partial class CaptureTexture
    {
        string _name;
        bool _framebufferCaptureEnabled;
        float[] _clearColor = new float[4];

        public CaptureTexture(string name, bool framebufferCaptureEnabled, float[] clearColor)
        {
            _name = name;
            _framebufferCaptureEnabled = framebufferCaptureEnabled;

            if (_clearColor.Length == clearColor.Length)
            {
                for (int i = 0; i < _clearColor.Length; ++i)
                {
                    _clearColor[i] = clearColor[i];
                }
            }
        }

        string Name { get { return _name; } }
        bool FramebufferCaptureEnabled { get { return _framebufferCaptureEnabled; } }
        float[] ClearColor { get { return _clearColor; } }
    }

    public class TexMapAdditionalInfo
    {
        int _bitFlags = 0;

        public enum InfoType
        {
            CaptureTexture = 0,
        };

        public void Enable(InfoType type)
        {
            _bitFlags |= (1 << (int)type);
        }

        public void Disable(InfoType type)
        {
            _bitFlags &= ~(1 << (int)type);
        }

        public int GetFlags()
        {
            return _bitFlags;
        }
    }

    public class TextureStage
    {
    }
}
