﻿// --------------------------------------------------------------------------------
// <copyright>
// Copyright (C)Nintendo. All rights reserved.
//
// These coded instructions, statements, and computer programs contain proprietary
// information of Nintendo and/or its licensed developers and are protected by
// national and international copyright laws. They may not be disclosed to third
// parties or copied or duplicated in any form, in whole or in part, without the
// prior written consent of Nintendo.
//
// The content herein is highly confidential and should be handled accordingly.
// </copyright>
// --------------------------------------------------------------------------------
using System;
using System.IO;
using System.Diagnostics;
using System.Collections.Generic;
using System.Linq;

namespace NW4F.LayoutBinaryConverter
{
    using Lyt = Schema.Flyt;
    using Tga = NW4C.Tga;
    using NW4F.LayoutBinaryConverter;

    /// <summary>
    /// TgaUtil の概要の説明です。
    /// </summary>
    public class TgaUtil
    {
        public static bool SkipIfSrcTextureNotExist { get; set; }
        static TexConv.ITexttureWriter _TexttureWriter;
        static string _ApiTypeName;

        static TgaUtil()
        {
        }

        /// <summary>
        /// プラットフォーム毎のテクスチャ書き出しクラスを初期化します。
        /// </summary>
        static public bool InitializeTexttureWriter(string apiTypeName, bool isLogSilent, string isGpuEncodigEnabled)
        {
            _ApiTypeName = apiTypeName;

            string platformSufix;
            if (string.IsNullOrEmpty(apiTypeName) || apiTypeName.Equals("Gl", StringComparison.CurrentCultureIgnoreCase))
            {
                platformSufix = "Generic";
            }
            else if (apiTypeName.Equals("Cafe", StringComparison.CurrentCultureIgnoreCase))
            {
                platformSufix = "Cafe";
            }
            else if (apiTypeName.Equals("Nvn", StringComparison.CurrentCultureIgnoreCase))
            {
                platformSufix = "NX";
            }
            else
            {
                platformSufix = "Unknown";
            }

            string platformDllFileName = "TextureConverter" + platformSufix;

            //// GX2TextureWriter をプラットフォーム中立なインタフェース経由で利用するよう変更して、実装を切り替えられるようにしています。
            Type type = PluginUtil.EnumrateInterfaceTypesFromDLL((dllFileName) => string.Compare(dllFileName, platformDllFileName, true) == 0, typeof(TexConv.ITexttureWriter)).FirstOrDefault();
            if (type != null)
            {
                _TexttureWriter = Activator.CreateInstance(type) as TexConv.ITexttureWriter;
            }else{
                // 指定したテクスチャコンバーターが発見できない場合は Generic コンバーターを初期化する。
                _TexttureWriter = Activator.CreateInstance(typeof(GenericTextureWriter)) as TexConv.ITexttureWriter;
            }

            if(_TexttureWriter is GenericTextureWriter)
            {
                // Generic用 エンコーダーを初期化する
                var genericTextureWriter = _TexttureWriter as GenericTextureWriter;
                string execPath = Path.GetDirectoryName(System.Reflection.Assembly.GetExecutingAssembly().Location);
                string dllPath = GetGenericEncoderDllLoadPath_(execPath);
                if (genericTextureWriter.InitializeEncoder(dllPath) == false)
                {
                    _TexttureWriter = null;
                }
            }

            if(_TexttureWriter != null)
            {
                _TexttureWriter.SetLogSilent(isLogSilent);
                _TexttureWriter.SetGpuEncodingEnabled(isGpuEncodigEnabled);
            }

            return _TexttureWriter != null;
        }

        /// <summary>
        /// Generic用 エンコーダー用 の DLL パスを取得する。
        /// </summary>
        static string GetGenericEncoderDllLoadPath_(string execPath)
        {
            // 正式な位置
            string dllPath = execPath + "\\..\\GraphicsTools";
            if(Directory.Exists(dllPath))
            {
                return dllPath;
            }

            // 開発時の出力フォルダ位置
            dllPath = execPath + "\\..\\..\\..\\..\\..\\..\\..\\..\\..\\..\\Tools\\Graphics\\GraphicsTools";
            if (Directory.Exists(dllPath))
            {
                return dllPath;
            }

            // 開発時の出力フォルダ位置
            // Encoder.vcxproj の Debug ビルドのビルド後イベントで下記の場所に 必要な dll 一式がコピーされます。
            string dllPathForDebug = execPath + "\\..\\..\\..\\..\\..\\..\\..\\Win32\\Tools\\Graphics\\GraphicsTools\\TextureConverter\\Debug";
            if (Directory.Exists(dllPathForDebug))
            {
                return dllPathForDebug;
            }

            Report.Err.WriteLine(string.Format("Can't initialize the encoder dll. path - {0} or {1}", dllPath, dllPathForDebug));

            return string.Empty;
        }

        /// <summary>
        /// TGA ファイルヘッダ
        /// </summary>
        struct TGAHeader
        {
            public readonly byte   IdSize;          // IDフィールド長
            public readonly byte   ColorMapExist;   // カラーマップ有り無し(0 = 無し、1 = 有り)
            public readonly byte   ImageFmt;        // 画像フォーマット
            // ( 0 = イメージ無し、
            //   1 = インデックスカラー（256色、
            //   2 = フルカラー、
            //   3 = 白黒 )
            public readonly short  ColorMapEntry;   // カラーマップエントリ
            public readonly short  ColorMapLength;  // カラーマップエントリ
            public readonly byte   ColorMapBitSize; // カラーマップエントリサイズ

            public readonly short PosX;             // X座標
            public readonly short PosY;             // Y座標
            public readonly short Width;            // 幅
            public readonly short Hegiht;           // 高さ

            public readonly byte  BitDepth;         // 色深度
            public readonly byte  TexelFmt;         // テクセルフォーマット
            // 0-3 : 属性
            // 4: 水平方向格納( 0 = 左 => 右 )
            // 5: 垂直方向格納( 0 = 上 => 下 )

            public TGAHeader( BinaryReader binReader )
            {
                IdSize          = binReader.ReadByte();
                ColorMapExist   = binReader.ReadByte();
                ImageFmt        = binReader.ReadByte();

                ColorMapEntry   = binReader.ReadInt16();
                ColorMapLength  = binReader.ReadInt16();
                ColorMapBitSize = binReader.ReadByte();

                PosX            = binReader.ReadInt16();
                PosY            = binReader.ReadInt16();
                Width           = binReader.ReadInt16();
                Hegiht          = binReader.ReadInt16();

                BitDepth        = binReader.ReadByte();
                TexelFmt        = binReader.ReadByte();
            }
        }

        public static bool HasAlpha(string filePath)
        {
            using (BinaryReader br = new BinaryReader(new FileStream(filePath, FileMode.Open, FileAccess.Read)))
            {
                TGAHeader tgaHeader = new TGAHeader(br);
                return tgaHeader.BitDepth == 32;
            }
        }

        /*>*******************************(*)*******************************<*/
        // returns num if num is a power of two.  returns 0 if num is not.
        // this function is only valid for numbers 1024 or less since this is
        // the maximum dimension of a texture.
        /*>*******************************(*)*******************************<*/

        /// <summary>
        ///
        /// </summary>
        static bool TCCheckPowerOfTwo(int num)
        {
            switch (num)
            {
            case 1024:      // max. dimension size is 1024
            case 512:
            case 256:
            case 128:
            case 64:
            case 32:
            case 16:
            case 8:
            case 4:
            case 2:
            case 1:
                return true;
            }

            return false;
        }

        /// <summary>
        /// アルファ付きのフォーマットか？
        /// </summary>
        static bool IsAlphaFormat_(Lyt.TexelFormat fmt)
        {
            if (fmt == Lyt.TexelFormat.A4 ||
                fmt == Lyt.TexelFormat.A8 ||
                fmt == Lyt.TexelFormat.BC3 ||
                fmt == Lyt.TexelFormat.BC7 ||
                fmt == Lyt.TexelFormat.BC4A ||
                fmt == Lyt.TexelFormat.BC5 ||
                fmt == Lyt.TexelFormat.LA4 ||
                fmt == Lyt.TexelFormat.LA8 ||
                fmt == Lyt.TexelFormat.RGB5A1 ||
                fmt == Lyt.TexelFormat.RGBA4 ||
                fmt == Lyt.TexelFormat.RGBA8)
            {
                return true;
            }
            else
            {
                return false;
            }
        }

        /// <summary>
        /// スイズルパターンを計算する
        /// </summary>
        static int CalcSwizzlePattern_(string inFileName)
        {
            byte[] bytes = System.Text.Encoding.ASCII.GetBytes(Path.GetFileNameWithoutExtension(inFileName));
            uint sum = 0;
            foreach (byte num in bytes)
            {
                sum += num;
            }
            return (int)(sum % 8);
        }

        /// <summary>
        ///
        /// </summary>
        static bool IsSrgbFormat_(Lyt::TexelFormat texFormat)
        {
            if (texFormat == Lyt::TexelFormat.RGBA8 ||
                texFormat == Lyt::TexelFormat.BC1 ||
                texFormat == Lyt::TexelFormat.BC2 ||
                texFormat == Lyt::TexelFormat.BC3 ||
                texFormat == Lyt::TexelFormat.BC7 ||

                texFormat == Lyt::TexelFormat.ASTC4x4 ||
                texFormat == Lyt::TexelFormat.ASTC5x4 ||
                texFormat == Lyt::TexelFormat.ASTC5x5 ||
                texFormat == Lyt::TexelFormat.ASTC6x5 ||
                texFormat == Lyt::TexelFormat.ASTC6x6 ||
                texFormat == Lyt::TexelFormat.ASTC8x5 ||
                texFormat == Lyt::TexelFormat.ASTC8x6 ||
                texFormat == Lyt::TexelFormat.ASTC8x8 ||
                texFormat == Lyt::TexelFormat.ASTC10x5 ||
                texFormat == Lyt::TexelFormat.ASTC10x6 ||
                texFormat == Lyt::TexelFormat.ASTC10x8 ||
                texFormat == Lyt::TexelFormat.ASTC10x10 ||
                texFormat == Lyt::TexelFormat.ASTC12x10 ||
                texFormat == Lyt::TexelFormat.ASTC12x12 )
            {
                // これらのフォーマットは実機側でsRGBフェッチするので、実データをリニアに変換する必要はない
                return true;
            }

            return false;
        }

        /// <summary>
        ///
        /// </summary>
        static TexConv.SrcImage LoadSrcImage_(string inFileName)
        {
            TexConv.SrcImage srcImage = new TexConv.SrcImage();
            byte[] buffer = null;
            using (FileStream stream = new FileStream(inFileName, FileMode.Open, FileAccess.Read, FileShare.Read))
            {
                int length = (int)stream.Length;
                buffer = new byte[length];
                stream.Read(buffer, 0, length);
                TexConv.TgaReader.Read(length, buffer, srcImage);
            }

            return srcImage;
        }

        /// <summary>
        ///
        /// </summary>
        private static TexConv.SrcImage CreateSourceImage_(Schema.Flyt.TextureFile texFile, string srcBasePath, bool isDegamma, bool isIndirect)
        {
            // 事前にファイルの有無をチェックして無ければ処理をスキップする。
            string srcPath = FileUtil.GetAbsolutePath(srcBasePath, texFile.imagePath);
            if (SkipIfSrcTextureNotExist && !File.Exists(srcPath))
            {
                return null;
            }

            // インダイレクト用は、ガンマ変換を行いません。
            bool isGammaEnabled = isIndirect ? false : isDegamma;

            TexConv.SrcImage srcImage = LoadSrcImage_(srcPath);

            // 読み込んだ画像のサイズを記録しておきます。
            texFile.ActualImageSize = new Lyt::Vec2(srcImage.lyColor.Width, srcImage.lyColor.Height);

            srcImage.name = Path.GetFileNameWithoutExtension(srcPath);
            srcImage.isIndirectTexture = isIndirect;
            srcImage.swizzlePattern = CalcSwizzlePattern_(srcPath);
            srcImage.isSrgbFormat = isGammaEnabled && IsSrgbFormat_(texFile.format);
            srcImage.isDegamma = isGammaEnabled && !srcImage.isSrgbFormat;
            srcImage.dstFormat = (TexConv.TexelFormat)texFile.format;

            return srcImage;
        }

        /// <summary>
        ///
        /// </summary>
        private static bool GenTextureFile_(TexConv.SrcImage srcImage, string dstDir, string cacheDir)
        {
            if (srcImage == null)
            {
                return false;
            }

            string dstPath = Path.Combine(dstDir, srcImage.name);

            try
            {
                if (!_TexttureWriter.ExportSrcImageToFtxbFile(srcImage, _ApiTypeName, dstDir, cacheDir))
                {
                    Report.Err.WriteLine(string.Format("Can't export ftxb - {0}", dstPath));
                    return false;
                }
            }
            catch (TexConv.TexConvException)
            {
                // 変換に失敗した
                if (File.Exists(dstPath))
                {
                    File.Delete(dstPath);   // 失敗したファイルが残っている場合は削除しておく
                }
                throw;
            }

            return true;
        }

        /// <summary>
        ///
        /// </summary>
        public static Dictionary<Schema.Flyt.TextureFile, TexConv.SrcImage> LoadTextureSourceFiles(string srcBasePath, ICollection<Schema.Flyt.TextureFile> texFiles, bool isDegamma)
        {
            var srcImageList = new Dictionary<Schema.Flyt.TextureFile, TexConv.SrcImage>();

            foreach (Schema.Flyt.TextureFile texFile in texFiles)
            {
                // キャプチャテクスチャはファイルを持っていないため読み込めない。
                if (texFile.IsCaptureTexture)
                {
                    continue;
                }

                bool isTexturePatternAnimTexture = (!texFile.IsIndirect && !texFile.IsNoIndirect);
                bool isUsedForNormalTexture = texFile.IsNoIndirect || isTexturePatternAnimTexture;

                srcImageList.Add(texFile, CreateSourceImage_(texFile, srcBasePath, isDegamma, !isUsedForNormalTexture));
            }

            return srcImageList;
        }


        /// <summary>
        ///
        /// </summary>
        public static bool GenTextureFiles(Dictionary<Schema.Flyt.TextureFile, TexConv.SrcImage> srcFiles, string dstBaseDir, string ftxbCacheDir, bool isDegamma)
        {
            if (srcFiles.Count == 0)
            {
                return true;
            }

            string dstDir = FileUtil.MakeResourceDirectory(dstBaseDir, BinaryLytWriter.TypeOfTextureResource);

            bool convertAllSuccessfully = true;
            foreach (KeyValuePair<Schema.Flyt.TextureFile, TexConv.SrcImage> srcFile in srcFiles)
            {
                // インダイレクトのテクスチャと非インダイレクトのテクスチャをそれぞれ出力する。
                // テクスチャパターンアニメーションのテクスチャの場合、どちらのフラグも立っていないので、その場合は非インダイレクトのテクスチャとして出力する。

                // 通常テクスチャとして利用されている
                bool isTexturePatternAnimTexture = (!srcFile.Key.IsIndirect && !srcFile.Key.IsNoIndirect);
                bool isUsedForNormalTexture = srcFile.Key.IsNoIndirect || isTexturePatternAnimTexture;
                bool isIndirect = !isUsedForNormalTexture;
                string name = srcFile.Value.name;
                {
                    // srcImage が持っている各種コンバート設定はキャッシュのハッシュ値を生成するために使用されている。
                    srcFile.Value.isIndirectTexture = isIndirect;

                    // インダイレクト用は、ガンマ変換を行いません。
                    bool isGammaEnabled = isIndirect ? false : isDegamma;

                    // 出力するデータの内容に合わせて SrcImage のパラメータを調整する。
                    srcFile.Value.name = name + Lyt::TextureFile.GetImagePostfix(srcFile.Key.format, isIndirect);
                    srcFile.Value.isIndirectTexture = isIndirect;
                    srcFile.Value.isSrgbFormat = isGammaEnabled && IsSrgbFormat_(srcFile.Key.format);
                    srcFile.Value.isDegamma = isGammaEnabled && !srcFile.Value.isSrgbFormat;
                    srcFile.Value.dstFormat = (TexConv.TexelFormat)srcFile.Key.format;

                    convertAllSuccessfully &= GenTextureFile_(srcFile.Value, dstDir, ftxbCacheDir);
                }

                // インダイレクト用途で利用されている
                if (srcFile.Key.IsIndirect)
                {
                    // インダイレクト用は、ガンマ変換を行いません。
                    bool isGammaEnabled = srcFile.Key.IsIndirect ? false : isDegamma;

                    // 出力するデータの内容に合わせて SrcImage のパラメータを調整する。
                    srcFile.Value.name = name + Lyt::TextureFile.GetImagePostfix(srcFile.Key.format, srcFile.Key.IsIndirect);
                    srcFile.Value.isIndirectTexture = srcFile.Key.IsIndirect;
                    srcFile.Value.isSrgbFormat = isGammaEnabled && IsSrgbFormat_(srcFile.Key.format);
                    srcFile.Value.isDegamma = isGammaEnabled && !srcFile.Value.isSrgbFormat;
                    srcFile.Value.dstFormat = (TexConv.TexelFormat)srcFile.Key.format;

                    convertAllSuccessfully &= GenTextureFile_(srcFile.Value, dstDir, ftxbCacheDir);
                }
            }

            return convertAllSuccessfully;
        }
    }
}
