﻿// --------------------------------------------------------------------------------
// <copyright>
// Copyright (C)Nintendo. All rights reserved.
//
// These coded instructions, statements, and computer programs contain proprietary
// information of Nintendo and/or its licensed developers and are protected by
// national and international copyright laws. They may not be disclosed to third
// parties or copied or duplicated in any form, in whole or in part, without the
// prior written consent of Nintendo.
//
// The content herein is highly confidential and should be handled accordingly.
// </copyright>
// --------------------------------------------------------------------------------
using System;
using System.Collections.Generic;
using System.Text;
using System.IO;

namespace NW4F.LayoutBinaryConverter
{
    using Schema.Flan;

    class LanInfo
    {
        AnimShare[] _animShare;
        AnimTag[] _animTagAry;
        LAN[] _lanAry;
        LytInfo _lytInfo;
        bool _bExtraFrameForOneTime;
        string _inputFileName;

        List<LAN> _rejectLans = new List<LAN>();

        public LanInfo(LanInfo obj)
        {
            _animShare = obj._animShare;
            _animTagAry = obj._animTagAry;
            _lanAry = obj._lanAry;
            _lytInfo = obj._lytInfo;
            _bExtraFrameForOneTime = obj._bExtraFrameForOneTime;
            _inputFileName = obj._inputFileName;
            _rejectLans = obj.RejectLans;
        }

        public LanInfo(Document doc, string inFileName, bool bNoCheckVer, bool bExtraFrameForOneTime, LytInfo lytInfo)
        {
            _inputFileName = inFileName;

            // 入力ファイル名のチェック
            if (! FileUtil.IsValidStringForFileName(Path.GetFileNameWithoutExtension(inFileName)))
            {
                throw new LayoutDataException(string.Format(Properties.Resources.ErrorInvalidAnimationFileName, inFileName));
            }

            if (doc == null)
            {
                throw new LayoutDataException(Properties.Resources.ErrorEmptyData);
            }

            if (! bNoCheckVer)
            {
                DataUtil.VerifyDocumentVersion(doc.version);
            }

            if (doc.body == null)
            {
                throw new LayoutDataException(Properties.Resources.ErrorEmptyData);
            }

            _bExtraFrameForOneTime = bExtraFrameForOneTime;
            _lytInfo = lytInfo;

            // LAN配列が1つも無い場合は、doc.body.flan は null になる。
            // その場合は、_rlanAry も null のままにしておく。
            if (doc.body.lan != null)
            {
                List<LAN> tmpLanList = new List<LAN>(doc.body.lan.Length);

                foreach (LAN lan in doc.body.lan)
                {
                    List<AnimContent> rejectContents = new List<AnimContent>();
                    if (! Triming(lan, ref rejectContents))
                    {
                        lan.animContent = rejectContents.ToArray();
                        _rejectLans.Add(lan);
                        continue;
                    }

                    tmpLanList.Add(lan);
                }

                if (tmpLanList.Count == doc.body.lan.Length)
                {
                    _lanAry = doc.body.lan;
                }
                else if (tmpLanList.Count > 0)
                {
                    _lanAry = new LAN[tmpLanList.Count];
                    tmpLanList.CopyTo(_lanAry);
                }
            }

            _animTagAry = doc.body.animTag;
            if (_animTagAry != null)
            {
                foreach (AnimTag animTag in _animTagAry)
                {
                    string fileName = animTag.GetFileName();
                    if ( fileName == null
                      || ! FileUtil.IsValidStringForFileName(fileName)
                    )
                    {
                        throw new LayoutDataException(string.Format(Properties.Resources.ErrorInvalidAnimSecTagName, fileName));
                    }

                    CalcFrameSize(animTag);

                    DataUtil.ResizeArray(ref animTag.group, UInt16.MaxValue);
                    foreach (GroupRef groupRef in animTag.group)
                    {
                        LytInfo.VerifyGroupName(groupRef.name);
                    }
                }
            }

            _animShare = doc.body.animShare;
        }

        public AnimShare[] AnimShare { get { return _animShare; } }

        public AnimTag[] AnimTagArray { get { return _animTagAry; } set { _animTagAry = value; } }

        public LAN[] LanArray { get { return _lanAry; } }

        public LytInfo GetLytInfo() { return _lytInfo; }

        public string InputFileName { get { return _inputFileName; } }

        public List<LAN> RejectLans { get { return _rejectLans; } }

        /// <summary>
        /// 外部からLanArrayを書き換えます。
        /// この関数はアニメーションコンバート時のみ使用されます。
        /// </summary>
        public void SetLanArray(LAN[] lanArray)
        {
            _lanAry = lanArray;
        }

        public AnimTag CreateAnimTagFromConvertFrameRange()
        {
            AnimTag animTag = new AnimTag();
            animTag.startFrame = _lanAry[0].convertStartFrame;
            animTag.endFrame = _lanAry[0].convertEndFrame;
            animTag.animLoop = _lanAry[0].animLoop;
            CalcFrameSize(animTag);
            animTag.group = new GroupRef[0];
            return animTag;
        }

        /// <summary>
        /// CalcFrameSize でのエラーメッセージを得る
        /// </summary>
        string GetCalcFrameErrorMessage_(AnimTag animTag)
        {
            return string.Format(
                Properties.Resources.ErrorIncorrectFrameRange,
                animTag.name, animTag.startFrame, animTag.endFrame, animTag.animLoop);
        }

        void CalcFrameSize(AnimTag animTag)
        {
            if (_bExtraFrameForOneTime)
            {
                if (animTag.animLoop == AnimLoopType.Loop)
                {
                    if (animTag.startFrame >= animTag.endFrame)
                    {
                        throw new LayoutDataException(GetCalcFrameErrorMessage_(animTag));
                    }
                    animTag.FrameSize = animTag.endFrame - animTag.startFrame;
                }
                else
                {
                    if (animTag.startFrame > animTag.endFrame)
                    {
                        throw new LayoutDataException(GetCalcFrameErrorMessage_(animTag));
                    }
                    animTag.FrameSize = animTag.endFrame - animTag.startFrame + 1;
                }
            }
            else
            {
                if (animTag.animLoop == AnimLoopType.Wait)
                {
                    if (animTag.startFrame != animTag.endFrame)
                    {
                        throw new LayoutDataException(GetCalcFrameErrorMessage_(animTag));
                    }
                }
                else
                {
                    if (animTag.startFrame >= animTag.endFrame)
                    {
                        throw new LayoutDataException(GetCalcFrameErrorMessage_(animTag));
                    }
                }
                animTag.FrameSize = animTag.endFrame - animTag.startFrame;
            }
        }

        /// <summary>
        /// 不要なデータを取り除きます。
        /// </summary>
        /// <returns>データが空出ない場合は true、空の場合は falseが返ります。</returns>
        bool Triming(LAN lan, ref List<AnimContent> rejects)
        {
            if (lan.animContent == null)
            {
                return false;
            }

            List<AnimContent> tmpAnimContents = new List<AnimContent>(lan.animContent.Length);

            foreach (AnimContent cont in lan.animContent)
            {
                List<AnimTarget> rejectTargets = new List<AnimTarget>();
                if (!Triming(cont, ref rejectTargets))
                {
                    cont.Items = rejectTargets.ToArray();
                    rejects.Add(cont);
                    continue;
                }
                tmpAnimContents.Add(cont);
            }

            if (tmpAnimContents.Count == 0)
            {
                lan.animContent = null;
            }
            else if (tmpAnimContents.Count != lan.animContent.Length)
            {
                lan.animContent = new AnimContent[tmpAnimContents.Count];
                tmpAnimContents.CopyTo(lan.animContent);
            }

            return lan.animContent != null;
        }

        /// <summary>
        /// 不要なデータを取り除きます。
        /// </summary>
        /// <returns>データが空出ない場合は true、空の場合は falseが返ります。</returns>
        bool Triming(AnimContent cont, ref List<AnimTarget> rejects)
        {
            if (cont.Items == null)
            {
                return false;
            }

            // contが関連付けられているマテリアルを取得する。
            Schema.Flyt.Material_CTR material = null;
            try
            {
                Schema.Flyt.Pane pane = _lytInfo.PaneDic[cont.name];
                if (pane.kind == Schema.Flyt.PaneKind.Picture)
                {
                    Schema.Flyt.Picture picture = (Schema.Flyt.Picture)pane.Item;
                    material = picture.materialCtr;
                }
                else if (pane.kind == Schema.Flyt.PaneKind.TextBox)
                {
                    Schema.Flyt.TextBox textBox = (Schema.Flyt.TextBox)pane.Item;
                    material = textBox.materialCtr;
                }
            }
            catch (System.Collections.Generic.KeyNotFoundException)
            {
                try
                {
                    // 見つからない場合はウィンドウのフレームの可能性があるため、後ろ二文字を削って探す
                    if(MergeAnimData.IsEndWithWindowMaterialSuffix(cont.name))
                    {
                        Schema.Flyt.Pane pane = _lytInfo.PaneDic[MergeAnimData.GetNameWithoutWindowMaterialSuffix(cont.name)];
                        if (pane.kind == Schema.Flyt.PaneKind.Window)
                        {
                            Schema.Flyt.Window window = (Schema.Flyt.Window)pane.Item;
                            if (window.useLTMaterial != null && window.useLTMaterial.value == true)
                            {
                                // 左上のマテリアルを使う設定
                                material = window.frame[0].materialCtr;
                            }
                            else
                            {
                                // そうでない場合は探す必要あり
                                foreach (Schema.Flyt.WindowFrame frame in window.frame)
                                {
                                    if (frame.materialCtr.name == cont.name)
                                    {
                                        material = frame.materialCtr;
                                        break;
                                    }
                                }
                                if (material == null)
                                {
                                    if (window.content.materialCtr.name == cont.name)
                                    {
                                        material = window.content.materialCtr;
                                    }
                                }
                            }
                        }
                    }
                }
                catch (System.Collections.Generic.KeyNotFoundException)
                {
                    // それでも見つからない場合はmaterialがnullのまま続行、ワーニングを出す
                    Report.Out.WriteLine(string.Format("cannot find target pane for {0}", cont.name));
                }
            }

            List<AnimTarget> tmpAnimTargets = new List<AnimTarget>(cont.Items.Length);

            foreach (AnimTarget target in cont.Items)
            {
                if (target is AnimPaneSRTTarget)
                {
                    switch (target.target)
                    {
                        case AnimTargetType.TranslateX:
                        case AnimTargetType.TranslateY:
                        case AnimTargetType.TranslateZ:
                        case AnimTargetType.RotateX:
                        case AnimTargetType.RotateY:
                        case AnimTargetType.RotateZ:
                        case AnimTargetType.ScaleX:
                        case AnimTargetType.ScaleY:
                        case AnimTargetType.SizeW:
                        case AnimTargetType.SizeH:
                            break;
                        default:
                            throw new LayoutDataException(string.Format(Properties.Resources.ErrorInvalidAnimTarget, target.target));
                    }
                }
                else if (target is AnimVisibilityTarget)
                {
                    switch (target.target)
                    {
                        case AnimTargetType.Visibility:
                            break;
                        default:
                            throw new LayoutDataException(string.Format(Properties.Resources.ErrorInvalidAnimTarget, target.target));
                    }
                }
                else if (target is AnimVertexColorTarget)
                {
                    switch (target.target)
                    {
                        case AnimTargetType.LT_r:
                        case AnimTargetType.LT_g:
                        case AnimTargetType.LT_b:
                        case AnimTargetType.LT_a:
                        case AnimTargetType.RT_r:
                        case AnimTargetType.RT_g:
                        case AnimTargetType.RT_b:
                        case AnimTargetType.RT_a:
                        case AnimTargetType.LB_r:
                        case AnimTargetType.LB_g:
                        case AnimTargetType.LB_b:
                        case AnimTargetType.LB_a:
                        case AnimTargetType.RB_r:
                        case AnimTargetType.RB_g:
                        case AnimTargetType.RB_b:
                        case AnimTargetType.RB_a:
                        case AnimTargetType.PaneAlpha:
                            break;
                        default:
                            throw new LayoutDataException(string.Format(Properties.Resources.ErrorInvalidAnimTarget, target.target));
                    }
                }
                else if (target is AnimMaterialColorTarget)
                {
                    switch (target.target)
                    {
                        case AnimTargetType.BlackColor_r:
                        case AnimTargetType.BlackColor_g:
                        case AnimTargetType.BlackColor_b:
                        case AnimTargetType.BlackColor_a:
                        case AnimTargetType.WhiteColor_r:
                        case AnimTargetType.WhiteColor_g:
                        case AnimTargetType.WhiteColor_b:
                        case AnimTargetType.WhiteColor_a:
                        // float 版
                        case AnimTargetType.BlackColorFloat_r:
                        case AnimTargetType.BlackColorFloat_g:
                        case AnimTargetType.BlackColorFloat_b:
                        case AnimTargetType.BlackColorFloat_a:
                        case AnimTargetType.WhiteColorFloat_r:
                        case AnimTargetType.WhiteColorFloat_g:
                        case AnimTargetType.WhiteColorFloat_b:
                        case AnimTargetType.WhiteColorFloat_a:
                            break;
                        case AnimTargetType.TevConstColor0_r:
                        case AnimTargetType.TevConstColor0_g:
                        case AnimTargetType.TevConstColor0_b:
                        case AnimTargetType.TevConstColor0_a:
                        case AnimTargetType.TevConstColor1_r:
                        case AnimTargetType.TevConstColor1_g:
                        case AnimTargetType.TevConstColor1_b:
                        case AnimTargetType.TevConstColor1_a:
                        case AnimTargetType.TevConstColor2_r:
                        case AnimTargetType.TevConstColor2_g:
                        case AnimTargetType.TevConstColor2_b:
                        case AnimTargetType.TevConstColor2_a:
                        case AnimTargetType.TevConstColor3_r:
                        case AnimTargetType.TevConstColor3_g:
                        case AnimTargetType.TevConstColor3_b:
                        case AnimTargetType.TevConstColor3_a:
                        case AnimTargetType.TevConstColor4_r:
                        case AnimTargetType.TevConstColor4_g:
                        case AnimTargetType.TevConstColor4_b:
                        case AnimTargetType.TevConstColor4_a:
                        // float 版
                        case AnimTargetType.TevConstColor0Float_r:
                        case AnimTargetType.TevConstColor0Float_g:
                        case AnimTargetType.TevConstColor0Float_b:
                        case AnimTargetType.TevConstColor0Float_a:
                        case AnimTargetType.TevConstColor1Float_r:
                        case AnimTargetType.TevConstColor1Float_g:
                        case AnimTargetType.TevConstColor1Float_b:
                        case AnimTargetType.TevConstColor1Float_a:
                        case AnimTargetType.TevConstColor2Float_r:
                        case AnimTargetType.TevConstColor2Float_g:
                        case AnimTargetType.TevConstColor2Float_b:
                        case AnimTargetType.TevConstColor2Float_a:
                        case AnimTargetType.TevConstColor3Float_r:
                        case AnimTargetType.TevConstColor3Float_g:
                        case AnimTargetType.TevConstColor3Float_b:
                        case AnimTargetType.TevConstColor3Float_a:
                        case AnimTargetType.TevConstColor4Float_r:
                        case AnimTargetType.TevConstColor4Float_g:
                        case AnimTargetType.TevConstColor4Float_b:
                        case AnimTargetType.TevConstColor4Float_a:

                            if (material == null || (!material.useLowLevelCombinerSettings && !material.useCombinerUserShaderSettings))
                            {
                                continue;
                            }
                            break;
                        case AnimTargetType.TevColor0_r:
                        case AnimTargetType.TevColor0_g:
                        case AnimTargetType.TevColor0_b:
                        case AnimTargetType.TevColor0_a:
                        case AnimTargetType.TevKonst0_r:
                        case AnimTargetType.TevKonst0_g:
                        case AnimTargetType.TevKonst0_b:
                        case AnimTargetType.TevKonst0_a:
                        case AnimTargetType.TevKonst1_r:
                        case AnimTargetType.TevKonst1_g:
                        case AnimTargetType.TevKonst1_b:
                        case AnimTargetType.TevKonst1_a:
                        case AnimTargetType.TevKonst2_r:
                        case AnimTargetType.TevKonst2_g:
                        case AnimTargetType.TevKonst2_b:
                        case AnimTargetType.TevKonst2_a:
                        case AnimTargetType.TevKonst3_r:
                        case AnimTargetType.TevKonst3_g:
                        case AnimTargetType.TevKonst3_b:
                        case AnimTargetType.TevKonst3_a:
                        case AnimTargetType.TevKonst4_r:
                        case AnimTargetType.TevKonst4_g:
                        case AnimTargetType.TevKonst4_b:
                        case AnimTargetType.TevKonst4_a:
                        case AnimTargetType.TevKonst5_r:
                        case AnimTargetType.TevKonst5_g:
                        case AnimTargetType.TevKonst5_b:
                        case AnimTargetType.TevKonst5_a:
                        // float 版
                        case AnimTargetType.TevColor0Float_r:
                        case AnimTargetType.TevColor0Float_g:
                        case AnimTargetType.TevColor0Float_b:
                        case AnimTargetType.TevColor0Float_a:
                        case AnimTargetType.TevKonst0Float_r:
                        case AnimTargetType.TevKonst0Float_g:
                        case AnimTargetType.TevKonst0Float_b:
                        case AnimTargetType.TevKonst0Float_a:
                        case AnimTargetType.TevKonst1Float_r:
                        case AnimTargetType.TevKonst1Float_g:
                        case AnimTargetType.TevKonst1Float_b:
                        case AnimTargetType.TevKonst1Float_a:
                        case AnimTargetType.TevKonst2Float_r:
                        case AnimTargetType.TevKonst2Float_g:
                        case AnimTargetType.TevKonst2Float_b:
                        case AnimTargetType.TevKonst2Float_a:
                        case AnimTargetType.TevKonst3Float_r:
                        case AnimTargetType.TevKonst3Float_g:
                        case AnimTargetType.TevKonst3Float_b:
                        case AnimTargetType.TevKonst3Float_a:
                        case AnimTargetType.TevKonst4Float_r:
                        case AnimTargetType.TevKonst4Float_g:
                        case AnimTargetType.TevKonst4Float_b:
                        case AnimTargetType.TevKonst4Float_a:
                        case AnimTargetType.TevKonst5Float_r:
                        case AnimTargetType.TevKonst5Float_g:
                        case AnimTargetType.TevKonst5Float_b:
                        case AnimTargetType.TevKonst5Float_a:

                        case AnimTargetType.TexBlendRatio0_c:
                        case AnimTargetType.TexBlendRatio1_c:
                        case AnimTargetType.TexBlendRatio2_c:
                            continue;
                        default:
                            throw new LayoutDataException(string.Format(Properties.Resources.ErrorInvalidAnimTarget, target.target));
                    }
                }
                else if (target is AnimTexSRTTarget)
                {
                    switch (target.target)
                    {
                        case AnimTargetType.TranslateS:
                        case AnimTargetType.TranslateT:
                        case AnimTargetType.Rotate:
                        case AnimTargetType.ScaleS:
                        case AnimTargetType.ScaleT:
                            break;
                        default:
                            throw new LayoutDataException(string.Format(Properties.Resources.ErrorInvalidAnimTarget, target.target));
                    }

                    if (material == null || material.texMap == null || target.id >= material.texMap.Length)
                    {
                        continue;
                    }
                }
                else if (target is AnimIndTexSRTTarget)
                {
                    switch (target.target)
                    {
                        case AnimTargetType.Rotate:
                        case AnimTargetType.ScaleS:
                        case AnimTargetType.ScaleT:
                            break;
                        default:
                            throw new LayoutDataException(string.Format(Properties.Resources.ErrorInvalidAnimTarget, target.target));
                    }

                    if (material == null || material.tevStage == null)
                    {
                        continue;
                    }
                    else
                    {
                        bool has_indirect = false;
                        foreach (Schema.Flyt.Material_CTRTevStage stage in material.tevStage)
                        {
                            if (stage.rgb.mode == Schema.Flyt.TevMode.Indirect || stage.rgb.mode == Schema.Flyt.TevMode.BlendIndirect || stage.rgb.mode == Schema.Flyt.TevMode.EachIndirect)
                            {
                                has_indirect = true;
                            }
                        }
                        if (!has_indirect)
                        {
                            continue;
                        }
                    }
                }
                else if (target is AnimTexPatternTarget)
                {
                    switch (target.target)
                    {
                        case AnimTargetType.Image:
                            break;
                        default:
                            throw new LayoutDataException(string.Format(Properties.Resources.ErrorInvalidAnimTarget, target.target));
                    }

                    if (material == null || target.id >= material.texMap.Length)
                    {
                        continue;
                    }
                }
                else if (target is AnimAlphaTestTarget)
                {
                    switch (target.target)
                    {
                        case AnimTargetType.AlphaTest:
                            break;
                        default:
                            throw new LayoutDataException(string.Format(Properties.Resources.ErrorInvalidAnimTarget, target.target));
                    }

                    if (material == null || material.useDefaultAlphaTestSettings)
                    {
                        continue;
                    }
                }
                else if (target is AnimFontShadowTarget)
                {
                    switch (target.target)
                    {
                        case AnimTargetType.FontShadowBlackColor_r:
                        case AnimTargetType.FontShadowBlackColor_g:
                        case AnimTargetType.FontShadowBlackColor_b:
                        case AnimTargetType.FontShadowWhiteColor_r:
                        case AnimTargetType.FontShadowWhiteColor_g:
                        case AnimTargetType.FontShadowWhiteColor_b:
                        case AnimTargetType.FontShadowWhiteColor_a:
                            break;
                        default:
                            throw new LayoutDataException(string.Format(Properties.Resources.ErrorInvalidAnimTarget, target.target));
                    }
                }
                else if (target is AnimPerCharacterTransformCurveTarget)
                {
                    switch (target.target)
                    {
                        case AnimTargetType.PerCharacterTransformScaleX:
                        case AnimTargetType.PerCharacterTransformScaleY:
                        case AnimTargetType.PerCharacterTransformTranslateX:
                        case AnimTargetType.PerCharacterTransformTranslateY:
                        case AnimTargetType.PerCharacterTransformTranslateZ:
                        case AnimTargetType.PerCharacterTransformRotateX:
                        case AnimTargetType.PerCharacterTransformRotateY:
                        case AnimTargetType.PerCharacterTransformRotateZ:
                        case AnimTargetType.PerCharacterTransformLT_r:
                        case AnimTargetType.PerCharacterTransformLT_g:
                        case AnimTargetType.PerCharacterTransformLT_b:
                        case AnimTargetType.PerCharacterTransformLT_a:
                        case AnimTargetType.PerCharacterTransformLB_r:
                        case AnimTargetType.PerCharacterTransformLB_g:
                        case AnimTargetType.PerCharacterTransformLB_b:
                        case AnimTargetType.PerCharacterTransformLB_a:
                            break;
                        default:
                            throw new LayoutDataException(string.Format(Properties.Resources.ErrorInvalidAnimTarget, target.target));
                    }
                }
                else if (target is AnimWindowTarget)
                {
                    switch (target.target)
                    {
                        case AnimTargetType.WindowFrameTop:
                        case AnimTargetType.WindowFrameBottom:
                        case AnimTargetType.WindowFrameLeft:
                        case AnimTargetType.WindowFrameRight:
                            break;
                        default:
                            throw new LayoutDataException(string.Format(Properties.Resources.ErrorInvalidAnimTarget, target.target));
                    }
                }
                else if (target is AnimExtUserDataTarget)
                {
                    // 想定している target type であることをあらわすため if 分だけ記述しておく。
                    // ないと想定していないデータと認識され ErrorInvalidAnimTarget 例外が発生する。
                }
                else if (target is AnimPerCharacterTransformEvalTarget)
                {
                    switch (target.target)
                    {
                        case AnimTargetType.PerCharacterTransformEvalTimeOffset:
                            break;
                        case AnimTargetType.PerCharacterTransformEvalTimeWidth:
                            break;
                        default:
                            throw new LayoutDataException(string.Format(Properties.Resources.ErrorInvalidAnimTarget, target.target));
                    }
                }
                else if (target is AnimMaskTarget)
                {
                    switch (target.target)
                    {
                        case AnimTargetType.MaskMtxScaleX:
                        case AnimTargetType.MaskMtxScaleY:
                        case AnimTargetType.MaskMtxRotate:
                        case AnimTargetType.MaskMtxTransX:
                        case AnimTargetType.MaskMtxTransY:
                            break;
                        default:
                            throw new LayoutDataException(string.Format(Properties.Resources.ErrorInvalidAnimTarget, target.target));
                    }
                }
                else if (target is AnimDropShadowTarget)
                {
                    switch (target.target)
                    {
                        case AnimTargetType.DropShadowColor_r:
                        case AnimTargetType.DropShadowColor_g:
                        case AnimTargetType.DropShadowColor_b:
                        case AnimTargetType.DropShadowColor_a:
                        case AnimTargetType.DropShadowAngle:
                        case AnimTargetType.DropShadowDistance:
                        case AnimTargetType.DropShadowSpread:
                        case AnimTargetType.DropShadowSize:
                        case AnimTargetType.OuterGlowColor_r:
                        case AnimTargetType.OuterGlowColor_g:
                        case AnimTargetType.OuterGlowColor_b:
                        case AnimTargetType.OuterGlowColor_a:
                        case AnimTargetType.OuterGlowSpread:
                        case AnimTargetType.OuterGlowSize:
                        case AnimTargetType.StrokeColor_r:
                        case AnimTargetType.StrokeColor_g:
                        case AnimTargetType.StrokeColor_b:
                        case AnimTargetType.StrokeColor_a:
                        case AnimTargetType.StrokeSize:
                            break;
                        default:
                            throw new LayoutDataException(string.Format(Properties.Resources.ErrorInvalidAnimTarget, target.target));
                    }
                }
                else if (target is AnimProceduralShapeTarget)
                {
                    switch (target.target)
                    {
                        case AnimTargetType.ProceduralShapeExp:
                        case AnimTargetType.ProceduralShapeRadius:
                        case AnimTargetType.ProceduralShapeInnerStrokeSize:
                        case AnimTargetType.ProceduralShapeInnerStrokeColor_r:
                        case AnimTargetType.ProceduralShapeInnerStrokeColor_g:
                        case AnimTargetType.ProceduralShapeInnerStrokeColor_b:
                        case AnimTargetType.ProceduralShapeInnerStrokeColor_a:
                        case AnimTargetType.ProceduralShapeInnerShadowColor_r:
                        case AnimTargetType.ProceduralShapeInnerShadowColor_g:
                        case AnimTargetType.ProceduralShapeInnerShadowColor_b:
                        case AnimTargetType.ProceduralShapeInnerShadowColor_a:
                        case AnimTargetType.ProceduralShapeInnerShadowAngle:
                        case AnimTargetType.ProceduralShapeInnerShadowDistance:
                        case AnimTargetType.ProceduralShapeInnerShadowSize:
                        case AnimTargetType.ProceduralShapeColorOverlayColor_r:
                        case AnimTargetType.ProceduralShapeColorOverlayColor_g:
                        case AnimTargetType.ProceduralShapeColorOverlayColor_b:
                        case AnimTargetType.ProceduralShapeColorOverlayColor_a:
                        case AnimTargetType.ProceduralShapeGradationOverlayControl0:
                        case AnimTargetType.ProceduralShapeGradationOverlayControl1:
                        case AnimTargetType.ProceduralShapeGradationOverlayControl2:
                        case AnimTargetType.ProceduralShapeGradationOverlayControl3:
                        case AnimTargetType.ProceduralShapeGradationOverlayColor0_r:
                        case AnimTargetType.ProceduralShapeGradationOverlayColor0_g:
                        case AnimTargetType.ProceduralShapeGradationOverlayColor0_b:
                        case AnimTargetType.ProceduralShapeGradationOverlayColor0_a:
                        case AnimTargetType.ProceduralShapeGradationOverlayColor1_r:
                        case AnimTargetType.ProceduralShapeGradationOverlayColor1_g:
                        case AnimTargetType.ProceduralShapeGradationOverlayColor1_b:
                        case AnimTargetType.ProceduralShapeGradationOverlayColor1_a:
                        case AnimTargetType.ProceduralShapeGradationOverlayColor2_r:
                        case AnimTargetType.ProceduralShapeGradationOverlayColor2_g:
                        case AnimTargetType.ProceduralShapeGradationOverlayColor2_b:
                        case AnimTargetType.ProceduralShapeGradationOverlayColor2_a:
                        case AnimTargetType.ProceduralShapeGradationOverlayColor3_r:
                        case AnimTargetType.ProceduralShapeGradationOverlayColor3_g:
                        case AnimTargetType.ProceduralShapeGradationOverlayColor3_b:
                        case AnimTargetType.ProceduralShapeGradationOverlayColor3_a:
                        case AnimTargetType.ProceduralShapeGradationOverlayAngle:
                        case AnimTargetType.ProceduralShapeOuterShadowColor_r:
                        case AnimTargetType.ProceduralShapeOuterShadowColor_g:
                        case AnimTargetType.ProceduralShapeOuterShadowColor_b:
                        case AnimTargetType.ProceduralShapeOuterShadowColor_a:
                        case AnimTargetType.ProceduralShapeOuterShadowAngle:
                        case AnimTargetType.ProceduralShapeOuterShadowDistance:
                        case AnimTargetType.ProceduralShapeOuterShadowSize:
                            break;
                        default:
                            throw new LayoutDataException(string.Format(Properties.Resources.ErrorInvalidAnimTarget, target.target));
                    }

                }
                else
                {
                    // 未知のtareget type
                    throw new LayoutDataException(string.Format(Properties.Resources.ErrorInvalidAnimTarget, target));
                }

                // キーを1つ以上ある場合、もしくはパラメタライズドアニメーションの場合
                if (target.key != null || target.parameterizedAnimParameter != null)
                {
                    tmpAnimTargets.Add(target);
                }
                else
                {
                    rejects.Add(target);
                }
            }

            if (tmpAnimTargets.Count == 0)
            {
                cont.Items = null;
            }
            else if (tmpAnimTargets.Count != cont.Items.Length)
            {
                cont.Items = new AnimTarget[tmpAnimTargets.Count];
                tmpAnimTargets.CopyTo(cont.Items);
            }

            return cont.Items != null;
        }

    }
}
