﻿// --------------------------------------------------------------------------------
// <copyright>
// Copyright (C)Nintendo. All rights reserved.
//
// These coded instructions, statements, and computer programs contain proprietary
// information of Nintendo and/or its licensed developers and are protected by
// national and international copyright laws. They may not be disclosed to third
// parties or copied or duplicated in any form, in whole or in part, without the
// prior written consent of Nintendo.
//
// The content herein is highly confidential and should be handled accordingly.
// </copyright>

// --------------------------------------------------------------------------------
using System;
using System.Collections.Generic;
using System.Text;
using System.IO;
using System.Linq;
using Nintendo.Foundation.IO;

namespace NW4F.LayoutBinaryConverter
{
    public enum BinaryConverterErrorType
    {
        Success,
        InternalError,
        ParameterError,
        FileIOError,
        ConvertError,
        BannerConvertError,

        MAX
    }

    public partial class BinaryConverter
    {
        public BinaryConverter()
        {
        }

        public BinaryConverterErrorType Convert(string[] strArgs)
        {
            CommandLineParser args = null;
            try
            {
                var setting = new CommandLineParserSettings()
                {
                    ApplicationDescription = Properties.Resources.ApplicationDescription,
                    // 例外処理に任せる
                    ErrorWriter = x => { },
                };

                if (!new Nintendo.Foundation.IO.CommandLineParser(setting).ParseArgs(strArgs, out args))
                {
                    return BinaryConverterErrorType.Success;
                }

                args.ValidateAndConvertArguments();

                // --platform オプションは削除予定です。
                if (!string.IsNullOrEmpty(args.TargetPlatformName))
                {
                    // ユーザーに廃止予定を警告します。
                    Report.Err.WriteLine(Properties.Resources.ErrorPlatformOptionObsolete);
                }

                // --api-type オプション、--code-type オプションは削除予定です。
                if (!string.IsNullOrEmpty(args.TargetApiTypeName) || !string.IsNullOrEmpty(args.TargetCodeTypeName))
                {
                    // ユーザーに廃止予定を警告します。
                    Report.Err.WriteLine(Properties.Resources.ErrorApiTypeAndCodeTypeOptionObsolete);
                }
            }
            catch (Exception ex)
            {
                // ちなみに ToolFoundation が投げる例外の型は internal なので、
                // ex の型による分岐はやりづらい。
                CommandLineParser.PrintProductInfo();
                Report.Err.WriteLine(ex.Message);
                return BinaryConverterErrorType.ParameterError;
            }

            if (args.IsOutLogFile)
            {
                Report.StartLogMode();
            }

            if (args.IsQuiet)
            {
                Report.SetQuiet();
            }
            CommandLineParser.PrintProductInfo();

            // 出力ディレクトリが存在するか?
            if (! Directory.Exists(args.outDirName))
            {
                Report.Err.WriteLine(string.Format("not found directory. - {0}", args.outDirName));
                return BinaryConverterErrorType.FileIOError;
            }

            // キャッシュディレクトリが存在するか？
            if (!string.IsNullOrEmpty(args.FtxbCacheDirectory) && !Directory.Exists(args.FtxbCacheDirectory))
            {
                Report.Err.WriteLine(string.Format("not found cache directory. - {0}", args.FtxbCacheDirectory));
                return BinaryConverterErrorType.FileIOError;
            }

            // 更新モードの場合はテンポラリフォルダを作り、そこに一旦出力する
            String tempDirName = string.Empty;
            if (args.IsUpdateMode)
            {
                tempDirName = FileUtil.CreateTempDirectory();
            }

            LayoutFileLoader layoutFileLoader = new LayoutFileLoader(args.IsBannerCompatible, args.IsNoCheckVersion, args.IsXsdValidate, args.IsExtraFrameForOneTime);
            DocumentManager docMan = new DocumentManager(layoutFileLoader);

            LytWriter rlytWriter = new LytWriter(
                args.IsImageFileWriteTimeSameOriginal,
                !args.IsNoConvertTagChar,
                args.IsDiscardTextboxText,
                args.IsBannerCompatible,
                !args.IsConvertAllTex,
                args.IsNoCopyFont,
                args.IsDegammaTexture,
                args.IsDegammaParameter,
                args.IsEnableViewerHidden,
                args.IsNoCopyFfnt,
                args.FtxbCacheDirectory,
                args.IsCompatible_0_12_x,
                args.ShaderEnvDirectories,
                args.CombinerEditorPath,
                args.IsEcmbEmptyErr,
                args.IsKeepingFontScaleEnabled);

            // プラットフォーム毎のテクスチャ書き出しクラスを初期化します。
            bool textureWriterInitResult = TgaUtil.InitializeTexttureWriter("Generic", args.IsQuiet, args.IsGpuEncodigEnabled);
            if (!textureWriterInitResult)
            {
                Report.Err.WriteLine(string.Format("Can't initialize the texture converter. platformName - {0}", args.TargetPlatformName));
                return BinaryConverterErrorType.ParameterError;
            }

            TgaUtil.SkipIfSrcTextureNotExist = args.IsSkipIfSrcTextureNotExist;
            var optimizeInfoList = new List<LytInfo.FillrateOptimizeInformation>();

            for (int i = 0; i < args.inFileNames.Length; ++i)
            {
                bool bInDirPath = false;
                string[] inFileNames;

                try
                {
                    // フィルレート最適化情報を取得するときは入力ディレクトリ以下のすべてのレイアウトデータについて情報を出力したいため再帰的にレイアウトファイルを検索する。
                    inFileNames = FileUtil.ExpandInPath(args.inFileNames[i], args.IsOutputOptimizationStatistics, out bInDirPath);
                }
                catch (LayoutConverterException ex)
                {
                    Report.Err.WriteLine(ex.Message);
                    return BinaryConverterErrorType.FileIOError;
                }
                catch (IOException ex)
                {
                    Report.Err.WriteLine(ex.ToString());
                    return BinaryConverterErrorType.FileIOError;
                }

                foreach (string inFileName in inFileNames)
                {
                    try
                    {
                        FileType fileType = FileUtil.GetFileType(inFileName);
                        if (fileType == FileType.unknown)
                        {
                            if (bInDirPath)
                            {
                                continue;
                            }
                            else
                            {
                                Report.Err.WriteLine(string.Format("unknown file extension. - {0}", inFileName));
                                return BinaryConverterErrorType.FileIOError;
                            }
                        }

                        Report.Out.WriteLine(string.Format("\"{0}\" -> \"{1}\"", inFileName, args.outDirName));

                        // 絶対パスに変換
                        string fullFileName = Path.GetFullPath(inFileName);

                        string outDirName = args.IsUpdateMode ? tempDirName : Path.GetFullPath(args.outDirName);

                        // 最適化ログを出力する場合は不要なファイルが出力されないように flyt だけ処理する。
                        if (args.IsOutputOptimizationStatistics &&
                            fileType != FileType.flyt)
                        {
                            continue;
                        }

                        switch (fileType)
                        {
                        case FileType.flyt:

                            LytInfo lytInfo = docMan.GetRlyt(fullFileName);
                            {
                                string animPath = FileUtil.GetAnimFileName(fullFileName);
                                if (File.Exists(animPath))
                                {
                                    lytInfo.LanInfo = new LanInfo(docMan.GetRlan(animPath));
                                    if(!args.IsNoOutputLayoutTagNameExport)
                                    {
                                        lytInfo.MakeAnimationTagNameSystemExtUserData();
                                    }
                                }
                            }

                            // コンバート元のテクスチャを読み込む。
                            // フィルレート最適化処理のためにコンバートに先立って読み込んでおく。
                            var srcFiles = TgaUtil.LoadTextureSourceFiles(
                                fullFileName,
                                !args.IsConvertAllTex ? lytInfo.RefTextureFileList.Values : lytInfo.TextureFileList.Values,
                                args.IsDegammaTexture);

                            // システム用拡張ユーザーデータを出力するために内部データを作成します。
                            // flyt を読み込んだ後、書き出す前に呼び出します。
                            lytInfo.MakeSystemExtUserData(args.IsDegammaParameter);

                            // 最適化が指定されていたらピクチャペインのフィルレート最適化フラグを設定する。
                            string fillOptMode = args.FillOptimizationMode;
                            if (fillOptMode != null ||
                                args.IsOutputOptimizationStatistics)
                            {

                                LytInfo.FillrateOptimizationMode mode = LytInfo.FillrateOptimizationMode.Auto;

                                if (fillOptMode != null)
                                {
                                    switch (fillOptMode.ToLower())
                                    {
                                        case "aabb":
                                            mode = LytInfo.FillrateOptimizationMode.AABB;
                                            break;
                                        case "obb":
                                            mode = LytInfo.FillrateOptimizationMode.OBB;
                                            break;
                                    }
                                }

                                int margin = 0;
                                if (args.FillOptimizationMargin != null)
                                {
                                    try
                                    {
                                        margin = Int32.Parse(args.FillOptimizationMargin);
                                        if (margin < 0)
                                        {
                                            margin = 0;
                                        }
                                    }
                                    catch (FormatException)
                                    {
                                        Report.Err.WriteLine("fill-optimization-margin number was not correct number format.");
                                    }
                                }

                                var optInfo = lytInfo.FillrateOptimization(srcFiles, mode, margin);
                                if (args.IsOutputOptimizationStatistics)
                                {
                                    //  最適化情報出力モード時はデータを解析して情報を出力する。
                                    //  コンバート処理は行わない。
                                    optimizeInfoList.Add(optInfo);
                                    continue;
                                }
                            }

                            // textureファイルの変換
                            // レイアウトの変換よりも先に行います。(テクスチャ書き出し時に取得した情報を利用するため)
                            if(!TgaUtil.GenTextureFiles(
                                srcFiles,
                                outDirName,
                                args.FtxbCacheDirectory,
                                args.IsDegammaTexture))
                                {
                                    return BinaryConverterErrorType.ConvertError;
                                }

                            rlytWriter.Write(outDirName, fullFileName, lytInfo, args.TargetTileMode, args.IsQuiet);
                            break;

                        case FileType.flan:
                        case FileType.clpa:
                        case FileType.clvi:
                        case FileType.clvc:
                        case FileType.clmc:
                        case FileType.clts:
                        case FileType.cltp:
                            {
                                LanInfo lanInfo = new LanInfo(docMan.GetRlan(fullFileName));

                                // 結合された状態のLANを書き出し
                                bool IsTagSplitAndWhole = args.IsTagSplitAndWhole;
                                LanWriter rlanWriter = null;
                                Dictionary<string, Schema.Flan.RefRes> cvtRefTexDic = null;
                                List<Schema.Flan.AnimTag> animTagList = new List<Schema.Flan.AnimTag>();
                                {
                                    // AnimTagの子供にLANが存在する場合(=分割モード)、
                                    // タグ毎の出力をやめる
                                    if (lanInfo.AnimTagArray != null)
                                    {
                                        // 分割モードの場合AnimTagごとにLANを持っているので集める
                                        foreach (Schema.Flan.AnimTag animTag in lanInfo.AnimTagArray)
                                        {
                                            if (animTag.lan != null)
                                            {
                                                animTagList.Add(animTag);
                                                IsTagSplitAndWhole = false;
                                            }
                                        }
                                    }

                                    rlanWriter = new LanWriter(args.IsTagSplit, IsTagSplitAndWhole, args.IsNoOutTagInfo, args.IsOmitSameKey, args.IsOmitSameKeyAll, args.IsOmitNoKeyAll, args.IsBakeInfinityAreaKey, args.IsConvertAllAnimContent, args.IsAllowedNoGroupAnimTag, args.IsDegammaTexture, args.IsDegammaParameter, args.IsDegammaParameterOldBug, args.IsBannerCompatible, args.FtxbCacheDirectory, args.IsCompatible_0_12_x);
                                    cvtRefTexDic = rlanWriter.Write(
                                        outDirName,
                                        fullFileName,
                                        lanInfo,
                                        0
                                    );
                                }

                                // 分割された状態のLANを書き出し
                                // 分割モードで作成されたファイルの場合のみ実行される
                                // 「キーフレーム区間以降の設定」をタグ区間ごとに適応するために利用します
                                {
                                    lanInfo.AnimTagArray = null;
                                    int outputAnimTagIndexOffset = 0;
                                    foreach (Schema.Flan.AnimTag animTag in animTagList)
                                    {
                                        List<Schema.Flan.AnimTag> newAnimTagList = new List<Schema.Flan.AnimTag>();
                                        newAnimTagList.Add(animTag);
                                        lanInfo.AnimTagArray = newAnimTagList.ToArray();
                                        lanInfo.SetLanArray(animTag.lan);

                                        LanWriter rlanWriterSub = new LanWriter(true, false, args.IsNoOutTagInfo, args.IsOmitSameKey, args.IsOmitSameKeyAll, args.IsOmitNoKeyAll, args.IsBakeInfinityAreaKey, args.IsConvertAllAnimContent, args.IsAllowedNoGroupAnimTag, args.IsDegammaTexture, args.IsDegammaParameter, args.IsDegammaParameterOldBug, args.IsBannerCompatible, args.FtxbCacheDirectory, args.IsCompatible_0_12_x);
                                        rlanWriterSub.Write(
                                            outDirName,
                                            fullFileName,
                                            lanInfo,
                                            outputAnimTagIndexOffset
                                        );
                                        outputAnimTagIndexOffset++;
                                    }
                                }

                                if ( cvtRefTexDic.Count > 0 )   // 実際に使用されているテクスチャのみ返還する
                                {
                                    // rlytのテクスチャリストを元にテクスチャパターンアニメーションで使用するテクスチャを変換する。
                                    // この段階でflanと対応するrlytがロードできなければエラーとなる。
                                    if(!rlanWriter.GenTexPatternAnimTexFiles(outDirName,cvtRefTexDic.Keys,lanInfo.GetLytInfo()))
                                    {
                                        return BinaryConverterErrorType.ConvertError;
                                    }
                                }
                            }
                            break;

                        case FileType.fcpx:
                            {
                                // fcpx が直接指定された場合に bfcpx を出力する
                                Schema.Flyt.FontFile fontFile = new Schema.Flyt.FontFile();
                                fontFile.path = fullFileName;
                                IEnumerable<Schema.Flyt.FontFile> fontFiles = new Schema.Flyt.FontFile[] { fontFile };
                                rlytWriter.WriteComplexFont(outDirName, fullFileName, fontFiles, args.TargetTileMode, args.IsQuiet);
                            }
                            break;
                        }
                    }
                    catch (BannerConvertException ex)
                    {
                        Report.Err.WriteLine(ex.Message);
                        return BinaryConverterErrorType.BannerConvertError;
                    }
                    catch (LayoutConverterException ex)
                    {
                        Report.Err.WriteLine(ex.Message);
                        return BinaryConverterErrorType.ConvertError;
                    }
                    catch (FileNotFoundException ex)
                    {
                        Report.Err.WriteLine(ex.Message);
                        return BinaryConverterErrorType.ConvertError;
                    }
                    catch (ApplicationException ex)
                    {
                        Report.Err.WriteLine(ex.ToString());
                        return BinaryConverterErrorType.ConvertError;
                    }
                    catch (IOException ex)
                    {
                        Report.Err.WriteLine(ex.ToString());
                        return BinaryConverterErrorType.ConvertError;
                    }
                    catch (UnauthorizedAccessException ex)
                    {
                        Report.Err.WriteLine(ex.ToString());
                        return BinaryConverterErrorType.ConvertError;
                    }
                }
            }

            if (args.IsUpdateMode)
            {
                // 内容が異なるファイルのみ移動
                FileUtil.MoveUpdateFiles(Path.GetFullPath(args.outDirName), tempDirName);
                Directory.Delete(tempDirName, true);
            }

            if (args.IsOutputOptimizationStatistics)
            {
                OutputFillrateOptimizeInformation(optimizeInfoList, args.inFileNames[0], args.IsHideIdentifyInformationInStatistics);
            }

            return BinaryConverterErrorType.Success;
        }
    }
}
