﻿// --------------------------------------------------------------------------------
// <copyright>
// Copyright (C)Nintendo. All rights reserved.
//
// These coded instructions, statements, and computer programs contain proprietary
// information of Nintendo and/or its licensed developers and are protected by
// national and international copyright laws. They may not be disclosed to third
// parties or copied or duplicated in any form, in whole or in part, without the
// prior written consent of Nintendo.
//
// The content herein is highly confidential and should be handled accordingly.
// </copyright>
// --------------------------------------------------------------------------------
using System;
using System.Collections;
using System.Collections.Generic;
using System.Linq;

namespace NW4F.LayoutBinaryConverter
{
    using Schema.Flan;

    /// <summary>
    /// flanに含まれるアニメーションの形式を変換するクラスです。
    /// </summary>
    public class AnimConverter
    {
        const int _sectionMargin = 10;

        LanInfo _lanInfo = null;

        /// <summary>
        /// コンストラクタです。
        /// </summary>
        public AnimConverter(object lanInfo)
        {
            _lanInfo = lanInfo as LanInfo;
        }

        /// <summary>
        /// アニメーションの変換を行ないます。（複数→単数）
        /// </summary>
        public void ConvertMultiToSingle()
        {
            if (!HasMultiAnimation_() ||
                _lanInfo.AnimTagArray == null ||
                _lanInfo.AnimTagArray.Count() <= 0)
            {
                return;
            }

            // 出力設定を変更（カーブが無い場合は出力OFFにする）
            CheckAndDisableOutputCurves_(_lanInfo.AnimTagArray);

            // テクスチャパターンアニメをマージする
            MergeTexturePatternAnim(_lanInfo);

            // アトリビュートを抽出
            var attrList = GetAttributeList_(_lanInfo.AnimTagArray);

            // range,marginの算出
            var maxRangeList = CalcSectionMaxRange_(attrList, _lanInfo.AnimTagArray);
            var maxMarginList = CalcSectionMaxMargin_(attrList, _lanInfo.AnimTagArray);

            // アニメーションのコンバートを行う
            ConvertMultiToSingle_(_lanInfo, attrList, maxRangeList, maxMarginList);

            // 区間タグの開始終了位置変更
            var sectionParam = CalcSectionParam(_lanInfo.AnimTagArray, maxRangeList, maxMarginList);
            ModifyAnimSectionTag_(_lanInfo.AnimTagArray, sectionParam);
        }

        /// <summary>
        /// アニメーションの変換を行ないます。（複数→単数）
        /// </summary>
        private void ConvertMultiToSingle_(
            LanInfo lanInfo,
            AnmAttribute[] attrList,
            Dictionary<string, int> rangeList,
            Dictionary<string, Tuple<int, int>> marginList)
        {
            // AnimType、AnimContentName、AnimTargetを洗い出し、
            // LAN、AnimContent、AnimTargetのインスタンスを生成する
            Dictionary<AnimationType, LAN> lanList = new Dictionary<AnimationType, LAN>();

            foreach (AnmAttribute attr in attrList)
            {
                // LAN
                LAN newLan;
                if (!lanList.TryGetValue(attr.animType, out newLan))
                {
                    newLan = new LAN();

                    newLan.animContent = null;
                    newLan.animType = attr.animType;
                    newLan.startFrame = attr.lan.startFrame;
                    newLan.endFrame = attr.lan.endFrame;
                    newLan.convertStartFrame = attr.lan.convertStartFrame;
                    newLan.convertEndFrame = attr.lan.convertEndFrame;
                    newLan.animLoop = attr.lan.animLoop;

                    lanList[attr.animType] = newLan;
                }

                // AnimContent
                AnimContent newContent;
                {
                    var hasLan = newLan.animContent?.Where(a => a.name == attr.animContent.name);
                    if (hasLan == null || hasLan.Count() <= 0)
                    {
                        newContent = new AnimContent();

                        newContent.Items = null;
                        newContent.name = attr.animContent.name;
                        newContent.extUserDataTargetName = attr.animContent.extUserDataTargetName;
                        newContent.inactive = attr.animContent.inactive;

                        // LAN.AnimContentの設定
                        List<AnimContent> addList = new List<AnimContent>();
                        if (newLan.animContent != null)
                        {
                            foreach (AnimContent content in newLan.animContent)
                            {
                                addList.Add(content);
                            }
                        }
                        addList.Add(newContent);
                        newLan.animContent = addList.ToArray();
                    }
                    else
                    {
                        newContent = hasLan.First();
                    }
                }

                // AnimTarget
                AnimTarget newTarget;
                {
                    var hasContent = newContent.Items?.Where(t => t.target == attr.animTarget.target);
                    if (hasContent == null || hasContent.Count() <= 0)
                    {
                        List<Hermite> keys = new List<Hermite>();
                        newTarget = attr.animTarget.Duplicate(keys.ToArray()); // 空のキーで複製

                        // AnimContent.Itemsの設定
                        List<AnimTarget> addList = new List<AnimTarget>();
                        if (newContent.Items != null)
                        {
                            foreach (AnimTarget target in newContent.Items)
                            {
                                addList.Add(target);
                            }
                        }

                        addList.Add(newTarget);
                        newContent.Items = addList.ToArray();
                    }
                    else
                    {
                        hasContent.First().postInfinityType = attr.animTarget.postInfinityType;
                    }
                }
            }

            // セット毎に仕分ける
            // animType、AnimContantName、AnimTargetTypeが同一のAnmAttributeでグルーピングされる
            Dictionary<Tuple<AnimationType, string, AnimTargetType>, List<AnmAttribute>> curveSet =
                new Dictionary<Tuple<AnimationType, string, AnimTargetType>, List<AnmAttribute>>();

            foreach (AnmAttribute attr in attrList)
            {
                var key = new Tuple<AnimationType, string, AnimTargetType>(attr.animType, attr.contentName, attr.animTargetType);

                List<AnmAttribute> list;
                if (!curveSet.TryGetValue(key, out list))
                {
                    list = new List<AnmAttribute>();
                    curveSet.Add(key, list);
                }

                list.Add(attr);
            }

            // 構築
            foreach (var curve in curveSet)
            {
                var animType = curve.Key.Item1;
                var contentName = curve.Key.Item2;
                var targetType = curve.Key.Item3;

                int offset = 0;
                List<Hermite> newKeys = new List<Hermite>();
                foreach (AnimTag animTag in lanInfo.AnimTagArray ?? Enumerable.Empty<AnimTag>())
                {
                    // maxRangeが無い場合はスキップ
                    int maxRange;
                    if (!rangeList.TryGetValue(animTag.name, out maxRange))
                    {
                        continue;
                    }

                    // marginが無い場合はスキップ
                    Tuple<int, int> margin;
                    if (!marginList.TryGetValue(animTag.name, out margin))
                    {
                        continue;
                    }

                    // タグ名で一致するAnmAttributeを参照する
                    var targetAttr = curve.Value.Where(attr => attr.tagName == animTag.name).FirstOrDefault();

                    if (targetAttr != null)
                    {
                        foreach (Hermite key in targetAttr.keys ?? Enumerable.Empty<Hermite>())
                        {
                            // オフセット更新
                            key.frame += offset + margin.Item1;

                            // キーを結合する
                            CombinKeys_(newKeys, key);
                        }
                    }

                    // オフセット更新
                    offset += maxRange + _sectionMargin; // セクション幅 + マージン

                    // 参照カーブの場合は即break
                    if (targetAttr != null)
                    {
                        if (IsReferenceCurve(targetAttr.animType))
                        {
                            break;
                        }
                    }
                }

                // AnimTargetにキーをセットする
                {
                    AnimTarget target = GetAnimTarget_(lanList, animType, contentName, targetType);
                    newKeys.Sort(new HermiteComparer()); // ソート
                    target.key = newKeys.ToArray();
                }
            }

            // flanを書き換える
            List<LAN> lanArray = new List<LAN>();
            foreach (LAN lan in lanList.Values)
            {
                lanArray.Add(lan);
            }
            lanInfo.SetLanArray(lanArray.ToArray());
        }

        /// <summary>
        /// 参照カーブかどうかを判定します。
        /// </summary>
        private bool IsReferenceCurve(AnimationType type)
        {
            return type == AnimationType.PerCharacterTransformCurve;
        }

        /// <summary>
        /// アニメーション区間タグのパラメータを変更します。
        /// </summary>
        public Dictionary<string, Tuple<int, int>> CalcSectionParam(
            AnimTag[] animTags,
            Dictionary<string, int> rangeList,
            Dictionary<string, Tuple<int, int>> marginList)
        {
            Dictionary<string, Tuple<int, int>> ret = new Dictionary<string, Tuple<int, int>>();
            int startFrame = 0;
            int endFrame = 0;
            foreach (AnimTag animTag in animTags)
            {
                if (rangeList.ContainsKey(animTag.name) && marginList.ContainsKey(animTag.name))
                {
                    int sectionRange = (rangeList[animTag.name] <= 1) ? rangeList[animTag.name] : animTag.endFrame - animTag.startFrame;

                    startFrame += marginList[animTag.name].Item1;
                    endFrame = startFrame + sectionRange;

                    ret.Add(animTag.name, new Tuple<int, int>(startFrame, endFrame));

                    startFrame = endFrame + marginList[animTag.name].Item2 + _sectionMargin;
                }
            }

            return ret;
        }

        /// <summary>
        /// アニメーション区間タグのパラメータを変更します。
        /// </summary>
        private void ModifyAnimSectionTag_(
            AnimTag[] animTags,
            Dictionary<string, Tuple<int, int>> sectionParam)
        {
            foreach (AnimTag animTag in animTags)
            {
                Tuple<int, int> param;
                if (sectionParam.TryGetValue(animTag.name, out param))
                {
                    // パラメータ設定
                    {
                        animTag.startFrame = param.Item1;
                        animTag.endFrame = param.Item2;
                    }
                }
            }
        }

        /// <summary>
        /// アニメーションを複数保持しているか。
        /// </summary>
        private bool HasMultiAnimation_()
        {
            if (_lanInfo.AnimTagArray != null)
            {
                foreach (AnimTag animTag in _lanInfo.AnimTagArray)
                {
                    if (animTag.lan != null && animTag.lan.Count() > 0)
                    {
                        return true;
                    }
                }
            }

            return false;
        }

        /// <summary>
        /// テクスチャパターンアニメをマージします。
        /// </summary>
        private void MergeTexturePatternAnim(LanInfo lanInfo)
        {
            // flanの読み込み時にrejectされたパターンアニメの情報をマージする
            foreach (LAN srcLan in lanInfo.RejectLans ?? Enumerable.Empty<LAN>())
            {
                foreach (AnimContent srcContent in srcLan.animContent ?? Enumerable.Empty<AnimContent>())
                {
                    foreach (AnimTarget srcTarget in srcContent.Items ?? Enumerable.Empty<AnimTarget>())
                    {
                        foreach (AnimTag animTag in lanInfo.AnimTagArray ?? Enumerable.Empty<AnimTag>())
                        {
                            // 同一種のAnimTargetを探す
                            foreach (LAN dstLan in animTag.lan ?? Enumerable.Empty<LAN>())
                            {
                                if (srcLan.animType != dstLan.animType)
                                {
                                    continue;
                                }

                                foreach (AnimContent dstContent in dstLan.animContent ?? Enumerable.Empty<AnimContent>())
                                {
                                    if (srcContent.name != dstContent.name)
                                    {
                                        continue;
                                    }

                                    foreach (AnimTarget dstTarget in dstContent.Items ?? Enumerable.Empty<AnimTarget>())
                                    {
                                        if (srcTarget.target != dstTarget.target)
                                        {
                                            continue;
                                        }

                                        dstTarget.refRes = srcTarget.refRes;
                                    }
                                }
                            }
                        }
                    }
                }
            }
        }

        /// <summary>
        /// アトリビュートを抽出します。
        /// </summary>
        private AnmAttribute[] GetAttributeList_(AnimTag[] animTagArray)
        {
            List<AnmAttribute> anmAttributeList = new List<AnmAttribute>();

            foreach (AnimTag animTag in animTagArray)
            {
                foreach (LAN lan in animTag.lan ?? Enumerable.Empty<LAN>())
                {
                    foreach (AnimContent animContent in lan.animContent ?? Enumerable.Empty<AnimContent>())
                    {
                        foreach (AnimTarget animTarget in animContent.Items ?? Enumerable.Empty<AnimTarget>())
                        {
                            anmAttributeList.Add(new AnmAttribute(
                                animTag.name,
                                lan,
                                animContent,
                                lan.animType,
                                animContent.name,
                                animTarget,
                                animTarget.target,
                                animTarget.key));
                        }
                    }
                }

                //animTag.lan = null; // lanを削除
            }

            return anmAttributeList.ToArray();
        }

        /// <summary>
        /// タグ区間にキーがひとつも無い場合、カーブ出力を無効に設定します。
        /// </summary>
        private void CheckAndDisableOutputCurves_(AnimTag[] animTagArray)
        {
            foreach (AnimTag animTag in animTagArray)
            {
                if (animTag.lan == null)
                {
                    DisableOutputCurves_(animTag);
                }
            }
        }

        /// <summary>
        /// カーブ出力を無効に設定します。
        /// </summary>
        private void DisableOutputCurves_(AnimTag animTag)
        {
            animTag.outputPaneSRT = false;
            animTag.outputVisibility = false;
            animTag.outputVertexColor = false;
            animTag.outputMaterialColor = false;
            animTag.outputTextureSRT = false;
            animTag.outputTexturePattern = false;
            animTag.outputIndTextureSRT = false;
            animTag.outputAlphaTest = false;
            animTag.outputPerCharacterTransform = false;
            animTag.outputWindow = false;
            animTag.outputExtUserData = false;
            animTag.emptyKeyOutputEnabled = false;
            animTag.outputDropShadow = false;
            animTag.outputMaskTextureSRT = false;
            animTag.outputProceduralShape = false;
        }

        /// <summary>
        /// タグ区間ごとの最大時間幅を算出します。
        /// </summary>
        private Dictionary<string, int> CalcSectionMaxRange_(AnmAttribute[] attrList, AnimTag[] animTags)
        {
            Dictionary<string, int> maxRangeList = new Dictionary<string, int>();
            Dictionary<string, Tuple<int, int>> keyInfoList = new Dictionary<string, Tuple<int, int>>();

            foreach (AnmAttribute attr in attrList)
            {
                // rangeの算出
                var rangeList = CalcSectionRange_(animTags, attr);

                // 最大最小値を求める
                foreach (KeyValuePair<string, Tuple<int, int>> pair in rangeList)
                {
                    if (keyInfoList.ContainsKey(pair.Key))
                    {
                        var compList = keyInfoList[pair.Key];

                        int min = (compList.Item1 > pair.Value.Item1) ? pair.Value.Item1 : compList.Item1;
                        int max = (compList.Item2 < pair.Value.Item2) ? pair.Value.Item2 : compList.Item2;

                        keyInfoList[pair.Key] = new Tuple<int, int>(min, max);
                    }
                    else
                    {
                        keyInfoList[pair.Key] = pair.Value;
                    }
                }

                // 最大最小値からrangeを求める
                foreach (KeyValuePair<string, Tuple<int, int>> pair in keyInfoList)
                {
                    int range = pair.Value.Item2 - pair.Value.Item1;
                    maxRangeList[pair.Key] = range;
                }
            }

            return maxRangeList;
        }

        /// <summary>
        /// タグ区間ごとの時間幅を算出します。
        /// </summary>
        private Dictionary<string, Tuple<int, int>> CalcSectionRange_(AnimTag[] animTags, AnmAttribute attr)
        {
            Dictionary<string, Tuple<int, int>> ret = new Dictionary<string, Tuple<int, int>>();

            foreach (AnimTag animTag in animTags)
            {
                string tagName = animTag.name;
                Hermite[] curve = attr.keys;

                if (curve == null || curve.Count() <= 0 || (string.Compare(attr.tagName, tagName) != 0))
                {
                    ret[tagName] = new Tuple<int, int>(animTag.startFrame, animTag.endFrame);
                    continue;
                }

                int firstTime = (int)Math.Ceiling(curve.First().frame);
                int lastTime = (int)Math.Ceiling(curve.Last().frame);

                ret[tagName] = new Tuple<int, int>(firstTime, lastTime);
            }

            return ret;
        }

        /// <summary>
        /// タグ区間ごとの最大マージンを算出します。
        /// </summary>
        private Dictionary<string, Tuple<int, int>> CalcSectionMaxMargin_(AnmAttribute[] attrList, AnimTag[] animTags)
        {
            Dictionary<string, Tuple<int, int>> maxMarginList = new Dictionary<string, Tuple<int, int>>();

            foreach (AnmAttribute attr in attrList)
            {
                // marginの算出
                var marginList = CalcSectionMargin_(animTags, attr);

                // 最大最小値を求める
                foreach (KeyValuePair<string, Tuple<int, int>> pair in marginList)
                {
                    if (maxMarginList.ContainsKey(pair.Key))
                    {
                        if (maxMarginList[pair.Key].Item1 < pair.Value.Item1)
                        {
                            maxMarginList[pair.Key] = new Tuple<int, int>(pair.Value.Item1, maxMarginList[pair.Key].Item2);
                        }

                        if (maxMarginList[pair.Key].Item2 < pair.Value.Item2)
                        {
                            maxMarginList[pair.Key] = new Tuple<int, int>(maxMarginList[pair.Key].Item1, pair.Value.Item2);
                        }
                    }
                    else
                    {
                        maxMarginList[pair.Key] = pair.Value;
                    }
                }
            }

            return maxMarginList;
        }

        /// <summary>
        /// タグ区間ごとのマージンを算出します。
        /// </summary>
        private Dictionary<string, Tuple<int, int>> CalcSectionMargin_(AnimTag[] animTags, AnmAttribute attr)
        {
            Dictionary<string, Tuple<int, int>> ret = new Dictionary<string, Tuple<int, int>>();

            foreach (AnimTag animTag in animTags)
            {
                string tagName = animTag.name;
                Hermite[] curve = attr.keys;

                if (curve == null || curve.Count() <= 0 || (string.Compare(attr.tagName, tagName) != 0))
                {
                    ret[tagName] = new Tuple<int, int>(0, 0);
                    continue;
                }

                // 左右のマージンを算出する
                int firstTime = (int)Math.Ceiling(curve.First().frame);
                int lastTime = (int)Math.Ceiling(curve.Last().frame);
                int startFrame = animTag.startFrame;
                int endFrame = animTag.endFrame;

                int leftMargin = firstTime < 0 ? (startFrame - firstTime) : 0;
                int rightMargin = lastTime > endFrame ? (lastTime - endFrame) : 0;

                ret.Add(tagName, new Tuple<int, int>(leftMargin, rightMargin));
            }

            return ret;
        }

        /// <summary>
        /// キーフレームを結合します。
        /// </summary>
        void CombinKeys_(List<Hermite> keys, Hermite key)
        {
            var check = keys.Where(k => (int)k.frame == (int)key.frame);
            if (check.Count() <= 0)
            {
                keys.Add(key);
            }
        }

        /// <summary>
        /// LANノードから特定のAnimTargetを取得します。
        /// </summary>
        private AnimTarget GetAnimTarget_(
            Dictionary<AnimationType, LAN> lanList,
            AnimationType animtype,
            string contentName,
            AnimTargetType targetType)
        {
            LAN lan;
            if (lanList.TryGetValue(animtype, out lan))
            {
                var contents = lan.animContent.Where(content => content.name == contentName);

                if (contents != null)
                {
                    foreach (AnimContent content in contents)
                    {
                        foreach (AnimTarget animTarget in content.Items)
                        {
                            if (animTarget.target == targetType)
                            {
                                return animTarget;
                            }
                        }
                    }
                }
            }

            return null;
        }

        /// <summary>
        /// アトリビュートのパラメータを保持するクラスです。
        /// </summary>
        internal class AnmAttribute
        {
            public string tagName { get; }
            public LAN lan { get; }
            public AnimContent animContent { get; }
            public AnimationType animType { get; }
            public string contentName { get; }
            public AnimTarget animTarget { get; }
            public AnimTargetType animTargetType { get; }
            public Hermite[] keys { get; }

            /// <summary>
            /// コンストラクタです。
            /// </summary>
            public AnmAttribute(
                string _tagName,
                LAN _lan,
                AnimContent _animContent,
                AnimationType _animType,
                string _contentName,
                AnimTarget _animTarget,
                AnimTargetType _animTargetType,
                Hermite[] _keys)
            {
                this.tagName = _tagName;
                this.lan = _lan;
                this.animContent = _animContent;
                this.animType = _animType;
                this.contentName = _contentName;
                this.animTarget = _animTarget;
                this.animTargetType = _animTargetType;
                this.keys = _keys;
            }
        }

        internal class HermiteComparer : IComparer<Hermite>
        {
            public int Compare(Hermite x, Hermite y)
            {
                Hermite _x = x as Hermite;
                Hermite _y = y as Hermite;

                return (int)_x.frame - (int)_y.frame;
            }
        }
    }
}
