﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include <cstdarg>
#include <cstddef>

#include <nn/gfx.h>
#include <nn/crypto/crypto_Md5Generator.h>

#include "sharcArchiveReader.h"
#include "sharcUtil.h"

namespace
{
    // 最大アライメントを割り出す
    int calcMaxAlignment(uint32_t addr)
    {
#if 0
        // 右から連続する0の個数を数える
        uint32_t x = static_cast<uint32_t>(nw::math::Rightmost1(static_cast<int32_t>(addr))) - 1;
        int bit = static_cast<int>(nw::math::CntBit1(x));

        // オフセット 0 で来た時の対応
        if( 32 == bit || bit < 1 ){
            return sharc::cDefaultAlignment;
        }

        return (1<<bit);
#else
        // 上コードの代替
        // 二つ目のコメントが間違っている気がする。

        // 一番右の値が 1 のビットを取り出す。なければ 0
        uint32_t x = addr & (-(int32_t)addr);

        if (x == 0 || x == 1)
        {
            return sharc::cDefaultAlignment;
        }

        return x;
#endif
    }

    std::string hashToString(uint32_t value)
    {
        char buffer[9];
        sprintf(buffer, "%8x", value);
        return buffer;
    }
}

namespace nw
{
    namespace ut
    {
        //---------------------------------------------------------------------------
        //! @brief       ポインタにオフセット値を加えます。(const版)
        //!
        //! @param[in]   ptr     ポインタ
        //! @param[in]   offset  オフセット値
        //---------------------------------------------------------------------------
        template <typename TOffset>
        inline const void*
            AddOffsetToPtr(const void* ptr, TOffset offset)
        {
            return reinterpret_cast<const void*>(GetIntPtr(ptr) + offset);
        }

        //---------------------------------------------------------------------------
        //! @brief       ポインタからIntPtrへのキャストをおこないます。
        //!
        //! @param[in]   ptr ポインタ
        //!
        //! @return      ptrをIntPtrにキャストした値を返します。
        //---------------------------------------------------------------------------
        inline std::intptr_t
            GetIntPtr(const void* ptr)
        {
            return reinterpret_cast<intptr_t>(ptr);
        }

        //---------------------------------------------------------------------------
        //! @brief       ２つのポインタアドレスのオフセット値をlongで取得します。
        //!
        //! @param[in]   start   開始アドレス
        //! @param[in]   end     終了アドレス
        //!
        //! @return      ２つのポインタのオフセット値
        //---------------------------------------------------------------------------
        template <typename TDiff>
        inline TDiff
            GetOffsetFromPtr(const void* start, const void* end)
        {
            return static_cast<TDiff>(GetIntPtr(end)) - static_cast<TDiff>(GetIntPtr(start));
        }

        inline std::ptrdiff_t
            GetOffsetFromPtr(const void* start, const void* end)
        {
            return GetOffsetFromPtr<std::ptrdiff_t>(start, end);
        }
    }
}

namespace sharc
{
//------------------------------------------------------------------------------

ArchiveReader::ArchiveReader()
: mArchiveBlockHeader( nullptr )
, mFATBlockHeader( nullptr )
, mFNTBlock( nullptr )
, mFATEntryPtrArray( nullptr )
, mDataBlock( nullptr )
, mIsExtracted( false )
, mIsLogSilent( false )
{
}

//------------------------------------------------------------------------------

ArchiveReader::~ArchiveReader()
{
    if (mFATEntryPtrArray != NULL)
    {
        free(mFATEntryPtrArray);
    }
}

//------------------------------------------------------------------------------

bool ArchiveReader::compare( const ArchiveReader& target )
{
    // アーカイブが正常にロードされていない
    if( !target.mIsExtracted || !mIsExtracted ){
        fprintf(stderr, "----\nfailed.\narchives are not ready to compare.\n");
        return false;
    }

    // エンディアン
    EndianTypes endian = markToEndian( mArchiveBlockHeader->byte_order );
    if( endian != markToEndian( target.mArchiveBlockHeader->byte_order ) ){
        fprintf(stderr, "----\nfailed.\nendian type is different.\n");
        return false;
    }

    // ハッシュキー
    if( mFATBlockHeader->hash_key != target.mFATBlockHeader->hash_key ){
        fprintf(stderr, "----\nfailed.\nhash key is different.\n");
        return false;
    }

    // エントリ数
    int entry_size = mFATEntrys.size();
    int entry_target = target.mFATEntrys.size();
    if( entry_size < entry_target ){
        // ターゲットと比べて自分の方が少ない
        fprintf(stderr, "----\nfailed.\nnumber of entry is different. smaller.\n");
        // 自分のファイルは全てターゲット内に含まれる可能性があるので、そこはハッシュ値が一致するかで簡易チェックする
        bool is_included = true;
        bool is_same_size = true;
        int entry = 0;
        for( int idx = 0; idx < entry_size; idx ++ ){
            for(;;){
                if( mFATEntrys[idx]->hash == target.mFATEntrys[entry]->hash ){
                    uint32_t offset_mine_end   = toHostU32( endian, mFATEntrys[idx]->data_end_offset );
                    uint32_t offset_mine_start = toHostU32( endian, mFATEntrys[idx]->data_start_offset );
                    uint32_t size_mine = offset_mine_end - offset_mine_start;

                    uint32_t offset_target_end   = toHostU32( endian, target.mFATEntrys[entry]->data_end_offset );
                    uint32_t offset_target_start = toHostU32( endian, target.mFATEntrys[entry]->data_start_offset );
                    uint32_t size_target = offset_target_end - offset_target_start;

                    if( size_mine != size_target ){
                        is_same_size = false;
                    }
                    break;
                }
                entry ++;
                // ターゲットの上限に達したら強制終了
                if( entry >= entry_target ){
                    idx = entry_size;
                    is_included = false;
                    break;
                }
            }
        }
        // ここにきたら少なくともファイルエントリが全て内包されているのは確か
        if( is_included ){
            if( is_same_size ){
                // サイズも一致している
                OutLogMessage("but, every file entries are included and also file sizes are matched.\n");
            } else {
                // エントリは一致しているが、サイズが異なるファイルがある
                OutLogMessage("but, every file entries are included. some file sizes are different.\n");
            }
        }
        return false;
    } else if( entry_size > entry_target ){
        // ターゲットと比べて自分の方が多い
        fprintf(stderr, "----\nfailed.\nnumber of entry is different. larger.\n");
        return false;
    }

    bool is_same_size = true;
    bool is_same_bin = true;
    bool is_first = true;
    for( int idx = 0; idx < entry_size; idx ++ ){
        // 同じ順番にハッシュが並んでいるか?
        if( mFATEntrys[idx]->hash != target.mFATEntrys[idx]->hash ){
            fprintf(stderr, "----\nfailed.\nhash order is different.\n");
            return false;
        }

        uint32_t offset_mine_end   = toHostU32( endian, mFATEntrys[idx]->data_end_offset );
        uint32_t offset_mine_start = toHostU32( endian, mFATEntrys[idx]->data_start_offset );
        uint32_t size_mine = offset_mine_end - offset_mine_start;

        uint32_t offset_target_end   = toHostU32( endian, target.mFATEntrys[idx]->data_end_offset );
        uint32_t offset_target_start = toHostU32( endian, target.mFATEntrys[idx]->data_start_offset );
        uint32_t size_target = offset_target_end - offset_target_start;

        bool is_size_problem = false;
        bool is_binary_problem = false;
        if( size_mine != size_target ){
            is_size_problem = true;
            is_same_size = false;
        }
        else{
            // サイズが一致していたらファイルごとのバイナリ同一性もきちんとチェックする
            if( memcmp( nw::ut::AddOffsetToPtr( mDataBlock, offset_mine_start ),
                        nw::ut::AddOffsetToPtr( target.mDataBlock, offset_target_start ),
                        size_mine ) != 0 ) {
                is_binary_problem = true;
                is_same_bin = false;
            }
        }

        if( is_size_problem || is_binary_problem ){
            if( is_first ){
                is_first = false;
                OutLogMessage("hash       reason   name \n");
            }
            std::string name = "-";
            if( nw::ut::GetOffsetFromPtr( mDataBlock, mFNTBlock ) != 0 ){
                // 自分自身のアーカイブにFNTがあるとき
                uint32_t name_offset = toHostU32( endian, mFATEntrys[idx]->name_offset );
                if( name_offset != 0 ){
                    name = reinterpret_cast<const char*>(mFNTBlock + ( name_offset & 0x00ffffff ) * SHARC_FNT_ALIGN );
                }
            } else if( nw::ut::GetOffsetFromPtr( target.mDataBlock, target.mFNTBlock ) != 0 ) {
                // 比較対象のアーカイブにFNTがあるとき
                uint32_t name_offset = toHostU32( endian, target.mFATEntrys[idx]->name_offset );
                if( name_offset != 0 ){
                    name = reinterpret_cast<const char*>(target.mFNTBlock + ( name_offset & 0x00ffffff ) * SHARC_FNT_ALIGN );
                }
            }

            // -nオプションの有無が異なるアーカイブ比較を主に想定しているので、相違の可能性を表示するときはパスがありそうな方を選択
            if( is_size_problem ){
                OutLogMessage("[%8x] [  size] %s\n", mFATEntrys[idx]->hash, name.c_str() );
            } else {
                OutLogMessage("[%8x] [binary] %s\n", mFATEntrys[idx]->hash, name.c_str() );
            }
        }
    }

    if( !is_same_size || !is_same_bin ){
        fprintf(stderr, "----\nfailed.\ndifferent files are displayed in above list.\n" );
        return false;
    }

    // ファイルイメージブロックサイズの比較
    const uint32_t block_size_mine   = toHostU32( endian, mArchiveBlockHeader->file_size ) - static_cast<uint32_t>(nw::ut::GetOffsetFromPtr(mArchiveBlockHeader, mDataBlock));
    const uint32_t block_size_target = toHostU32( endian, target.mArchiveBlockHeader->file_size ) - static_cast<uint32_t>(nw::ut::GetOffsetFromPtr(target.mArchiveBlockHeader, target.mDataBlock));
    if( block_size_mine != block_size_target ){
        // ここで異なるということは、データ書き込み時のアライメントが違う可能性がある
        fprintf(stderr, "----\nfailed.\nprobably, alignment is different.\n" );
        return false;
    }

    // 最終的なバイナリの同一性保証, FNTが無くてもファイルイメージブロックは一致しているはず
    if( memcmp( mDataBlock, target.mDataBlock, block_size_mine ) != 0 ){
        fprintf(stderr, "----\nfailed.\nbinary is different.\n" );
        return false;
    }

    return true;
}

//------------------------------------------------------------------------------

bool ArchiveReader::resolve(const std::string& work, const std::string& path)
{
    // ヘッダ部分などの基本ロード
    if( !extractCommon(path) ){
        return false;
    }

    PathString dir_name;
    dir_name = "_" + path;

    bool is_exist = false;
    if( tryIsExistDirectory( &is_exist, dir_name.c_str() ) && is_exist ){
        // 解凍先が既に存在しているならば削除を促す
        fprintf(stderr, "----\nfailed.\n output directory is already exist.\n[ %s ]\n", dir_name.c_str());
        return false;
    }

    // 解凍先ディレクトリを作成
    makeDirectory( dir_name );

    FILE* handle;

    int entry_size = mFATEntrys.size();

    EndianTypes endian = markToEndian( mArchiveBlockHeader->byte_order );

    PathString file_name;
    dir_name = "_" + path;
    for( int i = 0; i < entry_size; i++ )
    {
        uint32_t data_start_offset = toHostU32( endian, mFATEntrys[i]->data_start_offset );
        uint32_t size = toHostU32( endian, mFATEntrys[i]->data_end_offset ) - data_start_offset;

        uint32_t name_offset = toHostU32( endian, mFATEntrys[i]->name_offset );
        std::string include_path = reinterpret_cast<const char*>(mFNTBlock + ( name_offset & 0x00ffffff ) * SHARC_FNT_ALIGN );
        if( nw::ut::GetOffsetFromPtr( mDataBlock, mFNTBlock ) != 0 && name_offset != 0 )    {
            // ファイル名がアーカイブ内にあるとき
            PathString virtual_dir_name;
            PathString virtual_file_name;
            getDirectoryName(virtual_dir_name, include_path);
            getFileName(virtual_file_name, include_path);
            std::string make_dir_name;
            // makeDirectoryWithParentは末尾に/が付くとエラーになるため、virtual_dir_nameが空文字のときは特別扱いする
            if (virtual_dir_name.empty()) {
                make_dir_name = work + dir_name;
            } else {
                make_dir_name = work + dir_name + "/" + virtual_dir_name;
            }

            if( makeDirectoryWithParent( make_dir_name ) ) {
                file_name = make_dir_name + "/" + virtual_file_name;
            } else {
                // フォルダの作成に失敗したとき, ハッシュ値+仮想ファイル名でネームを決定して書き出す
                file_name = dir_name + "/[" + hashToString(mFATEntrys[i]->hash) + "]" + virtual_file_name;
            }
            OutLogMessage("%s\n", file_name.c_str());
            errno_t res_code = fopen_s(&handle, file_name.c_str(), "wb");
            if (res_code == 0)
            {
                fwrite(nw::ut::AddOffsetToPtr(mDataBlock, data_start_offset), 1, size, handle);
                fclose(handle);
            }
            else
            {
                fprintf(stderr, "can't open file.\n[ %s ]\n", file_name.c_str());
            }
        } else {
            // ファイル名がアーカイブ内にないとき, ハッシュ値でネームを決定して書き出す
            file_name = dir_name + "/[" + hashToString(mFATEntrys[i]->hash);
            errno_t res_code = fopen_s(&handle, file_name.c_str(), "wb");
            if (res_code == 0)
            {
                fwrite(nw::ut::AddOffsetToPtr(mDataBlock, data_start_offset), 1, size, handle);
                fclose(handle);
            }
            else
            {
                fprintf(stderr, "can't open file.\n[ %s ]\n", file_name.c_str());
            }
        }
    }

    return true;
}

//------------------------------------------------------------------------------

void CalcMd5HashString_(char* pHashStr, const size_t HashStrSize, const void* pData, const size_t dataSize)
{
    NN_SDK_ASSERT(HashStrSize == nn::crypto::Md5Generator::HashSize * 2 + 1);

    unsigned char hash[nn::crypto::Md5Generator::HashSize];
    nn::crypto::Md5Generator gen;
    gen.Initialize();
    gen.Update(pData, dataSize);
    gen.GetHash(hash, nn::crypto::Md5Generator::HashSize);

    memset(pHashStr, 0x0, HashStrSize);
    for (int idx = 0; idx < nn::crypto::Md5Generator::HashSize; idx++)
    {
        sprintf(&pHashStr[idx * 2], "%02X", hash[idx]);
    }
}

//------------------------------------------------------------------------------

void ArchiveReader::extractBntxList(void* pResTexFile)
{
    const nn::gfx::ResTextureFile* pResTextureFile = nn::gfx::ResTextureFile::ResCast(pResTexFile);
    const int TexCount = pResTextureFile->GetTextureDic()->GetCount();

    for (int index = 0; index < TexCount; index++)
    {
        const nn::gfx::ResTexture* pResTex = pResTextureFile->GetResTexture(index);
        if (pResTex == NULL)
        {
            continue;
        }

        const size_t HashStrSize = nn::crypto::Md5Generator::HashSize * 2 + 1;
        char hashStr[HashStrSize];
        CalcMd5HashString_(hashStr, HashStrSize, &pResTex->ToData(), pResTex->ToData().blockHeader.GetBlockSize());

        OutLogMessage("[%s] [%6d] [%8d] %s \n", hashStr, pResTex->ToData().alignment, pResTex->ToData().textureDataSize, pResTex->GetName());
    }
}

//------------------------------------------------------------------------------

bool ArchiveReader::extractFileList(const std::string& path)
{
    // ヘッダ部分などの基本ロード
    if( !extractCommon(path) ){
        return false;
    }

    int entry_size = mFATEntrys.size();

    EndianTypes endian = markToEndian( mArchiveBlockHeader->byte_order );

    OutLogMessage("archive  [ %s ]\nhash key [ %d ], ", path.c_str(), toHostU32( endian, mFATBlockHeader->hash_key ) );
    if( endian == EndianTypes::cBig ){
        OutLogMessage("endian [ big ]\n");
    } else {
        OutLogMessage("endian [ little ]\n");
    }

    uint32_t data_block_offset = toHostU32( endian, mArchiveBlockHeader->data_block_offset );

    OutLogMessage("----\nhash       align    size       name \n");
    for( int i = 0; i < entry_size; i++ )
    {
        uint32_t name_offset = toHostU32( endian, mFATEntrys[i]->name_offset );
        std::string include_path = reinterpret_cast<const char*>(mFNTBlock + ( name_offset & 0x00ffffff ) * SHARC_FNT_ALIGN );
        uint32_t data_start_offset = toHostU32( endian, mFATEntrys[i]->data_start_offset );
        uint32_t size = toHostU32( endian, mFATEntrys[i]->data_end_offset ) - data_start_offset;
        int align = calcMaxAlignment( data_start_offset + data_block_offset );
        if( nw::ut::GetOffsetFromPtr( mDataBlock, mFNTBlock ) != 0 && name_offset != 0 )    {
            // ファイル名がアーカイブ内にあるとき
            OutLogMessage("[%8x] [%6d] [%8d] %s \n", mFATEntrys[i]->hash, align, size, include_path.c_str() );

            // エントリ名が timg/__Combined.bntx なら詳しくレポートする。
            if (strcmp(include_path.c_str(), "timg/__Combined.bntx") == 0)
            {
                void* pResTexFile = const_cast<void*>(nw::ut::AddOffsetToPtr(mDataBlock, mFATEntrys[i]->data_start_offset));
                extractBntxList(pResTexFile);
            }
        } else {
            // ファイル名がアーカイブ内にないとき
            OutLogMessage("[%8x] [%6d] [%8d] - \n", mFATEntrys[i]->hash, align, size );
        }
    }

    OutLogMessage("\n%d files\n", entry_size);

    return true;
}

//------------------------------------------------------------------------------

bool ArchiveReader::extractCommon(const std::string& path)
{
    // 既にロードされているときは二重呼び出しを防ぐ
    if( mIsExtracted ){
        return true;
    }

    FILE* handle;
    errno_t res_code = fopen_s(&handle, path.c_str(), "rb");
    if( res_code != 0 )
    {
        fprintf(stderr, "can't open archive.\n[ %s ]\n", path.c_str());
        return false;
    }
    fseek(handle, 0, SEEK_END);
    uint32_t size = static_cast<uint32_t>(ftell(handle));
    fseek(handle, 0, SEEK_SET);
    uint8_t* data = static_cast<uint8_t*>(malloc(size));
    fread(data, 1, size, handle);
    fclose(handle);

    const uint8_t* archive8 = reinterpret_cast<const uint8_t*>( data );
    mArchiveBlockHeader = reinterpret_cast<const ArchiveBlockHeader*>( archive8 );
    if ( strncmp( mArchiveBlockHeader->signature, "SARC", 4 ) != 0 )
    {
        fprintf(stderr, "Invalid ArchiveBlockHeader" );
        goto error;
    }
    EndianTypes endian = markToEndian( mArchiveBlockHeader->byte_order );

    if ( toHostU16( endian, mArchiveBlockHeader->version ) != SHARC_ARCHIVE_VERSION )
    {
        fprintf(stderr, "unmatching version ( expect: %x, actual: %x )", SHARC_ARCHIVE_VERSION, mArchiveBlockHeader->version );
        goto error;
    }
    if( toHostU16( endian, mArchiveBlockHeader->header_size ) != sizeof( ArchiveBlockHeader ) )
    {
        fprintf(stderr, "Invalid ArchiveBlockHeader" );
        goto error;
    }
    mFATBlockHeader = reinterpret_cast<const FATBlockHeader*>( archive8 + toHostU16( endian, mArchiveBlockHeader->header_size ) );
    if ( strncmp( mFATBlockHeader->signature, "SFAT", 4 ) != 0 )
    {
        fprintf(stderr, "Invalid FATBlockHeader" );
        return false;
    }

    if( toHostU16( endian, mFATBlockHeader->header_size ) != sizeof( FATBlockHeader ) )
    {
        fprintf(stderr, "Invalid FATBlockHeader" );
        goto error;
    }
    if( toHostU16( endian, mFATBlockHeader->file_num ) > cMaxPathNum )
    {
        fprintf(stderr, "Invalid FATBlockHeader" );
        goto error;
    }
    uint16_t fileNum = toHostU16( endian, mFATBlockHeader->file_num);

    const uint8_t* pFATEntryTop = archive8 + toHostU16( endian, mArchiveBlockHeader->header_size ) + toHostU16( endian, mFATBlockHeader->header_size );

    // mFATEntrys では FATEntry* の配列を管理するため、配列の実体を作成する。
    if (mFATEntryPtrArray != NULL)
    {
        free(mFATEntryPtrArray);
    }
    mFATEntryPtrArray = static_cast<FATEntry**>(malloc(fileNum * sizeof(FATEntry*)));
    for (int i = 0; i < fileNum;i++)
    {
        mFATEntryPtrArray[i] = const_cast<FATEntry*>(reinterpret_cast<const FATEntry*>( pFATEntryTop + i * sizeof(FATEntry)));
    }
    // mFATEntrys に管理する配列を設定する。
    mFATEntrys.assign(mFATEntryPtrArray, mFATEntryPtrArray + toHostU16(endian, mFATBlockHeader->file_num));
    const FNTBlockHeader* fnt_header = reinterpret_cast<const FNTBlockHeader*>( pFATEntryTop  + toHostU16( endian, mFATBlockHeader->file_num ) * sizeof( FATEntry ) );
    if ( strncmp( fnt_header->signature, "SFNT", 4 ) != 0 )
    {
        fprintf(stderr, "Invalid FNTBlockHeader" );
        goto error;
    }
    if( toHostU16( endian, fnt_header->header_size ) != sizeof(FNTBlockHeader) )
    {
        fprintf(stderr, "Invalid FNTBlockHeader" );
        goto error;
    }
    mFNTBlock = reinterpret_cast<const char*>( reinterpret_cast<const uint8_t*>( fnt_header ) + toHostU16( endian, fnt_header->header_size ) );
    if( toHostU32( endian, mArchiveBlockHeader->data_block_offset ) < static_cast<uint32_t>(nw::ut::GetOffsetFromPtr(mArchiveBlockHeader, mFNTBlock)) )
    {
        fprintf(stderr, "Invalid data block offset" );
        goto error;
    }
    mDataBlock = archive8 + toHostU32( endian, mArchiveBlockHeader->data_block_offset );

    mIsExtracted = true;
    return true;

error:
    free(data);
    return false;
}

//------------------------------------------------------------------------------

void ArchiveReader::OutLogMessage(char* pMsg, ...)
{
    if(getLogSilent())
    {
        return;
    }

    va_list va;
    va_start(va, pMsg);

    vprintf(pMsg, va);

    va_end(va);
}

//------------------------------------------------------------------------------
}   // namespace sharc
