﻿// --------------------------------------------------------------------------------
// <copyright>
// Copyright (C)Nintendo All rights reserved.
//
// These coded instructions, statements, and computer programs contain proprietary
// information of Nintendo and/or its licensed developers and are protected by
// national and international copyright laws. They may not be disclosed to third
// parties or copied or duplicated in any form, in whole or in part, without the
// prior written consent of Nintendo.
//
// The content herein is highly confidential and should be handled accordingly.
// </copyright>
// --------------------------------------------------------------------------------
using System;
using System.Collections.Generic;
using System.ComponentModel;
using System.Drawing;
using System.Data;
using System.Linq;
using System.Text;
using System.Threading.Tasks;
using System.Windows.Forms;

namespace FcpxMaker.Panel.Property
{
    using DataModel;
    using FcpxMaker.Node;
    using static FcpxMaker.Type;

    public partial class FontPropertyPanel : PropertyPanelBase
        , IEventListener
    {
        /// <summary>
        /// 操作対象のデータモデルです。
        /// </summary>
        private FontDataModel TargetDataModel
        {
            get
            {
                return Target as FontDataModel;
            }
        }

        /// <summary>
        /// 操作対象のデータモデルです。
        /// </summary>
        public FontFileType FontType { get; set; } = FontFileType.All;

        /// <summary>
        /// UIを更新します。
        /// </summary>
        override public void UpdateProperty()
        {
            if (TargetDataModel != null)
            {
                _tbxFontFilePath.Text = TargetDataModel.RelativeFontFilePath;
            }
        }

        /// <summary>
        /// データモデルを関連付けます。
        /// </summary>
        override public void BindTarget(DataModelBase target)
        {
            this.Target = target;
            if (TargetDataModel != null)
            {
                this.FontType = TargetDataModel.FontType;
            }
        }

        /// <summary>
        /// コンストラクタです。
        /// </summary>
        public FontPropertyPanel()
        {
            InitializeComponent();
        }

        /// <summary>
        /// イベントを通知します。
        /// </summary>
        public void dispatch(object sender, EventArgs arg)
        {
            EventParam param = arg as EventParam;

            switch (param.Type)
            {
                case EventType.AddFont:
                    {
                        DataNode node = param.Option as DataNode;
                        FontDataModel fontDataModel = node.DataModel as FontDataModel;
                        BindTarget(fontDataModel);

                        TargetDataModel.AbsoluteFontFilePath = fontDataModel.AbsoluteFontFilePath;

                        UpdateProperty();
                    }
                    break;
                case EventType.ModifyFont:
                case EventType.PasteParams:
                    {
                        UpdateProperty();
                    }
                    break;
            }
        }

        /// <summary>
        /// フォントの変更ボタンを押した時のイベントハンドラです。
        /// </summary>
        private void Event_FontFileButtonClick(object sender, EventArgs e)
        {
            if (FileIOManager.ShowFontOpenDialog(FontType) == DialogResult.OK)
            {
                if (this.Target == null)
                {
                    DataNode node = DataNodeFactory.CreateDataNode(DataModelFactory.CreateDataModel(DataType.Font));
                    (node.DataModel as FontDataModel).AbsoluteFontFilePath = FileIOManager.FontOpenFilePath;
                    (node.DataModel as FontDataModel).FontType = FontType;

                    EventParam args = new EventParam(EventType.AddFont, node);
                    EventManager.entry(sender, args);
                }
                else
                {
                    (this.Target as FontDataModel).AbsoluteFontFilePath = FileIOManager.FontOpenFilePath;
                    (this.Target as FontDataModel).FontType = FontType;

                    EventParam args = new EventParam(EventType.ModifyFont, this.Target);
                    EventManager.entry(sender, args);
                }

                // ツリービューの更新要求を通知します
                NotifyUpdateTreeView();

                // データの変更を通知します
                NotifyDataModified();
            }
        }
    }
}
