﻿// --------------------------------------------------------------------------------
// <copyright>
// Copyright (C)Nintendo All rights reserved.
//
// These coded instructions, statements, and computer programs contain proprietary
// information of Nintendo and/or its licensed developers and are protected by
// national and international copyright laws. They may not be disclosed to third
// parties or copied or duplicated in any form, in whole or in part, without the
// prior written consent of Nintendo.
//
// The content herein is highly confidential and should be handled accordingly.
// </copyright>
// --------------------------------------------------------------------------------
using System;
using System.Collections.Generic;
using System.Linq;
using System.Text;
using System.Threading.Tasks;

namespace FcpxMaker.DataModel
{
    using static FcpxMaker.Type;

    /// <summary>
    /// マルチスケーラブルフォントのデータモデルです。
    /// </summary>
    [Serializable]
    class MultiScalableFontDataModel : DataModelBase
    {
        List<DataModelBase> _scalableFontDescriptionSet = new List<DataModelBase>();

        /// <summary>
        /// フォントのサイズです。
        /// </summary>
        public float FontSize { get; set; } = 40;

        /// <summary>
        /// 代替文字を 10 進数で指定します。
        /// 省略した場合は 63 ('?' の文字)になります。
        /// </summary>
        public uint AlternateChar { get; set; } = '?';

        /// <summary>
        /// 行間のオフセット値を指定します。
        /// 省略した場合は 0 になります。
        /// </summary>
        public int LineFeedOffset { get; set; } = 0;

        /// <summary>
        /// スケーラブルフォントの設定です。
        /// </summary>
        public List<DataModelBase> ScalableFontDescriptionSet
        {
            get { return _scalableFontDescriptionSet; }
        }

        /// <summary>
        /// データの種類です。
        /// </summary>
        override public DataType DataType
        {
            get { return DataType.MultiScalableFont; }
        }

        /// <summary>
        /// データの検証結果です。
        /// </summary>
        override public bool Validate
        {
            get
            {
                // ScalableFontDescriptionが1つ以上存在する
                if ((_scalableFontDescriptionSet.Count() == 0) ||
                    (AlternateChar == 0))
                {
                    return false;
                }

                return true;
            }
        }

        /// <summary>
        /// データモデルを比較します。
        /// </summary>
        override public bool Equals(DataModelBase dataModel)
        {
            MultiScalableFontDataModel dm = dataModel as MultiScalableFontDataModel;
            if (dm == null) return false;

            if ((this.FontSize != dm.FontSize) ||
                (this.AlternateChar != dm.AlternateChar) ||
                (this.LineFeedOffset != dm.LineFeedOffset) ||
                (this.ScalableFontDescriptionSet.Count != dm.ScalableFontDescriptionSet.Count))
            {
                return false;
            }

            for (int i = 0; i < this.ScalableFontDescriptionSet.Count; i++)
            {
                if (!this.ScalableFontDescriptionSet[i].Equals(dm.ScalableFontDescriptionSet[i]))
                {
                    return false;
                }
            }

            return true;
        }

        /// <summary>
        /// データモデルを複製します。
        /// </summary>
        override public DataModelBase Clone()
        {
            MultiScalableFontDataModel newDataModel = new MultiScalableFontDataModel();
            newDataModel.FontSize = this.FontSize;
            newDataModel.AlternateChar = this.AlternateChar;
            newDataModel.LineFeedOffset = this.LineFeedOffset;

            for (int i = 0; i < this.ScalableFontDescriptionSet.Count; i++)
            {
                newDataModel.ScalableFontDescriptionSet.Add(this.ScalableFontDescriptionSet[i].Clone());
            }

            return newDataModel;
        }

        /// <summary>
        /// データモデルをコピーします。
        /// </summary>
        override public bool CopyParam(DataModelBase dataModel, bool copyChild)
        {
            MultiScalableFontDataModel dm = dataModel as MultiScalableFontDataModel;
            if (dm == null) return false;

            this.FontSize = dm.FontSize;
            this.AlternateChar = dm.AlternateChar;
            this.LineFeedOffset = dm.LineFeedOffset;

            if (copyChild)
            {
                if (dm.ScalableFontDescriptionSet != null && dm.ScalableFontDescriptionSet.Count > 0)
                {
                    this.ScalableFontDescriptionSet.Clear();
                    this.ScalableFontDescriptionSet.AddRange(dm.ScalableFontDescriptionSet);
                }
            }

            return true;
        }

        /// <summary>
        /// 子ノードとして追加可能かどうかを返します。
        /// </summary>
        override public bool CanAddChild(DataModelBase dataModel)
        {
            if (dataModel is ScalableFontDescriptionDataModel)
            {
                return true;
            }

            return false;
        }

        /// <summary>
        /// 表示名です。
        /// </summary>
        override public string ToString()
        {
            return DataType.ToString();
        }
    }
}
