﻿// --------------------------------------------------------------------------------
// <copyright>
// Copyright (C)Nintendo All rights reserved.
//
// These coded instructions, statements, and computer programs contain proprietary
// information of Nintendo and/or its licensed developers and are protected by
// national and international copyright laws. They may not be disclosed to third
// parties or copied or duplicated in any form, in whole or in part, without the
// prior written consent of Nintendo.
//
// The content herein is highly confidential and should be handled accordingly.
// </copyright>
// --------------------------------------------------------------------------------
using System;
using System.Collections.Generic;
using System.Linq;
using System.Text;
using System.Threading.Tasks;

namespace FcpxMaker.DataModel
{
    using static FcpxMaker.Type;

    /// <summary>
    /// ビットマップフォントのデータモデルです。
    /// </summary>
    [Serializable]
    public class BitmapFontDataModel : DataModelBase
    {
        DataModelBase _charCodeRange = new CharCodeRangeDataModel();
        DataModelBase _fontDataModel = null;

        /// <summary>
        /// フォントのデータモデルです。
        /// </summary>
        public DataModelBase FontDataModel
        {
            get { return _fontDataModel; }
            set { _fontDataModel = value; }
        }

        /// <summary>
        /// 文字範囲のデータモデルです。
        /// </summary>
        public DataModelBase CharCodeRangeDataModel
        {
            get { return _charCodeRange; }
            set { _charCodeRange = value; }
        }

        /// <summary>
        /// フォントファイルのパスを指定します。
        /// </summary>
        public string FontFilePath
        {
            get
            {
                if (_fontDataModel == null) return string.Empty;

                return (_fontDataModel as FontDataModel).RelativeFontFilePath;
            }
        }

        /// <summary>
        /// フォントを追加します。
        /// </summary>
        public void SetFontDataModel(DataModelBase dataModel)
        {
            _fontDataModel = dataModel;
        }

        /// <summary>
        /// フォントを削除します。
        /// </summary>
        public void RemoveFontDataModel()
        {
            _fontDataModel = null;
        }

        /// <summary>
        /// データの種類です。
        /// </summary>
        override public DataType DataType
        {
            get { return DataType.BitmapFont; }
        }

        /// <summary>
        /// データの検証結果です。
        /// </summary>
        override public bool Validate
        {
            get
            {
                return _charCodeRange.Validate;
            }
        }

        /// <summary>
        /// データモデルを比較します。
        /// </summary>
        override public bool Equals(DataModelBase dataModel)
        {
            BitmapFontDataModel dm = dataModel as BitmapFontDataModel;
            if (dm == null) return false;

            if (this.FontDataModel == null && dm.FontDataModel != null)
            {
                return false;
            }

            if (this.CharCodeRangeDataModel == null && dm.CharCodeRangeDataModel != null)
            {
                return false;
            }

            if ((this.FontDataModel != null && !this.FontDataModel.Equals(dm.FontDataModel)) ||
                (this.CharCodeRangeDataModel != null && !this.CharCodeRangeDataModel.Equals(dm.CharCodeRangeDataModel)))
            {
                return false;
            }

            return true;
        }

        /// <summary>
        /// データモデルを複製します。
        /// </summary>
        override public DataModelBase Clone()
        {
            BitmapFontDataModel newDataModel = new BitmapFontDataModel();
            newDataModel.FontDataModel = this.FontDataModel?.Clone();
            newDataModel.CharCodeRangeDataModel = this.CharCodeRangeDataModel?.Clone();

            return newDataModel;
        }

        /// <summary>
        /// データモデルをコピーします。
        /// </summary>
        override public bool CopyParam(DataModelBase dataModel, bool copyChild)
        {
            BitmapFontDataModel dm = dataModel as BitmapFontDataModel;
            if (dm == null) return false;

            if (copyChild)
            {
                this.FontDataModel = dm.FontDataModel;
            }

            this.CharCodeRangeDataModel.CopyParam(dm.CharCodeRangeDataModel, copyChild);

            return true;
        }

        /// <summary>
        /// 子ノードとして追加可能かどうかを返します。
        /// </summary>
        override public bool CanAddChild(DataModelBase dataModel)
        {
            if (dataModel is FontDataModel)
            {
                return
                    ((dataModel as FontDataModel).FontType == FontFileType.Bitmap &&
                     _fontDataModel == null);
            }

            return false;
        }

        /// <summary>
        /// 表示名です。
        /// </summary>
        override public string ToString()
        {
            return DataType.ToString();
        }
    }
}
