﻿// --------------------------------------------------------------------------------
// <copyright>
// Copyright (C)Nintendo All rights reserved.
//
// These coded instructions, statements, and computer programs contain proprietary
// information of Nintendo and/or its licensed developers and are protected by
// national and international copyright laws. They may not be disclosed to third
// parties or copied or duplicated in any form, in whole or in part, without the
// prior written consent of Nintendo.
//
// The content herein is highly confidential and should be handled accordingly.
// </copyright>
// --------------------------------------------------------------------------------

using System;
using System.Diagnostics;

using LECore.Structures.Nsrif;
using LECore.Structures.Nsrif.Attributes;
using LECore.Structures.Nsrif.Elements;

namespace LayoutEditor.Utility
{
    /// <summary>
    /// ＵＩデータバインダクラス。
    /// </summary>
    public sealed class UIDataBinder
    {
        #region フラグテキスト
        /// <summary>
        /// フラグテキストを取得（On/Off）。
        /// </summary>
        public static string GetFlagTextOnOff(bool flag)
        {
            if (flag) { return StringResMgr.Get("FLAG_ON");  }
            else      { return StringResMgr.Get("FLAG_OFF"); }
        }

        /// <summary>
        /// フラグテキストを取得（はい/いいえ）。
        /// </summary>
        public static string GetFlagTextYesNo(bool flag)
        {
            if (flag) { return StringResMgr.Get("FLAG_YES"); }
            else      { return StringResMgr.Get("FLAG_NO");  }
        }

        /// <summary>
        /// フラグテキストを取得（有効/無効）。
        /// </summary>
        public static string GetFlagTextEnableDisable(bool flag)
        {
            if (flag) { return StringResMgr.Get("FLAG_ENABLE");  }
            else      { return StringResMgr.Get("FLAG_DISABLE"); }
        }

        /// <summary>
        /// フラグテキストを取得（あり/なし）。
        /// </summary>
        public static string GetFlagTextExistOrNot(bool flag)
        {
            if (flag) { return StringResMgr.Get("FLAG_EXIST");    }
            else      { return StringResMgr.Get("FLAG_NOTEXIST"); }
        }
        #endregion

        #region 日時テキスト
        /// <summary>
        /// 日時テキスト取得
        /// </summary>
        public static string GetDateTimeText(string date)
        {
            string result = "????????";
            try
            {
                DateTime dt = DateTime.Parse(date);
                result = dt.ToString();
            }
            catch (FormatException)
            {
            }
            return result;
        }
        #endregion

        #region AttrScalingRule
        private static string[] _strIDScalingRule = {
            "MODEL_SR_STD",
            "MODEL_SR_SI",
            "MODEL_SR_MAYA",
        };

        /// <summary>
        /// 属性テキストを取得。
        /// </summary>
        public static string GetAttributeText(AttrScalingRule attr)
        {
            DebugAssertCheckArrayToEnum(_strIDScalingRule, attr.GetType());
            return StringResMgr.Get(_strIDScalingRule[(int)attr]);
        }
        #endregion

        #region AttrTexMatrixMode
        private static string[] _strIDTexMatrixMode = {
            "MODEL_TMM_MAYA",
            "MODEL_TMM_XSI",
            "MODEL_TMM_MAX",
        };

        /// <summary>
        /// 属性テキストを取得。
        /// </summary>
        public static string GetAttributeText(AttrTexMatrixMode attr)
        {
            DebugAssertCheckArrayToEnum(_strIDTexMatrixMode, attr.GetType());
            return StringResMgr.Get(_strIDTexMatrixMode[(int)attr]);
        }
        #endregion

        #region AttrBillborad
        private static string[] _strIDBillborad = {
            "NODE_BB_OFF",
            "NODE_BB_STD",
            "NODE_BB_PERSPSTD",
            "NODE_BB_ROT",
            "NODE_BB_PERSPROT",
            "NODE_BB_Y",
            "NODE_BB_PERSPY",
        };

        /// <summary>
        /// 属性テキストを取得。
        /// </summary>
        public static string GetAttributeText(AttrBillboard attr)
        {
            DebugAssertCheckArrayToEnum(_strIDBillborad, attr.GetType());
            return StringResMgr.Get(_strIDBillborad[(int)attr]);
        }
        #endregion

        #region AttrPrimitiveType
        private static string[] _strIDPrimitiveType = {
            "POLY_PT_POINTS",
            "POLY_PT_LINESTRIP",
            "POLY_PT_TRIANGLES",
            "POLY_PT_TRIANGLEFAN",
            "POLY_PT_TRIANGLESTRIP",
        };

        /// <summary>
        /// 属性テキストを取得。
        /// </summary>
        public static string GetAttributeText(AttrPrimitiveType attr)
        {
            DebugAssertCheckArrayToEnum(_strIDPrimitiveType, attr.GetType());
            return StringResMgr.Get(_strIDPrimitiveType[(int)attr]);
        }
        #endregion

        #region AttrDisplayFace
        private static string[] _strIDDisplayFace = {
            "POLY_DF_FRONT",
            "POLY_DF_BACK",
            "POLY_DF_BOTH",
        };

        /// <summary>
        /// 属性テキストを取得。
        /// </summary>
        public static string GetAttributeText(AttrDisplayFace attr)
        {
            DebugAssertCheckArrayToEnum(_strIDDisplayFace, attr.GetType());
            return StringResMgr.Get(_strIDDisplayFace[(int)attr]);
        }
        #endregion

        #region AttrTransparencyMode
        private static readonly string[] _strIDTransparencyMode = {
            "MAT_TM_OPA",
            "MAT_TM_XLU",
        };

        /// <summary>
        /// 属性テキストを取得。
        /// </summary>
        public static string GetAttributeText(AttrTransparencyMode attr)
        {
            DebugAssertCheckArrayToEnum(_strIDTransparencyMode, attr.GetType());
            return StringResMgr.Get(_strIDTransparencyMode[(int)attr]);
        }
        #endregion

        #region AttrColorAssignTarget
        private static readonly string[] _strIDColorAssignTarget = {
            "MAT_CAT_MAT0_RGB",
            "MAT_CAT_MAT0_A",
            "MAT_CAT_MAT1_RGB",
            "MAT_CAT_MAT1_A",
            "MAT_CAT_AMB0_RGB",
            "MAT_CAT_AMB0_A",
            "MAT_CAT_AMB1_RGB",
            "MAT_CAT_AMB1_A",
            "MAT_CAT_TEVC0_RGB",
            "MAT_CAT_TEVC0_A",
            "MAT_CAT_TEVC1_RGB",
            "MAT_CAT_TEVC1_A",
            "MAT_CAT_TEVC2_RGB",
            "MAT_CAT_TEVC2_A",
            "MAT_CAT_TEVK0_RGB",
            "MAT_CAT_TEVK0_A",
            "MAT_CAT_TEVK1_RGB",
            "MAT_CAT_TEVK1_A",
            "MAT_CAT_TEVK2_RGB",
            "MAT_CAT_TEVK2_A",
            "MAT_CAT_TEVK3_RGB",
            "MAT_CAT_TEVK3_A",
        };

        /// <summary>
        /// 属性テキストを取得。
        /// </summary>
        public static string GetAttributeText(AttrColorAssignTarget attr)
        {
            DebugAssertCheckArrayToEnum(_strIDColorAssignTarget, attr.GetType());
            return StringResMgr.Get(_strIDColorAssignTarget[(int)attr]);
        }
        #endregion

        #region AttrImageFormat
        private static string[] _strIDImageFormat = {
            "TEX_FMT_I4",
            "TEX_FMT_IA4",
            "TEX_FMT_I8",
            "TEX_FMT_IA8",
            "TEX_FMT_RGB565",
            "TEX_FMT_RGB5A3",
            "TEX_FMT_RGBA8",
            "TEX_FMT_CMPR",
            "TEX_FMT_C4",
            "TEX_FMT_C8",
            "TEX_FMT_C14",
        };

        /// <summary>
        /// 属性テキストを取得。
        /// </summary>
        public static string GetAttributeText(AttrImageFormat attr)
        {
            DebugAssertCheckArrayToEnum(_strIDImageFormat, attr.GetType());
            return StringResMgr.Get(_strIDImageFormat[(int)attr]);
        }
        #endregion

        #region AttrPaletteFormat
        private static string[] _strIDPaletteFormat = {
            "PLTT_FMT_IA8",
            "PLTT_FMT_RGB565",
            "PLTT_FMT_RGB5A3",
        };

        /// <summary>
        /// 属性テキストを取得。
        /// </summary>
        public static string GetAttributeText(AttrPaletteFormat attr)
        {
            DebugAssertCheckArrayToEnum(_strIDPaletteFormat, attr.GetType());
            return StringResMgr.Get(_strIDPaletteFormat[(int)attr]);
        }
        #endregion

        /// <summary>
        /// 配列要素数と列挙定義数の対応確認。
        /// </summary>
        [Conditional("DEBUG")]
        private static void DebugAssertCheckArrayToEnum(Array array, Type enumType)
        {
            // _Unknownを除いた数で比較する
            Debug.Assert(
                array.Length == Enum.GetValues(enumType).Length - 1,
                string.Format("{0} 型と対応する配列要素数が合いません。", enumType.FullName)
            );
        }

        /// <summary>
        /// コンストラクタ。
        /// </summary>
        private UIDataBinder() {}
    }
}
