﻿// --------------------------------------------------------------------------------
// <copyright>
// Copyright (C)Nintendo. All rights reserved.
//
// These coded instructions, statements, and computer programs contain proprietary
// information of Nintendo and/or its licensed developers and are protected by
// national and international copyright laws. They may not be disclosed to third
// parties or copied or duplicated in any form, in whole or in part, without the
// prior written consent of Nintendo.
//
// The content herein is highly confidential and should be handled accordingly.
// </copyright>
// --------------------------------------------------------------------------------
using System;
using System.Collections.Generic;
using System.Text;
using System.Drawing;
using System.Diagnostics;
using System.Windows.Forms;

namespace LayoutEditor.Utility
{
    using LECore.Structures;

    class PaneItemImageHelper
    {
        // アイコンセット
        static ImageList         _paneItemImageList = null;
        static Image _animationOverrayIcon = null;
        static Image _commentOverrayIcon = null;
        static Image _animCommentOverrayIcon = null;
        static Image _warningOverrayIcon = null;
        static Image _prunedPaneOverrayIcon = null;


       #region private

        static void AddItemImage_(  PaneKind kind, bool locked, bool hidden, string iconName )
        {
            Debug.Assert( _paneItemImageList != null );
            _paneItemImageList.Images.Add(GetPaneImageKindAsString(kind, locked, hidden, false), ImageResMgr.GetManifestResourcePng(iconName));
        }

        static void AddReadOnlyItemImage_(PaneKind kind, string iconName)
        {
            Debug.Assert(_paneItemImageList != null);
            _paneItemImageList.Images.Add(GetPaneImageKindAsString(kind, false, false, true), ImageResMgr.GetManifestResourcePng(iconName));
        }
       #endregion private

        /// <summary>
        /// コンストラクタ
        /// </summary>
        static PaneItemImageHelper()
        {
            if( _paneItemImageList == null )
            {
                _paneItemImageList = new ImageList();
                _paneItemImageList.ColorDepth = ColorDepth.Depth24Bit;

                // null
                AddItemImage_( PaneKind.Null, false, false, "null_pane_nrm_16.png" );
                AddItemImage_(PaneKind.Null, true, false, "null_pane_lck_16.png");
                AddItemImage_(PaneKind.Null, false, true, "null_pane_hdn_16.png");
                AddItemImage_(PaneKind.Null, true, true, "null_pane_hdn_16.png");
                AddReadOnlyItemImage_(PaneKind.Null, "null_pane_roy_16.png");

                // picture
                AddItemImage_(PaneKind.Picture, false, false, "picture_pane_nrm_16.png");
                AddItemImage_(PaneKind.Picture, true, false, "picture_pane_lck_16.png");
                AddItemImage_(PaneKind.Picture, false, true, "picture_pane_hdn_16.png");
                AddItemImage_(PaneKind.Picture, true, true, "picture_pane_hdn_16.png");
                AddReadOnlyItemImage_(PaneKind.Picture, "picture_pane_roy_16.png");

                // textbox
                AddItemImage_(PaneKind.Textbox, false, false, "textbox_pane_nrm_16.png");
                AddItemImage_(PaneKind.Textbox, true, false, "textbox_pane_lck_16.png");
                AddItemImage_(PaneKind.Textbox, false, true, "textbox_pane_hdn_16.png");
                AddItemImage_(PaneKind.Textbox, true, true, "textbox_pane_hdn_16.png");
                AddReadOnlyItemImage_(PaneKind.Textbox, "textbox_pane_roy_16.png");

                // window
                AddItemImage_(PaneKind.Window, false, false, "window_pane_nrm_16.png");
                AddItemImage_(PaneKind.Window, true, false, "window_pane_lck_16.png");
                AddItemImage_(PaneKind.Window, false, true, "window_pane_hdn_16.png");
                AddItemImage_(PaneKind.Window, true, true, "window_pane_hdn_16.png");
                AddReadOnlyItemImage_(PaneKind.Window, "window_pane_roy_16.png");

                // region
                AddItemImage_(PaneKind.Bounding, false, false, "bounding_pane_nrm_16.png");
                AddItemImage_(PaneKind.Bounding, true, false, "bounding_pane_lck_16.png");
                AddItemImage_(PaneKind.Bounding, false, true, "bounding_pane_hdn_16.png");
                AddItemImage_(PaneKind.Bounding, true, true, "bounding_pane_hdn_16.png");
                AddReadOnlyItemImage_(PaneKind.Bounding, "bounding_pane_roy_16.png");

                // parts
                AddItemImage_(PaneKind.Parts, false, false, "parts_pane_nrm_16.png");
                AddItemImage_(PaneKind.Parts, true, false, "parts_pane_lck_16.png");
                AddItemImage_(PaneKind.Parts, false, true, "parts_pane_hdn_16.png");
                AddItemImage_(PaneKind.Parts, true, true, "parts_pane_hdn_16.png");
                AddReadOnlyItemImage_(PaneKind.Parts, "parts_pane_roy_16.png");

                // capture
                AddItemImage_(PaneKind.Capture, false, false, "capture_pane_nrm_16.png");
                AddItemImage_(PaneKind.Capture, true, false, "capture_pane_lck_16.png");
                AddItemImage_(PaneKind.Capture, false, true, "capture_pane_hdn_16.png");
                AddItemImage_(PaneKind.Capture, true, true, "capture_pane_hdn_16.png");
                AddReadOnlyItemImage_(PaneKind.Capture, "capture_pane_roy_16.png");

                // Alignment
                AddItemImage_(PaneKind.Alignment, false, false, "alignment_pane_nrm_16.png");
                AddItemImage_(PaneKind.Alignment, true, false, "alignment_pane_lck_16.png");
                AddItemImage_(PaneKind.Alignment, false, true, "alignment_pane_hdn_16.png");
                AddItemImage_(PaneKind.Alignment, true, true, "alignment_pane_hdn_16.png");
                AddReadOnlyItemImage_(PaneKind.Alignment, "alignment_pane_roy_16.png");

                // Scissor
                AddItemImage_(PaneKind.Scissor, false, false, "scissor_pane_nrm_16.png");
                AddItemImage_(PaneKind.Scissor, true, false, "scissor_pane_lck_16.png");
                AddItemImage_(PaneKind.Scissor, false, true, "scissor_pane_hdn_16.png");
                AddItemImage_(PaneKind.Scissor, true, true, "scissor_pane_hdn_16.png");
                AddReadOnlyItemImage_(PaneKind.Scissor, "scissor_pane_roy_16.png");
            }

            _animationOverrayIcon = ImageResMgr.GetManifestResourcePng("overray_animation.png");
            _commentOverrayIcon = ImageResMgr.GetManifestResourcePng("overray_comment.png");
            _warningOverrayIcon = ImageResMgr.GetManifestResourcePng("overray_warning.png");
            _prunedPaneOverrayIcon = ImageResMgr.GetManifestResourcePng("overray_pruned.png");

            _animCommentOverrayIcon = new Bitmap(_animationOverrayIcon);
            using (Graphics gfx = Graphics.FromImage(_animCommentOverrayIcon))
            {
                gfx.DrawImage(_commentOverrayIcon, Point.Empty);
            }
        }

        /// <summary>
        /// ImageList にペインアイテムイメージをコピーします。
        /// </summary>
        static public void CopyPaneItemImage( ImageList dstList )
        {
            Debug.Assert( dstList != null );
            foreach( string imageKeyString in _paneItemImageList.Images.Keys )
            {
                dstList.Images.Add( imageKeyString, _paneItemImageList.Images[imageKeyString] );
            }
        }

        /// <summary>
        ///
        /// </summary>
        static public Image GetAnimationOverrayIcon()
        {
            return _animationOverrayIcon;
        }

        /// <summary>
        /// コメント用オーバーレイアイコン
        /// </summary>
        static public Image GetCommentOverrayIcon()
        {
            return _commentOverrayIcon;
        }

        /// <summary>
        /// アニメーションとコメント用オーバーレイアイコン
        /// </summary>
        static public Image GetAnimAndCommentOverrayIcon()
        {
            return _animCommentOverrayIcon;
        }

        /// <summary>
        /// 警告用オーバーレイアイコン
        /// </summary>
        static public Image GetWarningOverrayIcon()
        {
            return _warningOverrayIcon;
        }

        /// <summary>
        /// 刈り込まれてなくなるペインオーバーレイアイコン
        /// </summary>
        static public Image GetPrunedPaneOverrayIcon()
        {
            return _prunedPaneOverrayIcon;
        }

        /// <summary>
        /// ImageList にアクセスするキー文字列を取得します。
        /// </summary>
        static public string GetPaneImageKindAsString( IPane pane )
        {
            Debug.Assert( pane != null );
            return GetPaneImageKindAsString( pane.PaneKind, pane.Locked, pane.Hidden, pane.IsReadOnlyLocked );
        }

        /// <summary>
        /// ImageList にアクセスするキー文字列を取得します。
        /// </summary>
        static public string GetPaneImageKindAsString(PaneKind kind, bool enabled)
        {
            return GetPaneImageKindAsString(kind, false, !enabled, false);
        }

        /// <summary>
        /// ImageList にアクセスするキー文字列を取得します。
        /// </summary>
        public static string GetPaneImageKindAsString(PaneKind kind, bool locked, bool hidden, bool readOnly)
        {
            return readOnly ? kind.ToString() + readOnly.ToString() : kind.ToString() + locked.ToString() + hidden.ToString();
        }

        /// <summary>
        /// ペインに対応するImageを取得します。
        /// </summary>
        static public Image GetPaneImage(IPane pane)
        {
            return _paneItemImageList.Images[GetPaneImageKindAsString(pane.PaneKind, pane.Locked, pane.Hidden, pane.IsReadOnlyLocked)];
        }

        /// <summary>
        /// ペインに対応するImageを取得します。
        /// </summary>
        static public Image GetPaneImage(PaneKind paneKind, bool enabled)
        {
            return _paneItemImageList.Images[GetPaneImageKindAsString(paneKind, false, !enabled, false)];
        }
    }
}
