﻿// --------------------------------------------------------------------------------
// <copyright>
// Copyright (C)Nintendo All rights reserved.
//
// These coded instructions, statements, and computer programs contain proprietary
// information of Nintendo and/or its licensed developers and are protected by
// national and international copyright laws. They may not be disclosed to third
// parties or copied or duplicated in any form, in whole or in part, without the
// prior written consent of Nintendo.
//
// The content herein is highly confidential and should be handled accordingly.
// </copyright>
// --------------------------------------------------------------------------------

using System;
using System.IO;
using System.Runtime.InteropServices;
using System.Windows.Forms;

namespace LayoutEditor.Utility
{
    /// <summary>
    /// アプリケーションメッセージボックスクラス。
    /// </summary>
    public sealed class MsgBox
    {
        // キャプション
        private static readonly string _caption;

        /// <summary>
        /// タイプコンストラクタ。
        /// </summary>
        static MsgBox()
        {
            // キャプション初期化
            _caption = Application.ProductName;
        }

        /// <summary>
        /// 情報メッセージ。
        /// </summary>
        public static void Information(string message)
        {
            Show(
                message,
                MessageBoxButtons.OK,
                MessageBoxIcon.Information,
                MessageBoxDefaultButton.Button1
            );
        }

        /// <summary>
        /// 情報メッセージ（文字列リソース版）。
        /// </summary>
        public static void InformationRes(string strID)
        {
            Information(StringResMgr.Get(strID));
        }

        /// <summary>
        /// 警告メッセージ。
        /// </summary>
        public static void Warning(string message)
        {
            Show(
                message,
                MessageBoxButtons.OK,
                MessageBoxIcon.Warning,
                MessageBoxDefaultButton.Button1
            );
        }

        /// <summary>
        /// 警告メッセージ（文字列リソース版）。
        /// </summary>
        public static void WarningRes(string strID)
        {
            Warning(StringResMgr.Get(strID));
        }

        /// <summary>
        /// エラーメッセージ。
        /// </summary>
        public static void Error(string message)
        {
            Show(
                message,
                MessageBoxButtons.OK,
                MessageBoxIcon.Error,
                MessageBoxDefaultButton.Button1
            );
        }

        /// <summary>
        /// エラーメッセージ（文字列リソース版）。
        /// </summary>
        public static void ErrorRes(string strID)
        {
            Error(StringResMgr.Get(strID));
        }

        /// <summary>
        /// ＯＫ、キャンセルメッセージ。
        /// </summary>
        public static bool OkCancel(string message)
        {
            DialogResult result = Show(
                message,
                MessageBoxButtons.OKCancel,
                MessageBoxIcon.Question,
                MessageBoxDefaultButton.Button2
            );
            return result == DialogResult.OK;
        }

        /// <summary>
        /// ＯＫ、キャンセルメッセージ（文字列リソース版）。
        /// </summary>
        public static bool OkCancelRes(string strID)
        {
            return OkCancel(StringResMgr.Get(strID));
        }

        /// <summary>
        /// はい、いいえメッセージ。
        /// </summary>
        public static bool YesNo(string message)
        {
            DialogResult result = Show(
                message,
                MessageBoxButtons.YesNo,
                MessageBoxIcon.Question,
                MessageBoxDefaultButton.Button2
            );
            return result == DialogResult.Yes;
        }

        /// <summary>
        /// はい、いいえメッセージ（文字列リソース版）。
        /// </summary>
        public static bool YesNoRes(string strID)
        {
            return YesNo(StringResMgr.Get(strID));
        }

        /// <summary>
        /// はい、いいえ、キャンセルメッセージ。
        /// </summary>
        public static DialogResult YesNoCancel(string message)
        {
            DialogResult result = Show(
                message,
                MessageBoxButtons.YesNoCancel,
                MessageBoxIcon.Question,
                MessageBoxDefaultButton.Button3
            );
            return result;
        }

        /// <summary>
        /// はい、いいえ、キャンセルメッセージ（文字列リソース版）。
        /// </summary>
        public static DialogResult YesNoCancelRes(string strID)
        {
            return YesNoCancel(StringResMgr.Get(strID));
        }

        /// <summary>
        /// コンストラクタ。
        /// </summary>
        private MsgBox() {}

        /// <summary>
        /// メッセージボックス表示
        /// </summary>
        private static DialogResult Show(
            string					text,
            MessageBoxButtons		buttons,
            MessageBoxIcon			icon,
            MessageBoxDefaultButton defaultButton
        )
        {
            // カーソルを保存
            Cursor cursor = Cursor.Current;

            // 表示
            DialogResult result = MessageBox.Show(text, _caption, buttons, icon, defaultButton);

            // 表示前のカーソルに戻す
            Cursor.Current = cursor;

            return result;
        }
    }
}
