﻿// --------------------------------------------------------------------------------
// <copyright>
// Copyright (C)Nintendo All rights reserved.
//
// These coded instructions, statements, and computer programs contain proprietary
// information of Nintendo and/or its licensed developers and are protected by
// national and international copyright laws. They may not be disclosed to third
// parties or copied or duplicated in any form, in whole or in part, without the
// prior written consent of Nintendo.
//
// The content herein is highly confidential and should be handled accordingly.
// </copyright>
// --------------------------------------------------------------------------------

using System;
using System.IO;
using System.Net;
using System.Net.Sockets;

namespace Nintendo.McsServer
{
    public enum ChannelRequestResult : uint
    {
        Success,                // 成功
        ProtocolError,          // 通信プロトコルのエラーです。
        AlreadyRegisted,        // 既に登録されているチャンネルです。
        SystemReserved,         // システムで予約しています。
        NotConnect              // ターゲットと接続されていません。
    }

    [Flags()]
    public enum ChannelRequestFlags : uint
    {
        NoCheckConnect  = 1 << 0    // ターゲットとの接続状態に依存しない
    }

    /// <summary>
    /// ChannelRequest の概要の説明です。
    /// </summary>
    public class ChannelRequest
    {
        public static ChannelRequestResult Request(Socket socket, uint channel)
        {
            return Request(socket, channel, 0);
        }

        public static ChannelRequestResult Request(Socket socket, uint channel, uint flag)
        {
            NetworkStream netsm = new NetworkStream(socket, FileAccess.ReadWrite, false);

            // 接続の確立のプロトコルを送信
            using (MemoryStream ms = new MemoryStream())
            {
                BinaryWriter bw = new BinaryWriter(ms);
                // ヘッダ
                uint val = 1;
                bw.Write(HostToNet(val));       // コマンド(ターゲットデバイスとの接続)
                val = 4 + 4;
                bw.Write(HostToNet(val));       // データ長

                // データ
                bw.Write(HostToNet(channel));   // 希望するチャンネル
                bw.Write(HostToNet(flag));      // フラグ
                bw.Flush();

                netsm.Write(ms.GetBuffer(), 0, (int)ms.Length);    // メッセージ長
            }

            // レスポンスをチェック
            BinaryReader br = new BinaryReader(netsm);
            uint chunkID = NetToHost(br.ReadUInt32());
            uint msgSize = NetToHost(br.ReadUInt32());
            if (msgSize > 0x10000)      // メッセージ長が大きすぎる
            {
                return ChannelRequestResult.ProtocolError;
            }
            if (msgSize != 4) // メッセージ長が期待している長さではない
            {
                for (int i = 0; i < msgSize; ++i)   // メッセージを空読み
                {
                    br.ReadByte();
                }
                return ChannelRequestResult.ProtocolError;
            }

            return (ChannelRequestResult)NetToHost(br.ReadUInt32());
        }

        static short HostToNet(short val)
        {
            return IPAddress.HostToNetworkOrder(val);
        }

        static ushort HostToNet(ushort val)
        {
            return (ushort)IPAddress.HostToNetworkOrder((short)val);
        }

        static int HostToNet(int val)
        {
            return IPAddress.HostToNetworkOrder(val);
        }

        static uint HostToNet(uint val)
        {
            return (uint)IPAddress.HostToNetworkOrder((int)val);
        }

        static short NetToHost(short val)
        {
            return IPAddress.NetworkToHostOrder(val);
        }

        static ushort NetToHost(ushort val)
        {
            return (ushort)IPAddress.NetworkToHostOrder((short)val);
        }

        static int NetToHost(int val)
        {
            return IPAddress.NetworkToHostOrder(val);
        }

        static uint NetToHost(uint val)
        {
            return (uint)IPAddress.NetworkToHostOrder((int)val);
        }

    }
}
