﻿// --------------------------------------------------------------------------------
// <copyright>
// Copyright (C)Nintendo. All rights reserved.
//
// These coded instructions, statements, and computer programs contain proprietary
// information of Nintendo and/or its licensed developers and are protected by
// national and international copyright laws. They may not be disclosed to third
// parties or copied or duplicated in any form, in whole or in part, without the
// prior written consent of Nintendo.
//
// The content herein is highly confidential and should be handled accordingly.
// </copyright>
// --------------------------------------------------------------------------------
using System;
using System.Collections.Generic;
using System.Text;
using System.Windows;
using System.Windows.Forms;
using System.Drawing;
using System.Diagnostics;

namespace LayoutEditor.Forms.ToolWindows.common
{
    using LECore.Manipulator;

    /// <summary>
    /// ツリーノードのドラッグドロップ編集処理の
    /// 状態管理を行うクラスです。
    /// </summary>
    class TreeNodeDragDropMgr
    {
        //------------------------------------------------------------
        // 型宣言
        /// <summary>
        /// 対象ノードに対して、ドロップ操作の有効なモードを判定します。
        /// </summary>
        public delegate TreeNodeDropHint CheckValidToDropHandler( TreeNode parentNode, TreeNode[] childCandidateNodeSet );
        /// <summary>
        /// ノードセットが、ドラッグ開始に適しているか判定します。
        /// </summary>
        public delegate bool OnBeginDragHandler( out TreeNode[] selectedNodeSet );
        /// <summary>
        /// ドロップ時の処理を行います。
        /// </summary>
        public delegate void OnDropHandler( TreeNode receiverNode, TreeNode[] dropedNodeSet, TreeNodeDropHint mode );

        /// <summary>
        /// TreeNodeドラッグドロップ操作の種類
        /// </summary>
        [FlagsAttribute]
        public enum TreeNodeDropHint
        {
            None = 0,
            InsertPrev = 1,
            AddChild = 2,
            InsertNext = 4,
            DropModeMask = InsertPrev | AddChild | InsertNext,
            CopyItemFlag = 8,
        }

        //------------------------------------------------------------
        // フィールド
        // ドロップモード
        TreeNodeDropHint _dropMode = TreeNodeDropHint.None;
        // ひとつ前のドロップ対象
        Rectangle			_prevTargetRect	= Rectangle.Empty;

        // 各種ユーザ拡張ハンドラ(コンストラクタで必ず指定します)
        readonly CheckValidToDropHandler		_checkValidToDropHandler;
        readonly OnBeginDragHandler			_onBeginDragHandler;
        readonly OnDropHandler				_onDropHandler;


        // 対象ツリーノード
        readonly TreeView _targetTreeView;
        // ツリースクロール管理クラス
        readonly TreeViewScroller _treeViewScroller;

        //------------------------------------------------------------
        // プロパティ
        /// <summary>
        /// ドラッグドロップによる複製機能の有効無効
        /// </summary>
        public bool EnableDragDropDuplicate
        {
            get; set;
        }

        private bool _IsPressAlt
        {
            get
            {
                return EnableDragDropDuplicate && ((Control.ModifierKeys & Keys.Alt) != 0);
            }
        }

        /// <summary>
        /// コンストラクタ
        /// 利用クラスで実装すべき、処理を
        /// デリゲータとして、指定する必要があります。
        ///
        /// Gof. テンプレートメソッドパターン（継承ベース）に
        /// リファクタクリングしたほうが自然かもしれません。）
        ///
        /// </summary>
        public TreeNodeDragDropMgr(
            TreeView targetTreeView,
            CheckValidToDropHandler checkValidToDropHandler,
            OnBeginDragHandler onBeginDragHandler,
            OnDropHandler onDropHandler )
        {
            _targetTreeView = targetTreeView;
            _treeViewScroller = new TreeViewScroller( targetTreeView );

            _checkValidToDropHandler = checkValidToDropHandler;
            _onBeginDragHandler = onBeginDragHandler;
            _onDropHandler = onDropHandler;

            _targetTreeView.DragDrop += Event_TvwTarget_DragDrop;
            _targetTreeView.ItemDrag += Event_TvwTarget_ItemDrag;
            _targetTreeView.DragOver += Event_TvwTarget_DragOver;
            _targetTreeView.DragLeave += Event_TvwTarget_DragLeave;

            _targetTreeView.MouseUp += Event_TvwTarget_MouseUp;
        }



        #region ドラッグ・ドロップ関連



        /// <summary>
        /// ドロップハンドラ
        /// </summary>
        private void Event_TvwTarget_DragDrop( object sender, System.Windows.Forms.DragEventArgs e )
        {
            // TreeNode以外のドロップは処理しません。
            if( !e.Data.GetDataPresent( typeof( TreeNode[] ).ToString(), false ) )
            {
                return;
            }


            Point pt = ( (TreeView)sender ).PointToClient( new Point( e.X, e.Y ) );

            TreeNode[] childNodeSet = e.Data.GetData( typeof( TreeNode[] ) ) as TreeNode[];
            TreeNode targetNode = ( (TreeView)sender ).GetNodeAt( pt );
            if(!EnableDragDropDuplicate && targetNode == null)
            {
                return;
            }

            _onDropHandler( targetNode, childNodeSet, _dropMode );
        }

        /// <summary>
        /// ノードアイテムドラッグハンドラ
        /// </summary>
        private void Event_TvwTarget_ItemDrag( object sender, System.Windows.Forms.ItemDragEventArgs e )
        {
            TreeView tv = sender as TreeView;
            tv.Focus();


            // ノードがドラッグ開始に有効か判定します。
            // _onBeginDragHandler は 利用クラスがカスタマイズしている点に注意
            TreeNode[] selectedNodeSet;
            if( !_onBeginDragHandler( out selectedNodeSet ) )
            {
                tv.DoDragDrop(selectedNodeSet, DragDropEffects.None);
                return;
            }


            // 親が含まれるノードを削除します。
            TreeNode[] draggedItemsSet =
                Array.FindAll( selectedNodeSet, delegate( TreeNode obj )
            {
                TreeNode node = obj.Parent;
                while( node != null )
                {
                    if( Array.IndexOf( selectedNodeSet, node ) != -1 )
                    {
                        return false;
                    }
                    node = node.Parent;
                }

                return true;
            } );

            // Y位置順にソートします。
            Array.Sort( draggedItemsSet, delegate( TreeNode lhs, TreeNode rhs )
            {
                return lhs.Bounds.Y.CompareTo( rhs.Bounds.Y );
            }
            );

            //ノードのドラッグを開始する
            DragDropEffects dde = tv.DoDragDrop( draggedItemsSet, DragDropEffects.All );
        }

        /// <summary>
        /// あるTreeNodeが別のTreeNodeの子ノードか調査します。
        /// </summary>
        private static bool IsChildNode_( TreeNode parent, TreeNode child )
        {
            if( child.Parent == parent )
            {
                return true;
            }
            else if( child.Parent != null )
            {
                return IsChildNode_( parent, child.Parent );
            }
            else
            {
                return false;
            }
        }

        #region ノードドラッグドロップ DragOver処理
        /// <summary>
        /// マウス点が、ノード上端に位置するか？
        /// </summary>
        bool InLowerBound_( Point posInTreeView, TreeNode node )
        {
            int nodeLocalY = posInTreeView.Y - node.Bounds.Top;
            return nodeLocalY > node.Bounds.Height * 3 / 4;
        }

        /// <summary>
        /// マウス点が、ノード下端に位置するか？
        /// </summary>
        bool InUpperBound_( Point posInTreeView, TreeNode node )
        {
            int nodeLocalY = posInTreeView.Y - node.Bounds.Top;
            return nodeLocalY < node.Bounds.Height / 4;
        }

        /// <summary>
        /// マウス点が、ノード中心より上か？
        /// </summary>
        bool InTopOfHalf_( Point posInTreeView, TreeNode node )
        {
            int nodeLocalY = posInTreeView.Y - node.Bounds.Top;
            return nodeLocalY > node.Bounds.Height * 0.5f;
        }

        /// <summary>
        /// ドラッグドロップによる複製が可能か？
        /// </summary>
        bool CanDragDropDuplicate(TreeView tv, TreeNode[] sourceNodes, int x, int y)
        {
            // Altが押されていなければ不可
            if (!_IsPressAlt)
            {
                return false;
            }

            foreach (TreeNode node in sourceNodes)
            {
                if (node.Parent == null)
                {
                    // Rootを含む場合は不可
                    return false;
                }
            }

            Rectangle bounds = new Rectangle(0, 0, tv.Width, tv.Height);
            return EnableDragDropDuplicate && bounds.Contains(x, y);
        }

        /// <summary>
        /// プレースホルダの描画を行います。
        /// </summary>
        void DrawInsertPlaceHolder_( TreeView tv, TreeNode target, TreeNodeDropHint dropMode, ref Rectangle drawRect )
        {
            Graphics g = tv.CreateGraphics();

            int left = target.Bounds.Left;
            int right = target.Bounds.Right + 8;

            if( dropMode == TreeNodeDropHint.AddChild )
            {


                int lineY = ( target.Bounds.Bottom + target.Bounds.Top ) / 2;


                Point[] RightTriangle = new Point[5]{
                                                        new Point(right, lineY - 4),
                                                        new Point(right, lineY + 4),
                                                        new Point(right - 4, lineY),
                                                        new Point(right - 4, lineY - 1),
                                                        new Point(right, lineY - 5)};

                g.DrawRectangle( new Pen( Color.FromArgb( 64, Color.DarkBlue ), 2.0f ), target.Bounds );
                g.FillPolygon( System.Drawing.Brushes.DarkBlue, RightTriangle );
                drawRect = Rectangle.FromLTRB( target.Bounds.Left, target.Bounds.Top, right, target.Bounds.Bottom );
                drawRect.Inflate( 2, 2 );
            }
            else
            {
                int lineY = ( dropMode == TreeNodeDropHint.InsertNext ) ?
                    target.Bounds.Bottom : target.Bounds.Top;


                Pen pen = new Pen( Brushes.Black, 2.0f );

                Point[] LeftTriangle = new Point[5]{
                                                       new Point(left, lineY - 4),
                                                       new Point(left, lineY + 4),
                                                       new Point(left + 4, lineY ),
                                                       new Point(left + 4, lineY - 1),
                                                       new Point(left, lineY - 5)};

                Point[] RightTriangle = new Point[5]{
                                                        new Point(right, lineY - 4),
                                                        new Point(right, lineY + 4),
                                                        new Point(right - 4, lineY),
                                                        new Point(right - 4, lineY - 1),
                                                        new Point(right, lineY - 5)};


                g.FillPolygon( System.Drawing.Brushes.Black, LeftTriangle );
                g.FillPolygon( System.Drawing.Brushes.Black, RightTriangle );

                g.DrawLine( pen, new Point( left, lineY ), new Point( right - 2, lineY ) );

                drawRect = Rectangle.FromLTRB( left, lineY - 4, right, lineY + 4 );
            }
        }

        /// <summary>
        /// マウス位置からドロップ動作の種類を取得します。
        /// </summary>
        TreeNodeDropHint GetDrapModeByPosition_(
            Point posInTreeView,
            TreeNode target,
            TreeNodeDropHint validMode )
        {
            if( validMode ==
                (TreeNodeDropHint.AddChild |
                TreeNodeDropHint.InsertPrev |
                TreeNodeDropHint.InsertNext) )
            {
                // すべて選択可能な場合は、領域を３分割して判定します。
                if( InLowerBound_( posInTreeView, target ) )
                {
                    return TreeNodeDropHint.InsertNext;
                }
                else if( InUpperBound_( posInTreeView, target ) )
                {
                    return TreeNodeDropHint.InsertPrev;
                }
                else
                {
                    return TreeNodeDropHint.AddChild;
                }
            }
            else if( validMode ==
                ( TreeNodeDropHint.InsertPrev |
                TreeNodeDropHint.InsertNext))
            {
                // 選択肢が２つの場合は、２分割して判定します。
                if( InTopOfHalf_( posInTreeView, target ) )
                {
                    return TreeNodeDropHint.InsertNext;
                }
                else
                {
                    return TreeNodeDropHint.InsertPrev;
                }
            }
            else if (validMode ==
                (TreeNodeDropHint.AddChild |
                TreeNodeDropHint.InsertPrev))
            {
                // 選択肢が２つの場合は、２分割して判定します。
                if (InTopOfHalf_(posInTreeView, target))
                {
                    return TreeNodeDropHint.InsertPrev;
                }
                else
                {
                    return TreeNodeDropHint.AddChild;
                }
            }
            else if (validMode ==
                (TreeNodeDropHint.AddChild |
                TreeNodeDropHint.InsertNext))
            {
                // 選択肢が２つの場合は、２分割して判定します。
                if (InTopOfHalf_(posInTreeView, target))
                {
                    return TreeNodeDropHint.AddChild;
                }
                else
                {
                    return TreeNodeDropHint.InsertNext;
                }
            }
            else
            {
                // 選択肢がひとつ以下なら、おのずと定まります。
                return validMode;
            }
        }

        /// <summary>
        /// マウスアップ
        /// </summary>
        private void Event_TvwTarget_MouseUp( object sender, System.Windows.Forms.MouseEventArgs e )
        {
            if( _treeViewScroller.Scrolling )
            {
                _treeViewScroller.StopScrolling();
            }
        }

        /// <summary>
        /// ノードアイテムドラッグオーバーハンドラ
        /// </summary>
        private void Event_TvwTarget_DragOver( object sender, System.Windows.Forms.DragEventArgs e )
        {
            e.Effect = DragDropEffects.None;

            //ドラッグされているデータがTreeNodeか調べる
            if( e.Data.GetDataPresent( typeof( TreeNode[] ) ) )
            {
                if( ( e.AllowedEffect & DragDropEffects.Move ) == DragDropEffects.Move )
                {
                    TreeView tv = sender as TreeView;
                    //ドラッグされているNodeを取得する
                    TreeNode[] sourceNodeSet = e.Data.GetData( typeof( TreeNode[] ) ) as TreeNode[];



                    // 自分以外からのツリーが発見されたらドロップ処理しません
                    foreach( TreeNode sourceNode in sourceNodeSet )
                    {
                        if( tv != sourceNode.TreeView )
                        {
                            e.Effect = DragDropEffects.None;
                            return;
                        }
                    }


                    Point posInTreeView = tv.PointToClient( new Point( e.X, e.Y ) );
                    TreeNode target = tv.GetNodeAt( posInTreeView );

                    // ノード以外には、ドロップを処理しません。
                    // ドラッグドロップによる複製機能が有効な場合、ツリービューの領域内であればドロップを処理します。
                    if (target == null)
                    {
                        if (CanDragDropDuplicate(tv, sourceNodeSet, posInTreeView.X, posInTreeView.Y))
                        {
                            e.Effect = DragDropEffects.Copy;
                        }
                        ClearPlaceHolder_();
                        return;
                    }

                    // スクロール判定をします。
                    _treeViewScroller.UpdateScrolling( new Point( e.X, e.Y ) );


                    // スクロール中は、ドロップを処理しません。
                    if( _treeViewScroller.Scrolling )
                    {
                        ClearPlaceHolder_();
                        return;
                    }

                    // 親子として妥当か調査します。
                    TreeNodeDropHint validDrapMode = _checkValidToDropHandler( target, sourceNodeSet );
                    TreeNodeDropHint dropModeFlag  = validDrapMode & TreeNodeDropHint.DropModeMask;

                    // 親子として妥当ならば...
                    if( dropModeFlag != TreeNodeDropHint.None )
                    {
                        // マウス位置から、ドロップ形式を判定します。
                        TreeNodeDropHint dropMode =
                            GetDrapModeByPosition_( posInTreeView, target, dropModeFlag );

                        Rectangle drawRect = Rectangle.Empty;
                        DrawInsertPlaceHolder_( tv, target, dropMode, ref drawRect );

                        // 再描画します。
                        UpdatePlaceHolder_( drawRect );

                        // ドロップ処理をを有効に設定します。
                        _dropMode = dropMode | ( validDrapMode & ~TreeNodeDropHint.DropModeMask );

                        // ドロップモードの変更。
                        if( ( _dropMode & TreeNodeDropHint.CopyItemFlag ) != 0 || _IsPressAlt )
                        {
                            e.Effect = DragDropEffects.Copy;
                        }
                        else
                        {
                            e.Effect = DragDropEffects.Move;
                        }
                    }
                    else
                    {
                        ClearPlaceHolder_();
                    }
                }
            }
        }

        #endregion ノードドラッグドロップ DragOver処理

        /// <summary>
        /// プレースホルダ位置を更新します。
        /// </summary>
        void UpdatePlaceHolder_( Rectangle newHolderBound )
        {
            if( _prevTargetRect != newHolderBound )
            {
                ClearPlaceHolder_();
                _prevTargetRect = newHolderBound;
            }
        }

        /// <summary>
        /// プレースホルダを消去します。
        /// </summary>
        void ClearPlaceHolder_()
        {
            if( _prevTargetRect != Rectangle.Empty )
            {
                _targetTreeView.Invalidate( _prevTargetRect );
                _prevTargetRect = Rectangle.Empty;
            }
        }

        /// <summary>
        /// DragLeave
        /// </summary>
        private void Event_TvwTarget_DragLeave( object sender, System.EventArgs e )
        {
            ClearPlaceHolder_();
        }

        #endregion ドラッグ・ドロップ関連
    }
}
