﻿// --------------------------------------------------------------------------------
// <copyright>
// Copyright (C)Nintendo. All rights reserved.
//
// These coded instructions, statements, and computer programs contain proprietary
// information of Nintendo and/or its licensed developers and are protected by
// national and international copyright laws. They may not be disclosed to third
// parties or copied or duplicated in any form, in whole or in part, without the
// prior written consent of Nintendo.
//
// The content herein is highly confidential and should be handled accordingly.
// </copyright>
// --------------------------------------------------------------------------------
using System;
using System.Collections.Generic;
using System.ComponentModel;
using System.Drawing;
using System.Data;
using System.Linq;
using System.Text;
using System.Windows.Forms;
using LECore.Structures;
using LayoutEditor.Controls;
using LayoutEditor.Forms.ToolWindows.PropertyEditWindow.Adapters;
using System.Diagnostics;
using LECore.Manipulator;

namespace LayoutEditor.src.Forms.ToolWindows.common
{
    //----------------------------------------------------------

    /// <summary>
    /// テクスチャ座標を編集するコントロール
    /// </summary>
    public partial class TextureCoordEdit : UserControl
    {
        /// <summary>
        /// コンストラクタです。
        /// </summary>
        public TextureCoordEdit()
        {
            InitializeComponent();

            _ctlTexCoordLU.OnFVec2Edit += Event_CtlTexCoord_OnFVec2Edit;
            _ctlTexCoordRU.OnFVec2Edit += Event_CtlTexCoord_OnFVec2Edit;
            _ctlTexCoordLB.OnFVec2Edit += Event_CtlTexCoord_OnFVec2Edit;
            _ctlTexCoordRB.OnFVec2Edit += Event_CtlTexCoord_OnFVec2Edit;

            _ctlPixCoordLU.OnFVec2Edit += Event_CtlPixCoord_OnFVec2Edit;
            _ctlPixCoordRU.OnFVec2Edit += Event_CtlPixCoord_OnFVec2Edit;
            _ctlPixCoordLB.OnFVec2Edit += Event_CtlPixCoord_OnFVec2Edit;
            _ctlPixCoordRB.OnFVec2Edit += Event_CtlPixCoord_OnFVec2Edit;
            _pictBoxSet = new PictureBox[] { _pic0, _pic1, _pic2, _pic3 };
        }

        Action                _onChangedEvent;
        ITexCoordmanipulator _targetMnp = null;
        ITexCoordmanipulator[] _targetMnps = null;
        ITexCoordHolder[] _targets = null;
        readonly PictureBox[] _pictBoxSet;
        bool _isSystemCalled = false;

        ITexCoordHolder Target
        {
            get
            {
                return _targetMnp.TexCoordOwner;
            }
        }

        ISubScene SubScene
        {
            get
            {
                return LECore.LayoutEditorCore.Scene.CurrentISubScene;
            }
        }

        /// <summary>
        /// 初期化
        /// </summary>
        public void Initialize(Action onChangedEvent)
        {
            Debug.Assert(_onChangedEvent == null);
            Debug.Assert(onChangedEvent != null);

            _onChangedEvent = onChangedEvent;
        }

        /// <summary>
        /// 対象を設定します。
        /// </summary>
        public void SetTarget(ITexCoordmanipulator targetMnp, IEnumerable<ITexCoordmanipulator> targetMnps, Color controlColor)
        {
            _targetMnp = targetMnp;
            _targetMnps = targetMnps.ToArray();
            _targets = _targetMnps.Select(mnp => mnp.TexCoordOwner).ToArray();

            // 複数同時編集
            _ctlNumTexCoord.BackColor = controlColor;
            _ctlTexCoordLU.TextBoxBackColor = controlColor;
            _ctlTexCoordRU.TextBoxBackColor = controlColor;
            _ctlTexCoordLB.TextBoxBackColor = controlColor;
            _ctlTexCoordRB.TextBoxBackColor = controlColor;
            _ctlPixCoordLU.TextBoxBackColor = controlColor;
            _ctlPixCoordRU.TextBoxBackColor = controlColor;
            _ctlPixCoordLB.TextBoxBackColor = controlColor;
            _ctlPixCoordRB.TextBoxBackColor = controlColor;
        }


        /// <summary>
        ///
        /// </summary>
        public void UpdateProperty()
        {
            if (Target == null || Target.NumTexCoord <= 0)
            {
                this.Enabled = false;

                foreach (var pictBox in _pictBoxSet)
                {
                    pictBox.Image = null;
                    pictBox.Visible = false;
                }

                return;
            }

            this.Enabled = true;

            // テクスチャ座標数が変更されていれば...
            if (_cmbVtxCorrd.Items.Count != Target.NumTexCoord)
            {
                _cmbVtxCorrd.Items.Clear();

                int numVtxTexCoords = Target.NumTexCoord;
                if (numVtxTexCoords >= 1) { _cmbVtxCorrd.AddItem("PROP_MAT_TEX_MAP_METHOD_UV0",     0,     true); }
                if (numVtxTexCoords >= 2) { _cmbVtxCorrd.AddItem("PROP_MAT_TEX_MAP_METHOD_UV1",     1,     true); }
                if (numVtxTexCoords >= 3) { _cmbVtxCorrd.AddItem("PROP_MAT_TEX_MAP_METHOD_UV2",     2,     true); }
                if (numVtxTexCoords >= 4) { _cmbVtxCorrd.AddItem("PROP_MAT_TEX_MAP_METHOD_UV3",     3,     true); }
                if (numVtxTexCoords >= 5) { _cmbVtxCorrd.AddItem("PROP_MAT_TEX_MAP_METHOD_UV4",     4,     true); }
                if (numVtxTexCoords >= 6) { _cmbVtxCorrd.AddItem("PROP_MAT_TEX_MAP_METHOD_UV5",     5,     true); }
                if (numVtxTexCoords >= 7) { _cmbVtxCorrd.AddItem("PROP_MAT_TEX_MAP_METHOD_UV6",     6,     true); }
                if (numVtxTexCoords >= 8) { _cmbVtxCorrd.AddItem("PROP_MAT_TEX_MAP_METHOD_UV7",     7,     true); }

                // 可能ならば、先頭の項目を選択します。
                if (numVtxTexCoords >= 1)
                {
                    _cmbVtxCorrd.SelectedIndex = 0;
                }
            }

            // テクスチャ座標使用数をコントロールに設定します。
            _ctlNumTexCoord.Value = Target.NumTexCoord;

            // 利用しているテクスチャの表示を設定します。
            UpdateusedTexture_((int)_cmbVtxCorrd.SelectedItemData);

            // テクスチャ座標をGUIコントロールに設定します。
            SetTexCoordToControl_();
        }

        /// <summary>
        /// テクスチャ座標をGUIコントロールに設定します。
        /// </summary>
        void SetTexCoordToControl_()
        {
            int coord4Idx = (int)_cmbVtxCorrd.SelectedItemData;

            UpdateusedTexture_(coord4Idx);

            // TexCoord4 srcCoord4 = _TargetPictureMpn.IPicture.GetTexCoord4(coord4Idx);
            TexCoord4 srcCoord4 = Target.GetTexCoord4(coord4Idx);

            if (srcCoord4 != null)
            {
                Fvec2EditUpDownPanel.InvokeEvent = false;

                // 選択されたUV座標値を設定します。
                _ctlTexCoordLU.FVec2Value = srcCoord4.LT;
                _ctlTexCoordRU.FVec2Value = srcCoord4.RT;
                _ctlTexCoordLB.FVec2Value = srcCoord4.LB;
                _ctlTexCoordRB.FVec2Value = srcCoord4.RB;

                Texture texture = GetPixelEditTargetTexture(coord4Idx, Target);
                if (texture != null)
                {
                    _pnlPix.Enabled = true;
                    _lblPixNG.Visible = false;

                    float width = texture.Size.Width;
                    float height = texture.Size.Height;

                    _ctlPixCoordLU.FVec2Value = new FVec2(srcCoord4.LT.X * width,
                                                           srcCoord4.LT.Y * height);
                    _ctlPixCoordRU.FVec2Value = new FVec2(srcCoord4.RT.X * width,
                                                           srcCoord4.RT.Y * height);
                    _ctlPixCoordLB.FVec2Value = new FVec2(srcCoord4.LB.X * width,
                                                           srcCoord4.LB.Y * height);
                    _ctlPixCoordRB.FVec2Value = new FVec2(srcCoord4.RB.X * width,
                                                           srcCoord4.RB.Y * height);
                }
                else
                {
                    _ctlPixCoordLU.FVec2Value = new FVec2(0,0);
                    _ctlPixCoordRU.FVec2Value = new FVec2(0,0);
                    _ctlPixCoordLB.FVec2Value = new FVec2(0,0);
                    _ctlPixCoordRB.FVec2Value = new FVec2(0,0);

                    _pnlPix.Enabled = false;
                    _lblPixNG.Visible = true;
                }

                Fvec2EditUpDownPanel.InvokeEvent = true;
            }
        }

        /// <summary>
        /// 利用しているテクスチャの表示
        /// </summary>
        private void UpdateusedTexture_(int coord4Idx)
        {
            var textures = Target.IMaterialTexMapSet.Where((matTex) => matTex.IsActive && (int)matTex.ITexGen.Method == coord4Idx).ToList();

            for (int i = 0; i < _pictBoxSet.Length; i++)
            {
                if (i < textures.Count)
                {
                    var texImg = TextureManager.Search(textures[i].TexImgName);
                    if (texImg != null)
                    {
                        _pictBoxSet[i].Image = texImg.RenderingImage;
                    }
                    else
                    {
                        _pictBoxSet[i].Image = null;
                    }

                    _pictBoxSet[i].Visible = true;
                }
                else
                {
                    _pictBoxSet[i].Image = null;
                    _pictBoxSet[i].Visible = false;
                }

                _lblUsedTextureNone.Visible = _pictBoxSet.All((box) => !box.Visible);
            }
        }

        /// <summary>
        ///
        /// </summary>
        private Texture GetPixelEditTargetTexture(int currentTexCoordIndex, ITexCoordHolder target)
        {
            // 利用座標から使っているテクスチャを列挙する
            // IPicture picture = _TargetPictureMpn.IPicture;
            // var textures = picture.IMaterial.IMaterialTexMapSet.Where((matTex) => matTex.IsActive && (int)matTex.ITexGen.Method == currentTexCoordIndex);
            var textures = target.IMaterialTexMapSet.Where((matTex) => matTex.IsActive && (int)matTex.ITexGen.Method == currentTexCoordIndex);

            int numUsedTex = textures.Count();

            if (numUsedTex == 0)
            {
                return null;
            }
            else if (numUsedTex == 1)
            {
                // 一つならそれがターゲット
                return TextureManager.Search(textures.First().TexImgName);
            }
            else
            {
                // ２つ以上ならサイズがすべて同じだったらOK。
                var firstTexImg = TextureManager.Search(textures.First().TexImgName);
                if (firstTexImg != null)
                {
                    bool areAllTextureSizeSame = textures.All((matTex) =>
                    {
                        var texImg = TextureManager.Search(matTex.TexImgName);
                        return texImg != null && texImg.Size == firstTexImg.Size;
                    });

                    if (areAllTextureSizeSame)
                    {
                        return firstTexImg;
                    }
                }
            }

            return null;
        }

        //----------------------------------------------------------
        // イベントハンドラ
        //----------------------------------------------------------

        /// <summary>
        /// コンボボックス選択アイテム変更ハンドラ
        /// </summary>
        private void Event_CmbVtxCorrd_SelectedIndexChanged(object sender, System.EventArgs e)
        {
            SetTexCoordToControl_();
        }

        /// <summary>
        /// 使用するステージ数
        /// </summary>
        private void Event_CtlNumTexCoord_ValueChanged(object sender, System.EventArgs e)
        {
            if (_isSystemCalled)
            {
                return;
            }

            IntTextBox intTextBox = sender as IntTextBox;
            int numTexCoordUsed = intTextBox.Value;
            int numTextureUsed = Target.IMaterialTexMapSet.Length;

            // 使用テクスチャ数よりも大きな数の、テクスチャ座標使用を設定した。
            if (numTexCoordUsed > numTextureUsed)
            {
                // 変更は却下され、GUIの値をリセットします。
                _isSystemCalled = true;
                intTextBox.Value = Target.NumTexCoord;
                _isSystemCalled = false;
                return;
            }

            SubScene.BeginMassiveModify();

            for (int i = 0; i < _targets.Count(); i++)
            {
                numTextureUsed = _targets.ElementAt(i).IMaterialTexMapSet.Length;

                if (numTexCoordUsed <= numTextureUsed)
                {
                    // 正常に変更された
                    _targetMnps.ElementAt(i).NumTexCoord = numTexCoordUsed;
                    NotifyPropertyChanged();
                }
            }

            SubScene.EndMassiveModify();
        }

        /// <summary>
        /// 座標コントロール変更ハンドラ
        /// GUIコントロールの変更をデータに反映します。
        /// </summary>
        private void Event_CtlTexCoord_OnFVec2Edit(object sender, FVec2EditEventArgs args)
        {
            int coord4Idx = (int)_cmbVtxCorrd.SelectedItemData;

            SubScene.BeginMassiveModify();

            for (int i = 0; i < _targets.Length; i++)
            {
                TexCoord4 dstCoord4 = _targets[i].GetTexCoord4(coord4Idx);

                // 操作対象が設定されていない場合は、何もしません。
                if (dstCoord4 == null)
                {
                    continue;
                }
                dstCoord4 = new TexCoord4(dstCoord4);

                Fvec2EditUpDownPanel udPanel = sender as Fvec2EditUpDownPanel;

                if (udPanel == _ctlTexCoordLU)
                {
                    dstCoord4.LT = udPanel.FVec2Value;
                }
                else if (udPanel == _ctlTexCoordRU)
                {
                    dstCoord4.RT = udPanel.FVec2Value;
                }
                else if (udPanel == _ctlTexCoordLB)
                {
                    dstCoord4.LB = udPanel.FVec2Value;
                }
                else if (udPanel == _ctlTexCoordRB)
                {
                    dstCoord4.RB = udPanel.FVec2Value;
                }

                _targetMnps[i].SetTexCoord(dstCoord4, coord4Idx);
                NotifyPropertyChanged();
            }

            SubScene.EndMassiveModify();
        }

        /// <summary>
        ///
        /// </summary>
        private int TargetTextureCount
        {
            get
            {
                return Target.IMaterialTexMapSet.Length;
            }
        }

        /// <summary>
        ///
        /// </summary>
        private void Event_CtlPixCoord_OnFVec2Edit(object sender, FVec2EditEventArgs args)
        {
            int coord4Idx = (int)_cmbVtxCorrd.SelectedItemData;

            bool modifyX = args.Name.Contains("X");
            bool modifyY = args.Name.Contains("Y");

            SubScene.BeginMassiveModify();

            for (int i = 0; i < _targets.Length; i++)
            {
                TexCoord4 dstCoord4 = _targets[i].GetTexCoord4(coord4Idx);

                // 操作対象が設定されていない場合は、何もしません。
                if (dstCoord4 == null)
                {
                    continue;
                }
                dstCoord4 = new TexCoord4(dstCoord4);

                Fvec2EditUpDownPanel panel = sender as Fvec2EditUpDownPanel;
                FVec2 vec = panel.FVec2Value;


                Texture texture = this.GetPixelEditTargetTexture(coord4Idx, _targets[i]);
                if (texture == null)
                {
                    continue;
                }

                vec.X = vec.X / texture.Size.Width;
                vec.Y = vec.Y / texture.Size.Height;

                if (panel == _ctlPixCoordLU)
                {
                    dstCoord4.LT = GetVecValue(dstCoord4.LT, vec, modifyX, modifyY);
                }
                else if (panel == _ctlPixCoordRU)
                {
                    dstCoord4.RT = GetVecValue(dstCoord4.RT, vec, modifyX, modifyY);
                }
                else if (panel == _ctlPixCoordLB)
                {
                    dstCoord4.LB = GetVecValue(dstCoord4.LB, vec, modifyX, modifyY);
                }
                else if (panel == _ctlPixCoordRB)
                {
                    dstCoord4.RB = GetVecValue(dstCoord4.RB, vec, modifyX, modifyY);
                }

                _targetMnps[i].SetTexCoord(dstCoord4, coord4Idx);
                NotifyPropertyChanged();
            }

            SubScene.EndMassiveModify();
        }

        FVec2 GetVecValue(FVec2 dst, FVec2 src, bool modifyX, bool modifyY)
        {
            var val = dst;
            if (modifyX) val.X = src.X;
            if (modifyY) val.Y = src.Y;
            return val;
        }

        /// <summary>
        /// 現在の捜査対象テクスチャ座標を更新します。
        /// </summary>
        /// <param name="modifyFunc"></param>
        void ModifyCurrentTragetTexCoord_(TexCoord4.TexCoord4ModifyFunc modifyFunc)
        {
            int coord4Idx = (int)_cmbVtxCorrd.SelectedItemData;

            SubScene.BeginMassiveModify();

            for (int i = 0; i < _targets.Length; i++)
            {
                TexCoord4 srcCoord4 = _targets[i].GetTexCoord4(coord4Idx);

                // 操作対象が設定されていない場合は、何もしません。
                if (srcCoord4 == null)
                {
                    continue;
                }

                // 更新処理を実行します。
                TexCoord4 newTexCoord = new TexCoord4();
                newTexCoord.Set(srcCoord4);
                modifyFunc(newTexCoord);

                // 新しいデータを設定
                _targetMnps[i].SetTexCoord(newTexCoord, coord4Idx);
            }

            SubScene.EndMassiveModify();
        }

        /// <summary>
        /// UV水平方向反転
        /// </summary>
        private void Event_BtnFlipH_Click(object sender, EventArgs e)
        {
            ModifyCurrentTragetTexCoord_(delegate(TexCoord4 texCoord) { texCoord.FlipHorizontal(); });
        }

        /// <summary>
        /// UV垂直方向反転
        /// </summary>
        private void Event_BtnFlipV_Click(object sender, EventArgs e)
        {
            ModifyCurrentTragetTexCoord_(delegate(TexCoord4 texCoord) { texCoord.FlipVertical(); });
        }

        /// <summary>
        /// UV 90度回転
        /// </summary>
        private void Event_BtnRotate90_Click(object sender, EventArgs e)
        {
            ModifyCurrentTragetTexCoord_(delegate(TexCoord4 texCoord) { texCoord.Rotate90(); });
        }

        /// <summary>
        /// 通知
        /// </summary>
        private void NotifyPropertyChanged()
        {
        }
    }
}
