﻿// --------------------------------------------------------------------------------
// <copyright>
// Copyright (C)Nintendo. All rights reserved.
//
// These coded instructions, statements, and computer programs contain proprietary
// information of Nintendo and/or its licensed developers and are protected by
// national and international copyright laws. They may not be disclosed to third
// parties or copied or duplicated in any form, in whole or in part, without the
// prior written consent of Nintendo.
//
// The content herein is highly confidential and should be handled accordingly.
// </copyright>
// --------------------------------------------------------------------------------
using System;
using System.Collections.Generic;
using System.Linq;
using System.Text;
using System.Drawing;
using LECore.Structures.Core;
using System.Windows.Forms;
using LECore;

namespace LayoutEditor.Forms.ToolWindows.TimiControlWindow
{
    /// <summary>
    /// 時間目盛りを表示するコントロールです。
    /// </summary>
    class TimeSliderControl: System.Windows.Forms.Control
    {
        /// <summary>
        /// 調整可能か？
        /// </summary>
        public bool Tweakable { get; set; }

        enum Mode
        {
            Idle,
            StartTweake,
            MoveTweake,
            EndTweake
        }

        Mode _currentMode = Mode.Idle;

        int _mouseDownStart;
        int _mouseDownEnd;
        float _mouseDownPos;

        Font _font = new Font("Tahoma", 8.25F);

        /// <summary>
        /// グローバル時間クラス（クラス中でよく使用するので）
        /// </summary>
        GlobalTime _GrbTime
        {
            get { return GlobalTime.Inst; }
        }

        //----------------------------------------------------------

        public TimeSliderControl()
        {
            InitializeComponent();
            SetStyle(ControlStyles.DoubleBuffer, true);
            this.Tweakable = true;
        }

        float TimeToClientPos_(float time)
        {
            float val = ((time - _GrbTime.AnimStartTime) / ((float)_GrbTime.AnimEndTime - _GrbTime.AnimStartTime)) * (float)this.Width;
            return val;
        }

        float ClientPosToTime_(float pos)
        {
            float val = (pos / (float)this.Width) * ((float)_GrbTime.AnimEndTime - _GrbTime.AnimStartTime);
            return val;
        }

        //----------------------------------------------------------

        protected override void OnPaint(PaintEventArgs pe)
        {
            Point myPos = this.PointToClient(Control.MousePosition);
            bool isMouseInside = new Rectangle(Point.Empty, this.Bounds.Size).Contains(myPos);
            Mode mouseLocationMode = this.Enabled && isMouseInside ? GetModeFromMousePos_(myPos.X) : Mode.Idle;

            var g = pe.Graphics;
            float start = TimeToClientPos_(_GrbTime.AnimPlayStartTime);
            float end = TimeToClientPos_(_GrbTime.AnimPlayEndTime);

            Brush buttonColor = new SolidBrush(this.BackColor);
            if (this.Enabled)
            {
                buttonColor = isMouseInside ? Brushes.LightSteelBlue : Brushes.LightGray;
            }
            g.FillRectangle(buttonColor, start, 1, end - start, this.Height - 2);

            var strSize = g.MeasureString(_GrbTime.AnimPlayEndTime.ToString(), _font);
            float offsetH = (this.Height - strSize.Height) * 0.5f;

            // Start を描画
            if (mouseLocationMode == Mode.StartTweake)
            {
                g.FillRectangle(Brushes.LightBlue, start, 1, 16, this.Height - 2);
            }

            g.DrawString(_GrbTime.AnimPlayStartTime.ToString(), _font, Brushes.DimGray, start, offsetH);

            // End を描画。文字のオーバーラップをチェック
            if (mouseLocationMode == Mode.EndTweake)
            {
                g.FillRectangle(Brushes.LightBlue, end - 16, 1, 16, this.Height - 2);
            }

            if (start + strSize.Width < end - strSize.Width)
            {
                g.DrawString(_GrbTime.AnimPlayEndTime.ToString(), _font, Brushes.DimGray, end - strSize.Width, offsetH);
            }
        }

        private void Event_TimeScaleControl_SizeChanged(object sender, System.EventArgs e)
        {
            Invalidate();
        }

        private Mode GetModeFromMousePos_(int x)
        {
            if(!this.Tweakable)
            {
                return Mode.Idle;
            }

            const float MaxAnchorSize = 16.0f;

            float end = TimeToClientPos_(_GrbTime.AnimPlayEndTime);
            float start = TimeToClientPos_(_GrbTime.AnimPlayStartTime);

            float size = Math.Max(end - start, 1.0f);

            float startAnchorL = start;
            float startAnchorR = start + Math.Min(MaxAnchorSize, (size / 3.0f));

            float endAnchorL = end - Math.Min(MaxAnchorSize, (size / 3.0f));
            float endAnchorR = end;

            if (x < startAnchorR)
            {
                return Mode.StartTweake;
            }
            else if (x > endAnchorL)
            {
                return Mode.EndTweake;
            }
            else
            {
                return Mode.MoveTweake;
            }
        }

        private void Event_TimeScaleControl_MouseDown(object sender, System.Windows.Forms.MouseEventArgs e)
        {
            if (_currentMode != Mode.Idle)
            {
                return;
            }

            this.Focus();

            _mouseDownPos = e.X;
            _mouseDownStart = _GrbTime.AnimPlayStartTime;
            _mouseDownEnd = _GrbTime.AnimPlayEndTime;

            _currentMode = GetModeFromMousePos_(e.X);
        }

        private void TimeSliderControl_MouseEnter(object sender, EventArgs e)
        {
            if ((Control.MouseButtons & MouseButtons.Left) == 0)
            {
                _currentMode = Mode.Idle;
            }
        }

        private void TimeSliderControl_MouseLeave(object sender, EventArgs e)
        {
            this.Invalidate();
        }

        private void Event_TimeScaleControl_MouseUp(object sender, System.Windows.Forms.MouseEventArgs e)
        {
            _currentMode = Mode.Idle;
        }

        private void Event_TimeScaleControl_MouseMove(object sender, System.Windows.Forms.MouseEventArgs e)
        {
            if (_currentMode == Mode.Idle)
            {
                this.Invalidate();
                return;
            }

            float diff = e.X - _mouseDownPos;
            float fDiffTime = ClientPosToTime_(diff);
            int iDiffTime = (int)Math.Floor(fDiffTime);

            if(_currentMode == Mode.MoveTweake)
            {
                int newStart = Math.Max(_GrbTime.AnimStartTime, _mouseDownStart + iDiffTime);
                int newEnd = Math.Min(_GrbTime.AnimEndTime, _mouseDownEnd + iDiffTime);

                if (newEnd == _GrbTime.AnimEndTime)
                {
                    newStart = _GrbTime.AnimEndTime - (_mouseDownEnd - _mouseDownStart);
                }
                else if (newStart == _GrbTime.AnimStartTime)
                {
                    newEnd = _GrbTime.AnimStartTime + (_mouseDownEnd - _mouseDownStart);
                }

                _GrbTime.Set(_GrbTime.AnimStartTime, _GrbTime.AnimEndTime, newStart, newEnd);
            }

            if (_currentMode == Mode.StartTweake)
            {
                int newStart = Math.Max(_GrbTime.AnimStartTime, _mouseDownStart + iDiffTime);
                _GrbTime.Set(_GrbTime.AnimStartTime, _GrbTime.AnimEndTime, newStart, _mouseDownEnd);
            }

            if (_currentMode == Mode.EndTweake)
            {
                int newEnd = Math.Min(_GrbTime.AnimEndTime, _mouseDownEnd + iDiffTime);
                _GrbTime.Set(_GrbTime.AnimStartTime, _GrbTime.AnimEndTime, _mouseDownStart, newEnd);
            }
        }

        //---------------------------------------------------------

        private void InitializeComponent()
        {
            this.SuspendLayout();
            //
            // TimeSliderControl
            //
            this.BackColor = System.Drawing.SystemColors.ControlDark;
            this.MouseMove += new System.Windows.Forms.MouseEventHandler(this.Event_TimeScaleControl_MouseMove);
            this.MouseDown += new System.Windows.Forms.MouseEventHandler(this.Event_TimeScaleControl_MouseDown);
            this.MouseUp += new System.Windows.Forms.MouseEventHandler(this.Event_TimeScaleControl_MouseUp);
            this.SizeChanged += new System.EventHandler(this.Event_TimeScaleControl_SizeChanged);
            this.MouseEnter += new System.EventHandler(this.TimeSliderControl_MouseEnter);
            this.MouseLeave += new System.EventHandler(this.TimeSliderControl_MouseLeave);
            this.ResumeLayout(false);

        }
    }
}
