﻿// --------------------------------------------------------------------------------
// <copyright>
// Copyright (C)Nintendo. All rights reserved.
//
// These coded instructions, statements, and computer programs contain proprietary
// information of Nintendo and/or its licensed developers and are protected by
// national and international copyright laws. They may not be disclosed to third
// parties or copied or duplicated in any form, in whole or in part, without the
// prior written consent of Nintendo.
//
// The content herein is highly confidential and should be handled accordingly.
// </copyright>
// --------------------------------------------------------------------------------
using System;
using System.Collections.Generic;
using System.ComponentModel;
using System.Drawing;
using System.Data;
using System.Linq;
using System.Text;
using System.Windows.Forms;
using LECore.Manipulator;
using LECore.Structures;
using LayoutEditor.Controls;
using LECore;
using LayoutEditor.Forms.ToolWindows.PropertyEditWindow.Adapters;

namespace LayoutEditor.Forms.ToolWindows.PropertyEditWindow
{
    /// <summary>
    ///
    /// </summary>
    public partial class PaneBasicPropertyPage : PropertyPage
    {
        //----------------------------------------------------------
        PaneGuiAdapterGroup _oldTarget = null;
        bool _guiUpdating = true;

        /// <summary>
        /// ページの表示内容と関係しているペインです。
        /// peneAdapterからマテリアル操作クラスを取得する際のキーとして利用します。
        /// </summary>
        private IPane _OwnerPane
        {
            get;
            set;
        }

        /// <summary>
        /// 操作対象ペインを操作するクラスを取得します。
        /// </summary>
        PaneManipulator _TargetPaneMpn
        {
            get
            {
                if (_OwnerPane != null)
                {
                    return Target.Active.PartsSubPaneManipulators.FirstOrDefault((mnp) => object.ReferenceEquals(mnp.IPane, _OwnerPane)) as PaneManipulator;
                }
                else
                {
                    return Target.Active.PaneManipulator;
                }
            }
        }

        //----------------------------------------------------------

        /// <summary>
        /// コンストラクタ
        /// </summary>
        public PaneBasicPropertyPage()
        {
            InitializeComponent();
        }

        protected override void InitializeProperty()
        {
            // 何もしません。
        }

        /// <summary>
        /// GUIを更新します。
        /// </summary>
        protected override void UpdateProperty()
        {
            // TODO : UIの有効無効を判断する。
            // ターゲットを参照してUIを更新する
            IPane targetPane = _TargetPaneMpn.IPane;
            if (targetPane != null)
            {
                try
                {
                    _guiUpdating = true;

                    _ctlPosition.FVec3Value = targetPane.Trans;
                    _ctlSize.FVec2Value = targetPane.Size.FVec2;
                    _ctlScale.FVec2Value = targetPane.Scale.FVec2;
                    _ctlRotate.FVec3Value = targetPane.RotAng;
                    _nudTranparency.Value = targetPane.Transparency;

                    _chkVisible.Checked = targetPane.Visible;
                    _tbxUserData.SetValue(targetPane.UserData);

                    // 連動ボタンの更新
                    if (_oldTarget != Target)
                    {
                        _oldTarget = Target;
                        UpdateLinkButton(targetPane);
                    }
                }
                finally
                {
                    _guiUpdating = false;
                }
            }
        }

        /// <summary>
        /// 連動ボタンを更新します
        /// </summary>
        private void UpdateLinkButton(IPane pane)
        {
            if (pane == null) return;

            _ctlScale.UpdateLinkButton(pane.Scale.X == pane.Scale.Y);
        }

        /// <summary>
        /// インスタンス作成。
        /// </summary>
        public static PropertyPage CreateInstance(object arg)
        {
            var page = new PaneBasicPropertyPage();
            var prop = arg as IPartsPropaerty;
            Ensure.Argument.NotNull(prop);
            Ensure.Argument.NotNull(prop.Paramater);

            page._OwnerPane = prop.Paramater.OwnerPane;

            page._ctlPosition.Enabled = (prop.Option & PartsPropertyUsageOptions.UsePanePosition) != 0;
            page._ctlSize.Enabled = (prop.Option & PartsPropertyUsageOptions.UsePaneSize) != 0;
            page._ctlScale.Enabled = (prop.Option & PartsPropertyUsageOptions.UsePaneScale) != 0;
            page._ctlRotate.Enabled = (prop.Option & PartsPropertyUsageOptions.UsePaneRotate) != 0;
            page._chkVisible.Enabled = (prop.Option & PartsPropertyUsageOptions.UsePaneVisibility) != 0;
            page._nudTranparency.Enabled = (prop.Option & PartsPropertyUsageOptions.UsePaneTransparency) != 0;
            page._tbxUserData.Enabled = (prop.Option & PartsPropertyUsageOptions.UsePaneBasicUserData) != 0;

            return page;
        }

        //----------------------------------------------------------
        //
        //----------------------------------------------------------

        /// <summary>
        ///
        /// </summary>
        private void Event_ctlPosition_OnFVec3Edit(object sender, FVec3EditEventArgs args)
        {
            if (_guiUpdating) return;

            if (_TargetPaneMpn.IPane.Trans != args.Value)
            {
                _TargetPaneMpn.Trans = args.Value;
            }
        }

        /// <summary>
        ///
        /// </summary>
        private void Event_ctlSize_OnFVec2Edit(object sender, FVec2EditEventArgs args)
        {
            if (_guiUpdating) return;

            var newVal3 = new FVec3(args.Value.X, args.Value.Y, _TargetPaneMpn.IPane.Size.Z);
            if (_TargetPaneMpn.IPane.Size != newVal3)
            {
                _TargetPaneMpn.Size = newVal3;
            }
        }

        /// <summary>
        ///
        /// </summary>
        private void Event_ctlScale_OnFVec2Edit(object sender, FVec2EditEventArgs args)
        {
            if (_guiUpdating) return;

            var newVal3 = new FVec3(args.Value.X, args.Value.Y, _TargetPaneMpn.IPane.Scale.Z);
            if (_TargetPaneMpn.IPane.Scale != newVal3)
            {
                _TargetPaneMpn.Scale = newVal3;
            }
        }

        /// <summary>
        ///
        /// </summary>
        private void Event_ctlRotate_OnFVec3Edit(object sender, FVec3EditEventArgs args)
        {
            if (_guiUpdating) return;

            if (_TargetPaneMpn.IPane.RotAng != args.Value)
            {
                _TargetPaneMpn.RotAng = args.Value;
            }
        }

        /// <summary>
        ///
        /// </summary>
        private void _nudTranparency_ValueChanged(object sender, EventArgs e)
        {
            if (_guiUpdating) return;

            NumericUpDown nud = sender as NumericUpDown;
            byte transparency = UIHelper.ClampDecimalToByte(nud.Value);

            if (_TargetPaneMpn.IPane.Transparency != transparency)
            {
                _TargetPaneMpn.Transparency = transparency;
            }
        }

        /// <summary>
        ///
        /// </summary>
        private void Event_chkVisible_CheckedChanged(object sender, EventArgs e)
        {
            if (_guiUpdating) return;

            CheckBox checkBox = sender as CheckBox;
            if (_TargetPaneMpn.IPane.Visible != checkBox.Checked)
            {
                _TargetPaneMpn.Visible = checkBox.Checked;
            }
        }

        /// <summary>
        ///
        /// </summary>
        private void Event_tbxUserData_ValueChanged(object sender, EventArgs e)
        {
            if (_guiUpdating) return;

            StringTextBox stb = sender as StringTextBox;

            if (_TargetPaneMpn.IPane.UserData != stb.Text)
            {
                // 英数字以外が使用されていないか確認します。
                if (PaneHelper.CheckUserDataValid(stb.Text))
                {
                    _TargetPaneMpn.UserData = stb.Text;
                }
                else
                {
                    // 不正な入力なので、以前のデータに戻します。
                    stb.SetValue(_TargetPaneMpn.IPane.UserData);
                }
            }
        }
    }
}
