﻿// --------------------------------------------------------------------------------
// <copyright>
// Copyright (C)Nintendo. All rights reserved.
//
// These coded instructions, statements, and computer programs contain proprietary
// information of Nintendo and/or its licensed developers and are protected by
// national and international copyright laws. They may not be disclosed to third
// parties or copied or duplicated in any form, in whole or in part, without the
// prior written consent of Nintendo.
//
// The content herein is highly confidential and should be handled accordingly.
// </copyright>
// --------------------------------------------------------------------------------
using System;
using System.Collections.Generic;
using System.Linq;
using System.Text;
using LayoutEditor.Forms.ToolWindows.PropertyEditWindow.Adapters;
using LECore.Structures;
using LayoutEditor.src.Forms.ToolWindows.PropertyEditWindow.Pane.Window;
using System.Windows.Forms;

namespace LayoutEditor.Forms.ToolWindows.PropertyEditWindow
{
    /// <summary>
    /// ウインドウ用のパネルです。
    /// </summary>
    public class WindowPropertyPanel : PaneBasicPanel
    {
        private readonly WindowKindPanel _pnlWindow = new WindowKindPanel();
        private IWindowPaneAdapterEntry _targetWindowPane = null;

        //----------------------------------------------------------

        /// <summary>
        /// コンストラクタです。
        /// </summary>
        public WindowPropertyPanel()
            : this(PropertyPanelType.Null, null, null)
        {
        }

        /// <summary>
        /// コンストラクタです。
        /// </summary>
        public WindowPropertyPanel(PropertyPanelType id, PaneGuiAdapterGroup target, Action<ViewManagerMessage> sendMessageToViewManagerAction)
            : base(id, target, sendMessageToViewManagerAction)
        {
            // 拡張パネル
            {
                _pnlWindow.OnTargetPartsChanged += new Action(() =>
                {
                    this.UpdateForm();

                    // カテゴリの更新を試行します。
                    this.OnTargetChangeCategoryUpdate();
                });
            }
        }

        //----------------------------------------------------------



        /// <summary>
        /// 捜査対象ペインアダプタ
        /// </summary>
        IWindowPaneAdapterEntry _TargetWindowPane
        {
            get
            {
                return _targetWindowPane;
            }

            set
            {
                if (_targetWindowPane != value)
                {
                    _targetWindowPane = value;
                    this.RightPageExtraPanel = _pnlWindow;
                }
            }
        }

        //----------------------------------------------------------

        /// <summary>
        /// 操作対象のマテリアルを取得します。
        /// </summary>
        protected override IRevHWMaterial GetTargetCoreMaterial()
        {
            if (_TargetWindowPane != null)
            {
                return _TargetWindowPane.GetTargetCoreMaterial();
            }

            return null;
        }

        /// <summary>
        /// アクティブカテゴリが変更された時の処理です。
        /// </summary>
        protected override void OnChangedActiveCategory(PropWndTreeNode categoryNode, PropertyPage newActivePage)
        {
            // ノードをさかのぼってウインドウノードを探す
            while (categoryNode != null)
            {
                if (categoryNode.GetPage() is WindowPanePage)
                {
                    break;
                }
                categoryNode = categoryNode.Parent as PropWndTreeNode;
            }

            // 見つかったウインドウノードを対象として設定する。
            var windowPage = (categoryNode != null) ? categoryNode.GetPage() as WindowPanePage : null;
            if (windowPage != null)
            {
                windowPage.Owner = this;
                _TargetWindowPane = windowPage.TargetEntry;
                _pnlWindow.TargetWindowPane = _TargetWindowPane;
            }

            base.OnChangedActiveCategory(categoryNode, newActivePage);
        }

        /// <summary>
        /// ターゲット変更時の更新処理
        /// </summary>
        public override void OnTargetChangeCategoryUpdate()
        {
            if (this.Owner.Target == null || this.Owner.Target.Active == null)
            {
                _TargetWindowPane = null;
                return;
            }

            try
            {
                if (this.Owner.Target.Active.FindTargetWindowPaneAdapterEntry(0) != null &&
                    this.Owner.Target.Active.FindTargetWindowPaneAdapterEntry(1) == null)
                {
                    _TargetWindowPane = this.Owner.Target.Active.FindTargetWindowPaneAdapterEntry(0);
                    return;
                }

                var windowPage = _ActivePage as WindowPanePage;
                if (windowPage != null)
                {
                    _TargetWindowPane = windowPage.TargetEntry;
                }
            }
            finally
            {
                // パーツ更新処理を行い、以前に編集対象だった部位に戻す。
                UpdateProperty();
                base.OnTargetChangeCategoryUpdate();
            }
        }

        /// <summary>
        ///
        /// </summary>
        public override void OnSceneModifyHandler(object sender, SceneModifyEventArgs e)
        {
            if (e.Target == SceneModifyEventArgs.Kind.SelectedSetClicked)
            {
                // 選択されたウインドウペインがクリックされた場合は、編集部位の変更を行う。
                SceneModifyEventArgs.SelectedPaneClickedEventArgs arg = e.Paramaters[0] as SceneModifyEventArgs.SelectedPaneClickedEventArgs;
                if (arg.IPane.PaneKind == PaneKind.Window)
                {
                    if (!this.Owner.Target.IsMulti &&
                        this.Owner.Target.Active?.Target == arg.IPane)
                    {
                        _targetWindowPane.CurrentTargetPartsID = ILEWindowHelper.GetPartIDByPosition(
                            _targetWindowPane.Manipulator.ILEWindow, arg.PosClicked);

                        _pnlWindow.TargetWindowpartsChanged(_targetWindowPane.CurrentTargetPartsID);
                        _pnlWindow.UpdateProperty();
                        _ActivePage.UpdateForm();

                        // カテゴリの更新を試行します。
                        this.OnTargetChangeCategoryUpdate();
                    }
                }
            }
            else if (e.Target == SceneModifyEventArgs.Kind.SelectedSetModify)
            {
                // 別のイベントで処理するので OnTargetChangeCategoryUpdate は不要
            }

            base.OnSceneModifyHandler(sender, e);
        }


        /// <summary>
        /// UI更新
        /// </summary>
        protected override void UpdateProperty()
        {
            base.UpdateProperty();
            _pnlWindow.TargetWindowPane = _TargetWindowPane;

            // 操作対象マテリアルを同期させる
            if (_TargetWindowPane != null && Owner.Target != null)
            {
                var partsId = _TargetWindowPane.CurrentTargetPartsID;
                this.Owner.Target.Active.UpdateWindowPaneAdaptorCurrentTargetParts(partsId);
            }

            _pnlWindow.UpdateProperty();
        }
    }
}
