﻿// --------------------------------------------------------------------------------
// <copyright>
// Copyright (C)Nintendo All rights reserved.
//
// These coded instructions, statements, and computer programs contain proprietary
// information of Nintendo and/or its licensed developers and are protected by
// national and international copyright laws. They may not be disclosed to third
// parties or copied or duplicated in any form, in whole or in part, without the
// prior written consent of Nintendo.
//
// The content herein is highly confidential and should be handled accordingly.
// </copyright>
// --------------------------------------------------------------------------------
using System;
using System.Collections;
using System.Collections.Generic;
using System.ComponentModel;
using System.Drawing;
using System.Diagnostics;
using System.Data;
using System.Windows.Forms;
using System.Linq;

namespace LayoutEditor.Forms.ToolWindows.PropertyEditWindow.Material.Texture
{
    using LayoutEditor.Forms.ToolWindows.PropertyEditWindow.Adapters;
    using LECore;
    using LECore.Structures.LECoreInterface;

    /// <summary>
    /// TexturePatternAnimPanel の概要の説明です。
    /// </summary>
    public class TexturePatternAnimAddPanel : System.Windows.Forms.UserControl
    {
        #region デザイナ変数
        private System.Windows.Forms.Button _btnAddToSubList;
        private System.Windows.Forms.Button _btnRemoveSubList;
        private System.Windows.Forms.ListView _lvwSubList;
        private System.Windows.Forms.Button _btnClearSubList;
        private System.Windows.Forms.Label _lblSubList;
        private System.Windows.Forms.ColumnHeader _clmSubListIdx;
        private System.Windows.Forms.ColumnHeader _clmSubListTexName;
        private System.Windows.Forms.Button _btnUP;
        private System.Windows.Forms.Button _btnDown;


        /// <summary>
        /// 必要なデザイナ変数です。
        /// </summary>
        private System.ComponentModel.Container components = null;
        #endregion デザイナ変数

        /// <summary>
        /// 使用されているリソースに後処理を実行します。
        /// </summary>
        protected override void Dispose( bool disposing )
        {
            if( disposing )
            {
                if(components != null)
                {
                    components.Dispose();
                }
            }
            base.Dispose( disposing );
        }

        #region コンポーネント デザイナで生成されたコード
        /// <summary>
        /// デザイナ サポートに必要なメソッドです。このメソッドの内容を
        /// コード エディタで変更しないでください。
        /// </summary>
        private void InitializeComponent()
        {
            System.ComponentModel.ComponentResourceManager resources = new System.ComponentModel.ComponentResourceManager(typeof(TexturePatternAnimAddPanel));
            this._btnAddToSubList = new System.Windows.Forms.Button();
            this._btnRemoveSubList = new System.Windows.Forms.Button();
            this._lvwSubList = new System.Windows.Forms.ListView();
            this._clmSubListIdx = ((System.Windows.Forms.ColumnHeader)(new System.Windows.Forms.ColumnHeader()));
            this._clmSubListTexName = ((System.Windows.Forms.ColumnHeader)(new System.Windows.Forms.ColumnHeader()));
            this._btnClearSubList = new System.Windows.Forms.Button();
            this._lblSubList = new System.Windows.Forms.Label();
            this._btnUP = new System.Windows.Forms.Button();
            this._btnDown = new System.Windows.Forms.Button();
            this.SuspendLayout();
            //
            // _btnAddToSubList
            //
            resources.ApplyResources(this._btnAddToSubList, "_btnAddToSubList");
            this._btnAddToSubList.Name = "_btnAddToSubList";
            this._btnAddToSubList.Click += new System.EventHandler(this.Event_BtnAddToSubList_Click);
            //
            // _btnRemoveSubList
            //
            resources.ApplyResources(this._btnRemoveSubList, "_btnRemoveSubList");
            this._btnRemoveSubList.Name = "_btnRemoveSubList";
            this._btnRemoveSubList.Click += new System.EventHandler(this.Event_BtnRemoveSubList_Click);
            //
            // _lvwSubList
            //
            resources.ApplyResources(this._lvwSubList, "_lvwSubList");
            this._lvwSubList.Columns.AddRange(new System.Windows.Forms.ColumnHeader[] {
            this._clmSubListIdx,
            this._clmSubListTexName});
            this._lvwSubList.FullRowSelect = true;
            this._lvwSubList.Name = "_lvwSubList";
            this._lvwSubList.UseCompatibleStateImageBehavior = false;
            this._lvwSubList.View = System.Windows.Forms.View.Details;
            this._lvwSubList.SelectedIndexChanged += new System.EventHandler(this.Event_LvwSubList_SelectedIndexChanged);
            //
            // _clmSubListIdx
            //
            resources.ApplyResources(this._clmSubListIdx, "_clmSubListIdx");
            //
            // _clmSubListTexName
            //
            resources.ApplyResources(this._clmSubListTexName, "_clmSubListTexName");
            //
            // _btnClearSubList
            //
            resources.ApplyResources(this._btnClearSubList, "_btnClearSubList");
            this._btnClearSubList.Name = "_btnClearSubList";
            this._btnClearSubList.Click += new System.EventHandler(this.Event_BtnClearSubList_Click);
            //
            // _lblSubList
            //
            resources.ApplyResources(this._lblSubList, "_lblSubList");
            this._lblSubList.Name = "_lblSubList";
            //
            // _btnUP
            //
            resources.ApplyResources(this._btnUP, "_btnUP");
            this._btnUP.Name = "_btnUP";
            this._btnUP.Click += new System.EventHandler(this.Event_BtnUP_Click);
            //
            // _btnDown
            //
            resources.ApplyResources(this._btnDown, "_btnDown");
            this._btnDown.Name = "_btnDown";
            this._btnDown.Click += new System.EventHandler(this.Event_BtnDown_Click);
            //
            // TexturePatternAnimAddPanel
            //
            this.Controls.Add(this._lblSubList);
            this.Controls.Add(this._lvwSubList);
            this.Controls.Add(this._btnAddToSubList);
            this.Controls.Add(this._btnRemoveSubList);
            this.Controls.Add(this._btnClearSubList);
            this.Controls.Add(this._btnUP);
            this.Controls.Add(this._btnDown);
            this.Name = "TexturePatternAnimAddPanel";
            resources.ApplyResources(this, "$this");
            this.ResumeLayout(false);

        }
        #endregion


        #region イベント
        /// <summary>
        /// 更新ハンドラ
        /// </summary>
        public EventHandler         OnTexMapTexSubListModify = null;
        #endregion イベント



        /// <summary>
        /// 操作対象テクスチャ
        /// </summary>
        TexMap        _targetTexMap = null;
        List<string> _texNameSet = new List<string>();
        List<string> _texSubSetList = new List<string>();
        DateTime _lastTextureListUpdateTime;


        #region プロパティ

        /// <summary>
        /// 操作対象テクスチャ
        /// </summary>
        public TexMap TargetTexMap
        {
            get{ return _targetTexMap; }
            set{ _targetTexMap = value; }
        }
        #endregion プロパティ


        /// <summary>
        /// GUI初期化
        /// </summary>
        void InitializeProperty_()
        {
            _clmSubListIdx.Text = LayoutEditor.StringResMgr.Get( "LISTVIEW_MAT_TEXPAT_SUB_IDX" );
            _clmSubListTexName.Text = LayoutEditor.StringResMgr.Get( "LISTVIEW_MAT_TEXPAT_SUB_NAME" );

            _lvwSubList.SmallImageList = new ImageList();
            _lvwSubList.SmallImageList.ColorDepth = ColorDepth.Depth32Bit;
            _lvwSubList.SmallImageList.ImageSize = new Size(16, 16);
        }

        /// <summary>
        /// コンストラクタ
        /// </summary>
        public TexturePatternAnimAddPanel()
        {
            // この呼び出しは、Windows.Forms フォーム デザイナで必要です。
            InitializeComponent();
            InitializeProperty_();
        }


        /// <summary>
        /// 指定アイテムが選択アイテムに含まれるか調べます。
        /// </summary>
        bool IsItemContainedBySelectedItems_(
            ListViewItem  item,
            Array         selectedItems )
        {
            foreach( ListViewItem selectedItem in selectedItems )
            {
                if( item.SubItems[1].Text == selectedItem.SubItems[1].Text )
                {
                    return true;
                }
            }
            return false;
        }

        /// <summary>
        ///
        /// </summary>
        bool CheckFullListNeedUpdate_()
        {
            // 最後に更新されてから、テクスチャマネージャがリセットされている。
            if (TextureManager.LastResetTime > _lastTextureListUpdateTime)
            {
                return true;
            }

            // 総数が一致しない
            if( _texNameSet.Count != TextureManager.TexturesOwnedByCurrentSubScene.Count )
            {
                return true;
            }

            // 名前が同一の順番で並んでいる
            for( int i = 0 ; i < _texNameSet.Count ; i++ )
            {
                if( _texNameSet[i] != TextureManager.TexturesOwnedByCurrentSubScene.ElementAt(i).Name )
                {
                    return true;
                }
            }

            return false;
        }

        bool CheckSubListNeedUpdate_()
        {
            // 最後に更新されてから、テクスチャマネージャがリセットされている。
            if (TextureManager.LastResetTime > _lastTextureListUpdateTime)
            {
                return true;
            }

            string[] currentSubSet = TargetTexMap.TextureSubList;

            // 総数が一致しない
            if( _texSubSetList.Count != currentSubSet.Length )
            {
                return true;
            }

            // 名前が同一の順番で並んでいる
            for( int i = 0 ; i < _texSubSetList.Count ; i++ )
            {
                if( currentSubSet[i] != _texSubSetList[i] )
                {
                    return true;
                }
            }

            return false;
        }

        /// <summary>
        /// UIを更新します。
        /// </summary>
        public void UpdateProperty()
        {
            bool isTexListUpdated = false;

            // サブリストの更新
            if( CheckSubListNeedUpdate_() )
            {
                isTexListUpdated = true;

                ListView.SelectedListViewItemCollection selectedCollection = _lvwSubList.SelectedItems;
                ListViewItem[] selectedItems = new ListViewItem[selectedCollection.Count];
                selectedCollection.CopyTo( selectedItems, 0 );

                _lvwSubList.BeginUpdate();
                _lvwSubList.Items.Clear();
                _lvwSubList.SmallImageList.Images.Clear();

                string[] texSubList = TargetTexMap.TextureSubList;
                for( int texIdx = 0 ; texIdx < texSubList.Length ; texIdx++ )
                {
                    ListViewItem lvItem = new ListViewItem();

                    lvItem.Text = texIdx.ToString();
                    lvItem.SubItems.Add( texSubList[texIdx] );

                    // 選択されていた項目の場合再び選択します。
                    lvItem.Selected = IsItemContainedBySelectedItems_( lvItem, selectedItems );

                    // イメージ設定
                    var texture = TextureManager.TexturesOwnedByCurrentSubScene.OfType<Texture>().FirstOrDefault(x => x.Name == texSubList[texIdx]);
                    if (texture != null)
                    {
                        lvItem.ImageIndex = _lvwSubList.SmallImageList.Images.Count;
                        Bitmap thumnail = new Bitmap(texture.ColorImage, new Size(16, 16));
                        _lvwSubList.SmallImageList.Images.Add(texture.Name, thumnail);
                    }

                    if (texture.SourceType == TextureSourceType.Dynamic)
                    {
                        lvItem.BackColor = LECore.AppConstants.ActiveCatpureTextureBackColor;
                    }

                    _lvwSubList.Items.Add( lvItem );
                }

                // 選択アイテムの表示
                foreach( int index in _lvwSubList.SelectedIndices )
                {
                    _lvwSubList.EnsureVisible( index );
                }

                _lvwSubList.EndUpdate();


                // リストの更新
                string[] currentSubSet = TargetTexMap.TextureSubList;
                _texSubSetList.Clear();
                _texSubSetList.Capacity = currentSubSet.Length;

                foreach( string textureName in currentSubSet )
                {
                    _texSubSetList.Add( textureName );
                }
            }

            // 最終更新時間を記録
            if (isTexListUpdated)
            {
                _lastTextureListUpdateTime = DateTime.Now;
            }

            UpdateButtonState_();
        }


        /// <summary>
        /// 各種ボタンの状態を更新します。
        /// </summary>
        void UpdateButtonState_()
        {
            // ボタンの状態を更新します。
            _btnRemoveSubList.Enabled = _lvwSubList.SelectedItems.Count != 0;
            _btnClearSubList.Enabled  = _lvwSubList.Items.Count != 0;

            // 上下ボタンの状態を更新します。
            // 単数の選択アイテムが存在する。
            _btnUP.Enabled   = false;
            _btnDown.Enabled = false;
            if( _lvwSubList.SelectedItems.Count == 1 )
            {
                ListViewItem selectedItem = _lvwSubList.SelectedItems[0] as ListViewItem;

                // 先頭の要素ではない。
                if( _lvwSubList.Items[0] != selectedItem )
                {
                    _btnUP.Enabled = true;
                }

                // 終端の要素ではない。
                if( _lvwSubList.Items[_lvwSubList.Items.Count - 1]  != selectedItem )
                {
                    _btnDown.Enabled = true;
                }
            }

        }

        /// <summary>
        /// テクスチャパターンアニメーションに登録可能なテクスチャのリストを取得します。
        /// </summary>
        /// <returns></returns>
        private IEnumerable<Texture> GetPatternAnimTextureAllowableList()
        {
            // SIGLO-75704
            // LayoutEditor/ランタイムともにキャプチャテクスチャをテクスチャパターンアニメーションとして使用するのが未対応なため、ファイルテクスチャのみ登録可能
            return TextureManager.TexturesOwnedByCurrentSubScene.OfType<Texture>().Where(x => !TargetTexMap.TextureSubList.Any(y => x.Name == y) && x.SourceType == TextureSourceType.File).ToArray();
        }

        #region メッセージハンドラ
        /// <summary>
        /// 全体リスト：選択アイテム更新ハンドラ
        /// </summary>
        private void Event_LvwFullList_SelectedIndexChanged(object sender, System.EventArgs e)
        {
            UpdateButtonState_();
        }

        /// <summary>
        /// サブリスト：選択アイテム更新ハンドラ
        /// </summary>
        private void Event_LvwSubList_SelectedIndexChanged(object sender, System.EventArgs e)
        {
            UpdateButtonState_();
        }


        /// <summary>
        /// 登録ボタンハンドラ
        /// </summary>
        private void Event_BtnAddToSubList_Click(object sender, System.EventArgs e)
        {
            using (var dialog = new TexturePatternAddDialog(GetPatternAnimTextureAllowableList()))
            {
                dialog.ShowDialog();
                if (dialog.DialogResult == DialogResult.OK)
                {
                    foreach (var texture in dialog.SelectedItems)
                    {
                        TargetTexMap.AddTextureToSubList(texture);
                    }

                    // 更新を通知します。
                    NotifyChange_();
                }
            }
        }

        /// <summary>
        /// 削除ボタンハンドラ
        /// </summary>
        private void Event_BtnRemoveSubList_Click(object sender, System.EventArgs e)
        {
            foreach( ListViewItem  lvItem in _lvwSubList.SelectedItems )
            {
                string filename = lvItem.SubItems[1].Text;
                // テクスチャリストに追加
                TargetTexMap.RemoveTextureFromSubList( filename );
            }
            // 更新を通知します。
            NotifyChange_();
        }

        /// <summary>
        /// クリアボタンハンドラ
        /// </summary>
        private void Event_BtnClearSubList_Click(object sender, System.EventArgs e)
        {
            // サブリストの内容をすべて消去します。
            string[] texSubList = TargetTexMap.TextureSubList;
            foreach( string texName in texSubList )
            {
                TargetTexMap.RemoveTextureFromSubList( texName );
            }

            // 更新を通知します。
            NotifyChange_();
        }

        /// <summary>
        /// 更新を通知します。
        /// </summary>
        void NotifyChange_()
        {
            if( OnTexMapTexSubListModify != null )
            {
                OnTexMapTexSubListModify( this, null );
            }
        }


        /// <summary>
        /// 選択されているサブテクスチャリストのテクスチャ名を取得します。
        /// </summary>
        string GetSubTexListSelectedTexName_()
        {
            Debug.Assert( _lvwSubList.SelectedItems.Count == 1 );

            ListViewItem  lvItem = _lvwSubList.SelectedItems[0];
            return lvItem.SubItems[1].Text;
        }

        /// <summary>
        /// 上ボタンハンドラ
        /// </summary>
        private void Event_BtnUP_Click(object sender, System.EventArgs e)
        {
            string texName = GetSubTexListSelectedTexName_();

            int index = TargetTexMap.GetTextureIndexOfSubList( texName );
            if( index != -1 && index > 0 )
            {
                TargetTexMap.RemoveTextureFromSubList( texName );
                TargetTexMap.InsertTextureToSubList( index - 1, texName );

                NotifyChange_();
            }
        }

        /// <summary>
        /// 下ボタンハンドラ
        /// </summary>
        private void Event_BtnDown_Click(object sender, System.EventArgs e)
        {
            string texName = GetSubTexListSelectedTexName_();

            int index = TargetTexMap.GetTextureIndexOfSubList( texName );
            if( index != -1 && index + 1 < TargetTexMap.CountOfSubList )
            {
                TargetTexMap.RemoveTextureFromSubList( texName );
                TargetTexMap.InsertTextureToSubList( index + 1, texName );

                NotifyChange_();
            }
        }

        #endregion メッセージハンドラ




    }
}
