﻿// --------------------------------------------------------------------------------
// <copyright>
// Copyright (C)Nintendo All rights reserved.
//
// These coded instructions, statements, and computer programs contain proprietary
// information of Nintendo and/or its licensed developers and are protected by
// national and international copyright laws. They may not be disclosed to third
// parties or copied or duplicated in any form, in whole or in part, without the
// prior written consent of Nintendo.
//
// The content herein is highly confidential and should be handled accordingly.
// </copyright>
// --------------------------------------------------------------------------------
using System;
using System.Collections.Generic;
using System.ComponentModel;
using System.Diagnostics;
using System.Drawing;
using System.Linq;
using System.Runtime.InteropServices;
using System.Windows.Forms;

namespace LayoutEditor.Forms.ToolWindows.LayoutWindow
{
    using LayoutEditor.Forms.ToolWindows.common;
    using LECore.Manipulator;
    using LECore.Structures;
    using LECore.Structures.Core;
    using LECore.Util;
    using Structures.SerializableObject;
    using LayoutEditor.Utility;
    using System.IO;
    using LECore;

    /// <summary>
    /// MainView の概要の説明です。
    ///
    /// ○本クラスのような、コントロールクラスが
    /// 時間変更イベント、値変更イベントなどを直接受け取り、
    /// 再描画などを行うことはしません。
    ///
    /// ○メッセージは一旦、ViewManager に 集約されます。
    /// ViewManager は適切なクラスに、再描画処理を発行します。
    ///
    /// </summary>
    public partial class MainView : System.Windows.Forms.Control, ISceneModifyListener
    {
        [DllImport("user32.dll")]
        private static extern short GetAsyncKeyState(System.Windows.Forms.Keys vKey);

        /// <summary>
        /// 必要なデザイナ変数です。
        /// </summary>
        private System.ComponentModel.Container components = null;

        #region デリゲータ
        public delegate void MessageReoprtHandler(string msg);
        public event MessageReoprtHandler OnReportMsg = null;
        public event Func<IPane, bool> IsPaneValidToPaste = null;
        #endregion デリゲータ

        #region フィールド

        // 倍率テーブル
        static float[] _MagnifyTbl =
        {
            0.125f,
            0.1667f,
            0.250f,
            0.333f,
            0.500f,
            0.6667f,
            1.000f,
            1.500f,
            2.000f,
            3.000f,
            4.000f,
            6.000f,
            8.000f
        };

        // スクリーン原点のシーン座標系での位置。(ウインドウ画面左上の位置)
        PointF _screenOrigInScene = PointF.Empty;
        // ビューの表示倍率
        float _magnify = 1.0f;

        // ビューの表示倍率範囲
        RangeF _magnifyRange = new RangeF(0.125f, 8.0f);

        // 描画モジュール
        IRenderer _renderer = null;

        // 描画オプション
        DrawableOption _drawableOption;

        // 範囲選択矩形
        SelectRect _selectRect = new SelectRect();

        // 背景
        AppSetting _appSetting;

        // ドラッグ中のペイン
        readonly PaneDragModifier _draggedPane;

        // 親子階層の構築を担当するクラスです。
        HierarchyManipulator _hierarchyMnp;

        // サブシーンに対する操作を担当するクラスです。
        SubSceneManipulator _subSceneMnp;

        // マウス操作モード
        MainViewMouseState _mouseState = null;

        // キャプチャテクスチャの定期更新の間隔(ms)です
        const int _captureTextureUpdateInterval = 1000;

        // キャプチャテクスチャの定期更新のためのタイマー
        System.Timers.Timer _captureTextureUpdateTimer = new System.Timers.Timer();

        /// <summary>
        /// 操作対象候補のサブシーン
        ///
        /// すべて描画されます。
        /// 今後、表示非表示の操作、透明度の設定などが可能になることが予想されます。
        /// </summary>
        ISubScene _targetSubScene = null;
        /// <summary>
        /// 操作対象となるサブシーン
        /// 現状は、ユーザに本フィールドの変更操作を提供していません。
        /// </summary>
        ISubScene _targetActiveSubScene = null;

        #endregion

        #region プロパティ

        /// <summary>
        /// 表示倍率
        /// </summary>
        public float Magnify
        {
            get { return _magnify; }
            set
            {
                // 有効な範囲内になければ、丸めます
                if (!_magnifyRange.Contains(value))
                {
                    _magnifyRange.Truncate(ref value);
                }

                if (_magnify != value)
                {
                    // ビューの中心位置を原点としたズームを行います。
                    float hW = this.ClientRectangle.Width * 0.5f;
                    float hH = this.ClientRectangle.Height * 0.5f;

                    _magnify = value;
                    InvokeMagnifyChanged_();
                    Invalidate();
                }

            }
        }

        /// <summary>
        /// スクリーン原点
        /// </summary>
        public PointF ScreenOrigInScene
        {
            get { return _screenOrigInScene; }
            set { _screenOrigInScene = value; }
        }

        /// <summary>
        ///
        /// </summary>
        float _WidthRate
        {
            get { return _renderer.PersepectiveRendering ? _BackGround.ScreenSize.X / this.Width : 1.0f; }
        }

        /// <summary>
        ///
        /// </summary>
        float _HeightRate
        {
            get{ return _renderer.PersepectiveRendering ? _BackGround.ScreenSize.Y / this.Height : 1.0f; }
        }

        /// <summary>
        /// 表示倍率の範囲。
        /// </summary>
        public RangeF MagnifyRange
        {
            get { return _magnifyRange; }
        }

        /// <summary>
        /// 安全フレームの描画
        /// </summary>
        bool _SafeFrameVisible
        {
            get { return _appSetting.SafeFrame.Visible; }
        }

        /// <summary>
        /// 拡大可能かどうか。
        /// </summary>
        public bool IsEnableMagnifyPlus
        {
            get { return _magnify < _magnifyRange.Max; }
        }

        /// <summary>
        /// 縮小可能かどうか。
        /// </summary>
        public bool IsEnableMagnifyMinus
        {
            get { return _magnify > _magnifyRange.Min; }
        }

        /// <summary>
        /// 階層関係を構築中かどうか？
        /// </summary>
        /// <returns></returns>
        public bool IsMakingHierarchy
        {
            get { return _mouseState.IsMakingHierarchy; }
        }
        /// <summary>
        /// 階層関係を構築中かどうか？
        /// </summary>
        /// <returns></returns>
        public bool IsBreakingHierarchy
        {
            get { return _mouseState.IsBreakingHierarchy; }
        }

        /// <summary>
        /// 背景パラメータを取得します。
        /// </summary>
        public BackGround BackGround
        {
            get { return _BackGround; }
        }


        /// <summary>
        /// 背景パラメータを取得します。
        /// </summary>
        BackGround _BackGround
        {
            get
            {
                if (TargetSubScene != null)
                {
                    return TargetSubScene.BackGround;
                }
                return BackGround.Default;
            }
        }

        /// <summary>
        /// グリッドサイズ
        /// </summary>
        float _GridSize
        {
            get
            {
                BackGround backGround = _TargetActiveSubScene.BackGround;
                return backGround.GridSize / backGround.GridDivisionNum;
            }
        }

        /// <summary>
        /// レンダラを取得します。
        /// </summary>
        public IRenderer Renderer
        {
            get { return _renderer; }
        }


        /// <summary>
        /// アプリケーション設定を設定します。
        /// </summary>
        [Browsable(false)]
        private AppSetting AppSetting
        {
            get
            {
                if (_appSetting != null)
                {
                    return _appSetting;
                }
                else
                {
                    return AppSetting.DefaultSetting;
                }
            }
            set
            {
                if (_appSetting != value)
                {
                    _appSetting = value;
                }
            }
        }

        public AppSetting GetAppSetting()
        {
            return this.AppSetting;
        }

        public void SetAppSetting(AppSetting setting)
        {
            this.AppSetting = setting;
        }

        /// <summary>
        /// 対象サブシーンを設定します。
        /// </summary>
        public ISubScene TargetSubScene
        {
            set
            {
                if (value == null) { return; }

                _targetSubScene = value;
                _TargetActiveSubScene = value;

                this.Focus();
                this.Invalidate();
            }

            get
            {
                return _targetSubScene;
            }
        }

        /// <summary>
        /// 操作対象のサブシーンを取得します。
        /// 複数の表示対象サブシーンの中から、
        /// アクティブなものがひとつ返されます。
        /// </summary>
        public ISubScene _TargetActiveSubScene
        {
            get { return _targetActiveSubScene; }
            set
            {
                if (value == null) { return; }

                // _targetSubScenes内に発見されるはず。
                Debug.Assert(_targetSubScene == value);

                _subSceneMnp.BindTarget(value);
                _targetActiveSubScene = value;
            }
        }

        /// <summary>
        /// 操作対象のシーンを操作する、操作クラスを取得します。
        /// </summary>
        public SubSceneManipulator ActiveSubSceneMnp
        {
            get { return _subSceneMnp; }
        }

        /// <summary>
        /// 描画オプション
        /// </summary>
        public DrawableOption DrawableOption
        {
            get { return _drawableOption; }
            set { _drawableOption = value; }
        }

        /// <summary>
        /// ペインの平行移動が子供に影響するか設定取得します。
        /// </summary>
        [Browsable(false)]
        public bool TransAffectsChildren
        {
            get { return _appSetting.TransAffectHierarchy; }
            set
            {
                if (_appSetting.TransAffectHierarchy != value)
                {
                    _appSetting.TransAffectHierarchy = value;
                    // 更新頻度が多いし、必要もないので _appSetting.RaiseModifyEvent() しない
                }
            }
        }

        [Browsable(false)]
        public bool TransAffectsChildrenFlip
        {
            get { return _appSetting.TransAffectsChildrenFlip; }
            set {
                if (_appSetting.TransAffectsChildrenFlip != value)
                {
                    _appSetting.TransAffectsChildrenFlip = value;
                    // 更新頻度が多いし、必要もないので _appSetting.RaiseModifyEvent() しない
                }
            }
        }

        [Browsable(false)]
        public bool PaneDragModifierChangesSize
        {
            get { return _appSetting.PaneDragModifierChangesSize; }
            set
            {
                if (_appSetting.PaneDragModifierChangesSize != value)
                {
                    _appSetting.PaneDragModifierChangesSize = value;
                    _appSetting.RaiseModifyEvent();
                }
            }
        }


        #endregion

        #region 公開イベント

        /// <summary>
        /// 表示倍率が変化したときに呼ばれるイベントです。
        /// </summary>
        public event EventHandler MagnifyChanged = null;
        public event EventHandler MouseStateChange = null;

        #endregion

        /// <summary>
        /// HandleCreated
        /// </summary>
        protected override void OnHandleCreated(EventArgs e)
        {
            base.OnHandleCreated(e);

            // ドッキングウインドウの場合はハンドルが再生成されることがある。
            if (_renderer != null)
            {
                _renderer.Dispose();
                _renderer = null;
            }

            // レンダラの初期化
            {
                _renderer = RendererFactory.Create(_appSetting.RendererKind);
                _renderer.LinearGammmaSRGBFetchEnabled = _appSetting.ProjectSettings.DegammaTextureConversionEnabled;
                _renderer.LinearGammmaSRGBWriteEnabled = _appSetting.ProjectSettings.SRGBWriteEnabled;
                _renderer.LinearGammmaColorParamatersEnabled = _appSetting.ProjectSettings.DegammaParameterConversionEnabled && _appSetting.ProjectSettings.SRGBWriteEnabled;

                _renderer.PersepectiveRendering = _appSetting.ProjectSettings.IsPerspectiveDefault;
                _renderer.FOV = MathUtil.DegToRad(_appSetting.ProjectSettings.PerspectiveFOV);
                _renderer.PerseNear = _appSetting.ProjectSettings.PerspectiveNear;
                _renderer.PerseFar = _appSetting.ProjectSettings.PerspectiveFar;
                _renderer.DepthWriteTestEnabled = _appSetting.ProjectSettings.DepthTestEnabled;

                SetStyle(RendererFactory.GetControlState(_appSetting.RendererKind), true);
                bool bInitSuccess = _renderer.Initialize(this);
                Trace.Assert(bInitSuccess);
            }

            _captureTextureUpdateTimer.Elapsed += new System.Timers.ElapsedEventHandler(Event_CaptureTextureTimer_Elapsed);
            _captureTextureUpdateTimer.Interval = _captureTextureUpdateInterval;
            _captureTextureUpdateTimer.Start();
        }

        protected override void OnHandleDestroyed(EventArgs e)
        {
            if (_renderer != null)
            {
                _renderer.Dispose();
                _renderer = null;
            }

            base.OnHandleDestroyed(e);
        }

        /// <summary>
        ///
        /// </summary>

        public MainView()
        {
            _appSetting = new AppSetting();
            _drawableOption = new DrawableOption();
            _hierarchyMnp = new HierarchyManipulator();
            _subSceneMnp = new SubSceneManipulator();
            // この呼び出しは、Windows.Forms フォーム デザイナで必要です。
            InitializeComponent();

            _mouseState = new MainViewMouseState(this);
            _draggedPane = new PaneDragModifier(this);
            this.MouseWheel += new MouseEventHandler(Event_MainView_MouseWheel);
            this.GotFocus += new EventHandler(Event_MainView_GotFocus);
        }

        /// <summary>
        /// 使用されているリソースに後処理を実行します。
        /// </summary>
        protected override void Dispose(bool disposing)
        {
            if (disposing)
            {
                if (components != null)
                {
                    components.Dispose();
                }
            }

            if (_renderer != null)
            {
                _renderer.Dispose();
            }

            base.Dispose(disposing);
        }


        /// <summary>
        /// 階層構造の構築を開始します。
        /// </summary>
        public void StartHierarchyMaking()
        {
            _mouseState.StartHierarchyMaking();
        }

        /// <summary>
        /// 階層の破壊を開始します。
        /// </summary>
        public void StartHierarchyBreaking()
        {
            _mouseState.StartHierarchyBreaking();
        }

        // テーブルから現在の値以上の
        // 最小の値を格納したテーブル番号を取得します。
        int FindMinimamValueTblIdx_(float currentVal)
        {
            int i = 0;
            for (i = 0; i < _MagnifyTbl.Length; i++)
            {
                if (_MagnifyTbl[i] >= currentVal)
                {
                    break;
                }
            }
            return i;
        }

        /// <summary>
        /// ズームインします。
        /// </summary>
        public void ZoomIn()
        {
            int index = FindMinimamValueTblIdx_(this.Magnify);
            if (_MagnifyTbl[index] == this.Magnify)
            {
                if (index < _MagnifyTbl.Length - 1)
                {
                    index++;
                }
            }
            // 倍率を増やす
            this.Magnify = _MagnifyTbl[index];
        }

        /// <summary>
        /// ズームアウトします。
        /// </summary>
        public void ZoomOut()
        {
            int index = FindMinimamValueTblIdx_(this.Magnify);
            if (index > 1)
            {
                index--;
            }
            // 倍率を減らす
            this.Magnify = _MagnifyTbl[index];
        }

        /// <summary>
        /// ビュー中心をレイアウト画面の中心に設定します。
        /// </summary>
        public void SetViewDefault()
        {
            Magnify = 1.0f;

            ScreenOrigInScene = PointF.Empty;
        }

        /// <summary>
        /// ビューを設定します。
        /// </summary>
        public void SetView(float cx, float cy, float newWidth, float newHeight)
        {
            Magnify = Math.Min(this.Size.Width / newWidth, this.Size.Height / newHeight);

            ScreenOrigInScene = new PointF(cx, cy);
        }

        /// <summary>
        /// スクリーン上の点をシーン座標系の点に変換します。
        /// 内部関数でしたが、公開します。
        /// </summary>
        public PointF ScreenToScene(PointF scrnPos)
        {
            return ScreenToScene_(scrnPos);
        }

        #region --------------------- 非公開メソッド ---------------------

        /// <summary>
        /// 倍率変更イベント発行。
        /// </summary>
        void InvokeMagnifyChanged_()
        {
            if (MagnifyChanged != null)
            {
                MagnifyChanged(this, EventArgs.Empty);
            }
        }



        /// <summary>
        /// スクリーン上の点をシーン座標系の点に変換します。
        /// </summary>
        PointF ScreenToScene_(PointF scrnPos)
        {
            PointF[] ps = { new PointF(scrnPos.X, scrnPos.Y) };

            this._renderer.PushMtx();
            {
                this._renderer.IdentityMtx();

                SetupViewTransforms_();

                this._renderer.TransformPointsToScene(ps);
            }
            this._renderer.PopMtx();

            return ps[0];
        }

        /// <summary>
        /// 矩形を使用してシーンのオブジェクトを選択します。
        /// </summary>
        /// <param name="selRectInScreen"></param>
        RectangleF ScreenRectToSceneRect_(RectangleF screenRect)
        {
            RectangleF rectInScene = screenRect;
            float invMag = 1.0f / _magnify;

            rectInScene.Location = ScreenToScene_(screenRect.Location);
            rectInScene.Size = new SizeF(screenRect.Size.Width * invMag * this._WidthRate, screenRect.Size.Height * invMag * this._HeightRate);

            //-------- TODO:
            rectInScene.Location = new PointF(rectInScene.Location.X, rectInScene.Location.Y - rectInScene.Size.Height);

            return rectInScene;
        }



        #endregion // --------------------- 非公開メソッド ---------------------

        #region コンポーネント デザイナで生成されたコード
        /// <summary>
        /// デザイナ サポートに必要なメソッドです。このメソッドの内容を
        /// コード エディタで変更しないでください。
        /// </summary>
        private void InitializeComponent()
        {
            this.SuspendLayout();
            //
            // MainView
            //
            this.AllowDrop = true;
            this.BackColor = System.Drawing.SystemColors.ControlDarkDark;
            this.SizeChanged += new System.EventHandler(this.MainView_SizeChanged);
            this.Paint += new System.Windows.Forms.PaintEventHandler(this.Event_MainView_Paint);
            this.KeyDown += new System.Windows.Forms.KeyEventHandler(this.Event_MainView_KeyDown);
            this.KeyUp += new System.Windows.Forms.KeyEventHandler(this.Event_MainView_KeyUp);
            this.MouseDown += new System.Windows.Forms.MouseEventHandler(this.Event_MainView_MouseDown);
            this.MouseMove += new System.Windows.Forms.MouseEventHandler(this.Event_MainView_MouseMove);
            this.MouseUp += new System.Windows.Forms.MouseEventHandler(this.Event_MainView_MouseUp);
            this.PreviewKeyDown += new System.Windows.Forms.PreviewKeyDownEventHandler(this.Event_MainView_PreviewKeyDown);
            this.ResumeLayout(false);

        }
        #endregion

        #region -------------------- イベントハンドラ --------------------

        private void Event_CaptureTextureTimer_Elapsed(object sender, System.Timers.ElapsedEventArgs e)
        {
            if (TargetSubScene != null)
            {
                TargetSubScene.UpdateCaptureTextureState();
            }
        }


        #region -------------------- 再描画 --------------------

        Color CalcSystemColor_(Color sRGBColor)
        {
            return !_appSetting.ProjectSettings.DegammaParameterConversionEnabled && _appSetting.ProjectSettings.SRGBWriteEnabled ?
                ColorHelper.ConvertToLinerGamma(sRGBColor) : sRGBColor;
        }

        /// <summary>
        /// 安全フレームを描画します。
        /// </summary>
        void DrawSafeFrame_(
            IRenderer renderer,
            FVec2 screenSize,
            SafeFrame safeFrame)
        {
            float innerTop = 0 + safeFrame.TopWidth;
            float innerBottom = screenSize.Y - safeFrame.BottomWidth;
            float innerRight = screenSize.X - safeFrame.RightWidth;
            float innerHeight = innerBottom - innerTop;

            Color color = renderer.Color;
            renderer.Color = CalcSystemColor_(safeFrame.RGBAColor.AsColor);

            // Top
            renderer.FillRectangle(0, 0, 0, screenSize.X, safeFrame.TopWidth);
            // Bottom
            renderer.FillRectangle(0, innerBottom, 0, screenSize.X, safeFrame.BottomWidth);
            // Left
            renderer.FillRectangle(0, innerTop, 0, safeFrame.LeftWidth, innerHeight);
            // right
            renderer.FillRectangle(innerRight, innerTop, 0, safeFrame.RightWidth, innerHeight);

            renderer.Color = color;
        }

        PointF CalcScreenSizeInScene_()
        {
            var screenSizeInScene = new PointF[] { new PointF(), new PointF(this.ClientRectangle.Width, this.ClientRectangle.Height) };
            _renderer.TransformPointsToScene(screenSizeInScene);

            return new PointF(
                Math.Abs(screenSizeInScene[1].X - screenSizeInScene[0].X),
                Math.Abs(screenSizeInScene[1].Y - screenSizeInScene[0].Y));
        }

        /// <summary>
        /// 背景を描画します。
        /// </summary>
        /// <param name="_renderer"></param>
        void DrawBackGround_(IRenderer renderer)
        {
            AppConfig config = LECore.LayoutEditorCore.Config;

            Color oldColor = renderer.Color;

            // 色の計算
            Color colInside = _BackGround.BackGroundColor;
            Color colOutside = ColorBlender.Blend(Color.Gray, colInside, 0.5f);

            bool currentDepthSetting = renderer.DepthWriteTestEnabled;
            renderer.DepthWriteTestEnabled = false;

            // 背景を塗りつぶします。(レイアウト対象外領域)
            renderer.PushMtx();
            {
                renderer.IdentityMtx();

                float widthRate = this._WidthRate;
                float heightRate = this._HeightRate;

                // ビュー変換
                SetupViewTransforms_();

                renderer.Color = CalcSystemColor_(colOutside);
                renderer.PushMtx();
                {
                    renderer.Trans(ScreenOrigInScene.X, ScreenOrigInScene.Y, _renderer.PerseNear);
                    var sceenSizeInScene = CalcScreenSizeInScene_();

                    // プロジェクションの設定によって、隙間が空くことがあるので、すこし大き目に塗りつぶします。
                    sceenSizeInScene = MathUtil.ScaleVec(sceenSizeInScene, 1.01f);

                    renderer.FillRectangle(-sceenSizeInScene.X / 2, -sceenSizeInScene.Y / 2, 0.0f, sceenSizeInScene.X, sceenSizeInScene.Y);
                }
                renderer.PopMtx();

                // 背景を塗りつぶします。(レイアウト対象領域)
                renderer.Color = CalcSystemColor_(colInside);
                renderer.FillRectangle(-_BackGround.ScreenSize.X / 2, -_BackGround.ScreenSize.Y / 2, 0.0f, _BackGround.ScreenSize.X, _BackGround.ScreenSize.Y);

                // 枠線を書きます
                renderer.Color = CalcSystemColor_(Color.Black);
                renderer.DrawRectangle(-_BackGround.ScreenSize.X / 2, -_BackGround.ScreenSize.Y / 2, 0.0f, _BackGround.ScreenSize.X, _BackGround.ScreenSize.Y);

                // 背景画像が設定されていれば、描画します。
                if (_BackGround.BackgroundImage != null)
                {
                    ITextureImage texImg = _BackGround.BackgroundImage;

                    // 状態リセット
                    renderer.SetMatrialColorBlend(FloatColor.White, FloatColor.Transparence, false);
                    renderer.SetTextureMtx(0, FVec2.Empty, FVec2.One, 0);

                    FVec2 imgSize = new FVec2(texImg.Size);
                    FVec3 centerPos = new FVec3((-imgSize.X) * 0.5f, (-imgSize.Y) * 0.5f, 0.0f);

                    renderer.PushMtx();
                    {
                        renderer.Scale(1.0f, -1.0f);

                        // デフォルトだとsRGB テクスチャとして扱われないためサンプラの設定する。
                        renderer.SetTextureSamplingState(
                            0,
                            LECore.Structures.Nsrif.Attributes.AttrTexWrap.Clamp,
                            LECore.Structures.Nsrif.Attributes.AttrTexWrap.Clamp,
                            LECore.Structures.Nsrif.Attributes.AttrTexFilterMin.Linear,
                            LECore.Structures.Nsrif.Attributes.AttrTexFilterMag.Linear, true);

                        renderer.DrawImage(texImg.GDIBitmap, centerPos, imgSize, new TexCoord4[] { new TexCoord4() });
                    }
                    renderer.PopMtx();
                }

                // グリッドを描画します。
                if (BackGround.GridVisible)
                {
                    DrawBackGround_Grid_(renderer);
                }
            }
            renderer.PopMtx();

            renderer.DepthWriteTestEnabled = currentDepthSetting;
            renderer.Color = oldColor;
        }

        /// <summary>
        /// フォームをズームする。
        /// </summary>
        public void ZoomForm(Form form, bool zoomIn)
        {
            if (zoomIn)
            {
                this.ZoomIn();
            }
            else
            {
                this.ZoomOut();
            }

            float magnity = this.Magnify;
            this.FitWindowSizeToLayoutScreenSize_(form, magnity);

            this.SetViewDefault();
            this.Magnify = magnity;
        }

        /// <summary>
        /// ウインドウサイズをレイアウトサイズに設定する。
        /// </summary>
        private void FitWindowSizeToLayoutScreenSize_(Form form, float magnity)
        {
            FVec2 layoutScrnSize = _TargetActiveSubScene.BackGround.ScreenSize;
            layoutScrnSize.Scale(1.2f);

            // サイズは、_mainView の拡大率を 乗算して求める
            float scaleX = (layoutScrnSize.X * magnity) / this.Size.Width;
            float scaleY = (layoutScrnSize.Y * magnity) / this.Size.Height;

            form.Size = new Size((int)(form.Size.Width * scaleX), (int)(form.Size.Height * scaleY));
        }

        /// <summary>
        /// グリッドの線の太さを計算します。
        /// </summary>
        bool IsSubGridLine_(float current, float start, float step, uint divNum)
        {
            float diff = current - start;
            int currentDivNum = (int)(diff / step);
            return ((currentDivNum & (divNum - 1)) != 0) ? true : false;
        }

        /// <summary>
        /// グリッドを描画します。
        /// </summary>
        /// <param name="renderer"></param>
        void DrawBackGround_Grid_(IRenderer renderer)
        {
            Color oldColor = renderer.Color;
            float oldLineWidth = renderer.LineWidth;

            renderer.LineWidth = 1.0f / _magnify;
            renderer.Color = CalcSystemColor_(BackGround.GridColor);

            // top, left は 境界に整列(値が小さくなるように)した値で...
            //            float top  = MathUtil.AlignValueOnTheLowerBoundary( _screenOrigInScene.Y, BackGround.GridSize );
            //            float left = MathUtil.AlignValueOnTheLowerBoundary( _screenOrigInScene.X, BackGround.GridSize );
            //            float right            = _screenOrigInScene.X + widthInScreen;
            //            float bottom           = _screenOrigInScene.Y + heightInScreen;

            var sceenSizeInScene = CalcScreenSizeInScene_();

            float left = MathUtil.AlignValueOnTheLowerBoundary(ScreenOrigInScene.X - sceenSizeInScene.X * 0.5f, BackGround.GridSize);
            float top = MathUtil.AlignValueOnTheHigherBoundary(ScreenOrigInScene.Y + sceenSizeInScene.Y * 0.5f, BackGround.GridSize);
            float right = ScreenOrigInScene.X + sceenSizeInScene.X * 0.5f;
            float bottom = ScreenOrigInScene.Y - sceenSizeInScene.Y * 0.5f;

            float step = BackGround.GridSize / (float)BackGround.GridDivisionNum;


            Color mainLineColor = renderer.Color;
            Color subLineColor = Color.FromArgb(renderer.Color.A / 2, renderer.Color);

            // 縦
            for (float x = left; x < right; x += step)
            {
                // 分割線と、非分割線で、線の太さを変更します。
                renderer.Color = CalcSystemColor_(IsSubGridLine_(x, left, step, BackGround.GridDivisionNum) ? subLineColor : mainLineColor);
                renderer.DrawLine(new PointF(x, top), new PointF(x, bottom));
            }
            // 横
            for (float y = top; y > bottom; y -= step)
            {
                // 分割線と、非分割線で、線の太さを変更します。
                renderer.Color = CalcSystemColor_(IsSubGridLine_(y, top, step, BackGround.GridDivisionNum) ? subLineColor : mainLineColor);
                renderer.DrawLine(new PointF(left, y), new PointF(right, y));
            }

            renderer.Color = oldColor;
            renderer.LineWidth = oldLineWidth;
        }

        /// <summary>
        /// ビュー変換を設定します。
        /// </summary>
        private void SetupViewTransforms_()
        {
            double halfFovy = (double)(_renderer.FOV) * 0.5;
            float cameraDist = _BackGround.ScreenSize.Y * 0.5f / (float)Math.Tan(halfFovy);
            _renderer.SetViewTransform(_magnify, _magnify, ScreenOrigInScene.X, ScreenOrigInScene.Y, 0.0f, cameraDist);

            _renderer.SetDefaultScreenSize(_BackGround.ScreenSize.X, _BackGround.ScreenSize.Y);
        }

        /// <summary>
        /// 再描画イベント
        /// </summary>
        private void Event_MainView_Paint(object sender, System.Windows.Forms.PaintEventArgs e)
        {
            // システムカラーの初期化
            _drawableOption.InitializeSystemColors(_appSetting.ProjectSettings.SRGBWriteEnabled, !_appSetting.ProjectSettings.DegammaParameterConversionEnabled);

            // キャプチャテクスチャの更新処理
            // キャプチャテクスチャの更新が重いためマウスドラッグでシーンを操作しているときはキャプチャテクスチャを更新しないようにする。
            if (TargetSubScene != null &&
                !_mouseState.IsTweaking)
            {
                TargetSubScene.RenderCaptureTexture(_renderer);
            }

            // GDIレンダラを使用して、シーンの再描画 をおこないます。
            _renderer.BeginRendering(e.Graphics);
            _renderer.IdentityMtx();

            Color oldColor = _renderer.Color;

            // ビュー変換
            SetupViewTransforms_();
            // ------------- シーン座標系のオブジェクトを描画します。
            // 背景を描画
            // 塗りつぶしを使用するため、
            // 一旦、アウトライン描画を強制的にOFFにします。
            uint tempFlag = _renderer.OptionFlag;
            _renderer.DisableOptionFlag(IRenderOptionFlag.DrawOutline);
            DrawBackGround_(_renderer);
            _renderer.OptionFlag = tempFlag;

            // 座標系原点の描画
            _renderer.Color = _drawableOption.SystemRed;
            _renderer.DrawLine(PointF.Empty, new PointF(20.0f, 0.0f));
            _renderer.DrawLine(PointF.Empty, new PointF(0.0f, 20.0f));

            // シーンの描画
            if (TargetSubScene != null)
            {
                ISubScene ss = TargetSubScene;
                if (ss != null)
                {
                    ss.Draw(_renderer, _drawableOption);
                }
            }

            // 安全フレームを描画します。
            if (_SafeFrameVisible)
            {
                AppConfig config = LECore.LayoutEditorCore.Config;
                _renderer.PushMtx();
                // レイアウト画面左上位置へ移動
                _renderer.Trans(-_BackGround.ScreenSize.X / 2, _BackGround.ScreenSize.Y / 2, 0.0f);
                // 画面内の座標系は2D座標系で扱うこととします
                _renderer.Scale(1.0f, -1.0f);

                DrawSafeFrame_(_renderer, _BackGround.ScreenSize, AppSetting.SafeFrame);
                _renderer.PopMtx();
            }

            // 移動セットの描画
            _renderer.Color = _drawableOption.SystemGreen;
            _draggedPane.Draw(_renderer, _drawableOption);

            // 階層構築中なら..
            if (_hierarchyMnp.IsBuilding || _hierarchyMnp.IsBreaking)
            {
                _hierarchyMnp.Draw(_renderer, _drawableOption);
            }

            // ------------- スクリーン座標系のオブジェクトを描画します。
            _renderer.PushMtx();
            {
                _renderer.IdentityMtx();
                _renderer.Color = _drawableOption.SystemRed;
                _selectRect.Draw(_renderer, _drawableOption);
                _renderer.Color = oldColor;
            }
            _renderer.PopMtx();

            //
            // 情報の報告をします。
            //
            if (_TargetActiveSubScene != null && OnReportMsg != null)
            {
                string msg = _draggedPane.ReportMsgString();
                if (msg != string.Empty)
                {
                    OnReportMsg(msg);
                }
                else
                {
                    OnReportMsg("");
                }
            }

            _renderer.EndRendering();
        }

        /// <summary>
        /// フォーカス取得イベント
        /// </summary>
        /// <param name="sender"></param>
        /// <param name="e"></param>
        private void Event_MainView_GotFocus(object sender, EventArgs e)
        {
            if (TargetSubScene != null)
            {
                TargetSubScene.RequestRenderCaptureTexture(CaptureTextureUpdateMode.All);
            }
        }

        #endregion -------------------- 再描画 --------------------


        #region --------- マウスイベント ---------
        /// <summary>
        /// イベントハンドラ：マウスダウン
        /// </summary>
        private void Event_MainView_MouseDown(object sender, System.Windows.Forms.MouseEventArgs e)
        {
            // フォーカスを移動します。
            Focus();

            if (e.Button != MouseButtons.Right)
            {
                // 右ボタンはコンテキストメニューに割り当てられているため、
                // 処理をおこないません。
                _mouseState.OnMouseDown(e);
            }
        }

        /// <summary>
        /// ホイールハンドラ
        /// </summary>
        private void Event_MainView_MouseWheel(object sender, MouseEventArgs e)
        {
            // フォーカスを移動します。
            Focus();

            if (e.Delta != 0)
            {
                _mouseState.OnMouseMove(e);
            }
        }

        /// <summary>
        /// マウスムーブイベント
        /// </summary
        private void Event_MainView_MouseMove(object sender, System.Windows.Forms.MouseEventArgs e)
        {
            if (e.Button != MouseButtons.Right)
            {
                // 右ボタンはコンテキストメニューに割り当てられているため、
                // 処理をおこないません。
                _mouseState.OnMouseMove(e);
            }
        }

        /// <summary>
        /// マウスアップイベント
        /// </summary>
        private void Event_MainView_MouseUp(object sender, System.Windows.Forms.MouseEventArgs e)
        {

            if (e.Button != MouseButtons.Right)
            {
                // 右ボタンはコンテキストメニューに割り当てられているため、
                // 処理をおこないません。
                _mouseState.OnMouseUp(e);
            }
        }
        #endregion --------- マウスイベント ---------

        #region --------- キーダウン等のイベント ---------
        // キーダウン等のイベントもマウスステートに処理が委譲されます
        /// <summary>
        /// キーダウン
        /// </summary>
        public void Event_MainView_KeyDown(object sender, System.Windows.Forms.KeyEventArgs e)
        {
            // シフトキーの押し下げ時は移動量をグリッドサイズ倍にする
            var vecScale =
                ((Control.ModifierKeys & Keys.Shift) == Keys.Shift && !BackGround.GridSnapEnalbed) ?
                _GridSize : 1.0f;

            // 矢印キーによる移動
            switch (e.KeyCode)
            {
                case Keys.Up: AddStepOffsetToSelectedPaneSet_(FVec2.Up * vecScale); break;
                case Keys.Down: AddStepOffsetToSelectedPaneSet_(FVec2.Down * vecScale); break;
                case Keys.Left: AddStepOffsetToSelectedPaneSet_(FVec2.Left * vecScale); break;
                case Keys.Right: AddStepOffsetToSelectedPaneSet_(FVec2.Right * vecScale); break;
            }

            _mouseState.OnKeyDown(e);
        }

        /// <summary>
        /// プレビューキーダウン
        /// </summary>
        public void Event_MainView_PreviewKeyDown(object sender, System.Windows.Forms.PreviewKeyDownEventArgs e)
        {
            // 修飾キーを押されている時のみPreviewKeyDownでハンドリングする
            if (e.Shift || e.Control || e.Alt)
            {
                this.Event_MainView_KeyDown(sender, new KeyEventArgs(e.KeyData));
            }
        }

        /// <summary>
        /// グリッドに整列するような、difference値を計算します。
        /// </summary>
        PointF CalcGridAlignedDifference_(
            PointF basePos,
            PointF difference)
        {
            if (_TargetActiveSubScene.BackGround.GridSnapEnalbed)
            {
                float moveScale = _GridSize;

                difference.X *= moveScale;
                difference.Y *= moveScale;

                difference = PaneDragTransModifier.CalcAlignedDifference(_GridSize, basePos, difference);
            }

            return difference;
        }

        /// <summary>
        /// キー操作によって選択ペインを指定量だけ移動します。
        /// LayoutWindow Formから移動
        /// </summary>
        void AddStepOffsetToSelectedPaneSet_(FVec2 step)
        {
            IPane[] panes = _TargetActiveSubScene.ISelectedSet.IPaneArray;
            if (panes.Length != 0 && !panes.Any((pane) => pane.IsReadOnlyLocked))
            {
                PointF posPane = new PointF(panes[0].XInWorld, panes[0].YInWorld);
                PointF diff = CalcGridAlignedDifference_(posPane, step.AsPointF);
                // グリッド吸着が有効なら、移動量を調整します。


                // 移動量を反映します。
                // シーンに大量の変更を通知します。
                _TargetActiveSubScene.BeginMassiveModify();

                _subSceneMnp.AddOffsetToSelectedSet(
                    diff,
                    TransAffectsChildren,
                    SubSceneManipulator.ValueRouwndDownEnable);

                _TargetActiveSubScene.EndMassiveModify();
            }
        }

        /// <summary>
        /// キーアップ
        /// </summary>
        public void Event_MainView_KeyUp(object sender, System.Windows.Forms.KeyEventArgs e)
        {
            _mouseState.OnKeyUp(e);
        }
        #endregion --------- キーダウン等のイベント ---------

        #region --------- ドラッグ・ドロップ
        /// <summary>
        /// ドラッグ・ドロップ・Enter
        /// (テクスチャウインドウからの)
        /// </summary>
        private void Event_MainView_DragEnter(object sender, System.Windows.Forms.DragEventArgs e)
        {

        }

        /// <summary>
        /// ドラッグ・ドロップ・DragDrop
        /// </summary>
        private void Event_MainView_DragDrop(object sender, System.Windows.Forms.DragEventArgs e)
        {

        }

        #endregion --------- ドラッグ・ドロップ

        #endregion // -------------------- イベントハンドラ --------------------

        #region ISceneModifyListener メンバ

        /// <summary>
        /// シーン変更イベントハンドラ
        /// </summary>
        public void OnSceneModifyHandler(object sender, SceneModifyEventArgs e)
        {
            // 選択セットが変更されたら...
            if (e.Target == SceneModifyEventArgs.Kind.SelectedSetModify)
            {
                // ドラッグアイテムの登録更新を行います。
                IPane[] selSet = e.Paramaters[0] as IPane[];

                ISubScene subScene = sender as ISubScene;
                if (sender is IScene)
                {
                    subScene = ((IScene)sender).CurrentISubScene;
                }

                if (selSet.Length > 0 && _targetActiveSubScene == subScene)
                {
                    _draggedPane.BeginDragging(
                                               PaneDragTargetAdapter.FromPaneSet(selSet),
                                               this._magnify,
                                               DragModifierHelper.InvalidMousePos,
                                               DragModifierHelper.OptionFlag.IgnoreShortDrag);

                    _draggedPane.AddOptionFlag(PaneDragModifier.Option.NeedToSetup);

                    // グリッド吸着が有効ならば、グリッドサイズを設定します。
                    if (BackGround.GridSnapEnalbed)
                    {
                        _draggedPane.SetGridSnapEnable(_GridSize);
                    }
                }
                else
                {
                    _draggedPane.Clear();
                    Invalidate();
                }
            }
        }
        #endregion

        private void MainView_SizeChanged(object sender, EventArgs e)
        {
            Invalidate();
        }
    }
}
