﻿// --------------------------------------------------------------------------------
// <copyright>
// Copyright (C)Nintendo All rights reserved.
//
// These coded instructions, statements, and computer programs contain proprietary
// information of Nintendo and/or its licensed developers and are protected by
// national and international copyright laws. They may not be disclosed to third
// parties or copied or duplicated in any form, in whole or in part, without the
// prior written consent of Nintendo.
//
// The content herein is highly confidential and should be handled accordingly.
// </copyright>
// --------------------------------------------------------------------------------
using System;
using System.Drawing;
using System.Diagnostics;
using System.Collections;
using System.Collections.Generic;
using System.ComponentModel;
using System.Windows.Forms;
using System.Linq;

namespace LayoutEditor.Forms.ToolWindows.GroupWindow
{
    using LECore;
    using LECore.Structures;
    using LECore.Manipulator;
    using Utility;
    using LayoutEditor.Structures.SerializableObject;

    using common;
    using Controls;

    using TreeNodeDropMode = common.TreeNodeDragDropMgr.TreeNodeDropHint;

    /// <summary>
    /// GroupWindow の概要の説明です。
    /// </summary>
    public class GroupWindow : LEToolWindow
    {
        #region デザイナ変数
        private PaneTreeStatePanel.PaneHierarchyTreeView _tvwGroup;
        private PaneTreeStatePanel _pnlState;
        private MenuStrip _mspMain;
        private ToolStripMenuItem _tmiGroup;
        private ToolStripMenuItem _tmiCut;
        private ToolStripMenuItem _tmiCopy;
        private ToolStripMenuItem _tmiPaste;
        private ToolStripMenuItem _tmiDelete;
        private ToolStripSeparator toolStripSeparator1;
        private ToolStripMenuItem _tmiGroupEdit;
        private ToolStripMenuItem _tmiGroupDelete;
        private ToolStripSeparator toolStripSeparator2;
        private ToolStripMenuItem _tmiSetGroupLocked;
        private ToolStripMenuItem _tmiSetGroupUnLocked;
        private ToolStripMenuItem _tmiSetGroupHidden;
        private ToolStripMenuItem _tmiSetGroupNotHidden;
        private ToolStripMenuItem _tmiShowSelectedEnable;
        private ToolStripSeparator toolStripSeparator3;
        private LayoutEditor.Controls.DockContentToolTip _ttpMain;
        private ToolStripMenuItem _tmiExpandAll;
        private ToolStripMenuItem _tmiCollopseAll;
        private IContainer components = null;
        #endregion デザイナ変数


        #region Windows フォーム デザイナで生成されたコード
        /// <summary>
        /// デザイナ サポートに必要なメソッドです。このメソッドの内容を
        /// コード エディタで変更しないでください。
        /// </summary>
        private void InitializeComponent()
        {
            this.components = new System.ComponentModel.Container();
            System.ComponentModel.ComponentResourceManager resources = new System.ComponentModel.ComponentResourceManager(typeof(GroupWindow));
            this._tvwGroup = new LayoutEditor.Forms.ToolWindows.common.PaneTreeStatePanel.PaneHierarchyTreeView();
            this._pnlState = new LayoutEditor.Forms.ToolWindows.common.PaneTreeStatePanel();
            this._mspMain = new System.Windows.Forms.MenuStrip();
            this._tmiGroup = new System.Windows.Forms.ToolStripMenuItem();
            this._tmiCut = new System.Windows.Forms.ToolStripMenuItem();
            this._tmiCopy = new System.Windows.Forms.ToolStripMenuItem();
            this._tmiPaste = new System.Windows.Forms.ToolStripMenuItem();
            this._tmiDelete = new System.Windows.Forms.ToolStripMenuItem();
            this.toolStripSeparator1 = new System.Windows.Forms.ToolStripSeparator();
            this._tmiGroupEdit = new System.Windows.Forms.ToolStripMenuItem();
            this._tmiGroupDelete = new System.Windows.Forms.ToolStripMenuItem();
            this.toolStripSeparator2 = new System.Windows.Forms.ToolStripSeparator();
            this._tmiSetGroupLocked = new System.Windows.Forms.ToolStripMenuItem();
            this._tmiSetGroupUnLocked = new System.Windows.Forms.ToolStripMenuItem();
            this._tmiSetGroupHidden = new System.Windows.Forms.ToolStripMenuItem();
            this._tmiSetGroupNotHidden = new System.Windows.Forms.ToolStripMenuItem();
            this.toolStripSeparator3 = new System.Windows.Forms.ToolStripSeparator();
            this._tmiShowSelectedEnable = new System.Windows.Forms.ToolStripMenuItem();
            this._tmiExpandAll = new System.Windows.Forms.ToolStripMenuItem();
            this._tmiCollopseAll = new System.Windows.Forms.ToolStripMenuItem();
            this._ttpMain = new LayoutEditor.Controls.DockContentToolTip(this.components);
            this._mspMain.SuspendLayout();
            this.SuspendLayout();
            //
            // _tvwGroup
            //
            this._tvwGroup.AllowDrop = true;
            resources.ApplyResources(this._tvwGroup, "_tvwGroup");
            this._tvwGroup.BorderStyle = System.Windows.Forms.BorderStyle.FixedSingle;
            this._tvwGroup.FullRowSelect = true;
            this._tvwGroup.ItemHeight = 16;
            this._tvwGroup.LabelEdit = true;
            this._tvwGroup.Name = "_tvwGroup";
            this._tvwGroup.ShowRootLines = false;
            this._tvwGroup.BeforeLabelEdit += new System.Windows.Forms.NodeLabelEditEventHandler(this.Event_TvwHierarchy_BeforeLabelEdit);
            this._tvwGroup.AfterLabelEdit += new System.Windows.Forms.NodeLabelEditEventHandler(this.Event_TvwGroup_AfterLabelEdit);
            this._tvwGroup.KeyDown += new System.Windows.Forms.KeyEventHandler(this.Event_TvwGroup_KeyDown);
            this._tvwGroup.MouseDown += new System.Windows.Forms.MouseEventHandler(this.Event_TvwGroup_MouseDown);
            this._tvwGroup.MouseUp += new System.Windows.Forms.MouseEventHandler(this.Event_TvwGroup_MouseUp);
            //
            // _pnlState
            //
            resources.ApplyResources(this._pnlState, "_pnlState");
            this._pnlState.BackColor = System.Drawing.SystemColors.Window;
            this._pnlState.BorderStyle = System.Windows.Forms.BorderStyle.FixedSingle;
            this._pnlState.Name = "_pnlState";
            //
            // _mspMain
            //
            this._mspMain.Items.AddRange(new System.Windows.Forms.ToolStripItem[] {
            this._tmiGroup,
            this._tmiExpandAll,
            this._tmiCollopseAll});
            resources.ApplyResources(this._mspMain, "_mspMain");
            this._mspMain.Name = "_mspMain";
            //
            // _tmiGroup
            //
            this._tmiGroup.DropDownItems.AddRange(new System.Windows.Forms.ToolStripItem[] {
            this._tmiCut,
            this._tmiCopy,
            this._tmiPaste,
            this._tmiDelete,
            this.toolStripSeparator1,
            this._tmiGroupEdit,
            this._tmiGroupDelete,
            this.toolStripSeparator2,
            this._tmiSetGroupLocked,
            this._tmiSetGroupUnLocked,
            this._tmiSetGroupHidden,
            this._tmiSetGroupNotHidden,
            this.toolStripSeparator3,
            this._tmiShowSelectedEnable});
            this._tmiGroup.Name = "_tmiGroup";
            resources.ApplyResources(this._tmiGroup, "_tmiGroup");
            this._tmiGroup.DropDownOpening += new System.EventHandler(this.Event_TmiGroup_DropDownOpening);
            //
            // _tmiCut
            //
            this._tmiCut.Name = "_tmiCut";
            resources.ApplyResources(this._tmiCut, "_tmiCut");
            this._tmiCut.Click += new System.EventHandler(this.Event_MniCut_Click);
            //
            // _tmiCopy
            //
            this._tmiCopy.Name = "_tmiCopy";
            resources.ApplyResources(this._tmiCopy, "_tmiCopy");
            this._tmiCopy.Click += new System.EventHandler(this.Event_MniCopy_Click);
            //
            // _tmiPaste
            //
            this._tmiPaste.Name = "_tmiPaste";
            resources.ApplyResources(this._tmiPaste, "_tmiPaste");
            this._tmiPaste.Click += new System.EventHandler(this.Event_MniPaste_Click);
            //
            // _tmiDelete
            //
            this._tmiDelete.Name = "_tmiDelete";
            resources.ApplyResources(this._tmiDelete, "_tmiDelete");
            this._tmiDelete.Click += new System.EventHandler(this.Event_MniGroupDelete_Click);
            //
            // toolStripSeparator1
            //
            this.toolStripSeparator1.Name = "toolStripSeparator1";
            resources.ApplyResources(this.toolStripSeparator1, "toolStripSeparator1");
            //
            // _tmiGroupEdit
            //
            this._tmiGroupEdit.Name = "_tmiGroupEdit";
            resources.ApplyResources(this._tmiGroupEdit, "_tmiGroupEdit");
            this._tmiGroupEdit.Click += new System.EventHandler(this.Event_MniGroupEdit_Click);
            //
            // _tmiGroupDelete
            //
            this._tmiGroupDelete.Name = "_tmiGroupDelete";
            resources.ApplyResources(this._tmiGroupDelete, "_tmiGroupDelete");
            this._tmiGroupDelete.Click += new System.EventHandler(this.Event_MniGroupDelete_Click);
            //
            // toolStripSeparator2
            //
            this.toolStripSeparator2.Name = "toolStripSeparator2";
            resources.ApplyResources(this.toolStripSeparator2, "toolStripSeparator2");
            //
            // _tmiSetGroupLocked
            //
            this._tmiSetGroupLocked.Name = "_tmiSetGroupLocked";
            resources.ApplyResources(this._tmiSetGroupLocked, "_tmiSetGroupLocked");
            this._tmiSetGroupLocked.Click += new System.EventHandler(this.Event_MniGroupSetLock_Click);
            //
            // _tmiSetGroupUnLocked
            //
            this._tmiSetGroupUnLocked.Name = "_tmiSetGroupUnLocked";
            resources.ApplyResources(this._tmiSetGroupUnLocked, "_tmiSetGroupUnLocked");
            this._tmiSetGroupUnLocked.Click += new System.EventHandler(this.Event_MniGroupSetUnlock_Click);
            //
            // _tmiSetGroupHidden
            //
            this._tmiSetGroupHidden.Name = "_tmiSetGroupHidden";
            resources.ApplyResources(this._tmiSetGroupHidden, "_tmiSetGroupHidden");
            this._tmiSetGroupHidden.Click += new System.EventHandler(this.Event_MniGroupSetHidden_Click);
            //
            // _tmiSetGroupNotHidden
            //
            this._tmiSetGroupNotHidden.Name = "_tmiSetGroupNotHidden";
            resources.ApplyResources(this._tmiSetGroupNotHidden, "_tmiSetGroupNotHidden");
            this._tmiSetGroupNotHidden.Click += new System.EventHandler(this.Event_MniGroupSetNotHidden_Click);
            //
            // toolStripSeparator3
            //
            this.toolStripSeparator3.Name = "toolStripSeparator3";
            resources.ApplyResources(this.toolStripSeparator3, "toolStripSeparator3");
            //
            // _tmiShowSelectedEnable
            //
            this._tmiShowSelectedEnable.Name = "_tmiShowSelectedEnable";
            resources.ApplyResources(this._tmiShowSelectedEnable, "_tmiShowSelectedEnable");
            this._tmiShowSelectedEnable.Click += new System.EventHandler(this.Event_MniShowSelectedEnable_Click);
            //
            // _tmiExpandAll
            //
            this._tmiExpandAll.AutoToolTip = true;
            this._tmiExpandAll.DisplayStyle = System.Windows.Forms.ToolStripItemDisplayStyle.Image;
            resources.ApplyResources(this._tmiExpandAll, "_tmiExpandAll");
            this._tmiExpandAll.Name = "_tmiExpandAll";
            this._tmiExpandAll.Click += new System.EventHandler(this.Event_tmiExpandAll_Click);
            //
            // _tmiCollopseAll
            //
            this._tmiCollopseAll.AutoToolTip = true;
            this._tmiCollopseAll.DisplayStyle = System.Windows.Forms.ToolStripItemDisplayStyle.Image;
            resources.ApplyResources(this._tmiCollopseAll, "_tmiCollopseAll");
            this._tmiCollopseAll.Name = "_tmiCollopseAll";
            this._tmiCollopseAll.Click += new System.EventHandler(this.Event_tmiCollopseAll_Click);
            //
            // GroupWindow
            //
            resources.ApplyResources(this, "$this");
            this.Controls.Add(this._pnlState);
            this.Controls.Add(this._tvwGroup);
            this.Controls.Add(this._mspMain);
            this.FormBorderStyle = System.Windows.Forms.FormBorderStyle.SizableToolWindow;
            this.MainMenuStrip = this._mspMain;
            this.Name = "GroupWindow";
            this.Activated += new System.EventHandler(this.GroupWindow_Activated);
            this._mspMain.ResumeLayout(false);
            this._mspMain.PerformLayout();
            this.ResumeLayout(false);
            this.PerformLayout();

        }
        #endregion

        #region 型宣言

        /// <summary>
        /// クリップボード用ペインセット
        /// </summary>
        sealed class GroupWindowClipboardPaneSet : List<IPane>{}

        /// <summary>
        /// クリップボードにデータを渡すための、インタフェース調整クラス
        /// </summary>
        class LEClipboardGroupConvertFunc : LEClipboardConvertFunc
        {
            /// <summary>
            /// コンストラクタ
            /// </summary>
            public LEClipboardGroupConvertFunc()
                : base( new DataConvertFunc( ToSavableFunc_ ),
            new DataConvertFunc( ToInternalFunc_ ),
            typeof( GroupWindowClipboardPaneSet ),
            typeof( GroupWindowClipboardPaneSet ) )
            {

            }

            /// <summary>
            /// シリアライズ可能なオブジェクトへ変換します。
            /// </summary>
            static object ToSavableFunc_( object obj )
            {
                return obj;
            }

            /// <summary>
            /// アプリケーション内部オブジェクトへ変換します。
            /// </summary>
            static object ToInternalFunc_( object obj )
            {
                return obj;
            }
        }
        #endregion 型宣言

        #region フィールド
        /// <summary>
        /// 複数選択処理モジュール
        /// </summary>
        readonly MultiSelectTreeNodeMgr _multiSelectTreeNodeMgr;

        /// <summary>
        /// ドラッグドロップ処理モジュール
        /// </summary>
        readonly TreeNodeDragDropMgr _treeNodeDragDropMgr;
        readonly TreeNodeStateHolder<object> _treeNodeStateHolder = new TreeNodeStateHolder<object>(
            (rootPane) => LayoutEditorCore.Scene.ContainsOwnerSubSceneOf(rootPane as IPane));

        TreeNodeFactory _treeNodeFactory = new TreeNodeFactory( new PaneStateCheckPolicy() );

        const string _RootGroupItemImageKeyStr = "RootGroup";
        const string _GroupItemImageKeyStr = "Group";
        const string _LockedGroupItemImageKeyStr = "LockedGroup";
        const string _HiddenGroupItemImageKeyStr = "HiddenGroup";
        const string _DerivativeGroupItemImageKeyStr = "DerivativeGroup";

        /// <summary>
        /// 選択ペインを自動表示するか？永続化されるパラメータです。
        /// </summary>
        bool _showSelectedEnable = true;

        private ShortcutHandler _shortcutHandler;
        #endregion フィールド

        #region プロパティ
        /// <summary>
        /// 現在の対象シーン
        /// </summary>
        ISubScene _CurrentISubScene
        {
            get { return LayoutEditorCore.Scene.CurrentISubScene; }
        }

        /// <summary>
        /// グループマネージャ
        /// </summary>
        ILEGroupMgr _ILEGroupMgr
        {
            get { return ( _CurrentISubScene != null ) ? _CurrentISubScene.ILEGroupMgr : null; }
        }

        /// <summary>
        /// クリップボード
        /// </summary>
        ILEClipboard _ILEClipboard
        {
            get { return LECore.LayoutEditorCore.Scene.ILEClipboard; }
        }

        /// <summary>
        /// 貼り付けは有効か?
        /// </summary>
        bool _PatseSrcEnalbe
        {
            get
            {
                return _ILEClipboard.CanPaste( typeof( GroupWindowClipboardPaneSet ) );
            }
        }

        /// <summary>
        /// グループのみが選択対象か？
        /// </summary>
        bool _GroupOnlySelected
        {
            get
            {
                if( _multiSelectTreeNodeMgr.Empty )
                {
                    return false;
                }
                else
                {
                    foreach( TreeNode node in _multiSelectTreeNodeMgr.SelectedNodeSet )
                    {
                        if( !( node.Tag is ILEGroup ) )
                        {
                            return false;
                        }
                    }
                    return true;
                }
            }
        }

        /// <summary>
        /// ペインのみが選択対象か？
        /// </summary>
        bool _PaneOnlySelected
        {
            get
            {
                if( _CurrentISubScene == null || _multiSelectTreeNodeMgr.Empty )
                {
                    return false;
                }
                else
                {
                    foreach( TreeNode node in _multiSelectTreeNodeMgr.SelectedNodeSet )
                    {
                        if( !( node.Tag is IPane ) )
                        {
                            return false;
                        }
                    }
                    return true;
                }
            }
        }

        #endregion プロパティ

        /// <summary>
        /// TreeViewの画像を初期化します。
        /// </summary>
        void InitializeTreeviewImage_()
        {
            _tvwGroup.ImageList = new ImageList();
            _tvwGroup.ImageList.Images.Add( _RootGroupItemImageKeyStr, ImageResMgr.GetManifestResourcePng( "group_nrm_16.png" ) );
            _tvwGroup.ImageList.Images.Add( _GroupItemImageKeyStr, ImageResMgr.GetManifestResourcePng( "group_nrm_16.png" ) );
            _tvwGroup.ImageList.Images.Add( _LockedGroupItemImageKeyStr, ImageResMgr.GetManifestResourcePng( "group_lck_16.png" ) );
            _tvwGroup.ImageList.Images.Add( _HiddenGroupItemImageKeyStr, ImageResMgr.GetManifestResourcePng( "group_hdn_16.png" ) );
            _tvwGroup.ImageList.Images.Add( _DerivativeGroupItemImageKeyStr, ImageResMgr.GetManifestResourcePng( "group_roy_16.png" ) );
            _tvwGroup.ImageList.ColorDepth = ColorDepth.Depth32Bit;
            PaneItemImageHelper.CopyPaneItemImage( _tvwGroup.ImageList );
        }

        /// <summary>
        /// コンストラクタ
        /// </summary>
        public GroupWindow()
        {
            InitializeComponent();

            InitializeTreeviewImage_();

            _multiSelectTreeNodeMgr = new MultiSelectTreeNodeMgr( _tvwGroup );
            _multiSelectTreeNodeMgr.ChangeSelecetItemByMouse += OnGroupSelectionChanged_;
            _multiSelectTreeNodeMgr.CheckNodeContentSelected += OnCheckSelectNode_;
            _multiSelectTreeNodeMgr.SetNodeApperanceSelected += SetNodeApperanceSelected_;
            _multiSelectTreeNodeMgr.ResetNodeApperanceSelected += ResetNodeApperanceSelected_;

            _treeNodeDragDropMgr = new TreeNodeDragDropMgr(
                _tvwGroup,
                CheckValidToDrop_,
                OnBeginDragHandler_,
                OnDropHandler_ );

            _pnlState.Initialize( _tvwGroup );

            InitializeDlgState();

            // クリップボード利用登録
            _ILEClipboard.RegisterConvertionFuncSet( new LEClipboardGroupConvertFunc() );

            // メッセージフィルタ設定
            ToolStripMenuItemHelper.ToolStripMessageFilter.BindMessageFilter( _mspMain );

            // ドッキングウインドウ用のショートカットの自前処理の設定
            _shortcutHandler = new ShortcutHandler(_mspMain);
        }

        #region TreeNodeDragDropMgr ハンドラ関連

        /// <summary>
        /// 追加モードとして有効か判定します。
        /// </summary>
        bool IsValidForAddMode_( TreeNode targetNode, TreeNode childCandidateNode )
        {
            // ドロップ先について...
            // ルートノードは不正
            if( targetNode.Parent == null )
            {
                return false;
            }

            // グループをあらわすノードでなければ不正
            if( !( targetNode.Tag is ILEGroup ) )
            {
                return false;
            }

            // ドラッグノードがペインをあらわすノードでなければ不正
            IPane childPane = childCandidateNode.Tag as IPane;
            if (childPane == null)
            {
                return false;
            }

            // ペインが属するグループが読み取り専用なら不正
            ILEGroup parentGroup = childCandidateNode.Parent != null ? childCandidateNode.Parent.Tag as ILEGroup : null;
            if (parentGroup != null && parentGroup.IsReadOnlyLocked)
            {
                return false;
            }

            // グループがすでに、ドラッグペインを含む場合は不正
            ILEGroup targetGroup = targetNode.Tag as ILEGroup;
            if( targetGroup.Contains( childCandidateNode.Tag as IPane ) )
            {
                return false;
            }

            // グループが読み取り専用なら不正
            if (targetGroup.IsReadOnlyLocked)
            {
                return false;
            }

            // 現在の親に対するドロップは不正
            if( childCandidateNode.Parent == targetNode )
            {
                return false;
            }

            return true;
        }

        /// <summary>
        /// 挿入モードとして有効か判定します。
        /// </summary>
        bool IsValidForInsertMode_( TreeNode targetNode, TreeNode childCandidateNode )
        {
            if( targetNode == childCandidateNode )
            {
                return false;
            }

            // 挿入モードの場合
            // ペインに対するペインの挿入なら
            if( ( targetNode.Tag is IPane ) &&
                 ( childCandidateNode.Tag is IPane ) )
            {
                Debug.Assert( targetNode.Parent != null );
                Debug.Assert( targetNode.Parent.Tag is ILEGroup );

                // 対象ペインの所属グループが読み取り専用なら不正
                if ((targetNode.Parent.Tag as ILEGroup).IsReadOnlyLocked)
                {
                    return false;
                }

                // ドラッグペインの所属グループが読み取り専用なら不正
                if ((childCandidateNode.Parent.Tag as ILEGroup).IsReadOnlyLocked)
                {
                    return false;
                }

                // 異なる親への移動の場合は、すでにドラッグ対象がふくまれてれば不正
                if( targetNode.Parent != childCandidateNode.Parent )
                {
                    ILEGroup targetGroup = targetNode.Parent.Tag as ILEGroup;
                    if( targetGroup.Contains( childCandidateNode.Tag as IPane ) )
                    {
                        return false;
                    }
                }

                return true;
            }
            // グループに対するグループの挿入なら有効
            else if( ( targetNode.Tag is ILEGroup ) &&
                      ( childCandidateNode.Tag is ILEGroup ) )
            {
                // どちらかが読み取り専用なら不正
                if ((targetNode.Tag as ILEGroup).IsReadOnlyLocked ||
                    (childCandidateNode.Tag as ILEGroup).IsReadOnlyLocked)
                {
                    return false;
                }

                // ドロップ先について...
                // ルートノードは不正
                if( targetNode.Parent == null )
                {
                    return false;
                }

                return true;
            }

            // 想定しないケースは不正。
            return false;
        }

        /// <summary>
        /// targetNode が childCandidateNode のドロップ先として正しいか判定します。
        /// </summary>
        TreeNodeDropMode CheckValidToDrop_(
            TreeNode targetNode,
            TreeNode[] childCandidateNodeSet )
        {
            TreeNodeDropMode validModeForAll = TreeNodeDropMode.DropModeMask;

            bool bSameParent = true;
            foreach( TreeNode childNode in childCandidateNodeSet )
            {
                TreeNodeDropMode validMode = TreeNodeDropMode.None;

                if( IsValidForAddMode_( targetNode, childNode ) )
                {
                    validMode |= TreeNodeDropMode.AddChild;
                }

                if( IsValidForInsertMode_( targetNode, childNode ) )
                {
                    validMode |= TreeNodeDropMode.InsertNext | TreeNodeDropMode.InsertPrev;
                }

                bSameParent &= childNode.Parent == targetNode.Parent;

                validModeForAll &= validMode;
            }

            // コピー可能フラグを指定します。
            if( ( Control.ModifierKeys & Keys.Control ) != 0 )
            {
                // 異なる親に対してのみ有効とします。
                if( !bSameParent )
                {
                    validModeForAll |= TreeNodeDropMode.CopyItemFlag;
                }
            }

            return validModeForAll;
        }

        /// <summary>
        /// ドラッグ開始ハンドラ
        /// </summary>
        bool OnBeginDragHandler_( out TreeNode[] selectedNodeSet )
        {
            if( !_multiSelectTreeNodeMgr.Empty )
            {
                TreeNode[] nodeSet = _multiSelectTreeNodeMgr.SelectedNodeSet;

                // 異なる種類のノードを同時に選択していたらNG
                if( nodeSet.Length > 0 )
                {
                    if( nodeSet[0].Tag is IPane )
                    {
                        for( int i = 1 ; i < nodeSet.Length ; i++ )
                        {
                            if( nodeSet[i].Tag is ILEGroup || nodeSet[i].Parent == null )
                            {
                                goto NG_Case;
                            }
                        }
                    }
                    else
                    {
                        for( int i = 1 ; i < nodeSet.Length ; i++ )
                        {
                            if( nodeSet[i].Tag is IPane || nodeSet[i].Parent == null )
                            {
                                goto NG_Case;
                            }
                        }
                    }
                }

                selectedNodeSet = nodeSet;
                return true;
            }

            // ドラッグを開始しません。
            NG_Case:
            selectedNodeSet = null;
            return false;
        }

        /// <summary>
        /// ノードがあらわす、グループ名を取得します。
        /// </summary>
        static string GetGroupName_( TreeNode node )
        {
            Debug.Assert( node.Tag is ILEGroup );
            return ( node.Tag as ILEGroup ).GrouprName;
        }

        /// <summary>
        /// ドロップ処理ハンドラ
        /// </summary>
        void OnDropHandler_(
            TreeNode targetNode,
            TreeNode[] childNodeSet,
            TreeNodeDropMode mode )
        {
            Debug.Assert( childNodeSet.Length != 0 );

            if( mode == TreeNodeDropMode.AddChild )
            {
                ILEGroup targetGroup = _ILEGroupMgr.FindGroupByName( GetGroupName_( targetNode ) );

                GroupMgrManipulator groupMgrMnp = new GroupMgrManipulator();
                groupMgrMnp.BindTarget( _ILEGroupMgr );

                // 編集開始
                groupMgrMnp.BeginEditGroup();


                //------------------------------------------
                // childNodeSet から ペインセットを抽出します。
                List<IPane> childPaneSet = new List<IPane>();
                foreach( TreeNode node in childNodeSet )
                {
                    Debug.Assert( node.Tag is IPane );
                    Debug.Assert( node.Parent != null );

                    IPane pane = node.Tag as IPane;
                    if( !targetGroup.Contains( pane ) )
                    {
                        // コピーで無い場合は現在のグループから削除します
                        if( ( mode & TreeNodeDropMode.CopyItemFlag ) == 0 )
                        {
                            groupMgrMnp.RemoveMember( node.Parent.Tag as ILEGroup, pane );
                        }

                        childPaneSet.Add( node.Tag as IPane );
                    }
                }


                //-------------------------------------------
                // targetNode に メンバとして加えます
                if( childPaneSet.Count > 0 )
                {
                    foreach( IPane pane in childPaneSet )
                    {
                        groupMgrMnp.AddMember( targetGroup, pane );
                    }
                }

                // 編集終了
                groupMgrMnp.EndEditGroup();
            }
            else
            {
                int nodeIndex = targetNode.Parent.Nodes.IndexOf( targetNode );
                Debug.Assert( nodeIndex != -1 );

                if( mode == TreeNodeDropMode.InsertNext )
                {
                    nodeIndex += 1;
                }

                ILEGroup targetGroup = targetNode.Parent.Tag as ILEGroup;

                GroupMgrManipulator groupMgrMnp = new GroupMgrManipulator();
                groupMgrMnp.BindTarget( _ILEGroupMgr );
                groupMgrMnp.BeginEditGroup();

                // ペインの挿入
                if( targetNode.Tag is IPane )
                {
                    Debug.Assert( targetNode.Parent != null &&
                                    targetNode.Parent.Tag is ILEGroup );

                    foreach( TreeNode childNode in childNodeSet )
                    {
                        IPane childPane = childNode.Tag as IPane;
                        Debug.Assert( childPane != null );

                        if( !targetGroup.Contains( childPane ) )
                        {
                            ILEGroup oldGroup = childNode.Parent.Tag as ILEGroup;
                            Debug.Assert( oldGroup != targetGroup );

                            // コピーで無い場合は元のグループから削除して...
                            if( ( mode & TreeNodeDropMode.CopyItemFlag ) == 0 )
                            {
                                groupMgrMnp.RemoveMember( oldGroup, childPane );
                            }
                            // 新しいグループに登録
                            groupMgrMnp.AddMember( targetGroup, childPane );
                        }
                        nodeIndex = groupMgrMnp.ChangeChildOrder( targetGroup, childPane, nodeIndex );
                    }
                }
                else
                {
                    // グループの挿入
                    Debug.Assert( targetNode.Tag is ILEGroup );

                    foreach( TreeNode childNode in childNodeSet )
                    {
                        ILEGroup childGroup = childNode.Tag as ILEGroup;
                        Debug.Assert( childGroup != null );

                        nodeIndex = groupMgrMnp.ChangeGroupOrder(
                            childGroup,
                            nodeIndex );
                    }
                }

                groupMgrMnp.EndEditGroup();
            }

        }

        #endregion TreeNodeDragDropMgr ハンドラ関連

        #region MultiSelectTreeNodeMgr ハンドラ
        /// <summary>
        ///
        /// </summary>
        void OnGroupSelectionChanged_( TreeNode[] selectedNode )
        {
            SubSceneManipulator subSceneMnp = new SubSceneManipulator();
            subSceneMnp.BindTarget( _CurrentISubScene );

            subSceneMnp.BeginSelectSetChange();
            subSceneMnp.ResetSelectedSet();

            foreach( TreeNode node in selectedNode )
            {
                object target = node.Tag;
                // target がペインだったら選択します。
                var pane = target as IPane;
                if (pane != null && !pane.IsRootPane())
                {
                    subSceneMnp.SelectPanesByPaneRef(target as IPane);
                }
                // それ以外は何もしません。
                // target がgroupだったら子供ペインを全選択する仕様は廃止になりました。
            }

            subSceneMnp.EndSelectSetChange();

            UpDataMenuItemEnableState_();
        }

        /// <summary>
        ///
        /// </summary>
        bool OnCheckSelectNode_( TreeNode node )
        {
            if( node.Tag is IPane )
            {
                IPane pane = node.Tag as IPane;
                if( pane.IsSelected )
                {
                    return true;
                }
            }
            return false;
        }

        /// <summary>
        /// 選択状態に設定します。
        /// </summary>
        void SetNodeApperanceSelected_( List<TreeNode> selectedSet )
        {
            _tvwGroup.BeginUpdate();

            foreach( TreeNode node in selectedSet )
            {
                UpdateTreeNode_( node, true );
            }

            _tvwGroup.EndUpdate();
        }

        /// <summary>
        /// 選択状態からリセットします。
        /// </summary>
        void ResetNodeApperanceSelected_( List<TreeNode> selectedSet )
        {
            _tvwGroup.BeginUpdate();

            foreach( TreeNode node in selectedSet )
            {
                UpdateTreeNode_( node, false );
            }

            _tvwGroup.EndUpdate();
        }
        #endregion MultiSelectTreeNodeMgr ハンドラ

        /// <summary>
        /// 使用されているリソースに後処理を実行します。
        /// </summary>
        protected override void Dispose( bool disposing )
        {
            if( disposing )
            {
                if( components != null )
                {
                    components.Dispose();
                }
            }
            base.Dispose( disposing );
        }



        #region Node新規作成

        /// <summary>
        /// ペインの画像イメージ番号を取得します。
        /// </summary>
        /// <param name="pane">ペイン</param>
        /// <returns>ペインの画像イメージ番号を返します。</returns>
        int GetPaneItemImageIndex_( IPane pane )
        {
            string keyString = PaneItemImageHelper.GetPaneImageKindAsString( pane );
            return _tvwGroup.ImageList.Images.IndexOfKey( keyString );
        }

        /// <summary>
        /// グループの画像イメージ番号を取得します。
        /// </summary>
        /// <param name="group">グループ</param>
        /// <returns>グループの画像イメージ番号を返します。</returns>
        int GetGroupItemImageIndex_( ILEGroup group )
        {
            if (group.IsReadOnlyLocked)
            {
                return _tvwGroup.ImageList.Images.IndexOfKey( _DerivativeGroupItemImageKeyStr );
            }

            bool allHidden = true;
            bool anyLocked = false;

            foreach ( IPane pane in group.Member )
            {
                if (!pane.Hidden)
                {
                    allHidden = false;
                }

                if (pane.Locked)
                {
                    anyLocked = true;
                }
            }

            if (allHidden)
            {
                return _tvwGroup.ImageList.Images.IndexOfKey( _HiddenGroupItemImageKeyStr );
            }
            else if (anyLocked)
            {
                return _tvwGroup.ImageList.Images.IndexOfKey( _LockedGroupItemImageKeyStr );
            }

            return _tvwGroup.ImageList.Images.IndexOfKey( _GroupItemImageKeyStr );
        }

        /// <summary>
        /// ツリービューアイテム画像インデックスを取得します。
        /// </summary>
        int GetImageIndex_( ILEGroup groupMember )
        {
            if( groupMember is IPane )
            {
                return GetPaneItemImageIndex_( groupMember as IPane );
            }
            else
            {
                return GetGroupItemImageIndex_( groupMember );
            }
        }

        /// <summary>
        /// グループ用TreeNodeを作成します。
        /// _treeNodeFactory による生成を行いません。
        /// Tagの内容で、ノードをルックアップすることが無いためです。
        ///
        /// 上記は、変更になりました。
        /// Groupで検索をする必要ができたため、_treeNodeFactory による生成を行います。
        /// </summary>
        TreeNode MakeNewNodeForGroup_( ILEGroup groupMember )
        {
            TreeNode newNode = _treeNodeFactory.MakeNewNode( groupMember );

            newNode.Text = groupMember.GrouprName;
            newNode.Tag = groupMember;
            newNode.ImageIndex = GetImageIndex_( groupMember );
            newNode.SelectedImageIndex = newNode.ImageIndex;
            UpdateTreeNode_(newNode, false);

            return newNode;
        }

        /// <summary>
        /// ペイン用TreeNodeを作成します。
        /// </summary>
        TreeNode MakeNewNodeForPane_( IPane pane )
        {
            TreeNode childNode = _treeNodeFactory.MakeNewNode( pane );

            childNode.ImageIndex = GetPaneItemImageIndex_( pane );
            childNode.SelectedImageIndex = childNode.ImageIndex;

            UpdateTreeNode_( childNode, false );

            return childNode;
        }

        /// <summary>
        /// グループをTreeViewのノードとして登録します。
        /// </summary>
        void RegisterGroupAsTreeNode_( ILEGroup groupMember, TreeNode parentNode )
        {
            TreeNode newNode = MakeNewNodeForGroup_( groupMember );
            parentNode.Nodes.Add( newNode );

            // メンバペインをノード登録します。
            foreach( IPane pane in groupMember.Member )
            {
                TreeNode childNode = MakeNewNodeForPane_( pane );
                newNode.Nodes.Add( childNode );
            }
        }
        #endregion Node新規作成

        /// <summary>
        /// ダイアログの状態を初期化します。
        /// </summary>
        public void InitializeDlgState()
        {
            LEToolWindow.DbgReportUpdate( this );

            _treeNodeFactory.Clear();

            if( _ILEGroupMgr != null )
            {
                TreeNode tempNode = new TreeNode( _ILEGroupMgr.RootNodeName );
                tempNode.ImageIndex = _tvwGroup.ImageList.Images.IndexOfKey( _RootGroupItemImageKeyStr );
                tempNode.SelectedImageIndex = tempNode.ImageIndex;

                // シーンのグループマネージャの状態をTreeViewに反映します。

                foreach( ILEGroup member in _ILEGroupMgr.GroupSet )
                {
                    RegisterGroupAsTreeNode_( member, tempNode );
                }
                tempNode.ExpandAll();

                // 元の状態があるなら、ノードの状態を保存コピーします。
                if( _tvwGroup.Nodes.Count > 0 )
                {
                    TreeNode srcGroupRoot = _tvwGroup.Nodes[0] as TreeNode;

                    _treeNodeStateHolder.PreserveNodeState(srcGroupRoot, _tvwGroup.Tag as IPane);
                    _treeNodeStateHolder.RestoreNodeState(tempNode, _CurrentISubScene.RootIPane);
                }

                _tvwGroup.BeginUpdate();

                // スクロール位置の保存。
                int scrollPos_V = LECore.Win32.User32.GetScrollPos( _tvwGroup.Handle, LECore.Win32.SB.SB_VERT );

                // クリアします。
                _tvwGroup.Nodes.Clear();
                // 登録します。
                _tvwGroup.Nodes.Add( tempNode );
                // ルートは必ず開いておく。
                tempNode.Expand();

                _tvwGroup.Tag = _CurrentISubScene.RootIPane;

                // スクロール位置の復元。
                LECore.Win32.User32.SetScrollPos( _tvwGroup.Handle, LECore.Win32.SB.SB_VERT, scrollPos_V, true );

                _tvwGroup.EndUpdate();
            }
            else
            {
                // クリアします。
                _tvwGroup.Nodes.Clear();
            }

            // 目玉アイコンの更新
            _pnlState.Invalidate();

            if( !_multiSelectTreeNodeMgr.Updating )
            {
                _multiSelectTreeNodeMgr.UpdateSelectedNodeByNodeContent();
            }
        }

        /// <summary>
        /// グループを選択状態に設定します。
        /// </summary>
        void SetGroupSelected_( SubSceneManipulator subSceneMnp, object target )
        {
            // target がペインだったら選択します。
            if( target is IPane )
            {
                subSceneMnp.SelectPanesByPaneRef( target as IPane );
            }
            else
            {
                ILEGroup group = target as ILEGroup;
                IPane[] members = group.Member;
                foreach( IPane memberPane in members )
                {
                    subSceneMnp.SelectPanesByPaneRef( memberPane );
                }
            }
        }


        #region イベントハンドラ

        /// <summary>
        /// アクティベート
        /// </summary>
        private void GroupWindow_Activated( object sender, EventArgs e )
        {
            // アクティブになる時にメニューの状態を更新します。
            // 逐一更新しなくても、ショートカット動作で問題は起こりません。
            UpDataMenuItemEnableState_();
        }

        #region マウスダウン
        /// <summary>
        /// コンテキストメニューの表示を処理します。
        /// クリック位置から、メニュー内容を変更します。
        /// </summary>
        void OnContextMenuShow_( object sender, System.Windows.Forms.MouseEventArgs e )
        {
            // メニュー項目の状態を更新します。
            UpDataMenuItemEnableState_();

            // コンテキストメニューにメニューを複製
            ToolStripItem[] itemSet = new ToolStripItem[_tmiGroup.DropDownItems.Count];
            _tmiGroup.DropDownItems.CopyTo( itemSet, 0 );

            ContextMenuStrip cm =
                ToolStripMenuItemHelper.MakeContextMenuStripFromToolStripMenuItem( itemSet );




            Control ctrl = sender as Control;
            Debug.Assert( ctrl != null );

            // メッセージ送り主の子供として、表示します。
            Point pntMouse = new Point( e.X, e.Y );
            cm.Show( ctrl, pntMouse );
        }

        /// <summary>
        /// 指定グループのパラメータ編集Dlgを表示します。
        /// </summary>
        void ShowEditGroupParamatersDlg_( object target )
        {
            // targetには、グループが指定されていること。
            Debug.Assert( _GroupOnlySelected );

            // 選択アイテムがグループならグループ編集ダイアログを表示する。
            GroupEditDlg groupEditDlg = new GroupEditDlg();
            groupEditDlg.TargetGroup = target as ILEGroup;

            groupEditDlg.ShowDialog();
        }

        /// <summary>
        /// マウスダウン
        /// </summary>
        private void Event_TvwGroup_MouseDown( object sender, System.Windows.Forms.MouseEventArgs e )
        {
            Point pntClient = new Point( e.X, e.Y );
            TreeNode node = _tvwGroup.GetNodeAt( pntClient );

            if( e.Clicks == 2 )
            {
                // ダブルクリック
                if( node != null )
                {
                    // 選択アイテムがペインならプロパティを表示するメッセージを送る。
                    if( !_GroupOnlySelected )
                    {
                        DoSendMessageToViewManager(ViewManagerMessage.ShowPropertyWndMsg);
                    }
                }
            }
        }

        /// <summary>
        /// マウスアップ
        /// </summary>
        private void Event_TvwGroup_MouseUp( object sender, MouseEventArgs e )
        {
            if( e.Button == MouseButtons.Right )
            {
                // コンテキストメニューの表示をします。
                OnContextMenuShow_( sender, e );
            }
        }

        #endregion マウスダウン

        #region ラベル編集
        /// <summary>
        /// ノード編集前
        /// </summary>
        private void Event_TvwHierarchy_BeforeLabelEdit(object sender, System.Windows.Forms.NodeLabelEditEventArgs e)
        {
            IPane tagetPane = e.Node.Tag as IPane;
            if (tagetPane != null && tagetPane.IsReadOnlyLocked)
            {
                e.CancelEdit = true;
                return;
            }

            ILEGroup tagetGroup = e.Node.Tag as ILEGroup;
            if (tagetGroup == null || (tagetGroup.IsReadOnlyLocked || tagetGroup.IsRootGroup()))
            {
                e.CancelEdit = true;
                return;
            }
        }

        /// <summary>
        /// ラベル編集
        /// </summary>
        private void Event_TvwGroup_AfterLabelEdit(object sender, System.Windows.Forms.NodeLabelEditEventArgs e)
        {
            ISubScene subScene = LECore.LayoutEditorCore.Scene.CurrentISubScene;

            if (subScene == null)
            {
                return;
            }

            e.CancelEdit = true;

            if (e.Label != null)
            {
                if (e.Node.Tag is ILEGroup)
                {
                    if (GroupHelper.CheckGroupNameValid(e.Label))
                    {
                        string uniqueName = GroupHelper.GetUniqueGroupName(subScene.ILEGroupMgr, e.Label);

                        // グループの場合
                        if (_ILEGroupMgr.FindGroupByName(uniqueName) == null)
                        {
                            _CurrentISubScene.BeginMassiveModify();
                            GroupManipulator groupMnp = new GroupManipulator();
                            groupMnp.BindTarget(e.Node.Tag as ILEGroup);
                            groupMnp.Name = uniqueName;
                            _CurrentISubScene.EndMassiveModify();
                        }
                    }
                }
                else if (e.Node.Tag is IPane)
                {
                    // ペインの場合
                    IPane pane = e.Node.Tag as IPane;

                    if (pane != subScene.RootIPane)
                    {
                        string uniqueName = subScene.GetUniqueNewPaneName(pane, pane.PaneName, e.Label);
                        if (PaneHelper.CheckPaneNameValid(uniqueName))
                        {
                            PaneManipulator paneMnp = new PaneManipulator();
                            paneMnp.BindTarget(pane);
                            paneMnp.PaneName = uniqueName;
                        }
                    }
                }
            }
        }

        /// <summary>
        /// キーダウン
        /// </summary>
        private void Event_TvwGroup_KeyDown( object sender, System.Windows.Forms.KeyEventArgs e )
        {
            // F2 でラベル編集
            switch( e.KeyData )
            {
                case Keys.F2:
                {
                    if( !_multiSelectTreeNodeMgr.Empty )
                    {
                        TreeNode treeNode = _multiSelectTreeNodeMgr.FirstSelectedNode;
                        treeNode.BeginEdit();
                    }
                    break;
                }
            }
        }
        #endregion ラベル編集

        #region キー処理
        /// <summary>
        /// コマンドキーの処理
        /// </summary>
        protected override bool ProcessCmdKey( ref Message msg, Keys keyData )
        {
            // ラベル編集中のノードがあれば、固有の処理(削除など）を実行しません。
            foreach( TreeNode node in _multiSelectTreeNodeMgr.SelectedNodeSet )
            {
                if( node.IsEditing )
                {
                    return false;
                }
            }

            if (_shortcutHandler.ProcessCmdKey(keyData))
            {
                return true;
            }

            switch (keyData)
            {
                case Keys.Control | Keys.A:
                    // メッセージを伝播します。
                    if (base.NotifyCmdKeyMessageToOwner(ref msg, keyData))
                    {
                        return true;
                    }
                    break;
            }


            return base.ProcessCmdKey( ref msg, keyData );
        }
        #endregion キー処理

        #endregion イベントハンドラ

        #region メニュー関連

        /// <summary>
        /// DropDownOpening
        /// </summary>
        private void Event_TmiGroup_DropDownOpening( object sender, EventArgs e )
        {
            UpDataMenuItemEnableState_();
        }

        /// <summary>
        /// メニュー項目の状態を更新します。
        /// </summary>
        void UpDataMenuItemEnableState_()
        {
            TreeNode[] selectedNodeSet = _multiSelectTreeNodeMgr.SelectedNodeSet;
            _tmiGroupDelete.Enabled = false;
            _tmiDelete.Enabled = selectedNodeSet.Length > 0;
            _tmiGroupEdit.Enabled = false;
            _tmiShowSelectedEnable.Checked = _showSelectedEnable;
            _tmiPaste.Enabled = _PatseSrcEnalbe;

            _tmiSetGroupNotHidden.Enabled = false;
            _tmiSetGroupHidden.Enabled = false;
            _tmiSetGroupLocked.Enabled = false;
            _tmiSetGroupUnLocked.Enabled = false;

            // コピー操作は、ペインのみ選択時に有効
            bool bCopyEnabled = _PaneOnlySelected;
            _tmiCopy.Enabled = bCopyEnabled;
            _tmiCut.Enabled = bCopyEnabled;

            if( !_multiSelectTreeNodeMgr.Empty && _CurrentISubScene != null )
            {
                if( _ILEGroupMgr != null )
                {
                    // 選択ノードの状態から、有効メニューを設定します。
                    foreach( TreeNode node in selectedNodeSet )
                    {
                        if( node.Tag is ILEGroup )
                        {
                            _tmiSetGroupNotHidden.Enabled = true;
                            _tmiSetGroupHidden.Enabled = true;
                            _tmiSetGroupLocked.Enabled = true;
                            _tmiSetGroupUnLocked.Enabled = true;

                            break;
                        }
                        else
                        {
                            IPane pane = node.Tag as IPane;
                            if( pane != null )
                            {
                                _tmiSetGroupHidden.Enabled |= !pane.Hidden;
                                _tmiSetGroupNotHidden.Enabled |= pane.Hidden;
                                _tmiSetGroupLocked.Enabled |= !pane.Locked;
                                _tmiSetGroupUnLocked.Enabled |= pane.Locked;
                            }
                        }
                    }

                    // 削除と編集
                    foreach (TreeNode node in selectedNodeSet)
                    {
                        if (node.Tag is ILEGroup)
                        {
                            if (!(node.Tag as ILEGroup).IsReadOnlyLocked)
                            {
                                _tmiGroupDelete.Enabled |= true;
                                _tmiGroupEdit.Enabled |= true;
                            }
                            else
                            {
                                // 一つでもReadonlyなら無効。
                                _tmiGroupDelete.Enabled = false;
                                _tmiGroupEdit.Enabled = false;
                                _tmiDelete.Enabled = false;
                                _tmiCut.Enabled = false;
                                _tmiPaste.Enabled = false;

                                break;
                            }
                        }
                        else
                        {
                            // 一つでもReadonlyなら無効。
                            IPane pane = node.Tag as IPane;
                            ILEGroup parentGroup = node.Parent != null ? node.Parent.Tag as ILEGroup : null;
                            if (parentGroup == null)
                            {
                                continue;
                            }

                            if (parentGroup.IsReadOnlyLocked)
                            {
                                _tmiGroupDelete.Enabled = false;
                                _tmiGroupEdit.Enabled = false;
                                _tmiDelete.Enabled = false;
                                _tmiCut.Enabled = false;
                                _tmiPaste.Enabled = false;

                                break;
                            }
                            else
                            {
                                _tmiGroupDelete.Enabled |= true;
                            }
                        }
                    }
                }
            }
        }

        /// <summary>
        /// メニューポップアップ
        /// </summary>
        private void Event_MniGroup_Popup( object sender, System.EventArgs e )
        {
            UpDataMenuItemEnableState_();
        }



        /// <summary>
        /// グループの編集
        /// </summary>
        private void Event_MniGroupEdit_Click( object sender, System.EventArgs e )
        {
            if( _GroupOnlySelected )
            {
                ShowEditGroupParamatersDlg_( _multiSelectTreeNodeMgr.FirstSelectedNode.Tag );
            }
        }

        #region 選択グループメンバを隠す

        /// <summary>
        /// グループメンバの隠す状態を設定します。
        /// </summary>
        void SetGroupMemberHidden_( object targetGroup, bool bHidden )
        {
            if( targetGroup == null )
            {
                return;
            }

            PaneManipulator paneMnp = new PaneManipulator();
            if( targetGroup is ILEGroup )
            {
                IPane[] memberSet = ( targetGroup as ILEGroup ).Member;
                foreach( IPane memberPane in memberSet )
                {
                    paneMnp.BindTarget( memberPane );
                    paneMnp.Hidden = bHidden;
                }
            }
            else
            {
                paneMnp.BindTarget( targetGroup as IPane );
                paneMnp.Hidden = bHidden;
            }
        }

        /// <summary>
        /// 選択グループを隠す
        /// </summary>
        private void Event_MniGroupSetHidden_Click( object sender, System.EventArgs e )
        {
            _CurrentISubScene.BeginMassiveModify();

            foreach( TreeNode node in _multiSelectTreeNodeMgr.SelectedNodeSet )
            {
                SetGroupMemberHidden_( node.Tag, true );
            }

            _CurrentISubScene.EndMassiveModify();
        }

        /// <summary>
        /// 選択グループをあらわす
        /// </summary>
        private void Event_MniGroupSetNotHidden_Click( object sender, System.EventArgs e )
        {
            _CurrentISubScene.BeginMassiveModify();

            foreach( TreeNode node in _multiSelectTreeNodeMgr.SelectedNodeSet )
            {
                SetGroupMemberHidden_( node.Tag, false );
            }

            _CurrentISubScene.EndMassiveModify();
        }

        #endregion 選択グループメンバを隠す

        #region 選択グループメンバをロック

        /// <summary>
        /// グループメンバのロック状態を設定します。
        /// </summary>
        void SetGroupMemberLocked_( object targetGroup, bool bLocked )
        {
            if( targetGroup == null )
            {
                return;
            }


            PaneManipulator paneMnp = new PaneManipulator();
            if( targetGroup is ILEGroup )
            {
                IPane[] memberSet = ( targetGroup as ILEGroup ).Member;
                foreach( IPane memberPane in memberSet )
                {
                    // グループのメンバは、必ずペインである。
                    paneMnp.BindTarget( memberPane );

                    paneMnp.Locked = bLocked;
                }
            }
            else
            {
                paneMnp.BindTarget( targetGroup as IPane );
                paneMnp.Locked = bLocked;
            }
        }

        /// <summary>
        /// 選択グループをロック
        /// </summary>
        private void Event_MniGroupSetLock_Click( object sender, System.EventArgs e )
        {
            _CurrentISubScene.BeginMassiveModify();

            foreach( TreeNode node in _multiSelectTreeNodeMgr.SelectedNodeSet )
            {
                SetGroupMemberLocked_( node.Tag, true );
            }

            _CurrentISubScene.EndMassiveModify();
        }

        /// <summary>
        /// 選択グループをアンロック
        /// </summary>
        private void Event_MniGroupSetUnlock_Click( object sender, System.EventArgs e )
        {
            _CurrentISubScene.BeginMassiveModify();

            foreach( TreeNode node in _multiSelectTreeNodeMgr.SelectedNodeSet )
            {
                SetGroupMemberLocked_( node.Tag, false );
            }

            _CurrentISubScene.EndMassiveModify();
        }

        #endregion 選択グループメンバをロック

        #region コピー・貼り付け
        /// <summary>
        ///
        /// </summary>
        private void Event_MniShowSelectedEnable_Click( object sender, EventArgs e )
        {
            ToolStripMenuItem mi = sender as ToolStripMenuItem;
            mi.Checked = !mi.Checked;

            _showSelectedEnable = mi.Checked;
        }


        /// <summary>
        /// 切り取り
        /// </summary>
        private void Event_MniCut_Click( object sender, EventArgs e )
        {
            if( CopyToClipBoard_() )
            {
                // コピーに成功したら、削除
                Event_MniGroupDelete_Click( sender, e );
            }
        }

        /// <summary>
        /// 選択アイテムをクリップボードにコピーします。
        /// </summary>
        bool CopyToClipBoard_()
        {
            GroupWindowClipboardPaneSet paneSet = new GroupWindowClipboardPaneSet();

            // すべての選択ノードについて
            // ペインへの参照を記憶します。
            TreeNode[] selectedNodeSet = _multiSelectTreeNodeMgr.SelectedNodeSet;
            foreach( TreeNode selectedNode in selectedNodeSet )
            {
                // 選択ノードがペインなら、収集。
                if( !( selectedNode.Tag is ILEGroup ) )
                {
                    if( selectedNode.Tag is IPane )
                    {
                        paneSet.Add( selectedNode.Tag as IPane );
                    }
                }
                else
                {
                    // ペイン以外が発見されたら。コピー処理を中断します。
                    return false;
                }
            }

            // クリップボードへ登録。
            _ILEClipboard.Copy( paneSet );
            return true;
        }

        /// <summary>
        /// コピー
        /// </summary>
        private void Event_MniCopy_Click( object sender, EventArgs e )
        {
            CopyToClipBoard_();
        }

        /// <summary>
        /// 貼り付け
        /// </summary>
        private void Event_MniPaste_Click( object sender, EventArgs e )
        {
            GroupWindowClipboardPaneSet paneSet =
                _ILEClipboard.Paste( typeof( GroupWindowClipboardPaneSet ) ) as GroupWindowClipboardPaneSet;

            // Clipboard に貼り付け元が存在しないければ中断。
            if( paneSet == null )
            {
                return;
            }

            //--------------------------------------------------
            // 選択ノードから貼り付けるグループを列挙します。
            TreeNode[] selectedNodeSet = _multiSelectTreeNodeMgr.SelectedNodeSet;
            List<ILEGroup> groupSet = new List<ILEGroup>();
            foreach( TreeNode selectedNode in selectedNodeSet )
            {
                if( selectedNode.Tag is ILEGroup )
                {
                    // ノードが読み取り専用ではないグループであるなら登録
                    ILEGroup group = selectedNode.Tag as ILEGroup;
                    if( !group.IsReadOnlyLocked && !groupSet.Contains( group ) )
                    {
                        groupSet.Add( group );
                    }
                }
                else
                {
                    // ノードがペインであるなら、ノードの親をたどって所属グループを取得して登録
                    if( selectedNode.Tag is IPane &&
                        selectedNode.Parent != null &&
                        selectedNode.Parent.Tag is ILEGroup )
                    {
                        ILEGroup group = selectedNode.Parent.Tag as ILEGroup;
                        if (!group.IsReadOnlyLocked && !groupSet.Contains(group))
                        {
                            groupSet.Add( group );
                        }

                    }
                }
            }

            // 有効な対象がなかった。
            if( groupSet.Count <= 0 )
            {
                return;
            }

            //----------------------------------------------
            // 貼り付けます。
            GroupMgrManipulator groupMgrMnp = new GroupMgrManipulator();
            groupMgrMnp.BindTarget( _ILEGroupMgr );

            // 編集開始
            groupMgrMnp.BeginEditGroup();

            // すべてのコピー先グループについて、
            foreach( ILEGroup pasteTargetGroup in groupSet )
            {
                // すべてのペインを
                foreach( IPane pane in paneSet )
                {
                    // 有効なら貼り付けます。
                    groupMgrMnp.AddMember( pasteTargetGroup, pane );
                }
            }

            // 編集終了
            groupMgrMnp.EndEditGroup();
        }

        /// <summary>
        /// Delete
        /// </summary>
        private void Event_MniGroupDelete_Click( object sender, System.EventArgs e )
        {
            // マネージャが存在しないときは、処理しません。
            if( _ILEGroupMgr != null )
            {
                if( !_multiSelectTreeNodeMgr.Empty )
                {
                    GroupMgrManipulator groupMgrMnp = new GroupMgrManipulator();
                    groupMgrMnp.BindTarget( _ILEGroupMgr );
                    groupMgrMnp.BeginEditGroup();

                    foreach( TreeNode node in _multiSelectTreeNodeMgr.SelectedNodeSet )
                    {
                        if( node.Tag is IPane )
                        {
                            Debug.Assert( node.Parent != null && node.Parent.Tag is ILEGroup );
                            if ((node.Parent.Tag as ILEGroup).IsReadOnlyLocked)
                            {
                                continue;
                            }

                            groupMgrMnp.RemoveMember( ( node.Parent.Tag as ILEGroup ), node.Tag as IPane );
                        }
                    }

                    foreach( TreeNode node in _multiSelectTreeNodeMgr.SelectedNodeSet )
                    {
                        if( node.Tag is ILEGroup )
                        {
                            if ((node.Tag as ILEGroup).IsReadOnlyLocked)
                            {
                                continue;
                            }

                            groupMgrMnp.RemoveGroup( node.Tag as ILEGroup );
                        }
                    }

                    // 区間タグのグループ登録を整理
                    AnimFrameSectionSetHelper.RemoveInvalidTargetGroupAll(_CurrentISubScene.IAnimFrameSectionSet, _ILEGroupMgr);

                    groupMgrMnp.EndEditGroup();
                }
            }
        }

        #endregion コピー・貼り付け

        /// <summary>
        /// すべて展開
        /// </summary>
        private void Event_tmiExpandAll_Click(object sender, EventArgs e)
        {
            _tvwGroup.BeginUpdate();
            _tvwGroup.ExpandAll();
            _tvwGroup.EndUpdate();
        }

        /// <summary>
        /// すべて閉じる
        /// </summary>
        private void Event_tmiCollopseAll_Click(object sender, EventArgs e)
        {
            _tvwGroup.BeginUpdate();
            _tvwGroup.CollapseAll();
            if (_tvwGroup.Nodes.Count > 0)
            {
                _tvwGroup.Nodes[0].Expand();
            }
            _tvwGroup.EndUpdate();
        }

        #endregion メニュー関連


        #region ILEToolWindow メンバ


        /// <summary>
        /// ツリーノードの表示状態を更新します。
        /// </summary>
        void UpdateTreeNode_( TreeNode node, bool bNodeSelected )
        {
            IPane updatedPane = node.Tag as IPane;
            if( updatedPane != null )
            {
                int oldImageIndex = node.ImageIndex;

                bool bActive = !updatedPane.Locked && !updatedPane.Hidden;

                node.ForeColor = !bNodeSelected && updatedPane.IsReadOnlyLocked ? LECore.AppConstants.InactiveForeColor : PaneTreeViewHelper.GetPaneNodeForeColor(bActive, bNodeSelected);
                node.BackColor = !bNodeSelected && updatedPane.IsReadOnlyLocked ? LECore.AppConstants.InactiveBackColor : PaneTreeViewHelper.GetPaneNodeBackColor(bNodeSelected);

                node.Text = PaneTreeViewHelper.GetPaneNodeText( updatedPane );
                node.ImageIndex = GetPaneItemImageIndex_( updatedPane );
                node.SelectedImageIndex = node.ImageIndex;

                // 文字を太字にする
                node.NodeFont = PaneTreeViewHelper.GetPaneNodeFont( bActive & bNodeSelected );
                node.Text = node.Text;

                // 選択されていたら、表示する
                if( !_multiSelectTreeNodeMgr.Updating )
                {
                    if( this.Visible & _showSelectedEnable & updatedPane.IsSelected )
                    {
                        node.EnsureVisible();
                    }
                }

                // ペインの画像が変わったら親グループの画像も更新する
                if (oldImageIndex != node.ImageIndex && node.Parent != null)
                {
                    UpdateTreeNode_(node.Parent, false);
                }
            }
            else if( node.Tag is ILEGroup )
            {
                ILEGroup updatedGroup = (ILEGroup)node.Tag;
                node.ForeColor = !bNodeSelected && updatedGroup.IsReadOnlyLocked ? LECore.AppConstants.InactiveForeColor : PaneTreeViewHelper.GetPaneNodeForeColor(true, bNodeSelected);
                node.BackColor = !bNodeSelected && updatedGroup.IsReadOnlyLocked ? LECore.AppConstants.InactiveBackColor : PaneTreeViewHelper.GetPaneNodeBackColor(bNodeSelected);

                node.Text = updatedGroup.GrouprName;
                node.ImageIndex = GetGroupItemImageIndex_( updatedGroup );

                // 子供ノードを再帰的に選択することはありません。
            }
        }

        /// <summary>
        /// シーン更新イベントハンドラ(ペイン)
        /// </summary>
        void OnSceneModifyHandler_PaneModify_( SceneModifyEventArgs e )
        {
            // グループウインドウの更新。
            List<TreeNode> nodeSet = null;

            bool bNodeUpdated = false;
            foreach( IPane changedPane in e.Paramaters )
            {
                List<TreeNode> changedNodeSet =
                    _treeNodeFactory.FindUpdatedNodeByContents( changedPane );

                if( changedPane.IsSelected )
                {
                    if( nodeSet == null )
                    {
                        nodeSet = new List<TreeNode>();
                    }
                    nodeSet.AddRange( changedNodeSet );
                }

                foreach( TreeNode node in changedNodeSet )
                {
                    if (!bNodeUpdated)
                    {
                        _tvwGroup.BeginUpdate();
                    }

                    UpdateTreeNode_( node, changedPane.IsSelected );
                    bNodeUpdated |= true;
                }
            }

            if( nodeSet != null )
            {
                _multiSelectTreeNodeMgr.AddSelectedNode( nodeSet );
            }

            if( bNodeUpdated )
            {
                _tvwGroup.EndUpdate();
                _pnlState.Invalidate();
            }
        }

        /// <summary>
        /// シーン更新イベントハンドラ(グループ)
        /// </summary>
        void OnSceneModifyHandler_Group_( SceneModifyEventArgs e )
        {
            ILEGroup group = null;
            if( e.Paramaters != null && e.Paramaters.Count > 0 )
            {
                group = e.Paramaters[0] as ILEGroup;
            }

            foreach( TreeNode node in _treeNodeFactory.FindUpdatedNodeByContents( group ) )
            {
                UpdateTreeNode_( node, false );
            }
        }

        /// <summary>
        /// シーン更新イベントハンドラ(選択セット)
        /// </summary>
        void OnSceneModifyHandler_SelectedSet_( SceneModifyEventArgs e )
        {
            IPane[] selSet = e.Paramaters[0] as IPane[];
            Debug.Assert( selSet != null );

            // 選択セットを更新します。
            if( !_multiSelectTreeNodeMgr.Updating )
            {
                _multiSelectTreeNodeMgr.UpdateSelectedNodeByNodeContent();
            }
        }

        /// <summary>
        /// シーン更新イベントハンドラ
        /// </summary>
        public override void OnSceneModifyHandler(
            object sender,
            SceneModifyEventArgs e )
        {
            switch( e.Target )
            {
                // ペイン更新
                case SceneModifyEventArgs.Kind.PaneModify:
                {
                    OnSceneModifyHandler_PaneModify_( e );
                    break;
                }
                // ペインの削除
                case SceneModifyEventArgs.Kind.PaneRemove:
                {
                    _pnlState.Invalidate();
                    break;
                }
                // グループの更新
                case SceneModifyEventArgs.Kind.GroupModify:
                {
                    OnSceneModifyHandler_Group_( e );
                    break;
                }
                case SceneModifyEventArgs.Kind.GroupMake:
                case SceneModifyEventArgs.Kind.GroupRemove:
                case SceneModifyEventArgs.Kind.GroupReorder:
                case SceneModifyEventArgs.Kind.GroupAddPane:
                case SceneModifyEventArgs.Kind.GroupRemovePane:
                case SceneModifyEventArgs.Kind.GroupReorderPane:
                {
                    InitializeDlgState();
                    break;
                }
                // カレントサブシーンの更新
                case SceneModifyEventArgs.Kind.CurrentSubSceneChanged:
                {
                    // 親子階層ビューの更新
                    // サブシーン固有のデータを表示しているViewは更新する必要があります。
                    // InitializeDlgState();
                    InitializeDlgState();
                    break;
                }
                case SceneModifyEventArgs.Kind.SubSceneRemoved:
                {
                    _treeNodeStateHolder.RemoveUnnecessaryNodeState();
                    break;
                }
                // 選択セット変更ハンドラ
                case SceneModifyEventArgs.Kind.SelectedSetModify:
                {
                    OnSceneModifyHandler_SelectedSet_( e );
                    break;
                }
                default:
                break;
            }
            base.OnSceneModifyHandler( sender, e );
        }

        #region データ保存
        /// <summary>
        /// 保存(カーブエディタ固有の情報を保存します。)
        /// </summary>
        public override void SaveSetting(LEToolFormSetting setting, SaveSettingOption option)
        {
            if (option.AlsoSaveOtherThanWorkspace)
            {
                LEControlUserDataChunk data = null;
                data = new LEControlUserDataChunk("ShowSelectNodeEnable", this._showSelectedEnable.ToString());
                setting.UserDataSet.Add(data);
            }

            base.SaveSetting(setting, option);
        }

        /// <summary>
        /// 読み込み
        /// </summary
        public override void LoadSetting(LEToolFormSetting setting, LoadSettingOption option)
        {
            base.LoadSetting(setting, option);

            if (option.AlsoLoadOtherThanWorkspace)
            {
                LEControlUserDataChunk data = null;
                data = setting.FindUserDataByName("ShowSelectNodeEnable");
                if (data != null)
                {
                    _showSelectedEnable = bool.Parse(data.Value);
                }
            }
        }

        #endregion データ保存

        public override Keys CustomShortcut { get { return Keys.F9; } }


        #endregion ILEToolWindow メンバ













    }
}
