﻿// --------------------------------------------------------------------------------
// <copyright>
// Copyright (C)Nintendo All rights reserved.
//
// These coded instructions, statements, and computer programs contain proprietary
// information of Nintendo and/or its licensed developers and are protected by
// national and international copyright laws. They may not be disclosed to third
// parties or copied or duplicated in any form, in whole or in part, without the
// prior written consent of Nintendo.
//
// The content herein is highly confidential and should be handled accordingly.
// </copyright>
// --------------------------------------------------------------------------------

using System;
using System.Collections;
using System.Drawing;
using System.Drawing.Drawing2D;
using System.Diagnostics;

namespace LayoutEditor.Forms.ToolWindows.CurveEditWindow
{
    using LECore.Structures;
    using LECore.Structures.Core;

    using LECore.Util;
    using LECore.Manipulator;
    using LayoutEditor.Forms.ToolWindows.common;


    /// <summary>
    /// アンカー調整中のキー
    /// </summary>
    class AnchorTweakedKey : IDragTarget
    {
        public enum TargetTangent
        {
            InTangent,
            OutTangent
        };

        readonly IAnmKeyFrame _keyFrame = null;
        readonly AnmKeyFrameManipulator _keyFrameMnp = new AnmKeyFrameManipulator();
        readonly float _defaultTangent = 0.0f;
        readonly TargetTangent _targetTangent = TargetTangent.InTangent;

        #region ----------------- プロパティ -----------------

        /// <summary>
        /// キー位置をPointFで取得します(グラフ座標系)
        /// </summary>
        public PointF AsPointF
        {
            get { return new PointF( _keyFrame.Time, _keyFrame.ValueAsFloat ); }
        }

        /// <summary>
        /// IAnmKeyFrame としての実体を取得します。
        /// </summary>
        public IAnmKeyFrame IAnmKeyFrame
        {
            get { return _keyFrame; }
        }

        /// <summary>
        /// 操作対象接線
        /// </summary>
        public float Tangent
        {
            get
            {
                if( _targetTangent == TargetTangent.InTangent )
                {
                    return _keyFrame.InTangent;
                }
                else
                {
                    return _keyFrame.OutTangent;
                }
            }

            set
            {
                // Undoコマンドを生成せずに値を変更します。
                _keyFrameMnp.BindTarget( _keyFrame );
                if( _targetTangent == TargetTangent.InTangent )
                {
                    _keyFrameMnp.InTangentWithoutCommand = value;
                }
                else
                {
                    _keyFrameMnp.OutTangentWithoutCommand = value;
                }
            }

        }

        /// <summary>
        /// 操作対象種類
        /// </summary>
        public TargetTangent TargetTangentKind
        {
            get { return _targetTangent; }
        }

        /// <summary>
        /// ドラッグ前のタンジェント
        /// </summary>
        public float InitialTangent
        {
            get { return _defaultTangent; }
        }

        #endregion // ----------------- プロパティ -----------------

        /// <summary>
        /// AnchorTweakedKey を作成します。
        /// </summary>
        static public AnchorTweakedKey[] FromKeyAanchor( KeyAanchor[] keyAnchorSet )
        {
            AnchorTweakedKey[] targetTangentSet = new AnchorTweakedKey[keyAnchorSet.Length];
            for( int i = 0 ; i < targetTangentSet.Length ; i++ )
            {
                AnchorTweakedKey.TargetTangent tangentType = keyAnchorSet[i].bLeft ?
                    AnchorTweakedKey.TargetTangent.InTangent : AnchorTweakedKey.TargetTangent.OutTangent;

                targetTangentSet[i] = new AnchorTweakedKey( keyAnchorSet[i].IAnmKeyFrame, PointF.Empty, tangentType );
            }
            return targetTangentSet;
        }

        /// <summary>
        ///  コンストラクタ
        ///
        ///  NOTICE:
        ///  キーの位置をコンストラクタで指定しているため、
        ///  選択後ビューを操作すると、不具合が起こるバグが確認されています。
        /// </summary>
        /// <param name="kfi"></param>
        /// <param name="pntKeyInScreen">キーの位置（スクリーン座標系)</param>
        public AnchorTweakedKey(
            IAnmKeyFrame keyFrame,
            PointF pntKeyInScreen,
            TargetTangent targetTangent )
        {

            _keyFrame = keyFrame;
            _keyFrameMnp.BindTarget( keyFrame );

            _targetTangent = targetTangent;

            _defaultTangent = this.Tangent;
        }

        /// <summary>
        /// 傾き情報を編集します。
        /// Undoシステムに反映されます。
        /// </summary>
        public void FixRotAngle()
        {
            // 一旦、デフォルトに戻し...
            float currentTangent = Tangent;
            Tangent = _defaultTangent;


            // Undo の適用される変更操作で...
            if( _targetTangent == TargetTangent.InTangent )
            {
                _keyFrameMnp.InTangent = currentTangent;
            }
            else
            {
                _keyFrameMnp.OutTangent = currentTangent;
            }
        }

        #region IDragTarget メンバ

        public float X
        {
            get { return AsPointF.X; }
        }

        public float Y
        {
            get { return AsPointF.Y; }
        }

        public float Width
        {
            get { return 0.0f; }
        }

        public float Height
        {
            get { return 0.0f; }
        }

        public RectangleF BoundingRectangle
        {
            get { return new RectangleF( X, Y, Width, Height ); }
        }

        #endregion

        #region IDrawable メンバ

        public void Draw( IRenderer renderer, DrawableOption option )
        {
            // TODO：
            renderer.DrawPoint( AsPointF );
        }

        #endregion
    }
}
