﻿// --------------------------------------------------------------------------------
// <copyright>
// Copyright (C)Nintendo. All rights reserved.
//
// These coded instructions, statements, and computer programs contain proprietary
// information of Nintendo and/or its licensed developers and are protected by
// national and international copyright laws. They may not be disclosed to third
// parties or copied or duplicated in any form, in whole or in part, without the
// prior written consent of Nintendo.
//
// The content herein is highly confidential and should be handled accordingly.
// </copyright>
// --------------------------------------------------------------------------------
using System;
using System.Collections.Generic;
using System.Text;
using System.Drawing;
using System.Drawing.Drawing2D;
using System.Diagnostics;

namespace LayoutEditor.Forms.ToolWindows.CurveEditWindow
{
    using LayoutEditor.Forms.ToolWindows.common;

    using LECore.Structures;
    using LECore.Manipulator;
    using LECore.Structures.Core;

    using DbgConsole = LECore.DbgConsole;
    using MathUtil = LECore.Util.MathUtil;
    using LEMath = LECore.Structures.LEMath;
    using AxisSnap = LayoutEditor.Forms.ToolWindows.common.DragModifierHelper.AxisSnap;
    using static LayoutWindow.PaneDragTargetAdapter;
    using System.Linq;

    public enum GraphValueSnap
    {
        None = 0,
        _10,
        _1,
        _01,
        _001,
        CalcByDiff,
    }

    /// <summary>
    /// キーフレームドラッグ調整クラス
    /// </summary>
    class DragKeyModifier : DragModifier<TweakedKey>
    {
        //---------------------------------------------------------
        #region フィールド
        PointF						_screenScale		= PointF.Empty;
        GraphValueSnap              _valueSnap		    = GraphValueSnap.None;
        AnmCurveManipulator		_anmCurveMnp			= new AnmCurveManipulator();
        #endregion フィールド

        //---------------------------------------------------------
        #region プロパティ

        /// <summary>
        /// スクリーンスケール：マウスドラッグ量を
        /// 実際の操作量に変換する際に利用します。
        /// </summary>
        public PointF ScreenScale
        {
            set { _screenScale = value; }
        }

        /// <summary>
        ///
        /// </summary>
        ISubScene _CoreScene
        {
            get { return LECore.LayoutEditorCore.Scene.CurrentISubScene; }
        }

        /// <summary>
        /// GraphView
        /// </summary>
        public GraphView GraphView { get; set; }

        /// <summary>
        /// ドラッグ対象のキーを含むカーブ内のキーのドラッグ前の状態
        /// </summary>
        private Dictionary<IAnmKeyFrame, KeyState> KeyStatesBeforeDrag { get; set; }

        /// <summary>
        /// ドラッグ対象のキー
        /// </summary>
        private HashSet<IAnmKeyFrame> DraggingKeys { get; set; }

        private IEnumerable<IAnmCurve> OwnerCurves => TargetSet.Select(x => x.IAnmKeyFrame.OwnerIAnmCurve).Distinct();

        #endregion プロパティ

        /// <summary>
        /// コンストラクタ
        /// </summary>
        public DragKeyModifier()
        {
            this._valueSnap = GraphValueSnap.None;
        }

        /// <summary>
        ///
        /// </summary>
        public void SetGraphValueSnap(GraphValueSnap valueClamp)
        {
            _valueSnap = valueClamp;
        }

        /// <summary>
        /// 特定小数点以下を丸めながら更新します。
        /// </summary>
        public void UpdateDraggingByDifferenceWithRoundDown( PointF differencePos )
        {
            GraphValueSnap current = _valueSnap;
            _valueSnap = GraphValueSnap.CalcByDiff;
            UpdateDraggingByDifference( differencePos );
            _valueSnap = current;
        }

        /// <summary>
        /// スナップ単位の指定に使う、小数点位置のオフセット値
        /// </summary>
        private int GetDecimalOffset_(GraphValueSnap valueClamp)
        {
            switch (valueClamp)
            {
                case GraphValueSnap._10: return -1;
                case GraphValueSnap._1: return 0;
                case GraphValueSnap._01: return 1;
                case GraphValueSnap._001: return 2;
                default: return 0;
            }
        }

        #region キー調整ハンドラ


        public override void BeginDragging(TweakedKey[] paneSet, PointF posInScreen, DragModifierHelper.OptionFlag optionFlags)
        {
            base.BeginDragging(paneSet, posInScreen, optionFlags);

            DraggingKeys = new HashSet<IAnmKeyFrame>(TargetSet.Select(x => x.IAnmKeyFrame));
            KeyStatesBeforeDrag = OwnerCurves.SelectMany(x => x.IKeyFrameSet).ToDictionary(x => x, x => new KeyState(x));
        }

        private struct KeyState
        {
            public float Value { get; set; }
            public float Time { get; set; }
            public float InSlope { get; set; }
            public float OutSlope { get; set; }
            public KeyState(IAnmKeyFrame key)
            {
                Value = key.ValueAsFloat;
                Time = key.Time;
                InSlope = key.InTangent;
                OutSlope = key.OutTangent;
            }
        }

        /// <summary>
        /// ドラッグ更新
        /// </summary>
        public override void UpdateDragging( PointF posInScreen )
        {
            base.UpdateDragging( posInScreen );

            if( Empty != false )
            {
                return;
            }

            _CoreScene.BeginMassiveModify();

            ModifyKeysTemporarily();

            var keyManipulator = new AnmKeyFrameManipulator();

            //--------------------------------------------------
            // 持ち主カーブ内のキーの順番と傾きを更新します。
            foreach ( IAnmCurve ownerCurve in OwnerCurves )
            {
                _anmCurveMnp.BindTarget( ownerCurve );
                _anmCurveMnp.Update( false );
                for (int i = 0; i < ownerCurve.IKeyFrameSet.Length; i++)
                {
                    var modifySlope = (i - 1 >= 0 && DraggingKeys.Contains(ownerCurve.IKeyFrameSet[i - 1])) ||
                        DraggingKeys.Contains(ownerCurve.IKeyFrameSet[i]) ||
                        (i + 1 < ownerCurve.IKeyFrameSet.Length && DraggingKeys.Contains(ownerCurve.IKeyFrameSet[i + 1]));

                    keyManipulator.BindTarget(ownerCurve.IKeyFrameSet[i]);
                    if (modifySlope)
                    {
                        // 傾きを更新します。
                        keyManipulator.UpdateSlopeWithoutCommand();
                    }
                    else
                    {
                        // 傾きを元に戻します。
                        var beforeState = KeyStatesBeforeDrag[ownerCurve.IKeyFrameSet[i]];
                        keyManipulator.InTangentWithoutCommand = beforeState.InSlope;
                        keyManipulator.OutTangentWithoutCommand = beforeState.OutSlope;

                    }
                }
            }

            _CoreScene.EndMassiveModify();
        }

        protected virtual void ModifyKeysTemporarily()
        {
            //--------------------------------------------------
            // 更新量を計算します。
            PointF difference;
            CalcDifference_(out difference);

            difference = new PointF(
                difference.X * _screenScale.X,
                difference.Y * _screenScale.Y);

            SizeF vDiff = new SizeF(difference);

            //--------------------------------------------------
            // すべてのキーを更新します。
            foreach (TweakedKey kf in TargetSet)
            {
                // 一時的な変更を行います。
                float newValue = kf.DefaultValue + vDiff.Height;

                // 指定された場合、小数点以下を丸めます。
                if (_valueSnap != GraphValueSnap.None)
                {
                    int decimalVal = _valueSnap == GraphValueSnap.CalcByDiff ?
                        LEMath.CalcDecimals(vDiff.Height) : GetDecimalOffset_(_valueSnap);

                    newValue = LEMath.ToRoundDownDecimals(newValue, decimalVal);
                }

                // 四捨五入して近い整数値
                int diifTime = (int)Math.Round(vDiff.Width, MidpointRounding.AwayFromZero);
                kf.ModifyTemporary(kf.DefaultTime + diifTime, newValue);
            }
        }

        /// <summary>
        /// ドラッグ終了
        /// </summary>
        public override void EndDragging()
        {
            if( _CoreScene == null || Empty != false )
            {
                return;
            }

            _CoreScene.BeginMassiveModify();

            bool isValid = true;
            foreach (var curve in OwnerCurves)
            {
                // 移動すると重複するかどうかを確認します。
                if (curve.IKeyFrameSet.Zip(
                    curve.IKeyFrameSet.Skip(1),
                    (x,y) => x.Time == y.Time && (DraggingKeys.Contains(x) || DraggingKeys.Contains(y))).Any(x => x))
                {
                    isValid = false;
                }
            }

            var keyManipulator = new AnmKeyFrameManipulator();
            foreach (var curve in OwnerCurves)
            {
                foreach (var key in curve.IKeyFrameSet)
                {
                    keyManipulator.BindTarget(key);
                    var beforeState = KeyStatesBeforeDrag[key];
                    var afterState = new KeyState(key);

                    // 元の状態に戻します。
                    keyManipulator.TimeWithoutCommand = beforeState.Time;
                    keyManipulator.ValueAsFloatWithoutCommand = beforeState.Value;
                    keyManipulator.InTangentWithoutCommand = beforeState.InSlope;
                    keyManipulator.OutTangentWithoutCommand = beforeState.OutSlope;

                    if (isValid)
                    {
                        // 更新して Undo/Redo の対象にします。
                        keyManipulator.ModifyForce(afterState.Value, afterState.Time, afterState.InSlope, afterState.OutSlope);
                    }
                }

                //--------------------------------------------------
                // 持ち主カーブの再評価を行います。
                _anmCurveMnp.BindTarget(curve);
                _anmCurveMnp.Update(true);
            }

            _CoreScene.EndMassiveModify();

            // 不要な情報を消去します。
            DraggingKeys = null;
            KeyStatesBeforeDrag = null;

            base.EndDragging();
        }



        #endregion キー調整ハンドラ
    }

    /// <summary>
    /// アンカードラッグ調整クラス
    /// </summary>
    class DragAnchorModifier : DragModifier<AnchorTweakedKey>
    {
        float aspectRatio = 1.0f;
        AnmCurveManipulator _anmCurveMnp = new AnmCurveManipulator();
        PointF _anchorTweakPivotPoint = PointF.Empty;

        /// <summary>
        ///
        /// </summary>
        public float AspectRatio
        {
            set { aspectRatio = value; }
        }

        /// <summary>
        ///
        /// </summary>
        public PointF AnchorTweakPivotPoint
        {
            set { _anchorTweakPivotPoint = value; }
        }

        /// <summary>
        ///
        /// </summary>
        ISubScene _CoreScene
        {
            get { return LECore.LayoutEditorCore.Scene.CurrentISubScene; }
        }

        /// <summary>
        /// コンストラクタ
        /// </summary>
        public DragAnchorModifier()
        {
        }

        #region アンカー調整ハンドラ

        /// <summary>
        /// ドラッグ更新
        /// </summary>
        public override void UpdateDragging( PointF posInScreen )
        {
            base.UpdateDragging( posInScreen );

            // キーの位置、編集開始点、現在の点 ( いずれもスクリーン座標系 )
            PointF p0 = MathUtil.NormalizeVec( MathUtil.SubVec( this.DragStartPointF, _anchorTweakPivotPoint ) );
            PointF p1 = MathUtil.NormalizeVec( MathUtil.SubVec( this.DragCurrentPointF, _anchorTweakPivotPoint ) );

            // line_pk_p0 と line_pk_p1 の角度を計算します。
            foreach( AnchorTweakedKey anchorTweakedKey in TargetSet )
            {
                float defAng = MathUtil.RadToDeg( (float)Math.Atan( anchorTweakedKey.InitialTangent / aspectRatio ) );
                float rotAngle = MathUtil.CalcAngleBtweenLines2( p0, p1 );
                // LECore.DbgConsole.WriteLine( "rotAngle = {0}", rotAngle );
                rotAngle += defAng;

                // 値が制限を越えるようなら調整をします。
                rotAngle = MathUtil.Clamp( rotAngle, -90.0f, 90.0f );


                // 傾き値を変更します。
                float tangent = (float)Math.Tan( MathUtil.DegToRad( rotAngle ) ) * aspectRatio;
                tangent = MathUtil.Clamp( tangent, -500.0f, 500.0f );// 適当な値！

                // 値を変更します。
                anchorTweakedKey.Tangent = tangent;
            }
        }

        /// <summary>
        ///
        /// </summary>
        public override void EndDragging()
        {


            _CoreScene.BeginMassiveModify();

            //--------------------------------------------------
            List<IAnmCurve> ownerSet = new List<IAnmCurve>();
            foreach( AnchorTweakedKey atk in TargetSet )
            {
                atk.FixRotAngle();
                IAnmCurve owner = atk.IAnmKeyFrame.OwnerIAnmCurve;
                if( !ownerSet.Contains( owner ) )
                {
                    ownerSet.Add( owner );
                }
            }

            //--------------------------------------------------
            // 持ち主カーブの再評価を行います。
            foreach( IAnmCurve ownerCurve in ownerSet )
            {
                _anmCurveMnp.BindTarget( ownerCurve );
                _anmCurveMnp.Update( true );
            }

            _CoreScene.EndMassiveModify();

            base.EndDragging();
        }



        #endregion アンカー調整ハンドラ

    }

    /// <summary>
    /// キーフレームドラッグスケール調整クラス
    /// </summary>
    class DragKeyScaleModifier : DragKeyModifier
    {
        //----------------------------------------------------
        PointF _transformOrigin = PointF.Empty;



        //----------------------------------------------------
        /// <summary>
        ///
        /// </summary>
        ISubScene _CoreScene
        {
            get { return LECore.LayoutEditorCore.Scene.CurrentISubScene; }
        }


        //----------------------------------------------------
        /// <summary>
        /// コンストラクタ
        /// </summary>
        public DragKeyScaleModifier()
        {
        }

        /// <summary>
        /// 変換原点を設定します。
        /// </summary>
        public void SetTransformOrigin( PointF posMouse, PointF dotSize )
        {
            DragModifierHelper.Contact contact;
            DragModifierHelper.CalcTransformOrigin(posMouse, DefaultBoundingRect, dotSize, out contact);

            _transformOrigin = contact.TransOrigin;
            AxisSnapKind = contact.AxisSnap;
        }

        //----------------------------------------------------
        #region ドラッグイベントハンドラ

        protected override void ModifyKeysTemporarily()
        {
            PointF difference;
            CalcDifference_(out difference);

            PointD scale;
            if (CalcScale_(difference, _transformOrigin, out scale))
            {
                ModifyTarget_(TargetSet, _transformOrigin, (float)scale.X, (float)scale.Y);
            }
        }

        /// <summary>
        /// 対象を変更します。
        /// </summary>
        void ModifyTarget_(
            TweakedKey[] tagetSet,
            PointF transformOrigin,
            float scaleX,
            float scaleY )
        {
            //-----------------------------------
            // 移動量を加算します。
            foreach( TweakedKey key in tagetSet )
            {
                PointF pos = key.DefaultAsPointF;

                Matrix mT = new Matrix();
                Matrix mS = new Matrix();
                Matrix mIT = new Matrix();

                mT.Translate( -transformOrigin.X, -transformOrigin.Y );
                mS.Scale( scaleX, scaleY );
                mIT.Translate( transformOrigin.X, transformOrigin.Y );

                mIT.Multiply( mS );
                mIT.Multiply( mT );

                pos = MathUtil.MtxTransformPoint( mIT, pos );

                key.ModifyTemporary( (int)pos.X, pos.Y );
            }
        }
        #endregion ドラッグイベントハンドラ

        //----------------------------------------------------
        /// <summary>
        /// 描画します。
        /// </summary>
        public override void Draw( IRenderer renderer, DrawableOption option )
        {
            if( this.Empty )
            {
                return;
            }

            float lineW = renderer.LineWidth;
            renderer.LineWidth = 1.0f;

            //---------------------------------------------
            // 変形前矩形
            RectangleF defaultRect = DefaultBoundingRect;
            if( defaultRect != RectangleF.Empty )
            {
                Color origColor = renderer.Color;
                renderer.Color = Color.FromArgb( 127, renderer.Color );
                RendererHelper.DrawRectangleWithoutTransform( renderer, defaultRect );

                renderer.Color = origColor;
            }


            //---------------------------------------------
            // 倍率計算
            PointF difference;
            PointD scale;
            CalcDifference_( out difference );
            bool bScaleValid = CalcScale_( difference, _transformOrigin, out scale );
            if( bScaleValid )
            {
                //---------------------------------------------
                // 矩形
                renderer.PushMtx();
                PointF transformOrigin = _transformOrigin;

                renderer.Trans( transformOrigin.X, transformOrigin.Y );
                renderer.Scale( (float)scale.X, (float)scale.Y );
                renderer.Trans( -transformOrigin.X, -transformOrigin.Y );

                DragKeyScaleModifier.DrawAnchorRectangle( renderer, defaultRect, 6.0f );
                renderer.PopMtx();

                //---------------------------------------------
                // 倍率文字列
                string str = String.Format( "sX = {0:F4} sY = {1:F4}", scale.X, scale.Y );

                RendererHelper.DrawStringWithoutTransform(
                        renderer,
                        SystemFonts.DefaultFont,
                        Color.Black,
                        str,
                        defaultRect.X, defaultRect.Y );
            }
            else
            {
                DragKeyScaleModifier.DrawAnchorRectangle( renderer, defaultRect, 6.0f );
            }

            renderer.LineWidth = lineW;

            base.Draw( renderer, option );
        }
    }
}
