﻿// --------------------------------------------------------------------------------
// <copyright>
// Copyright (C)Nintendo. All rights reserved.
//
// These coded instructions, statements, and computer programs contain proprietary
// information of Nintendo and/or its licensed developers and are protected by
// national and international copyright laws. They may not be disclosed to third
// parties or copied or duplicated in any form, in whole or in part, without the
// prior written consent of Nintendo.
//
// The content herein is highly confidential and should be handled accordingly.
// </copyright>
// --------------------------------------------------------------------------------
using System;
using System.Collections.Generic;
using System.ComponentModel;
using System.Drawing;
using System.Data;
using System.Text;
using System.Windows.Forms;

namespace LayoutEditor.Forms.ToolWindows.AnimSectionTagWindow
{
    using LECore.Structures;
    using LECore.Structures.LECoreInterface;

    public partial class AnimSectionBasicPanel : UserControl
    {
        IAnimFrameSection       _target = null;
        const bool _AdjustEndFrame = true;
        int _textLengthXPos = 0;
        const int _textLengthXPosAddDialog = 5;
        ScaleType _scaleType = ScaleType.Scale_Off;
        bool _dialogDispMode = false;

        public event EventHandler OnChanged = null;

        public enum ScaleType
        {
            Scale_Off,
            Scale_Float,
            Scale_Int,
        }

       #region プロパティ

        public string TragetName { get { return _tbxName.Text; } set { _tbxName.SetValue( value ); }  }
        public string Comment     { get { return _tbxComment.Text; } }
        public int StartFrame     { get { return _tbxStartFrame.Value; }}
        public int EndFrame       { get { return _tbxEndFrame.Value; }}
        public bool ScaleLinkVisible { set { _pnlScale.Visible = value; } }
        public ScaleType ScaleLink
        {
            get
            {
                return _scaleType;
            }
        }
        public bool DialogDispMode
        {
            set
            {
                _dialogDispMode = value;
                if (_dialogDispMode)
                {
                    _pnlScale.Visible = false;
                    _tbxLength.Location = new Point(this.Width - (_tbxLength.Width + 10), _tbxLength.Location.Y);
                }
            }
        }

       #endregion プロパティ

        /// <summary>
        /// 表示対象を設定、取得します。
        /// </summary>
        public IAnimFrameSection Target
        {
            set
            {
                if( !AnimFrameSectionSetHelper.IsAnimFrameSectionSame( _target, value ) )
                {
                    _target = value;
                    UpdataProperty();
                }
            }
            get { return _target; }
        }


        /// <summary>
        /// 変更イベントを通知します。
        /// </summary>
        void NotifyChangeEvent_()
        {
            if( OnChanged != null )
            {
                OnChanged( this, null );
            }
        }

        /// <summary>
        /// コンストラクタ
        /// </summary>
        public AnimSectionBasicPanel()
        {
            InitializeComponent();

            _radScaleOff.Checked = true;
        }

        /// <summary>
        /// 更新
        /// </summary>
        public void UpdataProperty()
        {
            if( _target == null )
            {
                _tbxName.SetValue( string.Empty );
                _tbxComment.Text = string.Empty;
                _tbxStartFrame.SetValue( 0 );
                _tbxEndFrame.SetValue( 1 );
                _tbxLength.SetValue(1);

                this.Enabled = false;
            }
            else
            {
                if( _tbxName.Value != _target.Name )
                {
                    _tbxName.SetValue( _target.Name );
                }

                if( _tbxComment.Text != _target.Comment )
                {
                    _tbxComment.Text = _target.Comment;
                }

                UpdateFrameEditBox();
                if ( _tbxStartFrame.Value != _target.StartFrame )
                {
                    _tbxStartFrame.SetValue( _target.StartFrame );
                }

                _tbxEndFrame.Enabled = this._target.BinarySettings.LoopType != AnimationLoopType.Wiat;
                if( _tbxEndFrame.Value != _target.EndFrame )
                {
                    _tbxEndFrame.SetValue( _target.EndFrame );
                }

                _tbxLength.SetValue(_target.EndFrame - _target.StartFrame);
            }
        }

        /// <summary>
        /// 更新：フレームエディットボックス
        /// </summary>
        public void UpdateFrameEditBox()
        {
            bool visible = !(LECore.LayoutEditorCore.Scene.CurrentISubScene.IsAnimEditSeparateMode());

            _lblStartFrame.Visible = visible;
            _tbxStartFrame.Visible = visible;
            panel1.Visible = visible;
            _tbxEndFrame.Visible = visible;

            if (!_dialogDispMode)
            {
                _pnlScale.Visible = !visible;
                if (!visible)
                {
                    _tbxLength.Location = new Point(_pnlScale.Location.X - _tbxLength.Width, _tbxLength.Location.Y);
                }
                else
                {
                    _tbxLength.Location = new Point(this.Width - _textLengthXPos, _tbxLength.Location.Y);
                }
            }
        }

        /// <summary>
        /// タグ名：テキスト変更
        /// </summary>
        private void Event_TbxName_TextChanged( object sender, EventArgs e )
        {
            NotifyChangeEvent_();
        }

        /// <summary>
        /// フレーム区間の調整
        /// </summary>
        private void AdjustFrameRange_(bool adjustEnd)
        {
            int startFrame = _tbxStartFrame.Value;
            int endFrame = _tbxEndFrame.Value;

            AnimationLoopType loopType = _target != null ? _target.BinarySettings.LoopType : AnimationLoopType.OneTime;
            AnimFrameSectionSetHelper.AdjustFrameRange(loopType, ref startFrame, ref endFrame, adjustEnd);

            _tbxStartFrame.SetValue(startFrame);
            _tbxEndFrame.SetValue(endFrame);
        }

        /// <summary>
        /// スタートフレーム：テキスト変更
        /// </summary>
        private void Event_TbxStartFrame_TextChanged( object sender, EventArgs e )
        {
            AdjustFrameRange_(_AdjustEndFrame);
            _tbxLength.SetValue(_tbxEndFrame.Value - _tbxStartFrame.Value);
            NotifyChangeEvent_();
        }

        /// <summary>
        /// エンドフレーム：テキスト変更
        /// </summary>
        private void Event_TbxEndFrame_TextChanged( object sender, EventArgs e )
        {
            AdjustFrameRange_(!_AdjustEndFrame);
            _tbxLength.SetValue(_tbxEndFrame.Value - _tbxStartFrame.Value);
            NotifyChangeEvent_();
        }

        /// <summary>
        /// フレーム長さ：テキスト変更
        /// </summary>
        private void Event_tbxLength_ValueChanged(object sender, EventArgs e)
        {
            _tbxEndFrame.SetValue(_tbxStartFrame.Value + _tbxLength.Value);
            AdjustFrameRange_(_AdjustEndFrame);
            NotifyChangeEvent_();
        }

        /// <summary>
        /// フレーム長さ：ロケーション変更
        /// </summary>
        private void Event_TbxLength_LocationChanged(object sender, EventArgs e)
        {
            if (_textLengthXPos <= 0)
            {
                _textLengthXPos = this.Width - _tbxLength.Location.X;
            }
        }

        /// <summary>
        /// コメント：正当性チェック
        /// </summary>
        private void Event_TbxComment_Validating( object sender, CancelEventArgs e )
        {
            ApplayChanges();
        }

        /// <summary>
        /// 拡大縮小ボタン(OFF)：押下
        /// </summary>
        private void Event_RadScaleOff_Click(object sender, EventArgs e)
        {
            _scaleType = ScaleType.Scale_Off;
        }

        /// <summary>
        /// 拡大縮小ボタン(Float)：押下
        /// </summary>
        private void Event_RadScaleFloat_Click(object sender, EventArgs e)
        {
            _scaleType = ScaleType.Scale_Float;
        }

        /// <summary>
        /// 拡大縮小ボタン(Int)：押下
        /// </summary>
        private void Event_RadScaleInt_Click(object sender, EventArgs e)
        {
            _scaleType = ScaleType.Scale_Int;
        }

        /// <summary>
        /// 変更を反映します。
        /// </summary>
        public void ApplayChanges()
        {
            // _tbxComment Validatingイベントを利用するので、変更されたか確認します。
            if (_target != null && _target.Comment != _tbxComment.Text)
            {
                NotifyChangeEvent_();
            }
        }
    }
}
