﻿// --------------------------------------------------------------------------------
// <copyright>
// Copyright (C)Nintendo. All rights reserved.
//
// These coded instructions, statements, and computer programs contain proprietary
// information of Nintendo and/or its licensed developers and are protected by
// national and international copyright laws. They may not be disclosed to third
// parties or copied or duplicated in any form, in whole or in part, without the
// prior written consent of Nintendo.
//
// The content herein is highly confidential and should be handled accordingly.
// </copyright>
// --------------------------------------------------------------------------------
using System;
using System.Collections.Generic;
using System.ComponentModel;
using System.Data;
using System.Drawing;
using System.Text;
using System.Windows.Forms;
using System.Linq;

namespace LayoutEditor.Forms
{
    using LayoutEditor.Forms.Dialogs;
    using LECore.Structures;
    using LECore.Structures.LECoreInterface;

    using LECore.Manipulator;

    /// <summary>
    /// 選択ダイアログ
    /// </summary>
    public partial class SelectFromListDlg : LEBaseDlg
    {
        [Flags]
        public enum Option
        {
            None = 0x00,
            SelectOneItem = 0x01
        }

        /// <summary>
        /// アイテムを唯一つだけ選択するかどうか
        /// </summary>
        readonly Option _Option = Option.None;



        /// <summary>
        /// コンストラクタ
        /// </summary>
        public SelectFromListDlg()
            :this(Option.None )
        {

        }

        /// <summary>
        /// コンストラクタ
        /// </summary>
        public SelectFromListDlg(Option  option )
        {
            InitializeComponent();
            _cmhGroupName.Width = this.Width;
            _Option = option;
        }

        /// <summary>
        /// 設定
        /// </summary>
        public void Setup<T>(
            IEnumerable<T> allTargetSet,
            Func<T, bool> detectCheckedFunc,
            Func<T, ListViewItem> createListItemFunc)
        {
            if (allTargetSet.Count() > 0)
            {
                _lvwTargetGroups.BeginUpdate();

                foreach (T target in allTargetSet)
                {
                    ListViewItem lvi = createListItemFunc(target);
                    lvi.Checked = detectCheckedFunc(target);
                    _lvwTargetGroups.Items.Add(lvi);
                }

                _lvwTargetGroups.EndUpdate();

                UpdateAllCheckState_();
            }
            else
            {
                _btnOK.Enabled = false;
                _chkCheckAll.Enabled = false;
                _lvwTargetGroups.Enabled = false;
            }
        }

        /// <summary>
        /// 選択グループを取得します。
        /// </summary>
        public T[] GetSelectedSet<T>()
        {
            List<T> groupSet = new List<T>();
            foreach (ListViewItem lvi in _lvwTargetGroups.Items)
            {
                if (lvi.Checked)
                {
                    groupSet.Add((T)lvi.Tag);
                }
            }
            return groupSet.ToArray();
        }

        #region イベントハンドラ

        private void UpdateAllCheckState_()
        {
            bool allChecked = true;

            foreach (ListViewItem item in _lvwTargetGroups.Items)
            {
                allChecked &= item.Checked;
            }

            _chkCheckAll.SetChecked(allChecked);
        }

        /// <summary>
        /// すべてをチェックハンドラ
        /// </summary>
        private void Event_chkCheckAll_CheckedChanged(object sender, EventArgs e)
        {
            _lvwTargetGroups.BeginUpdate();

            foreach (ListViewItem lvi in _lvwTargetGroups.Items)
            {
                lvi.Checked = _chkCheckAll.Checked;
            }

            _lvwTargetGroups.EndUpdate();
        }

        /// <summary>
        /// OK
        /// </summary>
        private void Event_BtnOK_Click( object sender, EventArgs e )
        {
            DialogResult = DialogResult.OK;
        }

        /// <summary>
        /// キャンセル
        /// </summary>
        private void Event_BtnCancel_Click( object sender, EventArgs e )
        {
            DialogResult = DialogResult.Cancel;
        }

        /// <summary>
        /// アイテムチェック状態変更ハンドラ
        /// </summary>
        private void Event_LvwTargetGroups_ItemCheck( object sender, ItemCheckEventArgs e )
        {
            if( (_Option & Option.SelectOneItem) != 0 )
            {
                // 新たにチェックされる場合は、
                if( e.NewValue == CheckState.Checked )
                {
                    //　現在チェックされているアイテムを非チェックに設定する。
                    foreach( ListViewItem lvi in _lvwTargetGroups.Items )
                    {
                        if( lvi != null &&
                            lvi.Index != e.Index &&
                            lvi.Checked )
                        {
                            lvi.Checked = false;
                        }
                    }
                }
            }
        }

        #endregion イベントハンドラ



    }
}
