﻿// --------------------------------------------------------------------------------
// <copyright>
// Copyright (C)Nintendo. All rights reserved.
//
// These coded instructions, statements, and computer programs contain proprietary
// information of Nintendo and/or its licensed developers and are protected by
// national and international copyright laws. They may not be disclosed to third
// parties or copied or duplicated in any form, in whole or in part, without the
// prior written consent of Nintendo.
//
// The content herein is highly confidential and should be handled accordingly.
// </copyright>
// --------------------------------------------------------------------------------
using System;
using System.Collections.Generic;
using System.ComponentModel;
using System.Data;
using System.Drawing;
using System.Linq;
using System.Text;
using System.Windows.Forms;
using LayoutEditor.Forms.Dialogs;
using LECore.Structures;

namespace LayoutEditor.src.Forms.Dialogs
{
    /// <summary>
    /// 機能アニメーションと同名の区間タグを作成＆設定
    /// </summary>
    public partial class PartsAnimToTagDlg : LEBaseDlg
    {
        /// <summary>
        /// コンストラクタです。
        /// </summary>
        public PartsAnimToTagDlg()
        {
            InitializeComponent();
        }

        //----------------------------------------------------------
        // プロパティ
        //----------------------------------------------------------

        /// <summary>
        /// 選択された機能アニメーションです。
        /// </summary>
        public IEnumerable<PartsAnimSetting> TargetAnims
        {
            get
            {
                foreach (ListViewItem item in this._lvwAnims.CheckedItems)
                {
                    var anim = (PartsAnimSetting)item.Tag;
                    anim.StartFrame = int.Parse(item.SubItems[1].Text);
                    anim.EndFrame = int.Parse(item.SubItems[2].Text);
                    anim.Desc = item.SubItems[3].Text;
                    yield return anim;
                }
            }
        }

        //----------------------------------------------------------
        // 初期化
        //----------------------------------------------------------

        /// <summary>
        /// ダイアログを初期化します。
        /// </summary>
        public void Initialize(IEnumerable<PartsAnimSetting> anims)
        {
            foreach (var anim in anims)
            {
                var item = new ListViewItem(new string[]
                {
                    anim.Anim.Name, anim.StartFrame.ToString(), anim.EndFrame.ToString(), anim.Desc,
                });
                item.Tag = anim;

                // 必須項目はデフォルトでチェックしておく。
                item.Checked = anim.Anim.IsRequired;

                _lvwAnims.Items.Add(item);
            }

            Event_LvwAnims_SelectedIndexChanged(null, null);
            Event_LvwAnims_ItemChecked(null, null);
        }

        //----------------------------------------------------------
        // イベント
        //----------------------------------------------------------

        /// <summary>
        /// OKボタン
        /// </summary>
        private void Event_BtnOK_Click(object sender, EventArgs e)
        {
            this.DialogResult = DialogResult.OK;
        }

        /// <summary>
        /// キャンセルボタン
        /// </summary>
        private void Event_BtnCancel_Click(object sender, EventArgs e)
        {
            this.DialogResult = DialogResult.Cancel;
        }

        /// <summary>
        /// リスト選択変更
        /// </summary>
        private void Event_LvwAnims_SelectedIndexChanged(object sender, EventArgs e)
        {
            if (_lvwAnims.SelectedIndices.Count > 0)
            {
                var item = _lvwAnims.SelectedItems[0];
                _lblSelectedAnimName.Text = string.Format("[{0}]", item.SubItems[0].Text);
                _tbxDesc.SetValue(item.SubItems[3].Text);
                _tbxStartFrame.SetValue(int.Parse(item.SubItems[1].Text));
                _tbxEndFrame.SetValue(int.Parse(item.SubItems[2].Text));
                _tbxLength.SetValue(_tbxEndFrame.Value - _tbxStartFrame.Value);
                _tbxStartFrame.Enabled = _tbxEndFrame.Enabled = _tbxLength.Enabled = _tbxDesc.Enabled =
                    !((PartsAnimSetting)_lvwAnims.SelectedItems[0].Tag).IsFoundTag;
            }
            else
            {
                _lblSelectedAnimName.Text = StringResMgr.Get("PARTS_TARGET_PARAM_NONE");
                _tbxDesc.SetValue(string.Empty);
                _tbxStartFrame.SetValue(0);
                _tbxEndFrame.SetValue(1);
                _tbxLength.SetValue(1);
                _tbxStartFrame.Enabled = _tbxEndFrame.Enabled = _tbxLength.Enabled = _tbxDesc.Enabled = false;
            }
        }

        /// <summary>
        /// チェックされているアイテムが1つ以上あればOKが押せる
        /// </summary>
        private void Event_LvwAnims_ItemChecked(object sender, ItemCheckedEventArgs e)
        {
            _btnOK.Enabled = _lvwAnims.CheckedItems.Count > 0;
        }

        /// <summary>
        /// 説明更新
        /// </summary>
        private void Event_TbxDesc_Validated(object sender, EventArgs e)
        {
            if (_lvwAnims.SelectedIndices.Count > 0)
            {
                _lvwAnims.SelectedItems[0].SubItems[3].Text = _tbxDesc.Text;
            }
        }

        /// <summary>
        /// 開始フレームテキストボックスの内容を適用
        /// </summary>
        private void _tbxStartFrame_Validated(object sender, EventArgs e)
        {
            if (_lvwAnims.SelectedIndices.Count > 0)
            {
                _lvwAnims.SelectedItems[0].SubItems[1].Text = _tbxStartFrame.Text;
                if (_tbxStartFrame.Value >= _tbxEndFrame.Value)
                {
                    if (_tbxLength.Value < 1)
                    {
                        _tbxLength.SetValue(1);
                    }

                    _tbxEndFrame.SetValue(_tbxStartFrame.Value + _tbxLength.Value);
                    _lvwAnims.SelectedItems[0].SubItems[2].Text = _tbxEndFrame.Text;
                }

                _tbxLength.SetValue(_tbxEndFrame.Value - _tbxStartFrame.Value);
            }
        }

        /// <summary>
        /// 終了フレームテキストボックスの内容を適用
        /// </summary>
        private void _tbxEndFrame_Validated(object sender, EventArgs e)
        {
            if (_lvwAnims.SelectedIndices.Count > 0)
            {
                _lvwAnims.SelectedItems[0].SubItems[2].Text = _tbxEndFrame.Text;
                if (_tbxStartFrame.Value >= _tbxEndFrame.Value)
                {
                    if (_tbxLength.Value < 1)
                    {
                        _tbxLength.SetValue(1);
                    }

                    _tbxStartFrame.SetValue(_tbxEndFrame.Value - _tbxLength.Value);
                    _lvwAnims.SelectedItems[0].SubItems[1].Text = _tbxStartFrame.Text;
                }

                _tbxLength.SetValue(_tbxEndFrame.Value - _tbxStartFrame.Value);
            }
        }

        /// <summary>
        /// 長さテキストボックスの内容を適用
        /// </summary>
        private void _tbxLength_Validated(object sender, EventArgs e)
        {
            if (_lvwAnims.SelectedIndices.Count > 0)
            {
                if (_tbxLength.Value > 0)
                {
                    _tbxEndFrame.SetValue(_tbxStartFrame.Value + _tbxLength.Value);
                    _lvwAnims.SelectedItems[0].SubItems[2].Text = _tbxEndFrame.Text;
                }
                else
                {
                    _tbxLength.SetValue(_tbxEndFrame.Value - _tbxStartFrame.Value);
                }
            }
        }
    }

    /// <summary>
    /// 機能アニメーションに関連する区間タグ情報
    /// </summary>
    public class PartsAnimSetting
    {
        /// <summary>
        /// アニメーション情報
        /// </summary>
        public IParamaterAnimaiton Anim { get; set; }

        /// <summary>
        /// 開始フレーム
        /// </summary>
        public int StartFrame { get; set; }

        /// <summary>
        /// 終了フレーム
        /// </summary>
        public int EndFrame { get; set; }

        /// <summary>
        /// アニメーションの説明
        /// </summary>
        public string Desc { get; set; }

        /// <summary>
        /// 同名タグが登録済みか否か
        /// </summary>
        public bool IsFoundTag { get; set; }
    }
}
