﻿// --------------------------------------------------------------------------------
// <copyright>
// Copyright (C)Nintendo. All rights reserved.
//
// These coded instructions, statements, and computer programs contain proprietary
// information of Nintendo and/or its licensed developers and are protected by
// national and international copyright laws. They may not be disclosed to third
// parties or copied or duplicated in any form, in whole or in part, without the
// prior written consent of Nintendo.
//
// The content herein is highly confidential and should be handled accordingly.
// </copyright>
// --------------------------------------------------------------------------------
using System;
using System.Collections;
using System.Diagnostics;
using System.IO;
using System.Threading;
using System.Windows.Forms;

namespace LayoutEditor.Forms
{
    using LECore;
    using LECore.Structures;
    using Structures.SerializableObject;
    /// <summary>
    /// ExportBinaryFileDlg の概要の説明です。
    /// </summary>
    public class ExportBinaryFileDlg : System.Windows.Forms.Form
    {
        #region 定数
        const string ConverterArgsString = "\"{0}\"\\ \"{1}\"\\";
        #endregion 定数

        #region デザイナ生成コード
        private System.Windows.Forms.GroupBox _grpOutDir;
        private System.Windows.Forms.Label _lblOutDir;
        private System.Windows.Forms.TextBox _tbxOutDir;
        private System.Windows.Forms.Button _btnOutDirSelect;
        private System.Windows.Forms.TextBox _tbxOutFileName;
        private System.Windows.Forms.Label _lblOutFileName;
        private System.Windows.Forms.CheckBox _chkMakeDir;
        private System.Windows.Forms.GroupBox _grpOutFileKind;
        private System.Windows.Forms.CheckBox _chkBrlyt;
        private System.Windows.Forms.CheckBox _chkBrlvi;
        private System.Windows.Forms.CheckBox _chkBrlvc;
        private System.Windows.Forms.CheckBox _chkBrlmc;
        private System.Windows.Forms.CheckBox _chkBrlpa;
        private System.Windows.Forms.CheckBox _chkBrlts;
        private System.Windows.Forms.CheckBox _chkBrltp;
        private System.Windows.Forms.Button _btnOutPut;
        private System.Windows.Forms.FolderBrowserDialog _folderBrowserDialog;
        private CheckBox _chkUseTagData;
        private Label _lblTagHelp;
        private CheckBox _chkFixInfinityCurve;
        private CheckBox _chkConvertTextboxControlCode;
        private LayoutEditor.Controls.UIRadioButton _radBakeKeyMode_Tool;
    private LayoutEditor.Controls.UIRadioButton _radBakeKeyMode_Converter;
    private LayoutEditor.Controls.UIRadioButton radSameKeyNone;
    private LayoutEditor.Controls.UIRadioButton radOmitSameKey;
    private LayoutEditor.Controls.UIRadioButton radOmitSameKeyAll;
    private GroupBox grpKey;
    private CheckBox _chkAll;
    private Label _lblNotice;
        private System.Windows.Forms.Button _btnCancle;

        #region Windows フォーム デザイナで生成されたコード
        /// <summary>
        /// デザイナ サポートに必要なメソッドです。このメソッドの内容を
        /// コード エディタで変更しないでください。
        /// </summary>
        private void InitializeComponent()
        {
            System.ComponentModel.ComponentResourceManager resources = new System.ComponentModel.ComponentResourceManager(typeof(ExportBinaryFileDlg));
            this._grpOutFileKind = new System.Windows.Forms.GroupBox();
            this._chkAll = new System.Windows.Forms.CheckBox();
            this._chkBrlyt = new System.Windows.Forms.CheckBox();
            this._chkBrlvi = new System.Windows.Forms.CheckBox();
            this._chkBrlvc = new System.Windows.Forms.CheckBox();
            this._chkBrlmc = new System.Windows.Forms.CheckBox();
            this._chkBrlpa = new System.Windows.Forms.CheckBox();
            this._chkBrlts = new System.Windows.Forms.CheckBox();
            this._chkBrltp = new System.Windows.Forms.CheckBox();
            this._grpOutDir = new System.Windows.Forms.GroupBox();
            this._lblTagHelp = new System.Windows.Forms.Label();
            this.grpKey = new System.Windows.Forms.GroupBox();
            this.radOmitSameKeyAll = new LayoutEditor.Controls.UIRadioButton();
            this.radOmitSameKey = new LayoutEditor.Controls.UIRadioButton();
            this.radSameKeyNone = new LayoutEditor.Controls.UIRadioButton();
            this._radBakeKeyMode_Tool = new LayoutEditor.Controls.UIRadioButton();
            this._radBakeKeyMode_Converter = new LayoutEditor.Controls.UIRadioButton();
            this._chkConvertTextboxControlCode = new System.Windows.Forms.CheckBox();
            this._chkFixInfinityCurve = new System.Windows.Forms.CheckBox();
            this._chkUseTagData = new System.Windows.Forms.CheckBox();
            this._chkMakeDir = new System.Windows.Forms.CheckBox();
            this._lblOutFileName = new System.Windows.Forms.Label();
            this._btnOutDirSelect = new System.Windows.Forms.Button();
            this._tbxOutDir = new System.Windows.Forms.TextBox();
            this._lblOutDir = new System.Windows.Forms.Label();
            this._tbxOutFileName = new System.Windows.Forms.TextBox();
            this._btnOutPut = new System.Windows.Forms.Button();
            this._folderBrowserDialog = new System.Windows.Forms.FolderBrowserDialog();
            this._btnCancle = new System.Windows.Forms.Button();
            this._lblNotice = new System.Windows.Forms.Label();
            this._grpOutFileKind.SuspendLayout();
            this._grpOutDir.SuspendLayout();
            this.grpKey.SuspendLayout();
            this.SuspendLayout();
            //
            // _grpOutFileKind
            //
            this._grpOutFileKind.Controls.Add(this._chkAll);
            this._grpOutFileKind.Controls.Add(this._chkBrlyt);
            this._grpOutFileKind.Controls.Add(this._chkBrlvi);
            this._grpOutFileKind.Controls.Add(this._chkBrlvc);
            this._grpOutFileKind.Controls.Add(this._chkBrlmc);
            this._grpOutFileKind.Controls.Add(this._chkBrlpa);
            this._grpOutFileKind.Controls.Add(this._chkBrlts);
            this._grpOutFileKind.Controls.Add(this._chkBrltp);
            resources.ApplyResources(this._grpOutFileKind, "_grpOutFileKind");
            this._grpOutFileKind.Name = "_grpOutFileKind";
            this._grpOutFileKind.TabStop = false;
            //
            // _chkAll
            //
            this._chkAll.Checked = true;
            this._chkAll.CheckState = System.Windows.Forms.CheckState.Checked;
            resources.ApplyResources(this._chkAll, "_chkAll");
            this._chkAll.Name = "_chkAll";
            this._chkAll.CheckedChanged += new System.EventHandler(this.Event_chkAll_CheckedChanged);
            //
            // _chkBrlyt
            //
            this._chkBrlyt.Checked = true;
            this._chkBrlyt.CheckState = System.Windows.Forms.CheckState.Checked;
            resources.ApplyResources(this._chkBrlyt, "_chkBrlyt");
            this._chkBrlyt.Name = "_chkBrlyt";
            //
            // _chkBrlvi
            //
            this._chkBrlvi.Checked = true;
            this._chkBrlvi.CheckState = System.Windows.Forms.CheckState.Checked;
            resources.ApplyResources(this._chkBrlvi, "_chkBrlvi");
            this._chkBrlvi.Name = "_chkBrlvi";
            //
            // _chkBrlvc
            //
            this._chkBrlvc.Checked = true;
            this._chkBrlvc.CheckState = System.Windows.Forms.CheckState.Checked;
            resources.ApplyResources(this._chkBrlvc, "_chkBrlvc");
            this._chkBrlvc.Name = "_chkBrlvc";
            //
            // _chkBrlmc
            //
            this._chkBrlmc.Checked = true;
            this._chkBrlmc.CheckState = System.Windows.Forms.CheckState.Checked;
            resources.ApplyResources(this._chkBrlmc, "_chkBrlmc");
            this._chkBrlmc.Name = "_chkBrlmc";
            //
            // _chkBrlpa
            //
            this._chkBrlpa.Checked = true;
            this._chkBrlpa.CheckState = System.Windows.Forms.CheckState.Checked;
            resources.ApplyResources(this._chkBrlpa, "_chkBrlpa");
            this._chkBrlpa.Name = "_chkBrlpa";
            //
            // _chkBrlts
            //
            this._chkBrlts.Checked = true;
            this._chkBrlts.CheckState = System.Windows.Forms.CheckState.Checked;
            resources.ApplyResources(this._chkBrlts, "_chkBrlts");
            this._chkBrlts.Name = "_chkBrlts";
            //
            // _chkBrltp
            //
            this._chkBrltp.Checked = true;
            this._chkBrltp.CheckState = System.Windows.Forms.CheckState.Checked;
            resources.ApplyResources(this._chkBrltp, "_chkBrltp");
            this._chkBrltp.Name = "_chkBrltp";
            //
            // _grpOutDir
            //
            this._grpOutDir.Controls.Add(this._lblTagHelp);
            this._grpOutDir.Controls.Add(this.grpKey);
            this._grpOutDir.Controls.Add(this._radBakeKeyMode_Tool);
            this._grpOutDir.Controls.Add(this._radBakeKeyMode_Converter);
            this._grpOutDir.Controls.Add(this._chkConvertTextboxControlCode);
            this._grpOutDir.Controls.Add(this._chkFixInfinityCurve);
            this._grpOutDir.Controls.Add(this._chkUseTagData);
            this._grpOutDir.Controls.Add(this._chkMakeDir);
            this._grpOutDir.Controls.Add(this._lblOutFileName);
            this._grpOutDir.Controls.Add(this._btnOutDirSelect);
            this._grpOutDir.Controls.Add(this._tbxOutDir);
            this._grpOutDir.Controls.Add(this._lblOutDir);
            this._grpOutDir.Controls.Add(this._tbxOutFileName);
            resources.ApplyResources(this._grpOutDir, "_grpOutDir");
            this._grpOutDir.Name = "_grpOutDir";
            this._grpOutDir.TabStop = false;
            //
            // _lblTagHelp
            //
            this._lblTagHelp.BackColor = System.Drawing.SystemColors.ControlLight;
            resources.ApplyResources(this._lblTagHelp, "_lblTagHelp");
            this._lblTagHelp.Name = "_lblTagHelp";
            //
            // grpKey
            //
            this.grpKey.Controls.Add(this.radOmitSameKeyAll);
            this.grpKey.Controls.Add(this.radOmitSameKey);
            this.grpKey.Controls.Add(this.radSameKeyNone);
            resources.ApplyResources(this.grpKey, "grpKey");
            this.grpKey.Name = "grpKey";
            this.grpKey.TabStop = false;
            //
            // radOmitSameKeyAll
            //
            this.radOmitSameKeyAll.Checked = true;
            resources.ApplyResources(this.radOmitSameKeyAll, "radOmitSameKeyAll");
            this.radOmitSameKeyAll.Name = "radOmitSameKeyAll";
            this.radOmitSameKeyAll.TabStop = true;
            //
            // radOmitSameKey
            //
            this.radOmitSameKey.Checked = true;
            resources.ApplyResources(this.radOmitSameKey, "radOmitSameKey");
            this.radOmitSameKey.Name = "radOmitSameKey";
            this.radOmitSameKey.TabStop = true;
            //
            // radSameKeyNone
            //
            this.radSameKeyNone.Checked = true;
            resources.ApplyResources(this.radSameKeyNone, "radSameKeyNone");
            this.radSameKeyNone.Name = "radSameKeyNone";
            this.radSameKeyNone.TabStop = true;
            //
            // _radBakeKeyMode_Tool
            //
            this._radBakeKeyMode_Tool.Checked = true;
            resources.ApplyResources(this._radBakeKeyMode_Tool, "_radBakeKeyMode_Tool");
            this._radBakeKeyMode_Tool.Name = "_radBakeKeyMode_Tool";
            this._radBakeKeyMode_Tool.TabStop = true;
            //
            // _radBakeKeyMode_Converter
            //
            this._radBakeKeyMode_Converter.Checked = true;
            resources.ApplyResources(this._radBakeKeyMode_Converter, "_radBakeKeyMode_Converter");
            this._radBakeKeyMode_Converter.Name = "_radBakeKeyMode_Converter";
            this._radBakeKeyMode_Converter.TabStop = true;
            //
            // _chkConvertTextboxControlCode
            //
            this._chkConvertTextboxControlCode.Checked = true;
            this._chkConvertTextboxControlCode.CheckState = System.Windows.Forms.CheckState.Checked;
            resources.ApplyResources(this._chkConvertTextboxControlCode, "_chkConvertTextboxControlCode");
            this._chkConvertTextboxControlCode.Name = "_chkConvertTextboxControlCode";
            //
            // _chkFixInfinityCurve
            //
            this._chkFixInfinityCurve.Checked = true;
            this._chkFixInfinityCurve.CheckState = System.Windows.Forms.CheckState.Checked;
            resources.ApplyResources(this._chkFixInfinityCurve, "_chkFixInfinityCurve");
            this._chkFixInfinityCurve.Name = "_chkFixInfinityCurve";
            this._chkFixInfinityCurve.CheckedChanged += new System.EventHandler(this.Event_ChkFixInfinityCurve_CheckedChanged);
            //
            // _chkUseTagData
            //
            this._chkUseTagData.Checked = true;
            this._chkUseTagData.CheckState = System.Windows.Forms.CheckState.Checked;
            resources.ApplyResources(this._chkUseTagData, "_chkUseTagData");
            this._chkUseTagData.Name = "_chkUseTagData";
            this._chkUseTagData.CheckedChanged += new System.EventHandler(this.Event_ChkUseTagData_CheckedChanged);
            //
            // _chkMakeDir
            //
            this._chkMakeDir.Checked = true;
            this._chkMakeDir.CheckState = System.Windows.Forms.CheckState.Checked;
            resources.ApplyResources(this._chkMakeDir, "_chkMakeDir");
            this._chkMakeDir.Name = "_chkMakeDir";
            //
            // _lblOutFileName
            //
            resources.ApplyResources(this._lblOutFileName, "_lblOutFileName");
            this._lblOutFileName.Name = "_lblOutFileName";
            //
            // _btnOutDirSelect
            //
            resources.ApplyResources(this._btnOutDirSelect, "_btnOutDirSelect");
            this._btnOutDirSelect.Name = "_btnOutDirSelect";
            this._btnOutDirSelect.Click += new System.EventHandler(this.Event_BtnOutDirSelect_Click);
            //
            // _tbxOutDir
            //
            resources.ApplyResources(this._tbxOutDir, "_tbxOutDir");
            this._tbxOutDir.Name = "_tbxOutDir";
            this._tbxOutDir.TextChanged += new System.EventHandler(this.Event_TbxOut_TextChanged);
            //
            // _lblOutDir
            //
            resources.ApplyResources(this._lblOutDir, "_lblOutDir");
            this._lblOutDir.Name = "_lblOutDir";
            //
            // _tbxOutFileName
            //
            resources.ApplyResources(this._tbxOutFileName, "_tbxOutFileName");
            this._tbxOutFileName.Name = "_tbxOutFileName";
            this._tbxOutFileName.TextChanged += new System.EventHandler(this.Event_TbxOut_TextChanged);
            //
            // _btnOutPut
            //
            resources.ApplyResources(this._btnOutPut, "_btnOutPut");
            this._btnOutPut.Name = "_btnOutPut";
            this._btnOutPut.Click += new System.EventHandler(this.Event_BtnOutPut_Click);
            //
            // _btnCancle
            //
            resources.ApplyResources(this._btnCancle, "_btnCancle");
            this._btnCancle.Name = "_btnCancle";
            this._btnCancle.Click += new System.EventHandler(this.Event_BtnCancel_Click);
            //
            // _lblNotice
            //
            this._lblNotice.BackColor = System.Drawing.SystemColors.ControlLight;
            resources.ApplyResources(this._lblNotice, "_lblNotice");
            this._lblNotice.ForeColor = System.Drawing.SystemColors.GrayText;
            this._lblNotice.Name = "_lblNotice";
            //
            // ExportBinaryFileDlg
            //
            resources.ApplyResources(this, "$this");
            this.Controls.Add(this._btnOutPut);
            this.Controls.Add(this._lblNotice);
            this.Controls.Add(this._grpOutDir);
            this.Controls.Add(this._grpOutFileKind);
            this.Controls.Add(this._btnCancle);
            this.FormBorderStyle = System.Windows.Forms.FormBorderStyle.FixedDialog;
            this.MaximizeBox = false;
            this.MinimizeBox = false;
            this.Name = "ExportBinaryFileDlg";
            this._grpOutFileKind.ResumeLayout(false);
            this._grpOutDir.ResumeLayout(false);
            this._grpOutDir.PerformLayout();
            this.grpKey.ResumeLayout(false);
            this.ResumeLayout(false);

        }
        #endregion

        #endregion デザイナ生成コード

        /// <summary>
        /// 必要なデザイナ変数です。
        /// </summary>
        private System.ComponentModel.Container components = null;

        readonly BinarySrcSaveFunction       _saveFunction = null;
        readonly AppSetting                   _appSetting = null;


        public delegate bool BinarySrcSaveFunction( string outPath, string outFileName, object saveOption );

        void UpdateProperty_Tag_( bool flag )
        {
            _grpOutFileKind.Enabled = !_chkUseTagData.Checked;
            _lblTagHelp.Visible = _chkUseTagData.Checked;
            grpKey.Enabled = _chkUseTagData.Checked;
        }

       #region プロパティ

        /// <summary>
        /// 出力パス
        /// </summary>
        public string OutFilePath
        {
            get{ return _tbxOutDir.Text;}
        }

        /// <summary>
        /// 出力ファイル名
        /// </summary>
        public string OutFileName
        {
            get{ return _tbxOutFileName.Text;}
        }


        /// <summary>
        /// キー固定モード
        /// </summary>
        public AppSetting.BakeKeyMode BakeKeyMode
        {
            get
            {
                if( _radBakeKeyMode_Tool.Checked )
                {
                    return AppSetting.BakeKeyMode.Tool;
                }
                else if( _radBakeKeyMode_Converter.Checked )
                {
                    return AppSetting.BakeKeyMode.Converter;
                }
                else
                {
                    Debug.Assert( false );
                    return AppSetting.BakeKeyMode.None;
                }
            }
        }

       #endregion プロパティ

        /// <summary>
        /// コンストラクタ
        /// </summary>
        /// <param name="saveFunc"></param>
        /// <param name="defaultOutDir"></param>
        /// <param name="defaultOutFileName"></param>
        /// <param name="appSetting"></param>
        public ExportBinaryFileDlg(
            BinarySrcSaveFunction saveFunc,
            string defaultOutDir,
            string defaultOutFileName,
            AppSetting appSetting )
        {
            Debug.Assert( appSetting != null );

            //
            // Windows フォーム デザイナ サポートに必要です。
            //
            InitializeComponent();

            // ファイル名が指定されていれば、ディレクトリ名だけ取り出して設定します。
            if( Directory.Exists( defaultOutDir ) )
            {
                _tbxOutDir.Text = defaultOutDir;
            }

            _tbxOutFileName.Text = defaultOutFileName;
            _saveFunction        =  saveFunc;
            _appSetting = appSetting;


            // キー固定モード
            _radBakeKeyMode_Tool.Tag = AppSetting.BakeKeyMode.Tool;
            _radBakeKeyMode_Converter.Tag = AppSetting.BakeKeyMode.Converter;
            _radBakeKeyMode_Tool.SetCheckedByTag( _appSetting.BakeKeyKind );

            UpdateOutputButtonEnableState_();

            UpdateProperty_Tag_( _chkUseTagData.Checked );
        }

        /// <summary>
        /// 使用されているリソースに後処理を実行します。
        /// </summary>
        protected override void Dispose( bool disposing )
        {
            if( disposing )
            {
                if(components != null)
                {
                    components.Dispose();
                }
            }
            base.Dispose( disposing );
        }

        #region イベントハンドラ

        /// <summary>
        /// すべてをチェック：ハンドラ
        /// </summary>
        private void Event_chkAll_CheckedChanged(object sender, EventArgs e)
        {
            _chkBrlyt.Checked = _chkAll.Checked;
            _chkBrlpa.Checked = _chkAll.Checked;
            _chkBrlvi.Checked = _chkAll.Checked;
            _chkBrlvc.Checked = _chkAll.Checked;
            _chkBrlts.Checked = _chkAll.Checked;
            _chkBrlmc.Checked = _chkAll.Checked;
            _chkBrltp.Checked = _chkAll.Checked;
        }

        /// <summary>
        /// 参照ボタンハンドラ
        /// </summary>
        private void Event_BtnOutDirSelect_Click(object sender, System.EventArgs e)
        {
            // 有効なディレクトリが入力済みであれば、
            // 初期ディレクトリとして設定します。
            if( Directory.Exists( _tbxOutDir.Text ) )
            {
                _folderBrowserDialog.SelectedPath = _tbxOutDir.Text;
            }

            if( _folderBrowserDialog.ShowDialog() == DialogResult.OK )
            {
                _tbxOutDir.Text = _folderBrowserDialog.SelectedPath;
            }

            UpdateOutputButtonEnableState_();
        }


        /// <summary>
        /// 完全な出力パスを取得します。
        /// </summary>
        string GetOutFullPath_()
        {
            string outFullPath = _tbxOutDir.Text;
            if( _chkMakeDir.Checked )
            {
                outFullPath += @"\" + OutFileName;
            }
            return outFullPath;
        }

        #region コンバータ実行部分(実機プレビュー部分の移植)
        /// <summary>
        /// 子プロセスの標準出力に書き込まれた内容をコンソールに出力
        /// </summary>
        /// <param name="state"></param>
        static void ConsoleOutFromChildStdOut(object state)
        {
            StreamReader stdout = (StreamReader)state;

            string line;
            while (null != (line = stdout.ReadLine()))
            {
                Debug.WriteLine(line);
            }
        }

        /// <summary>
        /// コンバータを実行します。
        /// </summary>
        bool DoConvert_( string inputDir, string outPutDir )
        {
            // レイアウトのコンバート
            ProcessStartInfo startInfo         = new ProcessStartInfo();
            startInfo.UseShellExecute          = false;
            startInfo.CreateNoWindow           = true;
            startInfo.RedirectStandardOutput   = true;
            startInfo.RedirectStandardError    = true;
            startInfo.FileName = AppConstants.LayoutConverterPath;
            startInfo.Arguments = string.Format( ConverterArgsString, inputDir, outPutDir );
            startInfo.WorkingDirectory = Application.StartupPath;

            var prefPf = LayoutEditorCore.PlatformDetail as IPlatformPreferenceOwner;
            PlatformPreference platformPreference = prefPf != null ? prefPf.PlatformPreference : null;

            // TODO: NX の場合、Pc にしておけば、--code-type は Binary_Ir になって問題がないが、必要なら実機向けも選択できるようにする。
            startInfo.Arguments += " " + (platformPreference != null ? platformPreference.PcConverterAdditionalArguments : string.Empty) + " ";

            // タグ情報を利用するオプション。
            if( _chkUseTagData.Checked )
            {
                // 全区間をあらわすアニメーションは必要がないはずなので
                //  --tag-split-and-whole は指定しません。
                startInfo.Arguments += " -g";

                if( radOmitSameKey.Checked != false )
                {
                    startInfo.Arguments += " --omit-samekey";
                }

                if( radOmitSameKeyAll.Checked != false )
                {
                    startInfo.Arguments += " --omit-samekey-all";
                }
            }

            // テキストボックスの制御コードを変換する
            if( !_chkConvertTextboxControlCode.Checked )
            {
                startInfo.Arguments += " --no-convert-cvtrchar";
            }

            // コンバータでキーを固定する必要がある場合
            if( _chkFixInfinityCurve.Checked &&
                _radBakeKeyMode_Converter.Checked )
            {
                startInfo.Arguments += " --bake-infinity";
            }

            // ログファイルを出力(nw4r_lytcvtr.log)
            startInfo.Arguments += " --logfile";

            try
            {
                using (Process process = Process.Start(startInfo))
                {
                    ProccesErrorReader procErrorReader = new ProccesErrorReader(process.StandardError);
                    ThreadPool.QueueUserWorkItem(new WaitCallback(ConsoleOutFromChildStdOut), process.StandardOutput );

                    if ( !process.WaitForExit( _appSetting.BinaryConvertTimeOutMiliSec ) )
                    {
                        // タイムアウト
                        process.Kill();
                        //-- CONVERTER_ERROR_TIMEOUT "バイナリコンバータの応答がないので、強制終了しました。\nログを次のファイルに出力しました。\n{0}"
                        string msg = LECoreStringResMgr.Get( "CONVERTER_ERROR_TIMEOUT", Path.GetFileNameWithoutExtension(startInfo.FileName) + ".log");
                        ShowErrorDialog( msg );
                        return false;
                    }

                    if (process.ExitCode != 0)
                    {
                        //-- CONVERTER_ERROR_FAILCONVERT "バイナリコンバートに失敗しました。 - ({0})\nログを次のファイルに出力しました。\n{1}\n\n{2}"
                        string msg = LECoreStringResMgr.Get("CONVERTER_ERROR_FAILCONVERT",
                            process.ExitCode,
                            Path.Combine(startInfo.WorkingDirectory, Path.GetFileNameWithoutExtension(startInfo.FileName) + ".log"),
                            procErrorReader.ReadErrorMsg());

                        ShowErrorDialog( msg );
                        return false;
                    }
                }
            }
            catch (System.ComponentModel.Win32Exception ex)
            {
                //-- CONVERTER_ERROR_FAILSTARTUP "バイナリコンバータ({0})の起動に失敗しました。\nNintendoWareが正しくインストールされているか確認してください。\n - ({1})"
                string msg = LECoreStringResMgr.Get( "CONVERTER_ERROR_FAILSTARTUP", Path.GetFileName(AppConstants.LayoutConverterPath), ex.Message);

                ShowErrorDialog(msg);
                return false;
            }
            return true;
        }

        void ShowErrorDialog( string message )
        {
            // PREVIEW_ERROR_DLG_TITLE "MCS 通信エラー - NW4R LayoutEditor"
            string title = StringResMgr.Get("EXPBINARY_MSG_DLG_TITLE");
            MessageBox.Show(
                /* this, */
                message,
                title,
                MessageBoxButtons.OK,
                MessageBoxIcon.Stop );
        }
        #endregion コンバータ実行部分(実機プレビュー部分の移植)

        /// <summary>
        /// 入力パラメータが有効か判定します。
        /// </summary>
        bool CheckParamaterValid_()
        {
            if( !Directory.Exists( _tbxOutDir.Text ) )
            {
                return false;
            }

            if( _tbxOutFileName.Text == string.Empty )
            {
                return false;
            }

            return true;
        }

         /// <summary>
         /// 出力ボタンの状態を更新します。
         /// </summary>
        void UpdateOutputButtonEnableState_()
        {
            _btnOutPut.Enabled = CheckParamaterValid_();
        }

        /// <summary>
        /// searchPatternにマッチする、ファイルを収集します。
        /// </summary>
        void GetAllFilesRecursive_( string folder, string searchPattern, ref ArrayList files)
        {
            try
            {
                //folderにあるファイルを取得する
                string[] fs =
                    System.IO.Directory.GetFiles(folder, searchPattern);
                //ArrayListに追加する
                files.AddRange(fs);

                //folderのサブフォルダを取得する
                string[] ds = System.IO.Directory.GetDirectories(folder);
                //サブフォルダにあるファイルも調べる
                foreach (string d in ds)
                {

                    GetAllFilesRecursive_(d, searchPattern, ref files);
                }
            }
            catch
            {
                // アクセス権がない場合などに例外になる。
                // 何もせず処理を続ける。
            }
        }

        /// <summary>
        /// タグ情報を使用するチャックボックス：ハンドラ
        /// </summary>
        private void Event_ChkUseTagData_CheckedChanged( object sender, EventArgs e )
        {
            // フラグを反転します。
            UpdateProperty_Tag_( _chkUseTagData.Checked );
        }

        /// <summary>
        /// 出力：ハンドラ
        /// </summary>
        private void Event_BtnOutPut_Click(object sender, System.EventArgs e)
        {
            // 設定が有効か確認する
            if( CheckParamaterValid_() )
            {
                // 出力処理
                // 出力パスを作成する(存在しない場合は作成する)
                string outFullPath = GetOutFullPath_();

                // 上書きの確認
                if( Directory.Exists( outFullPath ) )
                {
                    ArrayList       files = new ArrayList();
                    GetAllFilesRecursive_( outFullPath, OutFileName + ".*", ref files );

                    // 上書きしそうなファイルが発見された。
                    if( files.Count != 0 )
                    {
                        string msg = StringResMgr.Get( "SYSTEM_FILE_OVERWRITE" );
                        string title = StringResMgr.Get( "EXPBINARY_MSG_DLG_TITLE" );

                        if( MessageBox.Show( msg, title, MessageBoxButtons.OKCancel ) == DialogResult.Cancel )
                        {
                            return;
                        }
                    }
                }

                // テンポラリの保存(フォルダ作成)
                string tempDir = outFullPath + @"\temp";

                Directory.CreateDirectory( tempDir );

                bool bToolHasToFixKeys =
                    _chkFixInfinityCurve.Checked && _radBakeKeyMode_Tool.Checked;
                bool result = _saveFunction( tempDir, OutFileName, bToolHasToFixKeys );
                if( result )
                {
                    // コンバータの実行
                    result &= DoConvert_( tempDir, outFullPath );
                }

                // テンポラリの削除
                Directory.Delete( tempDir, true );

                if( result )
                {
                    // 出力は無事成功した。
                    // ファイルが出力されたなら、
                    // レイアウトファイルのバイナリ出力先情報を更新します。
                    _appSetting.LastExportedBinaryFilePath = this.OutFilePath;
                    _appSetting.BakeKeyKind = this.BakeKeyMode;

                    DialogResult = DialogResult.OK;
                }
                else
                {
                    DialogResult = DialogResult.Cancel;
                }
            }
        }

        /// <summary>
        /// キャンセル：ハンドラ
        /// </summary>
        private void Event_BtnCancel_Click(object sender, System.EventArgs e)
        {
            DialogResult = DialogResult.Cancel;
        }

        /// <summary>
        /// 出力情報テキストボックス文字列変更ハンドラ
        /// </summary>
        private void Event_TbxOut_TextChanged(object sender, System.EventArgs e)
        {
            UpdateOutputButtonEnableState_();
        }

        /// <summary>
        /// チェック状態更新ハンドラ
        /// </summary>
        private void Event_ChkFixInfinityCurve_CheckedChanged( object sender, EventArgs e )
        {
            _radBakeKeyMode_Tool.Enabled = _chkFixInfinityCurve.Checked;
            _radBakeKeyMode_Converter.Enabled = _chkFixInfinityCurve.Checked;
        }
        #endregion イベントハンドラ
    }
}
