﻿// --------------------------------------------------------------------------------
// <copyright>
// Copyright (C)Nintendo All rights reserved.
//
// These coded instructions, statements, and computer programs contain proprietary
// information of Nintendo and/or its licensed developers and are protected by
// national and international copyright laws. They may not be disclosed to third
// parties or copied or duplicated in any form, in whole or in part, without the
// prior written consent of Nintendo.
//
// The content herein is highly confidential and should be handled accordingly.
// </copyright>
// --------------------------------------------------------------------------------
using System;
using System.Drawing;
using System.Collections;
using System.Collections.Generic;
using System.Linq;
using System.ComponentModel;
using System.Windows.Forms;
using System.IO;

namespace LayoutEditor.Forms
{
    using LECore;
    using LECore.Structures;
    using LECore.Structures.Core;

    using LECore.Util;
    using LECore.Manipulator;
    using System.Diagnostics;

    /// <summary>
    /// CreateTextBoxDlg の概要の説明です。
    /// </summary>
    public class CreateTextBoxDlg : System.Windows.Forms.Form
    {
        private System.Windows.Forms.Label     _lblContentsText;
        private System.Windows.Forms.TextBox   _tbxContentsText;
        private System.Windows.Forms.Button    _btnOK;
        private Label _lblFont;
        private Controls.UIComboBox _cmbFont;
        private Panel _pnlFont;
        private Panel _pnlFile;
        private Button _btnReference;
        private Label _lblFontFile;
        private TextBox _tbxFontFile;
        private Controls.UIRadioButton _radFile;
        private Controls.UIRadioButton _radFont;
        /// <summary>
        /// 必要なデザイナ変数です。
        /// </summary>
        private System.ComponentModel.Container components = null;

        private HashSet<string> _fontNamesWithFontSettings = new HashSet<string>();
        #region プロパティ

        /// <summary>
        /// フォントファイルパス
        /// </summary>
        public string FontName
        {
            get { return _radFont.Checked ? _cmbFont.Text : string.Empty; }
        }

        /// <summary>
        /// フォントファイルパス
        /// </summary>
        public string FontFilePath
        {
            get { return _radFile.Checked ? _tbxFontFile.Text : string.Empty; }
        }

        /// <summary>
        /// テキストボックスの内容文字列
        /// </summary>
        public string TextBoxContentsText
        {
            get{ return _tbxContentsText.Text;}
        }
       #endregion プロパティ

        /// <summary>
        /// コンストラクタ
        /// </summary>
        public CreateTextBoxDlg()
        {
            //
            // Windows フォーム デザイナ サポートに必要です。
            //
            InitializeComponent();

            _radFile.Checked = true;
        }

        /// <summary>
        /// 初期化をします。
        /// </summary>
        public void Setup(string defaultFontPath, IEnumerable<string> fontNames, IEnumerable<string> fontNamesInProject)
        {
            // 規定のパスを設定
            _tbxFontFile.Text = defaultFontPath;

            // コンボの初期化
            {
                string currentText = _cmbFont.Text;

                _cmbFont.Items.Clear();
                foreach (var fontName in fontNames)
                {

                    _cmbFont.Items.Add(fontName);
                }

                _cmbFont.SelectedIndex = fontNames.Count() == 0 ? -1 : 0;
                _cmbFont.SetSelectedByText(currentText);
            }

            if (fontNames.Count() == 0)
            {
                _radFont.Enabled = false;
                _radFile.Checked = true;
            }
            else
            {
                _radFont.Enabled = true;
            }

            _fontNamesWithFontSettings.Clear();
            foreach (var fontName in fontNamesInProject)
            {
                _fontNamesWithFontSettings.Add(fontName);
            }

            UpdateOKButtonState_();
        }

        /// <summary>
        /// 使用されているリソースに後処理を実行します。
        /// </summary>
        protected override void Dispose( bool disposing )
        {
            if( disposing )
            {
                if(components != null)
                {
                    components.Dispose();
                }
            }
            base.Dispose( disposing );
        }

        #region Windows フォーム デザイナで生成されたコード
        /// <summary>
        /// デザイナ サポートに必要なメソッドです。このメソッドの内容を
        /// コード エディタで変更しないでください。
        /// </summary>
        private void InitializeComponent()
        {
            System.ComponentModel.ComponentResourceManager resources = new System.ComponentModel.ComponentResourceManager(typeof(CreateTextBoxDlg));
            this._lblContentsText = new System.Windows.Forms.Label();
            this._tbxContentsText = new System.Windows.Forms.TextBox();
            this._btnOK = new System.Windows.Forms.Button();
            this._lblFont = new System.Windows.Forms.Label();
            this._pnlFont = new System.Windows.Forms.Panel();
            this._cmbFont = new LayoutEditor.Controls.UIComboBox();
            this._pnlFile = new System.Windows.Forms.Panel();
            this._btnReference = new System.Windows.Forms.Button();
            this._lblFontFile = new System.Windows.Forms.Label();
            this._tbxFontFile = new System.Windows.Forms.TextBox();
            this._radFont = new LayoutEditor.Controls.UIRadioButton();
            this._radFile = new LayoutEditor.Controls.UIRadioButton();
            this._pnlFont.SuspendLayout();
            this._pnlFile.SuspendLayout();
            this.SuspendLayout();
            //
            // _lblContentsText
            //
            resources.ApplyResources(this._lblContentsText, "_lblContentsText");
            this._lblContentsText.Name = "_lblContentsText";
            //
            // _tbxContentsText
            //
            resources.ApplyResources(this._tbxContentsText, "_tbxContentsText");
            this._tbxContentsText.Name = "_tbxContentsText";
            this._tbxContentsText.TextChanged += new System.EventHandler(this.Event_TbxContentsText_TextChanged);
            //
            // _btnOK
            //
            resources.ApplyResources(this._btnOK, "_btnOK");
            this._btnOK.Name = "_btnOK";
            this._btnOK.Click += new System.EventHandler(this.EventEvent_BtnOK_Click);
            //
            // _lblFont
            //
            resources.ApplyResources(this._lblFont, "_lblFont");
            this._lblFont.Name = "_lblFont";
            //
            // _pnlFont
            //
            this._pnlFont.BackColor = System.Drawing.SystemColors.ControlLight;
            this._pnlFont.Controls.Add(this._lblFont);
            this._pnlFont.Controls.Add(this._cmbFont);
            resources.ApplyResources(this._pnlFont, "_pnlFont");
            this._pnlFont.Name = "_pnlFont";
            //
            // _cmbFont
            //
            this._cmbFont.FormattingEnabled = true;
            resources.ApplyResources(this._cmbFont, "_cmbFont");
            this._cmbFont.Name = "_cmbFont";
            this._cmbFont.SelectedItemData = null;
            //
            // _pnlFile
            //
            this._pnlFile.BackColor = System.Drawing.SystemColors.ControlLight;
            this._pnlFile.Controls.Add(this._btnReference);
            this._pnlFile.Controls.Add(this._lblFontFile);
            this._pnlFile.Controls.Add(this._tbxFontFile);
            resources.ApplyResources(this._pnlFile, "_pnlFile");
            this._pnlFile.Name = "_pnlFile";
            //
            // _btnReference
            //
            resources.ApplyResources(this._btnReference, "_btnReference");
            this._btnReference.Name = "_btnReference";
            this._btnReference.Click += new System.EventHandler(this.Event_BtnReference_Click);
            //
            // _lblFontFile
            //
            resources.ApplyResources(this._lblFontFile, "_lblFontFile");
            this._lblFontFile.Name = "_lblFontFile";
            //
            // _tbxFontFile
            //
            resources.ApplyResources(this._tbxFontFile, "_tbxFontFile");
            this._tbxFontFile.Name = "_tbxFontFile";
            //
            // _radFont
            //
            resources.ApplyResources(this._radFont, "_radFont");
            this._radFont.Name = "_radFont";
            this._radFont.UseVisualStyleBackColor = true;
            this._radFont.CheckedChanged += new System.EventHandler(this.Event_radFile_CheckedChanged);
            //
            // _radFile
            //
            resources.ApplyResources(this._radFile, "_radFile");
            this._radFile.Name = "_radFile";
            this._radFile.UseVisualStyleBackColor = true;
            this._radFile.CheckedChanged += new System.EventHandler(this.Event_radFile_CheckedChanged);
            //
            // CreateTextBoxDlg
            //
            resources.ApplyResources(this, "$this");
            this.Controls.Add(this._pnlFont);
            this.Controls.Add(this._pnlFile);
            this.Controls.Add(this._radFont);
            this.Controls.Add(this._radFile);
            this.Controls.Add(this._btnOK);
            this.Controls.Add(this._tbxContentsText);
            this.Controls.Add(this._lblContentsText);
            this.FormBorderStyle = System.Windows.Forms.FormBorderStyle.FixedDialog;
            this.MaximizeBox = false;
            this.MinimizeBox = false;
            this.Name = "CreateTextBoxDlg";
            this.ShowInTaskbar = false;
            this._pnlFont.ResumeLayout(false);
            this._pnlFont.PerformLayout();
            this._pnlFile.ResumeLayout(false);
            this._pnlFile.PerformLayout();
            this.ResumeLayout(false);
            this.PerformLayout();

        }
        #endregion


       #region public メソッド

        /// <summary>
        /// リセットします。
        /// </summary>
        public void Reset()
        {
            _tbxFontFile.Text = string.Empty;
        }

       #endregion public メソッド


       #region イベントハンドラ
        /// <summary>
        /// 有効なフォントパスか？
        /// </summary>
        bool IsValidFontPath_()
        {
            if (_radFile.Checked)
            {
                return _tbxFontFile.Text != string.Empty && File.Exists(_tbxFontFile.Text);
            }
            else if (_radFont.Checked)
            {
                return _cmbFont.Text != string.Empty;
            }

            return false;
        }

        /// <summary>
        /// ダイアログのパラメータが妥当か？
        /// </summary>
        bool IsValidToCreateTextBox_()
        {
            if( IsValidFontPath_() && _tbxContentsText.Text != string.Empty )
            {
                return true;
            }
            return false;
        }

        /// <summary>
        /// 参照ボタンクリック
        /// </summary>
        private void Event_BtnReference_Click(object sender, System.EventArgs e)
        {
            // ファイルダイアログを表示してフォントファイルを読み込みます。
            OpenFileDialog      openFileDlg = new OpenFileDialog();

            //------------------------------------------------------
            // ユーザが拡張子を指定しない場合、拡張子を追加する。
            openFileDlg.AddExtension = true;
            openFileDlg.Filter = LEFontHelper.OPEN_FONT_FILE_FILTER_STR;

            // 初期ディレクトリの設定
            if( _tbxFontFile.Text != string.Empty &&
                Directory.Exists( Path.GetDirectoryName( _tbxFontFile.Text ) ) )
            {
                openFileDlg.InitialDirectory = SaveLoadUIHelper.GetAbsolutePathForExistDirectory(Path.GetDirectoryName( _tbxFontFile.Text ));
                openFileDlg.FileName = Path.GetFileName( _tbxFontFile.Text );
            }

            if( openFileDlg.ShowDialog() == DialogResult.OK )
            {
                _tbxFontFile.Text   = openFileDlg.FileName;
                UpdateOKButtonState_();
            }
        }

        /// <summary>
        /// OKボタンクリック
        /// </summary>
        private void EventEvent_BtnOK_Click(object sender, System.EventArgs e)
        {
           // 入力されている内容が正しいかチェックします。
            if( IsValidToCreateTextBox_() )
            {
                if (_radFile.Checked)
                {
                    try
                    {
                        // プロジェクトに登録されているものは選べない
                        var fileName = Path.GetFileName(_tbxFontFile.Text);
                        var key = Path.GetFileNameWithoutExtension(_tbxFontFile.Text);
                        if (_fontNamesWithFontSettings.Contains(key))
                        {
                            MessageBox.Show(
                                string.Format(StringResMgr.Get("CREATE_TEXTBOX_DLG_USEDNAME"), key, fileName),
                                this.Text);
                            return;
                        }
                    }
                    catch
                    {
                        Debug.Assert(false);
                    }
                }

                DialogResult = DialogResult.OK;
                return;
            }



            // 正しく作成されなかった
            DialogResult = DialogResult.Cancel;
        }

        /// <summary>
        /// OKボタンの状態を更新します。
        /// </summary>
        void UpdateOKButtonState_()
        {
            _btnOK.Enabled = IsValidToCreateTextBox_();
        }

        /// <summary>
        /// フォントファイル文字列変更
        /// </summary>
        private void Event_TbxFontFile_TextChanged(object sender, System.EventArgs e)
        {
            UpdateOKButtonState_();
        }

        /// <summary>
        /// TextBox 内容文字列変更
        /// </summary>
        private void Event_TbxContentsText_TextChanged(object sender, System.EventArgs e)
        {
            UpdateOKButtonState_();
        }

        /// <summary>
        /// ラジオボタンの変更
        /// </summary>
        private void Event_radFile_CheckedChanged(object sender, EventArgs e)
        {
            _pnlFile.Enabled = _radFile.Checked;
            _pnlFont.Enabled = _radFont.Checked;

            UpdateOKButtonState_();
        }
        #endregion イベントハンドラ
    }
}
