﻿// --------------------------------------------------------------------------------
// <copyright>
// Copyright (C)Nintendo All rights reserved.
//
// These coded instructions, statements, and computer programs contain proprietary
// information of Nintendo and/or its licensed developers and are protected by
// national and international copyright laws. They may not be disclosed to third
// parties or copied or duplicated in any form, in whole or in part, without the
// prior written consent of Nintendo.
//
// The content herein is highly confidential and should be handled accordingly.
// </copyright>
// --------------------------------------------------------------------------------

using System;
using System.Collections;
using System.ComponentModel;
using System.Drawing;
using System.Drawing.Drawing2D;
using System.Drawing.Imaging;
using System.Data;
using System.Windows.Forms;

namespace LayoutEditor.Controls
{
    using Win32 = LECore.Win32;
    /// <summary>
    /// ToolButton の概要の説明です。
    /// </summary>
    public class ToolButton : System.Windows.Forms.Control
    {
        //	ボタンコントロール用
        private Image		_image		= null;
        private ImageList	_imageList	= null;
        private int			_imageIndex	= -1;
        private Style		_btnstyle	= Style.Normal;
        private bool		_checked	= false;
        private bool		_isFocus	= false;

        //	動作用
        private bool		_mouseDown	= false;
        private bool		_keyDown	= false;
        private ButtonStateType	_buttonState	= ButtonStateType.Normal;

        /// <summary>
        /// ボタンスタイル
        /// </summary>
        public enum	Style
        {
            /// <summary>通常ボタン</summary>
            Normal	= 0,
            /// <summary>トグルボタン</summary>
            Toggle	= 1
        }

        private enum ButtonStateType
        {
            /// <summary>通常ボタン</summary>
            Normal,
            /// <summary>ボタンにフォーカス</summary>
            Focus,
            /// <summary>ボタンダウン</summary>
            Click,
            /// <summary>トグルダウン</summary>
            Down
        }

        /// <summary>
        /// コンストラクタ
        /// </summary>
        public ToolButton()
        {
            this.TabStop	= false;
            //	ダブルクリックをシングルクリックへ
            this.SetStyle( ControlStyles.StandardClick, true );
            this.SetStyle( ControlStyles.StandardDoubleClick, false );
        }

        #region プロパティ
        /// <summary>
        /// イメージ
        /// </summary>
        [DefaultValue(null)]
        public Image Image
        {
            get	{ return _image;	}
            set
            {
                _image = value;
                if( this.Visible )
                    this.Invalidate();
            }
        }
        /// <summary>
        /// イメージリスト
        /// </summary>
        [DefaultValue(null)]
        public ImageList ImageList
        {
            get	{ return _imageList;	}
            set
            {
                _imageList = value;
                if( _imageIndex >= 0 && _imageList != null && _imageList.Images.Count > _imageIndex )
                    _image	= _imageList.Images[_imageIndex];
                if( this.Visible )
                    this.Invalidate();
            }
        }
        /// <summary>
        /// イメージインデックス
        /// </summary>
        [DefaultValue(-1)]
        public int ImageIndex
        {
            get	{ return _imageIndex;	}
            set
            {
                _imageIndex = value;
                if( _imageIndex >= 0 && _imageList != null && _imageList.Images.Count > _imageIndex )
                    _image	= _imageList.Images[_imageIndex];
                if( this.Visible )
                    this.Invalidate();
            }
        }
        /// <summary>
        /// ボタンスタイル
        /// </summary>
        [DefaultValue(Style.Normal)]
        public Style	ButtonStyle
        {
            get	{ return _btnstyle;		}
            set { _btnstyle	= value;	}
        }

        /// <summary>
        /// トグルか否か
        /// </summary>
        public bool IsToggle
        {
            get	{ return (_btnstyle & Style.Toggle ) != 0;	}
        }
        /// <summary>
        /// フォーカス取るスタイルか否か
        /// </summary>
        [DefaultValue(false)]
        public bool IsFocus
        {
            get	{ return _isFocus;		}
            set { _isFocus	= value;	}
        }
        /// <summary>
        /// ボタンのチェック状態
        /// </summary>
        [DefaultValue(false)]
        public bool Checked
        {
            get	{ return _checked;	}
            set
            {
                _checked = value;
                _buttonState	= ( _checked ) ? ButtonStateType.Down : ButtonStateType.Normal;
                if( this.Visible )
                    this.Invalidate();
            }
        }
        private ButtonStateType ButtonState
        {
            get	{ return _buttonState;	}
            set
            {
                if( _buttonState != value )
                {
                    _buttonState = value;
                    if( this.Visible )
                        this.Invalidate();
                }
            }
        }

        #endregion

        #region オーバーライド
        /// <summary>
        /// オーバーライド
        /// </summary>
        protected override void OnCreateControl()
        {
            if( Image == null )
            {
                if( _imageIndex >= 0 && _imageList != null && _imageList.Images.Count > _imageIndex )
                    _image	= _imageList.Images[_imageIndex];
            }
            if( !IsToggle )
                ButtonState	= ButtonStateType.Normal;
            else if( Checked )
                ButtonState	= ButtonStateType.Down;
            else
                ButtonState	= ButtonStateType.Normal;
            base.OnCreateControl ();
        }
        /// <summary>
        /// オーバーライド
        /// </summary>
        protected override void OnMouseDown(MouseEventArgs e)
        {
            if( ( e.Button & MouseButtons.Left ) != 0 )
            {
                _mouseDown	= true;
                ButtonState	= ButtonStateType.Click;
            }
            if( IsFocus )
                this.Focus();
            base.OnMouseDown (e);
        }
        /// <summary>
        /// オーバーライド
        /// </summary>
        protected override void OnMouseUp(MouseEventArgs e)
        {
            if( ( e.Button & MouseButtons.Left ) != 0 )
            {
                _mouseDown	= false;
                ButtonState	= ButtonStateType.Focus;
            }
            base.OnMouseUp (e);
        }
        /// <summary>
        /// オーバーライド
        /// </summary>
        protected override void OnMouseMove(MouseEventArgs e)
        {
            if((e.Button != MouseButtons.None) && this._mouseDown)
            {
                if(!base.ClientRectangle.Contains(e.X, e.Y))
                {
                    ButtonState	= ButtonStateType.Normal;
                }
                else
                {
                    ButtonState	= ButtonStateType.Click;
                }
            }
            base.OnMouseMove (e);
        }
        /// <summary>
        /// オーバーライド
        /// </summary>
        protected override void OnMouseEnter(EventArgs e)
        {
            if( !_mouseDown )
                ButtonState	= ButtonStateType.Focus;
            else
                ButtonState	= ButtonStateType.Click;
            base.OnMouseEnter (e);
        }
        /// <summary>
        /// オーバーライド
        /// </summary>
        protected override void OnMouseLeave(EventArgs e)
        {
            if( !IsToggle )
                ButtonState	= ButtonStateType.Normal;
            else if( Checked )
                ButtonState	= ButtonStateType.Down;
            else
                ButtonState	= ButtonStateType.Normal;
            base.OnMouseLeave (e);
        }
        /// <summary>
        /// オーバーライド
        /// </summary>
        protected override void OnPaint(PaintEventArgs e)
        {
            e.Graphics.Clear( this.BackColor );
            Rectangle	rect	= this.ClientRectangle;
            rect.Inflate( -4, -4 );
            switch( ButtonState )
            {
                case ButtonStateType.Focus:
                {
                    //	ボタンにフォーカスあるときはグラデーション
                    Rectangle	rect2	= this.ClientRectangle;
                    rect2.Inflate( -1, -1 );
                    using( LinearGradientBrush myBrush = new LinearGradientBrush( rect2, SystemColors.ControlLightLight,
                               SystemColors.ControlLight, LinearGradientMode.Vertical))
                    {
                        e.Graphics.FillRectangle(myBrush, rect2 );
                    }
                    //	トグルボタンでチェックされてるときはイメージずらす
                    if( IsToggle && Checked )
                        rect.Offset( 1, 1 );
                    LECore.Util.GraphicsUtil.DrawRoundishRectangle( e.Graphics, SystemPens.ControlDark, this.ClientRectangle );
                    DrawImage( e.Graphics, rect );
                    break;
                }
                case ButtonStateType.Click:
                    //	マウス＆キーダウン時
                    LECore.Util.GraphicsUtil.DrawRoundishRectangle( e.Graphics, SystemPens.ControlDark, this.ClientRectangle );
                    rect.Offset( 1, 1 );
                    DrawImage( e.Graphics, rect );
                    break;
//				case ButtonStateType.Up:
//					//	トグルアップ時
//					LECore.Util.GraphicsUtil.Draw3DRectangle( e.Graphics, SystemPens.ControlLightLight, SystemPens.ControlDark, this.ClientRectangle );
//					DrawImage( e.Graphics, rect );
//					break;
                case ButtonStateType.Down:
                {
                    //	トグルダウン時
                    Rectangle	rect2	= this.ClientRectangle;
                    rect2.Inflate( -1, -1 );
                    e.Graphics.FillRectangle(SystemBrushes.ControlLightLight, rect2 );
                    LECore.Util.GraphicsUtil.DrawRoundishRectangle( e.Graphics, SystemPens.ControlDark, this.ClientRectangle );
                    rect.Offset( 1, 1 );
                    DrawImage( e.Graphics, rect );
                    break;
                }
                default:
                    //	通常時
                    DrawImage( e.Graphics, rect );
                    break;
            }
            //	フォーカス枠
            if( this.Focused )
            {
                rect	= this.ClientRectangle;
                rect.Inflate( -2, -2 );
                LECore.Util.GraphicsUtil.DrawFocusRectangle( e.Graphics, rect );
            }
//			base.OnPaint (e);
        }
        /// <summary>
        /// オーバーライド
        /// </summary>
        protected override void OnClick(EventArgs e)
        {
            if( IsToggle )
            {
                _checked	= ( !_checked );
            }
            base.OnClick (e);
        }
        /// <summary>
        /// オーバーライド
        /// </summary>
        protected override void OnGotFocus(EventArgs e)
        {
            if( this.Visible )
                this.Invalidate();
            base.OnGotFocus (e);
        }
        /// <summary>
        /// オーバーライド
        /// </summary>
        protected override void OnLostFocus(EventArgs e)
        {
            StatueClear();
            if( this.Visible )
                this.Invalidate();
            base.OnLostFocus (e);
        }
        /// <summary>
        /// オーバーライド。
        /// </summary>
        protected override void OnKeyDown(KeyEventArgs kevent)
        {
            if( kevent.KeyCode == Keys.Space )
            {
                if(!_mouseDown)
                {
                    ButtonState	= ButtonStateType.Click;
                    _keyDown	= true;
                    this.Invalidate();
                    kevent.Handled = true;
                }
                else
                    kevent.Handled = false;
            }
            base.OnKeyDown (kevent);
        }
        /// <summary>
        /// オーバーライド。
        /// </summary>
        protected override void OnKeyUp(KeyEventArgs kevent)
        {
            if( ButtonState	== ButtonStateType.Click && _keyDown )
            {
                StatueClear();
                this.OnClick(EventArgs.Empty);
                kevent.Handled = true;
            }
            base.OnKeyUp (kevent);
        }
        /// <summary>
        /// オーバーライド。
        /// </summary>
        protected override void OnEnabledChanged(EventArgs e)
        {
            base.OnEnabledChanged(e);
            StatueClear();
            if( this.Enabled )
                this.Invalidate();
        }
        /// <summary>
        /// オーバーライド
        /// </summary>
        protected override void WndProc(ref Message m)
        {
            switch (m.Msg)
            {
                case Win32.WM.WM_SETFOCUS:
                case Win32.WM.WM_KILLFOCUS:
                    if( !IsFocus )
                        return;
                    break;
                case LECore.Win32.WM.WM_KEYDOWN:
                case LECore.Win32.WM.WM_KEYUP:
                    if( !this.Focused )
                        return;
                    Keys keyCode = (Keys)(int)m.WParam & Keys.KeyCode;
                    if( _mouseDown && keyCode == Keys.Space )
                    {
                        return;
                    }
                    break;
                //	無視するボタン
                case LECore.Win32.WM.WM_MBUTTONDBLCLK:
                case LECore.Win32.WM.WM_MBUTTONDOWN:
                case LECore.Win32.WM.WM_RBUTTONDBLCLK:
                case LECore.Win32.WM.WM_RBUTTONDOWN:
                    return;
                default:
                    break;
            }
            base.WndProc(ref m);
        }

        #endregion

        /// <summary>
        ///	各ステータスをクリアする
        /// </summary>
        private void StatueClear()
        {
            _keyDown	= false;
            _mouseDown	= false;
            if( !IsToggle )
                ButtonState	= ButtonStateType.Normal;
            else if( this.ClientRectangle.Contains(this.PointToClient(MousePosition)))
                ButtonState	= ButtonStateType.Focus;
            else if( Checked )
                ButtonState	= ButtonStateType.Down;
            else
                ButtonState	= ButtonStateType.Normal;
        }

        /// <summary>
        /// イメージの描画
        /// </summary>
        private void DrawImage( Graphics g, Rectangle rect )
        {
            if( Image == null )
                return;
            if( this.Enabled )
            {
                g.DrawImage(Image, rect);
            }
            else
            {
                //	グレースケール化
                ColorMatrix cm = new ColorMatrix();

                cm.Matrix00 = 0.3f;
                cm.Matrix10 = 0.3f;
                cm.Matrix20 = 0.3f;
                cm.Matrix30 = 0.0f;
                cm.Matrix40 = 0.0f;

                cm.Matrix01 = 0.3f;
                cm.Matrix11 = 0.3f;
                cm.Matrix21 = 0.3f;
                cm.Matrix31 = 0.0f;
                cm.Matrix41 = 0.0f;

                cm.Matrix02 = 0.3f;
                cm.Matrix12 = 0.3f;
                cm.Matrix22 = 0.3f;
                cm.Matrix32 = 0.0f;
                cm.Matrix33 = 0.5f;
                cm.Matrix42 = 0.0f;

                using (ImageAttributes ia = new ImageAttributes())
                {
                    ia.SetColorMatrix(cm);
                    g.DrawImage(Image, rect, 0, 0, Image.Width, Image.Height, GraphicsUnit.Pixel, ia);
                }
            }
        }
    }
}
